; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; $Id: caa_edicrf.pro,v 1.17 2007/05/11 16:04:23 awe Exp $
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; Cluster Active Archive : Program to produce EDI code repetition frequencies
; ****************************************************************************

@readlib.pro
@hkutil.pro
@myfwrite.pro

; ===========================================================================
  FUNCTION create_header_crf, sc, date, version, gendate, iso_time_range, $
                              program_version
; ===========================================================================
;
; ***************************************************************************

   if n_params() ne 6 then begin
      message, 'Expecting six parameters (sc,date,version,gendate,iso_time_range,program_version)', /cont
      return,-1
   endif

   name = 'C'+sc+'_CP_EDI_CRF__'+date+'_V'+version

   header_lines = $
      [ 'FILE_NAME="'+name+'.cef"', $
        'FILE_FORMAT_VERSION="CEF-2.0"', $
        '!', $
        'START_META = LOGICAL_FILE_ID', $
        ' ENTRY = "'+name+'"', $
        'END_META = LOGICAL_FILE_ID', $
        '!', $
        'START_META = VERSION_NUMBER', $
        ' ENTRY = "'+version+'"', $
        'END_META = VERSION_NUMBER', $
        '!', $
        'START_META = FILE_TIME_SPAN', $
        ' VALUE_TYPE = ISO_TIME_RANGE', $
        ' ENTRY = '+iso_time_range, $
        'END_META = FILE_TIME_SPAN', $
        '!', $
        'START_META = GENERATION_DATE', $
        ' VALUE_TYPE = ISO_TIME', $
        ' ENTRY = '+gendate, $
        'END_META = GENERATION_DATE', $
        '!', $
        'START_META = FILE_CAVEATS', $
        ' ENTRY = "'+program_version+'"', $
        'END_META = FILE_CAVEATS', $
        '!', $
        '! include EGD (DATASET) HEADER File for Cluster-'+sc, $
        '! with variable definitions, metadata_type AND _version', $
        'include="C'+sc+'_CH_EDI_CRF_DATASET.ceh"', $
        '!', $
        'data_until=EOF', $
        '!' ]

   return, header_lines

END


; ============================================================================
  PRO CAA_EDICRF, sc, year, month, day, version, debug=debug
; ============================================================================
; Parameters
;    sc     int   (1,2,3,4)
;    year   int   (2001,...)
;    month  int   (1,2,...,12)
;    day    int   (1,2,...,31)
;
; Keywords
;    debug  0/1     print debugging information
; ****************************************************************************

@paths.inc
@vector.inc

  program_version = 'CAA_EDICRF_V1_02  2007-04-02'
 readerr=0
  if n_params() ne 5 then begin
     message, 'Call syntax: caa_edicrf, sc, year, month, day, version, debug=debug', /cont
     readerr=1 & goto,endlbl
  endif

  ; Conversion factors for code dividers n and m, and code length in units of chips
  n_conv = [ 1, 2, 4, 8, 8, 16, 32, 64 ]
  m_conv = [ 16, 8, 4, 2 ]
  c_length = [ 15, 127 ]

  i_sc      = fix(sc)
  i_year    = fix(year)
  i_month   = fix(month)
  i_day     = fix(day)
  i_version = fix(version)

  if i_version lt 0 or i_version gt 99 then begin
     message, 'Error: version must be between 0 and 99', /cont
     readerr=1 & goto,endlbl
  endif

  s_sc      = strtrim(i_sc,2)
  s_year    = strtrim(i_year,2)
  s_month   = string(format='(I2.2)', i_month)
  s_day     = string(format='(I2.2)', i_day)
  s_version = string(format='(I2.2)', i_version)


  if not keyword_set(debug) then debug = 0 else debug = 1

  date     = s_year + s_month + s_day
  iso_date = s_year + '-' + s_month + '-' + s_day
  ctr = CluTimeStr2Val(iso_date) + [16, 24.*3600 - 16]

  ; Read EDI housekeeping data
  ; --------------------------
  edi_hk = GetHkdata(i_sc, ctr=ctr, debug=debug)
  if edi_hk.hdr.status eq 1 then begin
     message, edi_hk.hdr.msg, /cont
     readerr=1 & goto,endlbl
  endif

  ; Read EDI 2nd-header data (science telemetry)
  ; --------------------------------------------
  shdr = GetShdrData(i_sc, ctr=ctr, debug=debug)
  if shdr.hdr.status eq 1 then begin
     message, shdr.hdr.msg, /cont
     readerr=1 & goto,endlbl
  endif

  if n_elements(edi_hk.data) ne n_elements(shdr.data) then begin
     message, 'hk and shdr data have different number of elements!', /cont
     print, n_elements(edi_hk.data), n_elements(shdr.data)
     readerr=1 & goto,endlbl
  endif


  submo = GetHkParam(edi_hk.data, 'SUBMO')
  if submo[0].ct lt 0 then begin
     message, 'Telemetry is invalid!', /cont
     readerr=1 & goto,endlbl
  endif
  modid = GetHkParam(edi_hk.data, 'MODID', xsel=xsel)
  if modid[0].ct lt 0 then begin
     message, 'Telemetry is invalid!', /cont
     readerr=1 & goto,endlbl
  endif

  submo = fix(submo.val+0.5)
  modid = fix(modid.val+0.5)

  bcnx1 = ISHFT(shdr.data[xsel].data[2], -8)            ; word 2, upper byte
  bcnx2 = ISHFT(ISHFT(shdr.data[xsel].data[2], 8), -8)  ; word 2, lower byte

  if i_sc eq 1 then begin
     bcnx1 = intarr(n_elements(bcnx2)) ; SC1 gun is off --> set bcnx1 to zero
  endif

  ; find all Windshield Wiper data (science mode 5)
  ; and exclude possibly bad TM (via SUBMO)
  x = where(modid eq 5 and $                              ; Windshield Wiper mode
             (submo eq 1 or submo eq 2 or submo eq 3 or $ ; NM 1/2/3
              submo eq 5 or submo eq 6), $                ; BM 1/2
            cnt)


  if cnt eq 0 then $
     nodata = 1 $
  else begin
     bcnx1 = bcnx1[x]
     bcnx2 = bcnx2[x]
     hk = edi_hk.data[x]

     nodata = 0
     n_raw = ishft(hk.data[10],-9) mod 8
     m_raw = ishft(hk.data[10],-2) mod 4
     ctype = ishft(hk.data[10],-15)

     ; code period in seconds
     code_period = n_conv[n_raw] * m_conv[m_raw] * 2.d^(-23) * c_length[ctype]

     ; code repetition frequency in kHz
     f_code = 1 / code_period  /  1.e3

     N_LINES = n_elements(hk)
     out_lines = strarr(N_LINES)
     fmt = '(A19,"Z,   ",F10.3,",   ",I1,",   ",I3,",   ",I3)'

     ; each line consists of: ut, frequency (kHz), code type
     for i=0L,N_LINES-1 do begin
        out_lines[i] = string(format=fmt, $
                              strmid(ct2str(hk[i].ct),0,19), $
                              f_code[i], $
                              ctype[i], $
                              bcnx1[i], bcnx2[i])
     endfor

  endelse

  ; Construct file name for output
  ; ------------------------------
  date = s_year + s_month + s_day
  out_filename = 'C'+s_sc+'_CP_EDI_CRF__'+date+'_V'+s_version+'.cef'

  iso_generation_date = strmid(CluTimeVal2Str(systime(1)),0,19) + 'Z'

  iso_this_day = s_year+'-'+s_month+'-'+s_day
  iso_next_day = strmid(CluTimeVal2Str(CluTimeStr2Val(iso_this_day+'t12')+86400.),0,10)
  iso_time_range = iso_this_day+'T00:00:00Z/'+iso_next_day+'T00:00:00Z'
  header_lines = create_header_crf(s_sc,date,s_version,iso_generation_date,$
                                   iso_time_range,program_version)

  ; Write data to output file
  ; -------------------------
  all_lines = header_lines
  if nodata eq 0 then all_lines = [ all_lines, out_lines ]
  MyFwrite, out_filename, all_lines

  if nodata eq 1 then print, 'No data! CEF header data written to ' + out_filename $
  else print, 'Data written to ' + out_filename
  
endlbl:
if readerr then print,'NO telemetry data'
END
