; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; $Id: boffs2.pro,v 1.7 2004/03/15 16:07:45 hav Exp $
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; Determination of magnetic field data offsets
; ****************************************************************************

@readlib.pro
@xout.pro

; ============================================================================
  FUNCTION SpinPlaneOffsetFit, X
; ============================================================================
; Model function for determination of spin plane offsets by
; minimization of variations at the spin frequency and the double spin
; frequency
; ****************************************************************************

COMMON boffs_data, dbx_sel, dby_sel, dbz_sel, wt, wt2
COMMON boffs_data2, wt_xy, wt_z, wt2_xy, wt2_z

amp = X[0] ; amplitude
eps = X[1] ; phase

amp2 = X[2]
eps2 = X[3]

func = total( ( dbx_sel - amp * sin(wt_xy + eps) - amp2 * sin(wt2_xy+eps2) )^2 ) + $
       total( ( dby_sel + amp * cos(wt_xy + eps) + amp2 * cos(wt2_xy+eps2) )^2 )

return, func

END

; ============================================================================
  FUNCTION SpinAxisOffsetFit, X
; ============================================================================
; Model function for determination of the spin axis offset. The spin
; axis component often shows slightly enhanced power at the spin
; frequency and the double spin frequency (Due to a slight
; misalignment of the axes ??). These variations will be
; removed by subtracting sine waves at fspin and 2*fspin with proper
; amplitude and phase, which are determined by minimizing this model function.
; ****************************************************************************

COMMON boffs_data, dbx_sel, dby_sel, dbz_sel, wt, wt2
COMMON boffs_data2, wt_xy, wt_z, wt2_xy, wt2_z

amp = X[0] ; amplitude
eps = X[1] ; phase

amp2 = X[2]
eps2 = X[3]

func = total( ( dbz_sel - amp * sin(wt_z + eps) - amp2 * sin(wt2_z+eps2) )^2 )

return, func

END


; ============================================================================
  FUNCTION GetFittedB, bs, sd, $
                       plotset=plotset, debug=debug, $
                       nofit=nofit, nofft=nofft, $
                       fitdata=fitdata
; ============================================================================
;
; ****************************************************************************

@vector.inc

COMMON boffs_data, dbx_sel, dby_sel, dbz_sel, wt, wt2
COMMON boffs_data2, wt_xy, wt_z, wt2_xy, wt2_z

  ; Check for parameters
  ; --------------------
  if n_params() eq 0 then begin
     message, 'Missing parameter. Call: b_out = GetFittedB(bs [, sd] )', /cont
     retall
  endif

  ax=axlabel(/reset)
  ct0 = min( bs.data.ct, max=ct1 )

  if n_params() lt 2 then $
     sd = GetSrpData(bs.hdr.sc_id, ctr=[ct0,ct1] )


  ; check for keywords
  ; ------------------
  if keyword_set(nofit) then goto, return_
  if not keyword_set(nofft) then nofft=0 else nofft=1
  if not keyword_set(plotset) then begin
     plotset = 0
     if nofft ne 0 then print, 'keyword nofft ignored!'
  endif


  bd = bs.data
  tspin_avg = total(sd.data.tspin) / n_elements(sd.data)
  x = where(sd.data.ct lt ct0)
  if x[0] eq -1 then srp0 = sd.data[0].ct $
  else srp0 = sd.data[x[n_elements(x)-1]].ct
  NN=n_elements(bd)


  bx = bd.val[0]
  by = bd.val[1]
  bz = bd.val[2]
  ct = bd.ct

  N_SPINS   = (ct1 - ct0)  /  tspin_avg
  DPS       = fix( 1.0 * NN / N_SPINS ) ; average number of data points per spin
                                        ; this assumes that there are no (major)
                                        ; data gaps !

  ; ----------------------------------------------------------------------
  ; Calculate deviation of high rate data from boxcar average
  ; The width used for the boxcar is the approximate number of points per spin
  ; ----------------------------------------------------------------------
  dbx = bx - smooth(bx, DPS, /EDGE_TRUNCATE)
  dby = by - smooth(by, DPS, /EDGE_TRUNCATE)
  dbz = bz - smooth(bz, DPS, /EDGE_TRUNCATE)

  ; calculate time tags relative to sun reference pulse,
  ; then modulo spin period
  ; ----------------------------------------------------
  tarr = ct - srp0
  tmod = ((tarr mod tspin_avg) + tspin_avg) mod tspin_avg
                        ; mod-add-mod takes care of negative values

  ; Sort data according to increasing spin phase
  ; --------------------------------------------
  sortindex = sort(tmod)
  tmod = tmod[sortindex]
  dbx  = dbx[sortindex]
  dby  = dby[sortindex]
  dbz  = dbz[sortindex]

  dbx_sel = dbx   ; these three, as well as wt and wt2, are global
  dby_sel = dby   ; and are used by the fitting functions
  dbz_sel = dbz


  ; determine outliers
  ; ------------------

  ; first get a standard deviation (use medians so that
  N_BINS = 20 & bin_width = tspin_avg / N_BINS
  for i=0,N_BINS-1 do begin
     x=where(tmod ge i*bin_width and tmod lt (i+1)*bin_width,N_INSIDE)
     if N_INSIDE lt 3 then begin
        message, 'Less than 3 data points within current bin!', /cont
        retall
     endif
     medx = median(dbx_sel[x])
     medy = median(dby_sel[x])
     medz = median(dbz_sel[x])

     sdvx = stddev(dbx_sel[x])
     sdvy = stddev(dby_sel[x])
     sdvz = stddev(dbz_sel[x])

     ; first pass at inlier calculation
     ; --------------------------------
     f = 1.7
     xin_x = where(dbx_sel[x] gt medx-f*sdvx and dbx_sel[x] lt medx+f*sdvx)
     xin_y = where(dby_sel[x] gt medy-f*sdvy and dby_sel[x] lt medy+f*sdvy)
     xin_z = where(dbz_sel[x] gt medz-f*sdvz and dbz_sel[x] lt medz+f*sdvz)


     ; recalculate median and standard deviation using only first pass inliers
     ; -----------------------------------------------------------------------
     medx = median(dbx_sel[x[xin_x]])  &  sdvx = stddev(dbx_sel[x[xin_x]])
     medy = median(dby_sel[x[xin_y]])  &  sdvy = stddev(dby_sel[x[xin_y]])
     medz = median(dbz_sel[x[xin_z]])  &  sdvz = stddev(dbz_sel[x[xin_z]])

     ; recalculate inliers (combine spin plane components because these are
     ; combined in minimization, too, and thus need common selection)
     ; ---------------------------------------------------------------------
     f = 2.0
     xin_xy = where(dbx_sel[x] gt medx-f*sdvx and dbx_sel[x] lt medx+f*sdvx  and  $
                    dby_sel[x] gt medy-f*sdvy and dby_sel[x] lt medy+f*sdvy)
     xin_z = where(dbz_sel[x] gt medz-f*sdvz and dbz_sel[x] lt medz+f*sdvz)


     if n_elements(all_xin_xy) eq 0 then begin
        all_xin_xy = x[xin_xy]
     endif else begin
        all_xin_xy = [ all_xin_xy, x[xin_xy] ]
     endelse

     if n_elements(all_xin_z) eq 0 then all_xin_z = x[xin_z] $
     else                               all_xin_z = [ all_xin_z, x[xin_z] ]

  endfor

;  help, all_xin_xy, all_xin_z

  xin_xy = all_xin_xy
  xin_z  = all_xin_z

  xout_xy = xout(NN, xin_xy)
  xout_z = xout(NN, xin_z)

;  help, xout_xy, xout_z
;retall

   ; calculate spin phase array
   ; --------------------------
   omega   = 2*!PI / tspin_avg

   wt      =     omega * tmod
   wt_ori  =     omega * tarr

   wt2     = 2 * wt
   wt2_ori = 2 * wt_ori

   wt_xy = wt    &  wt_z = wt
   wt2_xy = wt2  &  wt2_z = wt2


  dbx_sel = dbx_sel[xin_xy]
  dby_sel = dby_sel[xin_xy]
  dbz_sel = dbz_sel[xin_z]

  wt_xy = wt[xin_xy]
  wt_z = wt[xin_z]

  wt2_xy = wt2[xin_xy]
  wt2_z = wt2[xin_z]

  ; ---------------------------------------------------------------------
  ; Determination of offsets is done separately for spin axis offset and
  ; spin plane offsets. The offsets are determined by fitting the
  ; deviations dbx, dby, dbz to a model. The model is based on FFT
  ; results of the real data, showing that there is often enhanced power
  ; at the spin frequency and at the double spin frequency. See the
  ; functions 'SpinPlaneOffsetFit' and 'SpinAxisOffsetFit' for the
  ; actual implementation of the model functions.
  ; ---------------------------------------------------------------------

  ; Spin Plane Offset Fit
  ; ---------------------
  P  = [ 0.1, 0.0, 0.1, 0.0 ] ; starting point
  Xi = fltarr(4,4)
  for i=0,3 do Xi[i,i] = 1.0
  Ftol = 1.0e-8
  Powell, P, Xi, Ftol, Fmin, 'SpinPlaneOffsetFit', iter=N_ITER, /double
  amp  = P[0]
  eps  = P[1]
  amp2 = P[2]
  eps2 = P[3]

  ; Spin Axis Offset Fit
  ; --------------------
  P  = [ 0.1, 0.0, 0.1, 0.0 ] ; starting point
  Xi = fltarr(4,4)
  for i=0,3 do Xi[i,i] = 1.0
  Ftol = 1.0e-8
  Powell, P, Xi, Ftol, Fmin, 'SpinAxisOffsetFit', iter=N_ITER, /double
  amp_z  = P[0]
  eps_z  = P[1]
  amp2_z = P[2]
  eps2_z = P[3]

  if keyword_set(debug) then begin
      print, '               amp          phase   amp        phase'
      print, format='("Spin plane: ",E12.3,F10.2,E12.3,F10.2)', $
             amp, eps*180/!PI, amp2, eps2 * 180/!PI
      print, format='("Spin axis : ",E12.3,F10.2,E12.3,F10.2)', $
             amp_z, eps_z*180/!PI, amp2_z, eps2_z*180/!PI

  endif

  fitdata = { BFIT_DATA_T, $
              pl_a1:amp, pl_a2:amp2, ax_a1:amp_z, ax_a2:amp2_z, $
              pl_ph1:eps*180/!PI, pl_ph2:eps2*180/!pi, $
              ax_ph1:eps_z*180/!pi, ax_ph2:eps2_z*180/!pi }


  ; calculate corrected B
  ; ---------------------
  Bx_cor = bx - amp * sin(wt_ori + eps) - amp2 * sin(wt2_ori + eps2)
  By_cor = by + amp * cos(wt_ori + eps) + amp2 * cos(wt2_ori + eps2)

  Bz_cor = bz - amp_z * sin(wt_ori + eps_z) - amp2_z * sin(wt2_ori + eps2_z)


  width = fix(NN/100.)
  if width ge 3 then begin
  fit_stdev = 1./NN * sqrt ( $
     total( (smooth(dbx,width,/edge_truncate) - amp*sin(wt+eps) $
                                       - amp2*sin(wt2+eps2)  )^2 ) + $
     total( (smooth(dby,width,/edge_truncate) + amp*cos(wt+eps) $
                                       + amp2*cos(wt2+eps2)  )^2 ) + $
     total( (smooth(dbz,width,/edge_truncate) - amp_z*sin(wt+eps_z) $
                                       - amp2_z*sin(wt2+eps2_z) )^2 ) )
  endif else $
     fit_stdev = -1.0



  if keyword_set(debug) then begin
     all_amp = sqrt( amp^2 + amp2^2 + amp_z^2 + amp2_z^2 )
     print, format='("fit quality: stdev = ",E10.2,"   normalized : ",' + $
                   'E10.2)', fit_stdev, fit_stdev / all_amp
  endif


; ------------------ Plotting ----------------------------------------------

if keyword_set(plotset) then begin

; Plot the data
; -------------
    !p.multi=[0,3,3,0,1]
    !x.style=1
    !p.charsize=2.0
    loadct, 39
@ct39.inc

    tmod_r = [ tmod[0], tmod[n_elements(tmod)-1] ]
    bmax = 10 * max(abs([amp,amp2,amp_z,amp2_z]))

bmax = 1.0

;    dbx_avg = total(abs(dbx)) / n_elements(dbx)
;    dby_avg = total(abs(dby)) / n_elements(dby)
;    dbz_avg = total(abs(dbz)) / n_elements(dbz)

;    sdev = sqrt( total((dbx-dbx_avg)^2) / n_elements(dbx) ) + $
;           sqrt( total((dby-dby_avg)^2) / n_elements(dby) ) + $
;           sqrt( total((dbz-dbz_avg)^2) / n_elements(dbz) )

;    bmax = total(dbx_avg+dby_avg+dbz_avg) / 1.5

    plot, tmod, dbx, yrange=[-bmax,bmax], psym=4
    oplot, tmod_r, [0,0], linestyle=2, color=red
    oplot, tmod, smooth(dbx, fix(NN/100.), /edge_truncate), color=green
    oplot, tmod, amp * sin(wt+eps) + amp2*sin(wt2+eps2), color=darkblue
    if xout_xy[0] ne -1 then oplot, tmod[xout_xy], dbx[xout_xy], psym=4, color=yellow


    plot, tmod, dby, yrange=[-bmax,bmax], psym=4
    oplot, tmod_r, [0,0], linestyle=2, color=red
    oplot, tmod, smooth(dby, fix(NN/100.), /edge_truncate), color=green
    oplot, tmod, -amp * cos(wt+eps) - amp2 * cos(wt2+eps2), color=darkblue
    if xout_xy[0] ne -1 then oplot, tmod[xout_xy], dby[xout_xy], psym=4, color=yellow

    plot, tmod, dbz, yrange=[-bmax,bmax], psym=4
    oplot, tmod_r, [0,0], linestyle=2, color=red
    oplot, tmod, smooth(dbz, fix(NN/100.), /edge_truncate), color=green
    oplot, tmod, amp_z * sin(wt+eps_z) + amp2_z * sin(wt2+eps2_z), $
           color=darkblue
    if xout_z[0] ne -1 then oplot, tmod[xout_z], dbz[xout_z], psym=4, color=yellow


if nofft eq 0 then begin

; -----------------------------------------------
; calculate and plot the FFT before and after fit
; -----------------------------------------------

    cnt = 0L
    while 2.d^cnt lt NN do cnt=cnt+1
    NR = 2L^(cnt-1)
    print, 'total number of data points : ', NN
    print, 'using for FFT : ', NR

    T_iv = ct[NR-1] - ct[0]
    t_old = ct - ct[0]
    t_int = findgen(NR) * T_iv/(NR-1)

    bx_int     = interpol(bx, t_old, t_int)
    by_int     = interpol(by, t_old, t_int)
    bz_int     = interpol(bz, t_old, t_int)

    bx_cor_int = interpol(bx_cor, t_old, t_int)
    by_cor_int = interpol(by_cor, t_old, t_int)
    bz_cor_int = interpol(bz_cor, t_old, t_int)



; Calculate Fast Fourier Transforms of old and new vector components
; ------------------------------------------------------------------
    if keyword_set(debug) then print, 'calculating FFT(bx)...'
    fft_x_0 = fft(bx_int[0:NR-1])
    if keyword_set(debug) then print, 'calculating FFT(by)...'
    fft_y_0 = fft(by_int[0:NR-1])
    if keyword_set(debug) then print, 'calculating FFT(bz)...'
    fft_z_0 = fft(bz_int[0:NR-1])

    if keyword_set(debug) then print, 'calculating FFT(bx_cor)...'
    fft_x_1 = fft(Bx_cor_int[0:NR-1])
    if keyword_set(debug) then print, 'calculating FFT(by_cor)...'
    fft_y_1 = fft(By_cor_int[0:NR-1])
    if keyword_set(debug) then print, 'calculating FFT(bz_cor)...'
    fft_z_1 = fft(Bz_cor_int[0:NR-1])

    fft_x_0_pwr = sqrt( fft_x_0 * conj(fft_x_0) )
    fft_y_0_pwr = sqrt( fft_y_0 * conj(fft_y_0) )
    fft_z_0_pwr = sqrt( fft_z_0 * conj(fft_z_0) )

    fft_x_1_pwr = sqrt( fft_x_1 * conj(fft_x_1) )
    fft_y_1_pwr = sqrt( fft_y_1 * conj(fft_y_1) )
    fft_z_1_pwr = sqrt( fft_z_1 * conj(fft_z_1) )


    x_spin = T_iv / tspin_avg
    mult = 5

    x0 = long(x_spin)
    ymax = 2.5 * max( [ fft_x_0_pwr[x0:x0+1], $
                        fft_y_0_pwr[x0:x0+1], $
                        fft_z_0_pwr[x0:x0+1] ] )
    plot, /nodata, fft_x_0_pwr, $
          yrange=[0,ymax], $
          xrange=[0, mult*x_spin]
    oplot, fft_x_0_pwr, color=red, psym=-4, symsize=1.2 ; before
    oplot, x_spin*[1,1], !y.crange, linestyle=2    ; vertical line at fSpin
    oplot, 2*x_spin*[1,1], !y.crange, linestyle=2  ; vertical line at 2*fSpin
    oplot, fft_x_1_pwr, color=green, psym=-4       ; after


    plot, /nodata, fft_y_0_pwr, $
          yrange=[0,ymax], $
          xrange=[0, mult*x_spin]
    oplot, fft_y_0_pwr, color=red, psym=-4
    oplot, x_spin*[1,1], !y.crange, linestyle=2
    oplot, 2*x_spin*[1,1], !y.crange, linestyle=2
    oplot, fft_y_1_pwr, color=green, psym=-4



    plot, /nodata, fft_z_0_pwr, $
          yrange=[0,ymax], $
          xrange=[0, mult*x_spin]
    oplot, fft_z_0_pwr, color=red, psym=-4
    oplot, x_spin*[1,1], !y.crange, linestyle=2
    oplot, 2*x_spin*[1,1], !y.crange, linestyle=2
    oplot, fft_z_1_pwr, color=green, psym=-4

endif ; nofft eq 0


    dbx = bx_cor - smooth(bx_cor, DPS, /EDGE_TRUNCATE)
    dby = by_cor - smooth(by_cor, DPS, /EDGE_TRUNCATE)
    dbz = bz_cor - smooth(bz_cor, DPS, /EDGE_TRUNCATE)

    dbx_avg = total(dbx) / NN
    dby_avg = total(dby) / NN
    dbz_avg = total(dbz) / NN


; Sort data according to increasing spin phase
; --------------------------------------------
    dbx  = dbx(sortindex)
    dby  = dby(sortindex)
    dbz  = dbz(sortindex)

    tmod_r = [ tmod[0], tmod[n_elements(tmod)-1] ]
    plot, tmod, dbx, yrange=[-bmax,bmax], psym=4
    oplot, tmod_r, [0,0], linestyle=2, color=red
    oplot, tmod, smooth(dbx, fix(NN/100.), /edge_truncate), color=green
    if xout_xy[0] ne -1 then oplot, tmod[xout_xy], dbx[xout_xy], psym=4, color=yellow

    plot, tmod, dby, yrange=[-bmax,bmax], psym=4
    oplot, tmod_r, [0,0], linestyle=2, color=red
    oplot, tmod, smooth(dby, fix(NN/100.), /edge_truncate), color=green
    if xout_xy[0] ne -1 then oplot, tmod[xout_xy], dby[xout_xy], psym=4, color=yellow

    plot, tmod, dbz, yrange=[-bmax,bmax], psym=4
    oplot, tmod_r, [0,0], linestyle=2, color=red
    oplot, tmod, smooth(dbz, fix(NN/100.), /edge_truncate), color=green
    if xout_z[0] ne -1 then oplot, tmod[xout_z], dbz[xout_z], psym=4, color=yellow


    if nofft ne 0 then begin
       dbx_abs = abs(dbx)
       dby_abs = abs(dby)
       dbz_abs = abs(dbz)

       plot, tmod, dbx_abs, yrange=[0,bmax], psym=4
       oplot, tmod, smooth(dbx_abs, fix(NN/100.), /edge_truncate), color=green

       plot, tmod, dby_abs, yrange=[0,bmax], psym=4
       oplot, tmod, smooth(dby_abs, fix(NN/100.), /edge_truncate), color=green

       plot, tmod, dbz_abs, yrange=[0,bmax], psym=4
       oplot, tmod, smooth(dbz_abs, fix(NN/100.), /edge_truncate), color=green
    endif


endif ; keyword_set(plotset)

; ----------------- end of plotting ------------------------------

return_:

if keyword_set(nofit) then begin
    ret    = bd
endif else begin
    ret = replicate( {VTS_T}, n_elements(bx_cor) )
    ret.val[0]  = bx_cor
    ret.val[1]  = by_cor
    ret.val[2]  = bz_cor
    ret.ct = bd.ct
endelse

return, ret

END









