/*
 * $Id: main.c,v 1.1 2001/09/26 14:07:17 simond Exp $
 *
 * Project:         CLUSTER WEC TED Package
 *
 * Item Name:       src/tmc/main.c
 *
 * Item Version:    (see RCS header)
 *
 * Item Type:       c-source
 *
 * Author:          Simon Davis
 *
 */

#define TEST_C
#define TMC_PROGRAM_VERSION "2.1"

#include "debug.h"
#include "main.h"
#include "convert.h"
#include "version.h"

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>


#define CCS_OPT	   't'		/* TMA from pre-launch system testing */
#define FLIGHT_OPT 'f'		/* FLIGHT data through ESOC */


/* {{{ debugging */

enum DEBUG_ENUM {
  DEBUG_PROC_NONE			= 0x00000,

  DEBUG_PROC_DAT_DIRECT			= 0x00001,
  DEBUG_PROC_DAT_SCID			= 0x00002,
  DEBUG_PROC_DAT_STREAM			= 0x02004,
  DEBUG_PROC_DAT_TCAL			= 0x00008,
  DEBUG_PROC_DAT_GSID			= 0x00010,
  DEBUG_PROC_DAT_SEQ			= 0x00020,

  DEBUG_PROC_DAT_TMP            	= 0x00100,
  DEBUG_PROC_DAT_DDSP           	= 0x00200,
  DEBUG_PROC_DAT_TRANS_DDSP		= 0x00400,
  DEBUG_PROC_DAT_TRANS_TMP		= 0x00800,

  DEBUG_PROC_DAT_TRANS_DDSP_ERROR	= 0x01000,
  DEBUG_PROC_DAT_WRTIE_DDSP_ERROR	= 0x02000,
  DEBUG_PROC_DAT_READ_DDSP_ERROR	= 0x04000,

  DEBUG_PROC_DAT_TRANS_TMP_ERROR	= 0x10000,
  DEBUG_PROC_DAT_WRITE_TMP_ERROR	= 0x20000,
  DEBUG_PROC_DAT_READ_TMP_ERROR		= 0x40000
};
typedef enum DEBUG_ENUM DEBUG_ENUM;

static long level = 0;


/* {{{ debug ! */

int debug(char value[])
{
  int index;
  int val;

  if(value == NULL) {
    level = 0L;
    return FALSE;
  }

  if(sscanf(value,"%i",&val) != 1) {
    level = 0L;
    return FALSE;
  }

  level =(long) val;

  return TRUE;
}

/* }}} */
/* {{{ diag ? */

int diag(DEBUG_ENUM dbg)
{
  return(dbg & level);
}

/* }}} */
/* {{{ report ! */

void
report(
DEBUG_ENUM rep,
...
)
{
  va_list list;

  unsigned char * packet = NULL;
  DEBUG_ENUM item;
  char *name;

  TED_SPACECRAFT scid;
  TED_TRANSMISSION stream;
  TED_QUALITY tcal;
  TED_GROUND gsid;
  char type;
  int seq;

  va_start(list,rep);
  
  switch(rep) {
  case DEBUG_PROC_NONE:
    break;
  case DEBUG_PROC_DAT_DIRECT:
    type = va_arg(list,char);
    fprintf(stderr,"(tmc-filter-direction %c)\n",type);
    break;
  case DEBUG_PROC_DAT_SCID:
    scid = va_arg(list,TED_SPACECRAFT);
    fprintf(stderr,"(tmc-filter-direction %s)\n",ted_scid_symbol(scid));
    break;
  case DEBUG_PROC_DAT_STREAM:
    stream = va_arg(list,TED_TRANSMISSION);
    fprintf(stderr,"(tmc-filter-direction %s)\n",ted_transmission_symbol(stream));
    break;
  case DEBUG_PROC_DAT_TCAL:
    tcal = va_arg(list,TED_QUALITY);
    fprintf(stderr,"(dat-tcal %s)\n", ted_quality_symbol(tcal));
    break;
  case DEBUG_PROC_DAT_GSID:
    gsid = va_arg(list,TED_GROUND);
    fprintf(stderr,"(dat-gsid %s)\n", ted_ground_symbol(gsid));
    break;
  case DEBUG_PROC_DAT_SEQ:
    seq = va_arg(list,int);
    fprintf(stderr,"(dat-sequence-counter %d)\n", seq);
    break;
  case DEBUG_PROC_DAT_TRANS_TMP:
  case DEBUG_PROC_DAT_TMP:
    packet = va_arg(list,unsigned char *);
    fprintf(stderr, "(tmc-debug \n");
    convert_display_tm_packet(packet);
    fprintf(stderr,")\n");
    break;
  case DEBUG_PROC_DAT_TRANS_DDSP:
  case DEBUG_PROC_DAT_DDSP:
    packet = va_arg(list,unsigned char *);
    fprintf(stderr, "(tmc-debug \n");
    convert_display_dds_packet(packet);
    fprintf(stderr,")\n");
    break;
  case DEBUG_PROC_DAT_TRANS_DDSP_ERROR:
    fprintf(stderr, "(tmc-debug tmp-two-ddsp-translation-error)\n");
    break;
  case DEBUG_PROC_DAT_WRTIE_DDSP_ERROR:
    packet = va_arg(list,unsigned char *);
    fprintf(stderr, "(tmc-debug ddsp-write-error\n");
    convert_display_dds_packet(packet);
    fprintf(stderr,")\n");
    break;
  case DEBUG_PROC_DAT_READ_DDSP_ERROR:
    fprintf(stderr, "(tmc-debug ddsp-read-error)\n");
    break;
  case DEBUG_PROC_DAT_TRANS_TMP_ERROR:
    fprintf(stderr, "(tmc-debug ddsp-two-tmp-translation-error)\n");
    break;
  case DEBUG_PROC_DAT_WRITE_TMP_ERROR:
    packet = va_arg(list,unsigned char *);
    fprintf(stderr, "(tmc-debug tmp-write-error\n");
    convert_display_tm_packet(packet);
    fprintf(stderr,")\n");
    break;
  case DEBUG_PROC_DAT_READ_TMP_ERROR:
    fprintf(stderr, "(tmc-debug tmp-read-error)\n");
    break;
  default:
    fprintf(stderr, "(tmc-debug unifentified-report-request)\n");
    break;
  }
  va_end(list);
  return;
}

/* }}} */

/* }}} */
/* {{{ usage */

void usage( int argc, char *argv [] )
{
#define USAGE "\
	-u \"print this message\"\n\
	-v \"report program version\"\n\
	-f \"translate ESA TM  -> CLUSTER DDS Format\"\n\
	-t \"translate CLUSTER DDS Format -> ESA TM\"\n\
	-a <opts> \"alter program default parameters\"\n\
	<opts> ::=\n\
	       scid     = 1|2|3|4\n\
	       channel  = RT|PB|RE|RP\n\
               tcal     = act|con|ext\n\
               tam|asid = 1 .. 15\n\
               gsid     = mal|red|kou|per|ode|gol\n\
               seq      = <seq no.>\n"

  fprintf( stderr, "USAGE: %s [-u | -v] -{f|t} [ -a <opts> ] \n", argv [ 0 ] );
  fprintf( stderr, USAGE );
  exit(-1);
}

/* }}} */
/* {{{ opt ground */

TED_GROUND OptGROUND( char value [] )
{
  int index;

  if(value == NULL)
    return TED_GROUND_ODENWALD;


  for(index = 1;index <(int) strlen(value);index++) {
    if(isalpha(value [index]))
      value [index] = tolower(value [index]);
  }

       if(! strncmp(value,"ode",3)) return TED_GROUND_ODENWALD;
  else if(! strncmp(value,"red",3)) return TED_GROUND_REDU;
  else if(! strncmp(value,"kou",3)) return TED_GROUND_KOUROU;
  else if(! strncmp(value,"per",3)) return TED_GROUND_PERTH;
  else if(! strncmp(value,"mal",3)) return TED_GROUND_MALINDI;
  else if(! strncmp(value,"can",3)) return TED_GROUND_CANBERRA;
  else if(! strncmp(value,"gol",3)) return TED_GROUND_GOLDSTONE;

  fprintf(stderr,"GSID: does not identify a known ground segment.\n");
  fprintf(stderr,"using default value.\n");
  return TED_GROUND_ODENWALD;
}

/* }}} */
/* {{{ opt tcal */

TED_QUALITY OptTCAL(char value [])
{
  int index;
 
  if(value == NULL)
    return TED_QUALITY_EXTRAPOLATED;

  for(index = 1;index <(int) strlen(value); index++) {
    if(isalpha(value [index]))
      value [index] = tolower(value [index]);
  }

       if(! strncmp("act",value,3)) return TED_QUALITY_ACTUAL;
  else if(! strncmp("ext",value,3)) return TED_QUALITY_EXTRAPOLATED;
  else if(! strncmp("con",value,3)) return TED_QUALITY_CONTINGENCY;

  fprintf(stderr,"TCAL: does not identify a known ground segment.\n");
  fprintf(stderr,"using default value.\n");
  return TED_QUALITY_EXTRAPOLATED;
}

/* }}} */
/* {{{ opt stream */

TED_TRANSMISSION OptSTREAM(char value [])
{
  int index;

  if(value == NULL)
    return TED_TRANSMISSION_RECALLPLAYBACK;
    
  for(index = 0;index <(int) strlen(value); index++) {
    if(isalpha(value [index]))
      value [index] = tolower(value [index]);
  }
  
       if(! strncmp("rt",value,2)) return TED_TRANSMISSION_REALTIME;
  else if(! strncmp("pb",value,2)) return TED_TRANSMISSION_PLAYBACK;
  else if(! strncmp("re",value,2)) return TED_TRANSMISSION_RECALL;
  else if(! strncmp("rp",value,2)) return TED_TRANSMISSION_RECALLPLAYBACK;

  fprintf(stderr,"CHANNEL CLASS: is not a defined name. \n");
  fprintf(stderr,"using default value. \n");
  return TED_TRANSMISSION_RECALLPLAYBACK;
}

/* }}} */
/* {{{ opt scid */

TED_SPACECRAFT OptSCID(char value [])
{
  int index;

  if(value == NULL)
    return CLUSTER_1;

       if(! strcmp(value,"1")) return CLUSTER_1;
  else if(! strcmp(value,"2")) return CLUSTER_2;
  else if(! strcmp(value,"3")) return CLUSTER_3;
  else if(! strcmp(value,"4")) return CLUSTER_4;

  fprintf(stderr,"SCID: in not in the required range.\n");
  fprintf(stderr,"using default value.\n");
  return CLUSTER_1;

}

/* }}} */
/* {{{ opt seq */

int OptSEQ( char value [] )
{
  int index;

  if(value == NULL)
    return TED_SPSC_INTERNAL;

  for(index = 0;index <(int) strlen(value); index++) {
    if(! isdigit(value [index])) {
      fprintf( stderr, "SEQ: has an error in the option syntax.\n" );
      fprintf( stderr, "using default value.\n" );
      return TED_SPSC_INTERNAL;
    }
  }
 
  return atoi(value);
}

/* }}} */
/* {{{ aux opt */

enum SUBOPT {
  AUX_SUBOPT_SCID,
  AUX_SUBOPT_GSID,
  AUX_SUBOPT_TCAL,
  AUX_SUBOPT_TAM,
  AUX_SUBOPT_STRE,
  AUX_SUBOPT_ASID,
  AUX_SUBOPT_SEQ
};


int aux_opt(
char arg[],
TED_SPACECRAFT *scid,
TED_QUALITY *tcal,
TED_TRANSMISSION *stream,
TED_GROUND *gsid,
int *seq
)
{
  char * options = arg;
  char * value;

  char *opts[] = { "scid", "gsid", "tcal", "tam", "stream", "asid", "seq", NULL };

  while(*options != '\0') {
    switch( getsubopt(&options,opts,&value)) {
    case AUX_SUBOPT_SCID: *scid   = OptSCID  (value); break;
    case AUX_SUBOPT_TCAL: *tcal   = OptTCAL  (value); break;
    case AUX_SUBOPT_GSID: *gsid   = OptGROUND(value); break;
    case AUX_SUBOPT_STRE: *stream = OptSTREAM(value); break;
    case AUX_SUBOPT_SEQ:  *seq    = OptSEQ   (value); break;
    default: return FALSE;
    }
  }	  
  return TRUE;
}

/* }}} */
/* {{{ process data */

int ProcessData(
FILE * ip,
FILE * op,
char type,
TED_SPACECRAFT scid,
TED_TRANSMISSION stream,
TED_QUALITY tcal,
TED_GROUND ground,
int seq)
{
  unsigned char *tmp;
  unsigned char *ddsp;
  TED_STATUS status;

  if(diag(DEBUG_PROC_DAT_DIRECT))
    report(DEBUG_PROC_DAT_DIRECT,type,NULL);

  if(diag(DEBUG_PROC_DAT_SCID))
    report(DEBUG_PROC_DAT_SCID,NULL);

  if(diag(DEBUG_PROC_DAT_STREAM))
    report(DEBUG_PROC_DAT_STREAM,stream,NULL);

  if(diag(DEBUG_PROC_DAT_TCAL))
    report(DEBUG_PROC_DAT_TCAL,tcal,NULL);

  if(diag(DEBUG_PROC_DAT_GSID))
    report(DEBUG_PROC_DAT_GSID,ground,NULL);

  if(diag(DEBUG_PROC_DAT_SEQ))
    report(DEBUG_PROC_DAT_SEQ,seq,NULL);

  if(type == 'f') {
    while(ted_convert_stream_read_tmp(ip,&tmp) == TED_OK) {

      if(diag(DEBUG_PROC_DAT_TMP))
	report(DEBUG_PROC_DAT_TMP,tmp,NULL);

      if((status = ted_convert_esatm2dds(tmp, &ddsp,
					   scid, stream,
					   tcal, ground)) != TED_OK) {

	report(DEBUG_PROC_DAT_TRANS_DDSP_ERROR,NULL);
	exit(status);
      }

      if(diag(DEBUG_PROC_DAT_TRANS_DDSP))
	report(DEBUG_PROC_DAT_TRANS_DDSP,ddsp,NULL);

      if((status = ted_convert_stream_write_ddsp(op, ddsp)) != TED_OK) {
	report(DEBUG_PROC_DAT_WRTIE_DDSP_ERROR,ddsp,NULL);
	exit(status);
      }
    }

    if(feof(stdin))
      return TRUE;
    else {
      report(DEBUG_PROC_DAT_READ_TMP_ERROR,NULL);
      return FALSE;
    }
  }
  else if(type == 't') {
    int spsc_inited = 0;


    while((ted_convert_stream_read_ddsp(ip,&ddsp)) == TED_OK) {

      if(spsc_inited)
	seq = TED_SPSC_INTERNAL;
      else
	spsc_inited = !0;

      if(diag(DEBUG_PROC_DAT_DDSP))
	report(DEBUG_PROC_DAT_DDSP,ddsp,NULL);

      if((status = ted_convert_dds2esatm(ddsp,&tmp,seq)) != TED_OK) {
	report(DEBUG_PROC_DAT_TRANS_TMP_ERROR,NULL);
	exit(status);
      }

      if(diag(DEBUG_PROC_DAT_TRANS_TMP))
	  report(DEBUG_PROC_DAT_TRANS_TMP,tmp,NULL);

      if((status = ted_convert_stream_write_tmp(op, tmp)) != TED_OK) {
	report(DEBUG_PROC_DAT_WRITE_TMP_ERROR,NULL);
	exit(status);
      }
    }

    if(feof(ip))
      return TRUE;
    else {
      report(DEBUG_PROC_DAT_READ_DDSP_ERROR,NULL);
      return FALSE;
    }
  }
  return FALSE;
}

/* }}} */
/* {{{ version*/

/*
 * This is a patch until the version module is updated
 * with a package wide standard version
 */
static char * VersionToString(int version,int revision,
			      int patch,int userpatch)
{
  static char string[40];

                 sprintf(string,"Version %d.%d",version,revision);
  if(patch)     sprintf(string + strlen(string)," Patch %d",patch);
  if(userpatch) sprintf(string + strlen(string),"(User Patch %d)",userpatch);
  return string;
}


void version()
{
  int version,
      revision,
      patch,
      userpatch;

 (void) ted_version_ted(&version,&revision,&patch,&userpatch);
  fprintf(stderr,
	  "TED Package %s\n",
	  VersionToString(version,revision,patch,userpatch));
  fprintf(stderr,
	  "TMC Program Version %s\n",
	  TMC_PROGRAM_VERSION);
}

/* }}} */

/* {{{ main */

void main( int argc, char *argv [] )
{
  int c;
  extern char *optarg;

  TED_SPACECRAFT   scid   = OptSCID  ((char *) getenv("TED_TMC_SCID"));
  TED_QUALITY      tcal   = OptTCAL  ((char *) getenv("TED_TMC_TCAL"));
  TED_TRANSMISSION stream = OptSTREAM((char *) getenv("TED_TMC_CHANNEL"));
  int              seq    = OptSEQ   ((char *) getenv("TED_TMC_SEQ_NO"));
  TED_GROUND       gsid   = OptGROUND((char *) getenv("TED_TMC_GROUND"));
  char             type   = 'n';

  while((c = getopt(argc,argv,"vd:ufta:")) != -1)
    switch(c) {
    case 'f':
    case 't':
      if((type != 'f')&&(type != 't'))
	type = c;
      break;
    case 'a':
      if(! aux_opt(optarg,&scid,&tcal,&stream,&gsid,&seq)) {
	fprintf(stderr, "error in auxillary options\n");
	usage(argc,argv);
	exit(-1);
      }
      break;
    case 'd':
      if(! debug(optarg)) {
	fprintf(stderr, "error in debug argument\n");
	usage(argc,argv);
	exit(-1);
      }
      break;
    case 'u':
      usage(argc,argv);
      exit(0);
    case 'v':
      version();
      exit(0);
    default:
      break;
    }

  ProcessData(stdin,stdout,type,scid,stream,tcal,gsid,seq);
}

/* }}} */


/* Local variables: */
/* folded-file: t */
/* end: */




