
/* $Id: main.c,v 1.1 2001/09/26 14:07:15 simond Exp $ */

#define _main_cc

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>

#include <main.h>
#include <convert.h>
#include <version.h>

#define SISTODDS_PROGRAM_VERSION "1.0"

/* {{{ usage */

void usage(int argc,char *argv [])
{
#define USAGE "\
	-u \"print this message\"\n\
	-v \"report program version\"\n\
	-i \"i/p\" <file opt>\n\
	-o \"o/p\" <file opt>\n\
	<file opt> ::= \n\
		hk = <filename>,sc = <filename>\n\
	-a <aux> \"alter program default parameters\"\n\
	<aux> ::=\n\
	       scid     = 1|2|3|4\n\
	       channel  = RT|PB|RE|RP\n\
               tcal     = act|con|ext\n\
               tam|asid = 1 .. 15\n\
               gsid     = mal|red|kou|per|ode|gol\n"

  fprintf(stderr,"USAGE: %s [-u | -v] [ -d ]\
 -i <file opt> -o <file opt> [-a <aux>] \n",argv [0]);
  fprintf(stderr,USAGE);
  exit(-1);
}

/* }}} */
/* {{{ debugging */

enum DEBUG_ENUM {
  DEBUG_PROC_NONE                    = 0x00000,

  DEBUG_PROC_IP                      = 0x00001,
  DEBUG_PROC_IP_HK                   = 0x00002,
  DEBUG_PROC_IP_SC                   = 0x00004,
  DEBUG_PROC_IP_PATCH                = 0x00008,

  DEBUG_PROC_TRANS_HK                = 0x00010,
  DEBUG_PROC_TRANS_SC                = 0x00020,
  DEBUG_PROC_TRANS_SC_TIME           = 0x00040,
  DEBUG_PROC_TRANS_HK_TIME           = 0x00080,

  DEBUG_PROC_DAT_IP_HK_FILENAME      = 0x00100,
  DEBUG_PROC_DAT_IP_SC_FILENAME      = 0x00200,
  DEBUG_PROC_DAT_OP_HK_FILENAME      = 0x00400,
  DEBUG_PROC_DAT_OP_SC_FILENAME      = 0x00800,

  DEBUG_PROC_DAT_SCID                = 0x01000,
  DEBUG_PROC_DAT_STREAM              = 0x02000,
  DEBUG_PROC_DAT_TCAL                = 0x04000,
  DEBUG_PROC_DAT_GSID                = 0x08000,

  DEBUG_PROC_OP                      = 0x10000,
  DEBUG_PROC_OP_HK                   = 0x20000,
  DEBUG_PROC_OP_SC                   = 0x40000
};
typedef enum DEBUG_ENUM DEBUG_ENUM;

static long level = 0;

/* {{{ debug ! */

int debug(char value[])
{
  int index;
  int val;

  if(value == NULL) {
    level = 0L;
    return FALSE;
  }
#if 0    
  for(index=0;index <(int) strlen(value); index++) {
    if(! isdigit(value[index])) {
      level = 0L;
      return FALSE;
    }
  }
#endif
  if(sscanf(value,"%i",&val) != 1) {
    level = 0L;
    return FALSE;
  }

  level =(long) val;

  return TRUE;
}

/* }}} */
/* {{{ diag ? */

int diag(DEBUG_ENUM dbg)
{
  return(dbg & level);
}

/* }}} */
/* {{{ report ! */

void
report(
DEBUG_ENUM rep,
...
)
{
  va_list list;

  unsigned char * packet = NULL;
  DEBUG_ENUM item;
  char *name;

  TED_SPACECRAFT scid;
  TED_TRANSMISSION stream;
  TED_QUALITY tcal;
  TED_GROUND gsid;

  va_start(list,rep);
  
  switch(rep) {
  case DEBUG_PROC_NONE:
    break;
  case DEBUG_PROC_IP:
    break;
  case DEBUG_PROC_IP_HK:
    packet = va_arg (list,unsigned char *);
    fprintf(stderr,"(sis-to-dds-debug(ip-hk\n");
    convert_display_tm_packet(packet);
    fprintf(stderr,"))\n");
    break;
  case DEBUG_PROC_IP_SC:
    packet = va_arg (list,unsigned char *);
    fprintf(stderr,"(sis-to-dds-debug(ip-sc\n");
    convert_display_tm_packet(packet);
    fprintf(stderr,"))\n");
    break;
  case DEBUG_PROC_IP_PATCH:
    fprintf(stderr,"(sis-to-dds-debug\n");

    packet = va_arg (list,unsigned char *);
    fprintf(stderr,"(ip-patch-hk\n");
    convert_display_tm_packet(packet);
    fprintf(stderr,")\n");

    packet = va_arg (list,unsigned char *);
    fprintf(stderr,"(ip-patch-sc\n");
    convert_display_tm_packet(packet);
    fprintf(stderr,")\n");

    fprintf(stderr,")\n");
    break;
  case DEBUG_PROC_TRANS_HK:
    packet = va_arg (list,unsigned char *);
    fprintf(stderr,"(sis-to-dds-debug\n");
    fprintf(stderr,"(trans-hk\n");
    convert_display_dds_packet(packet);
    fprintf(stderr,"))\n");
    break;
  case DEBUG_PROC_TRANS_SC:
    packet = va_arg (list,unsigned char *);
    fprintf(stderr,"(sis-to-dds-debug\n");
    fprintf(stderr,"(trans-sc\n");
    convert_display_dds_packet(packet);
    fprintf(stderr,"))\n");
    break;
  case DEBUG_PROC_TRANS_SC_TIME:
    fprintf(stderr,"(trans-sc-time\n");
    rep = va_arg(list,DEBUG_ENUM);
    packet = va_arg (list,unsigned char *);
    switch(rep) {
    case DEBUG_PROC_IP_SC:
       fprintf(stderr,"(ip-pkt-time\n");
      convert_display_tm_packet_time(packet);
      break;
    case DEBUG_PROC_OP_SC:
       fprintf(stderr,"(op-pkt-time\n");
      convert_display_dds_packet_time(packet);
      break;
    default:
      fprintf(stderr,"trans-sc-time-error\n");
    }
    fprintf(stderr,"))\n");
    break;
  case DEBUG_PROC_TRANS_HK_TIME:
    fprintf(stderr,"(trans-hk-time\n");
    rep = va_arg(list,DEBUG_ENUM);
    packet = va_arg (list,unsigned char *);
    switch(rep) {
    case DEBUG_PROC_IP_HK:
       fprintf(stderr,"(ip-pkt-time\n");
      convert_display_tm_packet_time(packet);
      break;
    case DEBUG_PROC_OP_HK:
       fprintf(stderr,"(op-pkt-time\n");
      convert_display_dds_packet_time(packet);
      break;
    default:
      fprintf(stderr,"trans-sc-time-error\n");
    }
    fprintf(stderr,"))\n");
    break;
  case DEBUG_PROC_DAT_IP_HK_FILENAME:
    name = va_arg(list,char *);
    fprintf(stderr,"(dat-ip-hk-filename %s)\n",name);
    break;
  case DEBUG_PROC_DAT_IP_SC_FILENAME:
    name = va_arg(list,char *);
    fprintf(stderr,"(dat-ip-sc-filename %s)\n",name);
    break;
  case DEBUG_PROC_DAT_OP_HK_FILENAME:
    name = va_arg(list,char *);
    fprintf(stderr,"(dat-op-hk-filename %s)\n",name);
    break;
  case DEBUG_PROC_DAT_OP_SC_FILENAME:
    name = va_arg(list,char *);
    fprintf(stderr,"(dat-op-sc-filename %s)\n",name);
    break;
  case DEBUG_PROC_DAT_SCID:
    scid = va_arg(list,TED_SPACECRAFT);
    fprintf(stderr,"(dat-spacecraft %s)\n",ted_scid_symbol(scid));
    break;
  case DEBUG_PROC_DAT_STREAM:
    stream = va_arg(list,TED_TRANSMISSION);
    fprintf(stderr,"(dat-stream %s)\n",ted_transmission_symbol(stream));
    break;
  case DEBUG_PROC_DAT_TCAL:
    tcal = va_arg(list,TED_QUALITY);
    fprintf(stderr,"(dat-tcal %s)\n",ted_quality_symbol(tcal));
    break;
  case DEBUG_PROC_DAT_GSID:
    gsid = va_arg(list,TED_GROUND);
    fprintf(stderr,"(dat-gsid %s)\n",ted_ground_symbol(gsid));
    break;
  case DEBUG_PROC_OP:
    break;
  case DEBUG_PROC_OP_HK:
    packet = va_arg (list,unsigned char *);
    fprintf(stderr,"(sis-to-dds-debug(ip-hk\n");
    convert_display_dds_packet(packet);
    fprintf(stderr,"))\n");
    break;
  case DEBUG_PROC_OP_SC:
    packet = va_arg (list,unsigned char *);
    fprintf(stderr,"(sis-to-dds-debug(ip-sc\n");
    convert_display_dds_packet(packet);
    fprintf(stderr,"))\n");
    break;
  default:
    break;
  }
  va_end(list);
  fflush(stderr);
  return;
}

/* }}} */

/* }}} */
/* {{{ string-to-gsid */

TED_GROUND OptGROUND(char value [])
{
  int index;

  if(value == NULL)
    return TED_GROUND_ODENWALD;

  for(index = 1; index <(int) strlen(value); index++)
    {
      if(isalpha(value [index]))
	value [index] = tolower(value [index]);
    }

       if(! strncmp(value,"ode",3)) return TED_GROUND_ODENWALD;
  else if(! strncmp(value,"red",3)) return TED_GROUND_REDU;
  else if(! strncmp(value,"kou",3)) return TED_GROUND_KOUROU;
  else if(! strncmp(value,"per",3)) return TED_GROUND_PERTH;
  else if(! strncmp(value,"mal",3)) return TED_GROUND_MALINDI;
  else if(! strncmp(value,"can",3)) return TED_GROUND_CANBERRA;
  else if(! strncmp(value,"gol",3)) return TED_GROUND_GOLDSTONE;

  fprintf(stderr,"GSID: does not identify a known ground segment.\n");
  fprintf(stderr,"using default value.\n");
  return TED_GROUND_ODENWALD;
}

/* }}} */
/* {{{ string-to-tcal */

TED_QUALITY OptTCAL(char value [])
{
  int index;

  if(value == NULL)
    return TED_QUALITY_EXTRAPOLATED;

  for(index = 1; index <(int) strlen(value); index++)
    if(isalpha(value [index]))
      value [index] = tolower(value [index]);

       if(! strncmp("act",value,3)) return TED_QUALITY_ACTUAL;
  else if(! strncmp("ext",value,3)) return TED_QUALITY_EXTRAPOLATED;
  else if(! strncmp("con",value,3)) return TED_QUALITY_CONTINGENCY;

  fprintf(stderr,"TCAL: does not identify a known ground segment.\n");
  fprintf(stderr,"using default value.\n");
  return TED_QUALITY_EXTRAPOLATED;
}

/* }}} */
/* {{{ string-to-stream */

TED_TRANSMISSION OptSTREAM(char value [])
{
  int index;

  if(value == NULL)
    return TED_TRANSMISSION_RECALLPLAYBACK;

  for(index = 0; index <(int) strlen(value); index++)
    if(isalpha(value [index]))
      value [index] = tolower(value [index]);
  
       if(! strncmp("rt",value,2)) return TED_TRANSMISSION_REALTIME;
  else if(! strncmp("pb",value,2)) return TED_TRANSMISSION_PLAYBACK;
  else if(! strncmp("re",value,2)) return TED_TRANSMISSION_RECALL;
  else if(! strncmp("rp",value,2)) return TED_TRANSMISSION_RECALLPLAYBACK;

  fprintf(stderr,"CHANNEL CLASS: is not a defined name. \n");
  fprintf(stderr,"using default value. \n");
  return TED_TRANSMISSION_RECALLPLAYBACK;
}

/* }}} */
/* {{{ string-to-scid */

TED_SPACECRAFT OptSCID(char value [])
{

  if(value == NULL)
    return CLUSTER_1;

       if(! strcmp(value,"1")) return CLUSTER_1;
  else if(! strcmp(value,"2")) return CLUSTER_2;
  else if(! strcmp(value,"3")) return CLUSTER_3;
  else if(! strcmp(value,"4")) return CLUSTER_4;

  fprintf(stderr,"SCID: in not in the required range.\n");
  fprintf(stderr,"using default value.\n");
  return CLUSTER_1;
}

/* }}} */
/* {{{ aux opt */

enum SUBOUT_ENUM {
  AUX_SUBOPT_SCID,
  AUX_SUBOPT_GSID,
  AUX_SUBOPT_TCAL,
  AUX_SUBOPT_TAM,
  AUX_SUBOPT_STRE,
  AUX_SUBOPT_ASID
};

int aux_opt(char arg[],
	     TED_SPACECRAFT *scid,
	     TED_QUALITY *tcal,
	     TED_TRANSMISSION *stream,
	     TED_GROUND *gsid)
{
  char * options = arg;
  char * value;

  char *opts[] = { "scid","gsid","tcal","tam","stream","asid",NULL };

  while(*options != '\0') {
    switch(getsubopt(&options,opts,&value)) {
    case AUX_SUBOPT_SCID: *scid    = OptSCID  (value); break;
    case AUX_SUBOPT_GSID: *gsid    = OptGROUND(value); break;
    case AUX_SUBOPT_TCAL: *tcal    = OptTCAL  (value); break;
    case AUX_SUBOPT_STRE: *stream  = OptSTREAM(value); break;
    default: return FALSE;
    }
  }	  
  return TRUE;
}

/* }}} */
/* {{{ tm open */

FILE * TMopen(char name[],char mode[])
{
  struct stat ssb; /* Struct Stat Buf */

  if(! strncmp(mode,"w",1)) {
    if(! stat(name,&ssb))
      return NULL;
  }

  if(! strncmp(mode,"r",1)) {
    if(stat(name,&ssb))
      return NULL;
  }

  return fopen(name,mode);
}

/* }}} */
/* {{{ file opt */

enum FILE_SUBOUT_ENUM {
  FILE_SUBOPT_HK,
  FILE_SUBOPT_SC
};

int file_opt(
char *arg,
char **hk,
char **sc
)
{
  char * options = arg;
  char * value;

  char *opts[] = { "hk","sc",NULL };

  while(*options != '\0') {
    switch(getsubopt(&options,opts,&value)) {
    case FILE_SUBOPT_HK: *hk =(char *) strdup(value); break;
    case FILE_SUBOPT_SC: *sc =(char *) strdup(value); break;
    default: break;
    }
  }
  return 1;
}

/* }}} */
/* {{{ debug opt */

void debug_opt(char arg[],float *debug)
{
  char * options = arg;
  char * value;

  if(sscanf(arg,"%f",debug) == 1)
    return;
  else
    *debug = 0.0;

  return;
}

/* }}} */
/* {{{ process-data */

enum PROC_DAT_ENUM {
  PROC_OK,			/* completed successfully */
  PROC_FILENAME_CONFLICT,	/* conflict in file names */
  PROC_OP_CREAT,		/* create o/p stream */
  PROC_IP_OPEN,			/* open i/p stream */
  PROC_IP_HK,			/* read an HK packet */
  PROC_IP_SC,			/* read an SC packet */
  PROC_OP_HK,			/* write an HK packet */
  PROC_OP_SC,			/* write an SC packet */
  PROC_TM_SIS_PATCH,		/* SIS TM Patch */
  PROC_TM_TRANSLATE		/* esatm2dds */
};
typedef enum PROC_DAT_ENUM PROC_DAT_ENUM;


PROC_DAT_ENUM proc_streams(FILE *hkif,
			    FILE *scif,
			    FILE *hkof,
			    FILE *scof,
			    TED_SPACECRAFT scid,
			    TED_TRANSMISSION stream,
			    TED_QUALITY tcal,
			    TED_GROUND ground)

{
  unsigned char *ddsp = NULL;
  unsigned char *hkp  = NULL;
  unsigned char *scp  = NULL;

  TED_STATUS status = TED_OK;

  if(hkif ==(FILE *) NULL) return PROC_IP_OPEN;
  if(scif ==(FILE *) NULL) return PROC_IP_OPEN;
  if(hkof ==(FILE *) NULL) return PROC_OP_CREAT;
  if(scof ==(FILE *) NULL) return PROC_OP_CREAT;

  for(;;) {
    status = ted_convert_stream_read_tmp(hkif,&hkp);
    if(status != TED_OK) {
      if(feof(hkif)) return PROC_OK;
    else
      return PROC_IP_HK;
    }

    if(diag(DEBUG_PROC_IP_HK))
      report(DEBUG_PROC_IP_HK,hkp,NULL);

      

    status = ted_convert_stream_read_sis_tmp(scif,&scp);
    if(feof(scif))
      return PROC_OK;
    switch(status)
      {
      case TED_OK:		break;
      case TED_READ_EOF:	return PROC_OK;
      default:			return PROC_IP_SC;
      }

    if(diag(DEBUG_PROC_IP_SC))
      report(DEBUG_PROC_IP_SC,scp,NULL);

    
    if(diag(DEBUG_PROC_TRANS_SC_TIME))
      report(DEBUG_PROC_TRANS_SC_TIME,DEBUG_PROC_IP_SC,scp,NULL);

    if(diag(DEBUG_PROC_TRANS_HK_TIME))
      report(DEBUG_PROC_TRANS_HK_TIME,DEBUG_PROC_IP_HK,hkp,NULL);

    status = ted_convert_sis_tm_patch(hkp,scp);
    if(status != TED_OK)
      return PROC_TM_SIS_PATCH;

    if(diag(DEBUG_PROC_IP_PATCH))
      report(DEBUG_PROC_IP_PATCH,hkp,scp,NULL);

    status = ted_convert_esatm2dds(hkp,  /* HK packet */
				    &ddsp,/* DDS packet */
				    scid, /* DDS scid */
				    stream,/* DDS stream class */
				    tcal, /* DDS time quality */
				    ground /* DDS ground segment */);
    if(status != TED_OK) return PROC_TM_TRANSLATE;

    if(diag(DEBUG_PROC_TRANS_HK))
      report(DEBUG_PROC_TRANS_HK,ddsp,NULL);

    if(diag(DEBUG_PROC_TRANS_HK_TIME))
      report(DEBUG_PROC_TRANS_HK_TIME,DEBUG_PROC_OP_HK,ddsp,NULL);

    status = ted_convert_stream_write_ddsp(hkof,ddsp);
    if(status != TED_OK) return PROC_OP_HK;
	
    status = ted_convert_esatm2dds(scp,  /* HK packet */
				    &ddsp,/* DDS packet */
				    scid, /* DDS scid */
				    stream,/* DDS stream class */
				    tcal, /* DDS time quality */
				    ground /* DDS ground segment */);
    if(status != TED_OK) return PROC_TM_TRANSLATE;
    
    if(diag(DEBUG_PROC_TRANS_SC))
      report(DEBUG_PROC_TRANS_SC,ddsp,NULL);

    if(diag(DEBUG_PROC_TRANS_SC_TIME))
      report(DEBUG_PROC_TRANS_SC_TIME,DEBUG_PROC_OP_SC,ddsp,NULL);
    
    status = ted_convert_stream_write_ddsp(scof,ddsp);
    if(status != TED_OK) return PROC_OP_SC;
  }
}

PROC_DAT_ENUM proc_dat(char hki[],
			char sci[],
			char hko[],
			char sco[],
			TED_SPACECRAFT scid,
			TED_TRANSMISSION stream,
			TED_QUALITY tcal,
			TED_GROUND ground)
{

  if(diag(DEBUG_PROC_DAT_SCID))
    report(DEBUG_PROC_DAT_SCID,scid,NULL);

  if(diag(DEBUG_PROC_DAT_STREAM))
    report(DEBUG_PROC_DAT_STREAM,stream,NULL);

  if(diag(DEBUG_PROC_DAT_TCAL))
    report(DEBUG_PROC_DAT_TCAL,tcal,NULL);

  if(diag(DEBUG_PROC_DAT_GSID))
    report(DEBUG_PROC_DAT_GSID,ground,NULL);

  if(diag(DEBUG_PROC_DAT_IP_HK_FILENAME))
    report(DEBUG_PROC_DAT_IP_HK_FILENAME,hki,NULL);

  if(diag(DEBUG_PROC_DAT_IP_SC_FILENAME))
    report(DEBUG_PROC_DAT_IP_SC_FILENAME,sci,NULL);

  if(diag(DEBUG_PROC_DAT_OP_HK_FILENAME))
    report(DEBUG_PROC_DAT_OP_HK_FILENAME,hko,NULL);

  if(diag(DEBUG_PROC_DAT_OP_SC_FILENAME))
    report(DEBUG_PROC_DAT_OP_SC_FILENAME,sco,NULL);

  if( !((strcmp(hki,sci) && strcmp(hki,hko) && strcmp(hki,sco)))) {
    fprintf(stderr,"error. A duplicate filename exists\n");
    return PROC_FILENAME_CONFLICT;
  }
  
  if( !((strcmp(sci,hki) && strcmp(sci,hko) && strcmp(sci,sco)))) {
    fprintf(stderr,"error. A duplicate filename exists\n");
    return PROC_FILENAME_CONFLICT;
  }
  
  if( !((strcmp(hko,hki) && strcmp(hko,sci) && strcmp(hko,sco)))) {
    fprintf(stderr,"error. A duplicate filename exists\n");
    return PROC_FILENAME_CONFLICT;
  }

  if( !((strcmp(sco,hki) && strcmp(sco,sci) && strcmp(sco,hko)))) {
    fprintf(stderr,"error. A duplicate filename exists\n");
    return PROC_FILENAME_CONFLICT;
  }

  return proc_streams(TMopen(hki,"r"),
		      TMopen(sci,"r"),
		      TMopen(hko,"w"),
		      TMopen(sco,"w"),
		      scid,
		      stream,
		      tcal,
		      ground);
}

/* }}} */
/* {{{ ted-version-to-string */

/*
 * This is a patch until the version module is updated
 * with a package wide standard version
 */
static char * VersionToString(int version,int revision,
			      int patch,int userpatch)
{
  static char string[40];

                 sprintf(string,"Version %d.%d",version,revision);
  if(patch)     sprintf(string + strlen(string)," Patch %d",patch);
  if(userpatch) sprintf(string + strlen(string),"(User Patch %d)",userpatch);
  return string;
}

/* }}} */
/* {{{ version */

void version()
{
  int version,
      revision,
      patch,
      userpatch;

 (void) ted_version_ted(&version,&revision,&patch,&userpatch);
  fprintf(stderr,
	  "TED Package %s\n",
	  VersionToString(version,revision,patch,userpatch));
  fprintf(stderr,
	   "SISTODDS Program Version %s\n",
	   SISTODDS_PROGRAM_VERSION);
}

/* }}} */
/* {{{ main */

void main(int argc,char *argv [])
{
  int c;
  extern char *optarg;

  TED_SPACECRAFT scid;
  TED_TRANSMISSION stream;
  TED_QUALITY tcal;
  TED_GROUND gsid;

  char *hki = NULL;
  char *sci = NULL;
  char *hko = NULL;
  char *sco = NULL;

  scid   = OptSCID  ((char *) getenv("TED_SIS_TO_DDS_SCID"   ));
  tcal   = OptTCAL  ((char *) getenv("TED_SIS_TO_DDS_TCAL"   ));
  stream = OptSTREAM((char *) getenv("TED_SIS_TO_DDS_CHANNEL"));
  gsid   = OptGROUND((char *) getenv("TED_SIS_TO_DDS_GROUND" ));

  while((c = getopt(argc,argv,"uvd:i:o:a:")) != -1)
    switch(c) {
    case 'd':
      if(! debug(optarg)) {
	fprintf(stderr,"error in debug argument\n");
	usage(argc,argv);
	exit(-1);
      }
      break;
    case 'i':
      if(! file_opt(optarg,&hki,&sci)) {
	fprintf(stderr,"error in i/p filename(s)\n");
	usage(argc,argv);
	exit(-1);
      }
      break;
    case 'o':
      if( ! file_opt(optarg,&hko,&sco)) {
	fprintf(stderr,"error in o/p filename(s)\n");
	usage(argc,argv);
	exit(-1);
      }
      break;
    case 'a':
      if( ! aux_opt(optarg,&scid,&tcal,&stream,&gsid)) {
	fprintf(stderr,"error in auxillary options\n");
	usage(argc,argv);
	exit(-1);
      }
      break;
    case 'u':
      usage(argc,argv);
      exit(0);
    case 'v':
      version();
      exit(0);
    case '?':
      fprintf(stderr,"error in command options\n");
      usage(argc,argv);
      exit(-1);
    default:
      usage(argc,argv);
      version();
      exit(-1);
    }


  if(argc < 5) {
    fprintf(stderr,"\nerror in command args\n");
    usage(argc,argv);
    version();
    exit(-1);
  }

  switch(proc_dat(hki,sci,hko,sco,scid,stream,tcal,gsid))
    {
    case PROC_OK:
      exit(0);
    case PROC_FILENAME_CONFLICT:
      fprintf(stderr,"error in filenames used\n");
      usage(argc,argv);
      exit(-1);
    case PROC_OP_CREAT:
      fprintf(stderr,"error opening an output file\n");
      usage(argc,argv);
      exit(-1);
    case PROC_IP_OPEN:
      fprintf(stderr,"error opening an input file\n");
      usage(argc,argv);
      exit(-1);
    case PROC_IP_HK:
      fprintf(stderr,"error while reading an ESA TM housekeeping packet\n");
      exit(-1);
    case PROC_IP_SC:
      fprintf(stderr,"error while reading an ESA TM science packet\n");
      exit(-1);
    case PROC_OP_HK:
      fprintf(stderr,"error while writing a DDS housekeeping packet\n");
      exit(-1);
    case PROC_OP_SC:
      fprintf(stderr,"error while writing a DDS science packet\n");
      exit(-1);
    case PROC_TM_SIS_PATCH:
      fprintf(stderr,"error patching ESA TM science packet\n");
      exit(-1);
    case PROC_TM_TRANSLATE:
      fprintf(stderr,"error translating packet from ESA TM to DDS format\n");
      exit(-1);
    default:
      fprintf(stderr,"undefined error caught\n");
      exit(-1);
    }
}

/* }}} */

/* Local variables: */
/* folded-file: t */
/* end: */




