 
/**
 * $Id: unpack.c,v 1.3 2001/09/28 13:28:51 simond Exp $
 */

#define _unpack_unpack_c

#include <generic.h>
#include <ted.h>
#include <tedsys.h>
#include <unpack.h>
#include <misc.h>
#include <dsd.h>
#include <version.h>
#include <show.h>
#include <xmacros.h>
#include <hkmacros.h>

#ifdef CL_WEC_TED_146
#include <ddsp.h>
#endif

#define debug_int(expr) debug_printf(#expr " = %d\n", (expr))
#ifndef NODEBUG
#define debug(message) debug_printf(#message "\n")
#else
#define debug(message)
#endif

#define minipd_udef0(mp) (minipd_count(mp) & 1)
#define minipd_udef1(mp) ((minipd_count(mp) & 2) >> 1)

#define WORD_SIZE 2


#define    DIAGNOSTIC(code, packet) enqueue_diagnostic(code, PacketToCCSDS(packet), NULL)
#define DIAGNOSTICEXP(code)         enqueue_diagnostic(code, &last_experiment_time, NULL)
#define  DIAGNOSTICHK(code)         enqueue_diagnostic(TED_DIAG_##code##, &last_hk_time, NULL)

#define SCIENCEPACKET_MAX ((474-1)*62)

#define DDSP_HEADER_SIZE 15

#define HKBLOCK_SIZE 192

#define DDSP_TIMESTAMP_SIZE 8

static jmp_buf return_from_ted_unpack_decom;

static int debug = 0;

static struct {
  int version;
  int revision;
  int patch;
  int userpatch;
} ted_version;

static struct {
  int spacecraft;
  int instruments;
  int ccs_converted;
  int model_tag_table[5];
} init_options;

static TED_TIME last_hk_time;

static TED_TIME last_experiment_time;

static char last_corresponding_hk_timestamp[8];

static module_initialised = FALSE;

typedef enum {
  NSD = 0,
  BSD = 1,
  HKD = 2
} DataFileType;

static char *text_DataFileType[] = {
  "NSD", "BSD", "HKD",
};

typedef enum {
  RealTime = 0,
  Playback = 1,
  Recall = 2,
  RecallPlayback = 3
} TransmissionMode;

static char *text_TransmissionMode[] = {
  "RT",
  "PB",
  "RE",
  "RP"
};

typedef enum {
  VC0 = 0,
  VC2 = 1,
  VC3 = 2
} VirtualChannel;

static char *text_VirtualChannel[] = {
  "VC0", "VC2", "VC3",
};

typedef enum {
  NormalMode1 = 0,
  NormalMode2 = 1,
  NormalMode3 = 2,
  BurstMode1 = 3,
  BurstMode2 = 4,
  BurstMode3 = 5
} AcqusitionMode;

static char *text_AcquisitionMode[] = {
  "NormalMode1",
  "NormalMode2",
  "NormalMode3",
  "BurstMode1",
  "BurstMode2",
  "BurstMode3",
};

enum AcqusitionSequence
{
HK1 = 0,
HK2 = 1,
NM1 = 2,
NM2 = 3,
NM3 = 4,
BM1 = 5,
BM2 = 6,
BM3 = 7
};
typedef enum AcqusitionSequence AcqusitionSequence;

static
char *
text_AcquisitionSequence[] =
{
  "HK1",
  "HK2",
  "NM1",
  "NM2",
  "NM3",
  "BM1",
  "BM2",
  "BM3"
};

#define DUMMY 0

static int BlockSizeForSequence[8] = {
  192,	/* HK1 */
  192,	/* HK2 */
  336,	/* NM1 */
  336,	/* NM2 */
  336,	/* NM3 */
  456,	/* BM1 */
  948,	/* BM2 */
  306	/* BM3 */
};

static int BlocksInMode[6] = { 10, 10, 10, 62, 62, 62 };

static int AcquisitionModeFromSequence[8] = {
  DUMMY,
  DUMMY,
  NormalMode1,
  NormalMode2,
  NormalMode3,
  BurstMode1,
  BurstMode2,
  BurstMode3,
};

static struct {
  int append_science;
  int ted_unpack_decom;
  int process_packet_header;
  int process_spacecraft_block;
  int process_exppacket;
  int process_hkpacket;
} reset_info;

#include <hktim.h>
static HKTIM *m_hktim = NULL;

/* 2006-06-19, KHY: Include TCOR header and instance of structure */
#include <hk_tcor.h>
static struct tcor_struct s_hktcor[1];

/* {{{ UTILITIES */

/* {{{ static void debug_printf(char message[], ...) */

static void debug_printf(char message[], ...)
{
  va_list args;

  va_start(args, message);
  if (debug)
    vfprintf(stderr, message, args);
  va_end(args);
}

/* }}} */
/* {{{ static char *must_malloc(unsigned size) */

static char *must_malloc(unsigned size)
{
  char *result;
  if ((result = (char *) malloc(size)) == NULL)
    longjmp(return_from_ted_unpack_decom, (int) TED_MEMORY);

  return result;
}

/* }}} */

/* }}} */
/* {{{ DIAGNOSTIC QUEUE */

struct queue_element {
  TED_UNPACK_DIAG diagnostic;
  struct queue_element *next;
};
typedef struct queue_element queue_element;

static struct {
  queue_element *front;
  queue_element *rear;
} queue;

/* {{{        char * static_Diagnostic_codeString(int) */

char * static_Diagnostic_codeString(int _code)
{
  static char *strings[] = {
    "DDSSOURCEINVALID",
    "DDSSPACECRAFTINVALID",
    "DDSSPACECRAFTMATCH",
    "DDSSTREAMINVALID",
    "DDSGROUNDINVALID",
    "DDSCHANNELTYPEMATCH",
    "DDSQUALITYINVALID",
    "DDSSEQUENCEINVALID",
    "DDSSPACECRAFTWRONG",
    "DDSDATATYPEWRONG",
    "DDSTIMESAME",
    "HKLENGTHWRONG",
    "HKEW5MOTAGINVALID",
    "HKEW5MOTAGMATCH",
    "DATALENGTHWRONG",
    "MPTRUNCATED",
    "MPSOURCEINVALID",
    "MPSOURCEWRONG",
    "MPCOUNTWRONG",
    "MPLENGTHEXTENDS",
    "MPMIDDLEFOUND",
    "SPTOOLONG",
	"TCORERROR",
    "MPLENGTHZERO"
  };
  if ((_code < TED_DIAG_DDSSOURCEINVALID) || (_code > TED_DIAG_MPLENGTHZERO))
    {
      return "?";
    }
  else
    {
      return strings[_code];
    }
}

/* }}} */
/* {{{        char * public_Diagnostic_toString(TED_UNPACK_DIAG *) */

char * public_Diagnostic_toString(TED_UNPACK_DIAG *this)
{
  static char buf[512];

  if (this == NULL) { return NULL; }

  sprintf(buf,
	  "%s %05ld/%08ld/%03ld %s",
	  CCSDSToString(&(this->time)),
	  (long) this->time.day,
	  (long) this->time.ms,
	  (long) this->time.us,
	  static_Diagnostic_codeString(this->code));

  if (this->details != NULL)
    {
      strcat(buf, " ");
      strcat(buf, this->details);
    }

  return buf;
}

/* }}} */
/* {{{        void   public_Diagnostic_show(TED_UNPACK_DIAG *,FILE *) */

void public_Diagnostic_show(TED_UNPACK_DIAG *this,FILE *_stream)
{
  if (_stream!= NULL) {
    if (this != NULL) {
      fprintf(_stream,"%s\n",public_Diagnostic_toString(this));
    }
  }
}

/* }}} */
/* {{{        void   public_Diagnostic_init(TED_UNPACK_DIAG *,TED_UNPACK_DIAGCODE,TED_TIME *,char *) */

void
public_Diagnostic_init(
TED_UNPACK_DIAG *this,
TED_UNPACK_DIAGCODE code,
TED_TIME *time,
char *details
)
{
  this->code = code;
  this->time.day = 0;
  this->time.ms  = 0;
  this->time.us  = 0;
  this->details = NULL;

  if (time != NULL)    this->time = *time;
  if (details != NULL) this->details = (char *) strdup(details);
}

/* }}} */
/* {{{        void   public_Diagnostic_final(TED_UNPACK_DIAG *) */

void public_Diagnostic_final(TED_UNPACK_DIAG *this)
{
  if (this != NULL)
  {
    if (this->details != NULL)
    {
      free(this->details);
      this->details = NULL;
    }
  }
}

/* }}} */
/* {{{        void   public_Diagnostic_dup(TED_UNPACK_DIAG *,TED_UNPACK_DIAG *) */

void
public_Diagnostic_dup(
TED_UNPACK_DIAG *this,
TED_UNPACK_DIAG *that
)
{
  if (this != NULL)
  {
    if (that != NULL)
    {
      public_Diagnostic_final(that);
      public_Diagnostic_init(that,this->code,&(this->time),this->details);
    }
  }
}

/* }}} */
/* {{{        char * public_Diagnostic_getDetails(TED_UNPACK_DIAG *this) */

char * public_Diagnostic_getDetails(TED_UNPACK_DIAG *this)
{
  if (this != NULL) {
    return this->details;
  }
  else {
    return NULL;
  }
}

/* }}} */

/* {{{        void              public_QueueElement_final  (queue_element **) */

void
public_QueueElement_final(
queue_element **p_this
)
{
  queue_element *this = NULL;
 
  if (p_this != NULL)
  {
    this = *p_this;
    if (this != NULL)
    {
      public_Diagnostic_final(&(this->diagnostic));
      free(this);
      *p_this = NULL;
    }
  }
}

/* }}} */
/* {{{        void              public_QueueElement_init   (queue_element **) */

void
public_QueueElement_init(
queue_element **p_this
)
{
  queue_element *this = NULL;
  if (p_this != NULL)
  {
    this = (queue_element *) malloc(sizeof(queue_element));
    if (this != NULL)
    {
      this->next = NULL;
      public_Diagnostic_init(&(this->diagnostic),
			     0,
			     NULL,
			     NULL);
    }
    *p_this = this;
  }
}

/* }}} */
/* {{{        TED_UNPACK_DIAG * public_QueueElement_getDiag(queue_element * ) */

TED_UNPACK_DIAG *
public_QueueElement_getDiag(
queue_element *this
)
{
  if (this != NULL)
  {
    return &(this->diagnostic);
  }
  return NULL;
}

/* }}} */
/* {{{        void              public_QueueElement_setDiag(queue_element *,TED_UNPACK_DIAG ) */

void
public_QueueElement_setDiag(
queue_element *this,
TED_UNPACK_DIAG diag
)
{
  if (this != NULL)
  {
    public_Diagnostic_dup(&diag,&(this->diagnostic));
  }
}

/* }}} */

/* {{{ static void enqueue_diagnostic(TED_UNPACK_DIAGCODE,TED_TIME *,char *, ...) */

static void enqueue_diagnostic(TED_UNPACK_DIAGCODE code, TED_TIME *time, char *_details, ...)
{
  va_list args;
  queue_element *that = NULL;

  public_QueueElement_init(&that);
  if (that != NULL)
  {
    TED_UNPACK_DIAG *l_diag = NULL;
    l_diag = (TED_UNPACK_DIAG *) public_QueueElement_getDiag(that);
    if (l_diag != NULL) public_Diagnostic_init(l_diag,code,time,NULL);
  }

  if (_details != NULL)
  {
    if (strlen(_details) > 0)
    {
      char buf[256];
      va_start(args, _details);
      vsprintf(buf, _details, args);
      va_end(args);
      public_Diagnostic_init(&(that->diagnostic),code,time,buf);
      public_Diagnostic_show(&(that->diagnostic),stderr);
    }
  }

  that->next = NULL;
  if (queue.front == NULL)
  {
    queue.front = that;
    queue.rear = that;
  }
  else
  {
    queue.rear->next = that;
    queue.rear = that;
  }

#ifndef NODEBUG
/*  debug_printf("unpack.diagnostic.queued %s\n", public_Diagnostic_toString(&(that->diagnostic))); */
#endif

}

/* }}} */
/* {{{ static void show_queue(void) */

static void show_queue(void)
{
  queue_element *q = queue.front;
  debug_printf("unpack.diagnostic.queue.entry\n");
  while (q != NULL) {
    debug_printf("unpack.diagnostic.queue.entry %s\n", DiagnosticToString(&q->diagnostic));
    q = q->next;
  }
}

/* }}} */
/* {{{ static void purge_queue(void) */

TED_STATUS
ted_unpack_diagnostic(
TED_UNPACK_DIAG **
);

static void purge_queue(void)
{
  TED_UNPACK_DIAG *d;
#ifndef NODEBUG
  debug_printf("unpack.diagnostic.purge\n");
#endif
  while (ted_unpack_diagnostic(&d) == TED_OK);
}

/* }}} */

/* }}} */
/* {{{ DDSP PACKET */

typedef struct {
  int transmission_mode;	/* eg RealTime */
  int virtual_channel;		/* eg VC2 */
  int data_type;		/* eg NSD */
  int acquisition_sequence;	/* eg NM2 */
  int acquisition_mode;		/* eg NormalMode2 */
} packet_info;

/* {{{ static int SpacecraftAndDataTypeFromSource(int,int *,int *) */

static int SpacecraftAndDataTypeFromSource(int source, int *data_type, int *spacecraft)
{
  switch (source) {
  case ddsp_source_wecnsd1: *data_type = NSD; *spacecraft = 1; break;
  case ddsp_source_wecbsd1: *data_type = BSD; *spacecraft = 1; break;
  case ddsp_source_wechkd1: *data_type = HKD; *spacecraft = 1; break;
  case ddsp_source_wecnsd2: *data_type = NSD; *spacecraft = 2; break;
  case ddsp_source_wecbsd2: *data_type = BSD; *spacecraft = 2; break;
  case ddsp_source_wechkd2: *data_type = HKD; *spacecraft = 2; break;
  case ddsp_source_wecnsd3: *data_type = NSD; *spacecraft = 3; break;
  case ddsp_source_wecbsd3: *data_type = BSD; *spacecraft = 3; break;
  case ddsp_source_wechkd3: *data_type = HKD; *spacecraft = 3; break;
  case ddsp_source_wecnsd4: *data_type = NSD; *spacecraft = 4; break;
  case ddsp_source_wecbsd4: *data_type = BSD; *spacecraft = 4; break;
  case ddsp_source_wechkd4: *data_type = HKD; *spacecraft = 4; break;
  default: return 1;
  }
  return 0;
}

/* }}} */
/* {{{ static int ModeAndChannelFromStream(int,int *,int *) */

static int ModeAndChannelFromStream(int stream, int *mode, int *channel)
{
  switch (stream) {
  case ddsp_stream_rtvc0: *mode = RealTime;       *channel = VC0; break;
  case ddsp_stream_rtvc2: *mode = RealTime;       *channel = VC2; break;
  case ddsp_stream_rtvc3: *mode = RealTime;       *channel = VC3; break;
  case ddsp_stream_pbvc0: *mode = Playback;       *channel = VC0; break;
  case ddsp_stream_pbvc2: *mode = Playback;       *channel = VC2; break;
  case ddsp_stream_pbvc3: *mode = Playback;       *channel = VC3; break;
  case ddsp_stream_revc0: *mode = Recall;         *channel = VC0; break;
  case ddsp_stream_revc2: *mode = Recall;         *channel = VC2; break;
  case ddsp_stream_revc3: *mode = Recall;         *channel = VC3; break;
  case ddsp_stream_rpvc0: *mode = RecallPlayback; *channel = VC0; break;
  case ddsp_stream_rpvc2: *mode = RecallPlayback; *channel = VC2; break;
  case ddsp_stream_rpvc3: *mode = RecallPlayback; *channel = VC3; break;
  default: return 1;
  }
  return 0;
}

/* }}} */
/* {{{ static int SequenceFromASIDOnChannel(int,int,int *) */

static int SequenceFromASIDOnChannel(int asid, int channel, int *sequence)
{
#ifdef CL_WEC_TED_146
  switch (channel)
  {
  case VC0:
    switch (asid)
    {
    case DDSP_ASID_1: *sequence = HK1; return 0;
    case DDSP_ASID_2: *sequence = HK2; return 0;
    default:          *sequence = HK1; return 0;
    }
    break;
  case VC2:
    switch (asid)
    {
    case DDSP_ASID_1: *sequence = NM1; return 0;
    case DDSP_ASID_2: *sequence = NM2; return 0;
    case DDSP_ASID_3: *sequence = NM3; return 0;
    default:          *sequence = NM3; return 0;
    }
    break;
  case VC3:
    switch (asid)
    {
    case DDSP_ASID_1: *sequence = BM1; return 0;
    case DDSP_ASID_2: *sequence = BM2; return 0;
    case DDSP_ASID_3: *sequence = BM3; return 0;
    default:          *sequence = -1;  return 1;
    }
    break;
  default:
    *sequence = -1;
    return 1;
  }
#else
  
   static int table[3][3] = {
              VC0  VC2  VC3
    ASID_1  { HK1, NM1, BM1 },
    ASID_2  { HK2, NM2, BM2 },
    ASID_3  {  -1, NM3, BM3 }
    };

   if (asid < 1 || asid > 3) return 1;
   *sequence = table[asid-1][channel];
   if (*sequence == -1) return 1;
   return 0;

#endif
}

/* }}} */
/* {{{ static int process_packet_header(char *,packet_info *) */

static
int
process_packet_header(
BYTE *packet,
packet_info *info
)
{
  /* TRUE iff this is the first packet received since a call to init() */
  static int first_packet;

  int spacecraft;

  /* resetting code */

  if (reset_info.process_packet_header) {
#ifndef NODEBUG
    debug_printf("unpack.packet_header.reset\n");
#endif
    first_packet = TRUE;
    reset_info.process_packet_header = FALSE;
  }

  if (SpacecraftAndDataTypeFromSource(ddsp_source(packet), &info->data_type, &spacecraft) != 0)
  {
    /* {{{ ERROR */

    DIAGNOSTIC(TED_DIAG_DDSSOURCEINVALID,packet);
    goto fatal;

    /* }}} */
  }

  if (!check_ddsp_spacecraft(ddsp_spacecraft(packet)))
  {
    /* {{{ ERROR */

    DIAGNOSTIC(TED_DIAG_DDSSPACECRAFTINVALID, packet);
    goto fatal;

    /* }}} */
  }

  if (spacecraft != ddsp_spacecraft(packet)) {
    /* {{{ ERROR */

    enqueue_diagnostic(TED_DIAG_DDSSPACECRAFTMATCH, PacketToCCSDS(packet),"%d != %d", spacecraft, ddsp_spacecraft(packet));
    goto fatal;

/* }}} */
  }

  if (init_options.spacecraft == CLUSTER_X && first_packet)
  {
    init_options.spacecraft = spacecraft;
    first_packet = FALSE;
  }
  else
  {
    if (spacecraft != init_options.spacecraft)
    {
      /* {{{ ERROR */

      DIAGNOSTIC(TED_DIAG_DDSSPACECRAFTWRONG, packet);
      goto fatal;

/* }}} */
    }
  }

  if (ModeAndChannelFromStream(ddsp_stream(packet),
			       &info->transmission_mode,
			       &info->virtual_channel) != 0)
  {
    /* {{{ ERROR */
    DIAGNOSTIC(TED_DIAG_DDSSTREAMINVALID, packet);
    goto fatal;
    /* }}} */
  }

  /* check for DDSCHANNELTYPEMATCH */

  if (!((info->data_type == NSD && info->virtual_channel == VC2)
	|| (info->data_type == BSD && info->virtual_channel == VC3)
	|| (info->data_type == HKD && info->virtual_channel == VC0)))
  {
    /* {{{ ERROR */
    DIAGNOSTIC(TED_DIAG_DDSCHANNELTYPEMATCH, packet);
    goto fatal;
    /* }}} */
  }

#if 0
  switch(info->virtual_channel) {
  case VC0: info->acquisition_sequence = HK1; break;
  case VC2: info->acquisition_sequence = NM1; break;
  case VC3: info->acquisition_sequence = BM1; break;
  }
#else
  if (SequenceFromASIDOnChannel(ddsp_sequence(packet),
				info->virtual_channel, &info->acquisition_sequence) != 0) {
    enqueue_diagnostic(TED_DIAG_DDSSEQUENCEINVALID, PacketToCCSDS(packet),"stream %s sequence %d",text_ddsp_stream(ddsp_stream(packet)),ddsp_sequence(packet));
    goto fatal;
  }
#endif

  /* calculate Acquisition Mode */

  info->acquisition_mode = AcquisitionModeFromSequence[info->acquisition_sequence];

#ifndef NODEBUG 
  debug_printf("unpack.packet_header.info %s %s %s %s\n",
	       text_DataFileType[info->data_type],
	       text_TransmissionMode[info->transmission_mode],
	       text_VirtualChannel[info->virtual_channel],
	       text_AcquisitionSequence[info->acquisition_sequence]);
#endif

  return 0;
 fatal:
  return 1;
}

/* }}} */

/* }}} */
/* {{{ SCIENCE PACKET */

static struct {
  BYTE *packet;	/* The packet */
  int length;	/* Length in words */
} science;

/* {{{ static int append_science(char *,int length) */

static int append_science(BYTE *minipacket, int length)
{
#ifdef DEBUG_DATA
  {
    int pos, val;
    for (pos = 0; pos < length; pos++) {
      val = (minipacket[pos*2+1] | minipacket[pos*2] << 8) & 0xffff;
      debug_printf("unpack.sciencebuffer.append.word %3d %04X\n", pos, val);
    }
  }
#endif

  /* Resetting code */

  if (reset_info.append_science) {
#ifndef NODEBUG
    debug_printf("unpack.sciencebuffer.reset\n");
#endif
    science.length = 0;
    reset_info.append_science = FALSE;
  }

#ifndef NODEBUG
  debug_printf("unpack.sciencebuffer.append %3d + %3d = %3d\n",
	       length, science.length, science.length + length);
#endif

  if (science.length + length > SCIENCEPACKET_MAX)
    return 1;
  memcpy(science.packet + science.length*2, minipacket, length*2);
  science.length += length;

  return 0;
}

/* }}} */
/* {{{ static void purge_science(void) */

static void purge_science(void)
{
#ifndef NODEBUG
  debug(unpack.sciencebuffer.purge);
#endif
  science.length = 0;
}

/* }}} */
/* {{{ static void flush_science(char **,int *) */

static
void
flush_science(
BYTE **packet,
int *length
)
{
#ifndef NODEBUG
  debug_printf("unpack.sciencebuffer.flush %d\n", science.length);
#endif
  *packet = science.packet;
  *length = science.length;
  science.length = 0;
}

/* }}} */
/* {{{ static int length_science(void) */

static int length_science(void)
{
  /* In case length_science() is called before any append_science(); */
  if (reset_info.append_science)
    science.length = 0;

  return science.length;
}

/* }}} */

/* }}} */
/* {{{ HK BLOCK */

/* {{{ static void show_hkblock_stuff(char *,char *) */

static void show_hkblock_stuff(char *heading, char *hkblock)
{
  debug_printf("%s EW5RSCNT=%d EW5MOTAG=%X EW5ACQMD=%s EW5PRCTL=%d EW5SSOFF=%d\n", heading,
	       (unsigned long) hkblock_EW5RSCNT(hkblock),
	       (unsigned long) hkblock_EW5MOTAG(hkblock),
	       text_hkblock_EW5ACQMD((unsigned long) hkblock_EW5ACQMD(hkblock)),
	       (unsigned long) hkblock_EW5PRCTL(hkblock),
	       (unsigned long) hkblock_EW5SSOFF(hkblock));
}

/* }}} */

/* }}} */
/* {{{ HK TABLE */

typedef struct {
  /* HKD packet containing HK block */
  BYTE packet[DDSP_HEADER_SIZE + HKBLOCK_SIZE];
  /* TRUE iff there is a HK block in this entry */
  int gotit;
  /* TRUE iff DDS timestamp is same as previous in file */
  int timesame;
  /* Master clock frequency calculated after hktim_update on HK block in this entry */
  double mcfreq;
} hkentry;

static hkentry hktable[8];

/* {{{ static void reset_hktable(void) */

static void reset_hktable(void)
{
  int i;
  for (i = 0; i < 8; i++) {
	  hktable[i++].gotit = FALSE;
	  hktable[i++].mcfreq = 900.0;
  }
}

/* }}} */
/* {{{ static void  show_hktable(int) */

static void show_hktable(int marked_entry)
{
  int i;
#ifndef NODEBUG
  debug_printf("unpack.hktable.entry ****************************************\n");
  for (i = 0; i < 8; i++) {
    debug_printf("unpack.hktable.entry %d %s ", i, i==marked_entry ? "-->" : "  :");
    if (hktable[i].gotit) {
      char *hkblock = ddsp_data(hktable[i].packet);
      debug_printf("%5d %s %s", hkblock_EW5RSCNT(hkblock),
		   CCSDSToString(PacketToCCSDS(hktable[i].packet)),
		   hktable[i].timesame ? "S" : "");
    } else
      debug_printf("----------------------------------");
    debug_printf("\n");
  }
#endif
}

/* }}} */
/* {{{ static void  fill_hktable(BYTE *,int) */

static
void
fill_hktable(
BYTE *packet,
int timesame,
double mcfreq
)
{
  BYTE *hkblock = ddsp_data(packet);
  hkentry *entry = &hktable[hkblock_EW5RSCNT(hkblock) % 8];

#ifndef NODEBUG
  debug_printf("unpack.hktable.fill count %ld (mod 8 = %ld)\n",
	       (unsigned long) hkblock_EW5RSCNT(hkblock),
	       ((unsigned long) hkblock_EW5RSCNT(hkblock)) % 8 );
#endif

  memcpy(entry->packet, packet, DDSP_HEADER_SIZE + ddsp_length(packet));
  entry->gotit = TRUE;
  entry->timesame = timesame;
  entry->mcfreq = mcfreq;

  show_hktable(hkblock_EW5RSCNT(hkblock) % 8);

  /* Clear the next two entries, just in case they are used before next HK packet arrives */
  entry = &hktable[(hkblock_EW5RSCNT(hkblock) + 1) % 8];
  entry->gotit = FALSE;
  entry = &hktable[(hkblock_EW5RSCNT(hkblock) + 2) % 8];
  entry->gotit = FALSE;

}

/* }}} */
/* {{{ static void   fix_hktable(int,int) */

static void fix_hktable(int previous, int pulses)
{
#ifndef NODEBUG
  debug_printf("unpack.hktable.fix previous=%d pulses=%d\n", previous, pulses);
#endif

  if (pulses > 0) {
    int first = previous + 1;	/* First one to clear */
    int last = first + (pulses > 7 ? 7 : pulses); /* Last one to clear */
    int pulse;
    for (pulse = first; pulse <= last; pulse++) {
#ifndef NODEBUG
      debug_printf("unpack.hktable.fix.clear %d\n", pulse % 8);
#endif
      hktable[pulse % 8].gotit = FALSE;
    }
  }
}

/* }}} */
/* {{{ static int    get_hktable(int,BYTE **,int *) */

static
int
get_hktable(
int entry, BYTE **packet,
int *timesame,
double *p_mcfreq
)
{
#ifndef NODEBUG
  debug_printf("unpack.hktable.get %d %s\n", entry,
	       hktable[entry].gotit ? "GOT IT" : "ENTRY BLANK");
#endif

  if (hktable[entry].gotit)
    {
      *packet = hktable[entry].packet;
      *timesame = hktable[entry].timesame;
	  *p_mcfreq = hktable[entry].mcfreq;
      return 0;
  } else
    return 1;
}

/* }}} */

/* }}} */
/* {{{ LVO PACKET TIME */

/* {{{ static double                          CCSDSToSeconds(TED_TIME *) */

static double CCSDSToSeconds(TED_TIME * ccsds)
{
  double seconds;

  seconds = (double) ccsds->day * 86400.0;
  seconds += (double) ccsds->ms / 1000.0;
  seconds += (double) ccsds->us / 1000000.0;

  return seconds;
}

/* }}} */
/* {{{ static int                              compare_times(TED_TIME *,TED_TIME *) */

static int compare_times(TED_TIME *time1, TED_TIME *time2)
{
  int result;
  result = (time1->day - time2->day);
  if (result)
    return result;
  result = (time1->ms - time2->ms);
  if (result)
    return result;
  return (time1->us - time2->us);
}

/* }}} */
/* {{{ static void                            add_timestamps(TED_TIME *,TED_TIME *,TED_TIME *,int *) */

static void add_timestamps(TED_TIME *a, TED_TIME *b, TED_TIME *result, int *carry)
{
  int sum;

  sum = a->us + b->us;
  *carry = sum / 1000;
  result->us = sum % 1000;

  sum = a->ms + b->ms + *carry;
  *carry = sum / 86400000;
  result->ms = sum % 86400000;

  sum = a->day + b->day + *carry;
  *carry = sum / 0x10000;
  result->day = sum % 0x10000;
}

/* }}} */
/* {{{ static int            estimate_pulses_from_timestamps(TED_TIME *,TED_TIME *) */

static int estimate_pulses_from_timestamps(TED_TIME *previous, TED_TIME *current)
{
  TED_TIME t = *previous;
  int pulses = 0;
  int carry;
#ifdef PULSE_INTERVAL_5
  static TED_TIME pulse_interval = {0,5000,000}; /* That's 5 seconds (just for testing) */
#else
  static TED_TIME pulse_interval = {0,5152,222}; /* That's 5.15222168 seconds rounded up */
#endif
  static TED_TIME zero_time = {0,0,0}; /* In-band special case */

#ifndef NODEBUG
   debug_printf("estimate_pulses_from_timestamps:\n"); 
   debug_printf("previous=%s\n", CCSDSToString(previous)); 
   debug_printf("current=%s\n", CCSDSToString(current));
#endif

  /* This algorithm doesn't take "carry" into account...because it doesn't need to */

  if (compare_times(previous, &zero_time) == 0) { return 1; }

#ifndef NODEBUG
   debug_printf("t=%s pulses=%d\n", CCSDSToString(&t), pulses);
#endif
  do {
    add_timestamps(&t, &pulse_interval, &t, &carry);
#ifndef NODEBUG
    debug_printf("t=%s pulses=%d\n", CCSDSToString(&t), pulses);
#endif
    pulses++;
  } while(compare_times(&t, current) < 0 && pulses <= 8);
  pulses--;

  return pulses;
}

/* }}} */
/* {{{ static int                estimate_pulses_from_counts(int previous, int current) */

/* estimate_pulses_from_counts()
 */
static int estimate_pulses_from_counts(int previous, int current)
{
  if (previous == -1)      { return 1; }
  if (current <= previous) { return 65536 - (previous-current); }
  else                     { return current-previous; }
}

/* }}} */
/* {{{ static int estimate_pulses_from_timestamps_and_counts(TED_TIME *,TED_TIME *,int,int) */

/* estimate_pulses_from_timestamps_and_counts()
 */
static int estimate_pulses_from_timestamps_and_counts(TED_TIME *previous_timestamp,TED_TIME *current_timestamp,int previous_count,int current_count)
{
  int timestamps_estimate = estimate_pulses_from_timestamps(previous_timestamp, current_timestamp);
  int counts_estimate = estimate_pulses_from_counts(previous_count, current_count);
  
#ifndef NODEBUG
  if (timestamps_estimate != counts_estimate)
    debug_printf("unpack.estimate.differ timestamps_estimate=%d counts_estimate=%d\n",
		 timestamps_estimate,
		 counts_estimate);
#endif

  if (timestamps_estimate > counts_estimate) {
    return timestamps_estimate;
  }
  else {
    return counts_estimate;
  }
}

/* }}} */

/* }}} */
/* {{{ WEC TIME */

/* {{{ static void calculate_sprt(char *,char *, double, TED_DSD_ITEMS *) */

static
void
calculate_sprt(
BYTE *hkpacket_for_timestamp,
BYTE *    hkpacket_for_block,
double mcfreq,
TED_DSD_ITEMS *dsd_items
)
{
  TED_TIME scet;

  time_t UT_sec;		/* SPRT seconds */
  double sprt;			/* SPRT seconds (and the rest) */
  unsigned int UT_msec;		/* SPRT milliseconds */
  unsigned int UT_usec;		/* SPRT microseconds */

  struct tm *UT_tm;		/* Time from UT_sec */

  if (hkpacket_for_timestamp != NULL)  scet = *PacketToCCSDS(hkpacket_for_timestamp);
  else if (hkpacket_for_block != NULL) scet = *PacketToCCSDS(hkpacket_for_block);
  else return;

#ifndef NODEBUG
  debug_printf("unpack.sprt.scet %s\n", CCSDSToString(PacketToCCSDS(hkpacket_for_timestamp)));
#endif

  /* sprt is in seconds */

  sprt  = scet.day * 86400;
  sprt -= 378691200;

  sprt += (double) scet.ms / 1000;
  sprt += (double) scet.us / 1000000;

  /* If no hk packet for timestamp, scet taken from next packet so must subtract one format */
  if (hkpacket_for_timestamp == NULL) sprt -= 5.15222168;

#ifndef DEBUG_UNPACK_SPRT
  if (hkpacket_for_block != NULL) {
    sprt += (double) hkblock_EW5SSOFF(ddsp_data(hkpacket_for_block)) * 1.0e-6;
  } else {
#ifndef NODEBUG
    debug_printf("unpack.sprt.no_snapshot\n");
#endif
  }
#ifndef DEBUG_UNPACK_SPRT_RAW_DWPCOUNT
  /* sprt += (double) dsd_items->dwpcount / 900.0; */
  sprt += (double) dsd_items->dwpcount / mcfreq;

#else
  debug_printf("unpack.sprt.raw_dwpcount\n");
  sprt += (double) dsd_items->dwpcount; /* Makes things simpler for testing */
#endif
#endif

  UT_sec = (time_t) sprt;
  sprt = (sprt - (double) UT_sec) * 1000;

  /* sprt is now milliseconds */

  UT_msec = (unsigned int) floor(sprt);
  sprt = (sprt - (double) UT_msec) * 1000;

  /* sprt is now microseconds */

  UT_usec = (unsigned int) floor(sprt);

  UT_tm = gmtime(&UT_sec);

  dsd_items->year = UT_tm->tm_year;
  dsd_items->month = UT_tm->tm_mon + 1;
  dsd_items->day = UT_tm->tm_mday;
  dsd_items->hour = UT_tm->tm_hour;
  dsd_items->min = UT_tm->tm_min;
  dsd_items->sec = UT_tm->tm_sec;
  dsd_items->msec = UT_msec;
  dsd_items->usec = UT_usec;
}

/* }}} */
/* {{{ static double DSDToSeconds(TED_DSD_ITEMS *dsd_items) */

static double DSDToSeconds(TED_DSD_ITEMS *dsd_items)
{
  double seconds;
  struct tm ut_tm;

  ut_tm.tm_year = dsd_items->year;
  ut_tm.tm_mon = dsd_items->month - 1;
  ut_tm.tm_mday = dsd_items->day;
  ut_tm.tm_hour = dsd_items->hour;
  ut_tm.tm_min = dsd_items->min;
  ut_tm.tm_sec = dsd_items->sec;

  seconds = mktime(&ut_tm);
  seconds += (double) dsd_items->msec / 1000.0;
  seconds += (double) dsd_items->usec / 1000000.0;

  return seconds;
}

/* }}} */
/* {{{ static void set_zero_timestamp(TED_DSD_ITEMS *dsd_items) */

static void set_zero_timestamp(TED_DSD_ITEMS *dsd_items)
{
  dsd_items->year = 0;
  dsd_items->month = 0;
  dsd_items->day = 0;
  dsd_items->hour = 0;
  dsd_items->min = 0;
  dsd_items->sec = 0;
  dsd_items->msec = 0;
  dsd_items->usec = 0;
}

/* }}} */

/* }}} */

/* {{{ HK PACKET */

/* {{{ static void process_hkpacket(char *,packet_info *) */

static
void
process_hkpacket(
BYTE *packet,
packet_info *info
)
{
  BYTE *hkblock;
  int timesame;
  static int had_packet;		/* TRUE iff we've seen a HK packet */
  static TED_TIME previous_timestamp;	/* Timestamp of previous packet if above is TRUE */

  /* 2006-06-20, KHY: Apply TCOR time corrections */
  hk_tcor_appl (s_hktcor, packet);
  switch (s_hktcor->status) {
  case 0: 
  case TCOR_STAT_NOPATH:
  case TCOR_STAT_BEFORE:
  case TCOR_STAT_PRECOR:
  case TCOR_STAT_ENDIDX:
	break;  /* These are not considered errors */
  default:
	enqueue_diagnostic(TED_DIAG_HKTCORERROR, PacketToCCSDS(packet), "%s", hk_tcor_statstr (s_hktcor));
  }

  /* Apply HKTIM corrections */
  {
    DDSP *l_ddsp = NULL;
    ddsp_init(&l_ddsp,0);
    ddsp_read_mem(l_ddsp,packet);

    hktim_update(m_hktim,l_ddsp);
    if (debug) hktim_show(m_hktim, stderr);

    ddsp_write_mem(l_ddsp,packet);
    ddsp_final(&l_ddsp);
  }

#ifndef NODEBUG
  debug(unpack.hkpacket);
#endif

  /* resetting code */

  if (reset_info.process_hkpacket) {
#ifndef NODEBUG
    debug_printf("unpack.hkpacket.reset\n");
#endif
    had_packet = 0;
    reset_info.process_hkpacket = FALSE;
  }

  timesame = (had_packet && (compare_times(PacketToCCSDS(packet), &previous_timestamp)) == 0);
  if (timesame)
    DIAGNOSTIC(TED_DIAG_DDSTIMESAME, packet);

  /* check for HKLENGTHWRONG */

  if (ddsp_length(packet) != BlockSizeForSequence[info->acquisition_sequence]) {
    DIAGNOSTIC(TED_DIAG_HKLENGTHWRONG, packet);
    goto discard;
  }

  hkblock = ddsp_data(packet);
#ifndef NODEBUG
  if (debug)
    show_hkblock_stuff("unpack.hkpacket.block.pars", hkblock);
#endif

  /* check for HKEW5MOTAGINVALID and HKEW5MOTAGMATCH */

  if (((unsigned long) hkblock_EW5MOTAG(hkblock)) != init_options.model_tag_table[ddsp_spacecraft(packet)]) {
    int spacecraft;
    for (spacecraft = 1; spacecraft < 5; spacecraft++) {
      if (((unsigned long) hkblock_EW5MOTAG(hkblock)) == init_options.model_tag_table[spacecraft]) {
	break;
      }
      if (spacecraft == 5) {
	/* No, EW5MOTAG doesn't match any entries */
	enqueue_diagnostic(TED_DIAG_HKEW5MOTAGINVALID, PacketToCCSDS(packet), "EW5MOTAG=%X", ((unsigned long) hkblock_EW5MOTAG(hkblock)));
	goto discard;
      }
      else {
	/* Yes, EW5MOTAG matches another entry -- should we let is pass? */
	if (init_options.ccs_converted)
	  debug(unpack.hkpacket.ignore_HKEW5MOTAGMATCH);
	else {
	  enqueue_diagnostic(TED_DIAG_HKEW5MOTAGMATCH, PacketToCCSDS(packet), "EW5MOTAG=%X", ((unsigned long)hkblock_EW5MOTAG(hkblock)));
	  goto discard;
	}
      }
    }
  }
  
  fill_hktable(packet, timesame, hktim_get_mcfreq (m_hktim));

 discard:

  had_packet = TRUE;
  previous_timestamp = *PacketToCCSDS(packet);

  return;
}

/* }}} */

/* }}} */
/* {{{ MINI PACKET */

/* {{{ static void show_minipacket_info(char *mp, int mp_number) */

static void show_minipacket_info(char *mp, int mp_number)
{
  int descriptor = (mp[1] & 0xff) | ((mp[0] & 0xff) << 8);

  fprintf(stderr, "unpack.spacecraft_block.minipacket %d %04X ", mp_number, descriptor);
  if (descriptor == 0x3000)
    fprintf(stderr, "--------------------------\n");
  else {
    fprintf(stderr, "%3d %-9s %-6s ",
	    minipd_length(mp),
	    text_minipd_source(minipd_source(mp)),
	    text_minipd_type(minipd_type(mp)));
    switch (minipd_type(mp)) {
    case minipd_type_single:
    case minipd_type_first:
      fprintf(stderr, "  %d %d\n", minipd_count(mp) & 2 >> 1, minipd_count(mp) & 1);
      break;
    case minipd_type_middle:
    case minipd_type_last:
      fprintf(stderr, "%d\n", minipd_count(mp));
      break;
    }
  }
}

/* }}} */

/* }}} */
/* {{{ SPACECRAFT BLOCK */

    /* determines whether the special case for zero-length mp's holds */

#define SpecialCaseForZeroHolds                          \
	   (                                                \
	     minipd_length(mpCur) == 0                 &&   \
	     minipd_type(mpCur)   == minipd_type_first &&   \
	     iscbMpCur            == iscbMax                \
	   )

#   define UnusedBlockRemainderConditionHolds              \
	  (                                                \
	    minipd_type(mpCur)   == minipd_type_single &&  \
	    minipd_source(mpCur) == 6                  &&  \
	    minipd_udef0(mpCur)  == 0                  &&  \
	    minipd_udef1(mpCur)  == 0                  &&  \
	    minipd_length(mpCur) == 0                      \
	  )


/* {{{ static void process_spacecraft_block(BYTE *,int,TED_DSD_ITEMS *,BOOL *,BOOL *,BYTE **) */

static
void
process_spacecraft_block(
BYTE *scb,
int iscbLim,
TED_DSD_ITEMS *pitms,
BOOL *pfGotScience,
BOOL *pfBlockNeededAgain,
BYTE **ppbScienceData
)
{


  static BOOL fStartingNewBlock;
  static int iscbMpCur;	   /* in bytes */
  static int cmpCur;	   /* count of this mp within the block */
  static BOOL fInReassemblySequence;
  static BOOL fAwaitingSecondMp;
  static int  nExpectedSequenceCount;
  static BYTE mpFirstDescriptor[WORD_SIZE];
  int iscbMax; /* first byte of the last word */
  BOOL fFinishedProcessingThisBlock;

  if (reset_info.process_spacecraft_block) {
    debug(unpack.spacecraft_block.reset);
    fStartingNewBlock = TRUE;
    fInReassemblySequence = FALSE;
    purge_science();
    reset_info.process_spacecraft_block = FALSE;
  }

  if (fStartingNewBlock) {
    cmpCur = 0;
    iscbMpCur = 0;
  }

  iscbMax = iscbLim - WORD_SIZE;

  *pfGotScience = FALSE;

  pitms->dw = 0;

  do {
    BYTE * mpCur = scb + iscbMpCur;  /* address of current mp */

#   ifndef NODEBUG
    if (debug)
      show_minipacket_info(mpCur, cmpCur);
#   endif

    fFinishedProcessingThisBlock = FALSE;

    if (UnusedBlockRemainderConditionHolds) {
      debug(unpack.spacecraft_block.unused);
      fFinishedProcessingThisBlock = TRUE;
      continue;
    }

    if (minipd_length(mpCur) == 0 && !SpecialCaseForZeroHolds) {
      DIAGNOSTICEXP(TED_DIAG_MPLENGTHZERO);
      if (fInReassemblySequence) {
	fInReassemblySequence = FALSE;
	if (length_science() > 0) {
	  pitms->dw     |= TED_DSD_DW_TRUNC;
	  *pfGotScience  = TRUE;
	} else
	  debug(unpack.spacecraft_block.zeropacket);
      }
      fFinishedProcessingThisBlock = TRUE;
      continue;
    }

    if (SpecialCaseForZeroHolds)
      debug(unpack.spacecraft_block.special);

    if (iscbMpCur + (minipd_length(mpCur) * WORD_SIZE) >= iscbLim) {
      DIAGNOSTICEXP(TED_DIAG_MPLENGTHEXTENDS);
      purge_science();
      fInReassemblySequence        = FALSE;
      fFinishedProcessingThisBlock = TRUE;
      continue;
    }

    switch (minipd_type(mpCur)) {
    case minipd_type_single:
    case minipd_type_first:
      if (fInReassemblySequence) {
	DIAGNOSTICEXP(TED_DIAG_MPTRUNCATED);
	fInReassemblySequence = FALSE;
	if (length_science() > 0) {
	  pitms->dw     |= TED_DSD_DW_TRUNC;
	  *pfGotScience  = TRUE;
	}
	else
	  debug(unpack.spacecraft_block.zeropacket);
	continue;
      }

      purge_science();

      fInReassemblySequence = TRUE;

      memcpy(mpFirstDescriptor, mpCur, WORD_SIZE);

      break;

    case minipd_type_middle:
    case minipd_type_last:

      if (!fInReassemblySequence) {
	DIAGNOSTICEXP(TED_DIAG_MPMIDDLEFOUND);
	goto LStepToNextMinipacket;
      }

      if (minipd_source(mpCur) != minipd_source(mpFirstDescriptor)) {
	DIAGNOSTICEXP(TED_DIAG_MPSOURCEWRONG);
	fInReassemblySequence = FALSE;
	if (length_science() > 0) {
	  pitms->dw     |= TED_DSD_DW_TRUNC;
	  *pfGotScience  = TRUE;
	}
	goto LStepToNextMinipacket;
      }

      if (fAwaitingSecondMp) {
	nExpectedSequenceCount = minipd_count(mpCur);
	fAwaitingSecondMp      = FALSE;
      }

      if (minipd_count(mpCur) != nExpectedSequenceCount) {
	DIAGNOSTICEXP(TED_DIAG_MPCOUNTWRONG);
	fInReassemblySequence = FALSE;
	if (length_science() > 0) {
	  pitms->dw     |= TED_DSD_DW_TRUNC;
	  *pfGotScience  = TRUE;
	}
	goto LStepToNextMinipacket;
      }

      break;
    }

    if (append_science(minipd_data(mpCur), minipd_length(mpCur))) {
      DIAGNOSTICEXP(TED_DIAG_SPTOOLONG);
      purge_science();
      fInReassemblySequence = FALSE;
      goto LStepToNextMinipacket;
    }

    switch (minipd_type(mpCur)) {
    case minipd_type_first:
      fAwaitingSecondMp = TRUE;
      break;
    case minipd_type_middle:
      nExpectedSequenceCount = (nExpectedSequenceCount + 1) % 4;
      break;
    case minipd_type_single:
    case minipd_type_last:
      fInReassemblySequence = FALSE;
      *pfGotScience         = TRUE;
      break;
    }

  LStepToNextMinipacket:
    
    iscbMpCur += WORD_SIZE + (minipd_length(mpCur) * WORD_SIZE);
    cmpCur++;
    
    if (iscbMpCur == iscbLim)
      fFinishedProcessingThisBlock = TRUE;
    
  } while (!(*pfGotScience || fFinishedProcessingThisBlock));

  if (*pfGotScience) {
    BYTE * bufSciencePacket;  /* science packet                        */
    int    cwSciencePacket;   /* length in words of the science packet */

    flush_science(&bufSciencePacket, &cwSciencePacket);

    pitms->udef0    = minipd_udef0(mpFirstDescriptor);
    pitms->udef1    = minipd_udef1(mpFirstDescriptor);
    pitms->source   = minipd_source(mpFirstDescriptor);
    pitms->dwpcount = sciencepacket_dwpcount(bufSciencePacket);
    pitms->obdhmod8 = sciencepacket_obdhmod8(bufSciencePacket);

    *ppbScienceData = sciencepacket_data(bufSciencePacket);

    /* the science data is the science packet minus the time tag word */
    pitms->length = cwSciencePacket - 1;
  }

  *pfBlockNeededAgain = *pfGotScience && !fFinishedProcessingThisBlock;

  fStartingNewBlock   = !(*pfBlockNeededAgain);

# ifndef NODEBUG
  debug_printf("unpack.spacecraft_block.return gotscience=%d again=%d\n",
	       *pfGotScience, *pfBlockNeededAgain);
# endif

  return;
}

/* }}} */

/* }}} */
/* {{{ DSD */

/* {{{ static void generate_dsd_version_items(TED_DSD_ITEMS *) */

static void generate_dsd_version_items(TED_DSD_ITEMS *dsd_items)
{
  dsd_items->version = ted_version.version;
  dsd_items->revision = ted_version.revision;
  dsd_items->patch = ted_version.patch;
  dsd_items->userpatch = ted_version.userpatch;
}

/* }}} */
/* {{{ static void generate_dsd_timestamp_items(TED_DSD_ITEMS *dsd_items, char **hkblock) */

static
void
generate_dsd_timestamp_items(
TED_DSD_ITEMS *dsd_items,
BYTE **hkblock
)
{
  BYTE *hkpacket_for_timestamp;	/* HK packet for timestamp */
  int got_hkpacket_for_timestamp; /* TRUE iff the HK packet for the timestamp is available */
  BYTE *hkpacket_for_block;	/* HK packet for HK block */
  int got_hkpacket_for_block;	/* TRUE iff the HK packet for the HK block is available */

  int timesame;
  int discard;			/* timesame from hkpacket_for_block will be discarded */
  int time_quality;     /* 4 bit time quality value from DDS packet header */
  double s_mcfreq;

#ifndef NODEBUG
  debug_printf("unpack.timestamp.timetag obdhmod8=%d dwpcount=%d\n",
	       dsd_items->obdhmod8, dsd_items->dwpcount);
#endif
  hkpacket_for_timestamp = NULL;
  got_hkpacket_for_timestamp = (!get_hktable(((int)(dsd_items->obdhmod8) % 8),&hkpacket_for_timestamp, &timesame, &s_mcfreq));
#ifndef NODEBUG
  debug_printf("unpack.timestamp.hkpacket_for_timestamp %d\n", got_hkpacket_for_timestamp);
#endif

  hkpacket_for_block = NULL;
  got_hkpacket_for_block = (!get_hktable(((int)(dsd_items->obdhmod8 + 1) % 8),&hkpacket_for_block, &discard, &s_mcfreq));
#ifndef NODEBUG
  debug_printf("unpack.timestamp.hkpacket_for_block %d\n", got_hkpacket_for_block);
#endif

  if (got_hkpacket_for_timestamp || got_hkpacket_for_block)
    {
      if (got_hkpacket_for_timestamp && timesame)	dsd_items->dw |= TED_DSD_DW_HKTIMESAME;

	  /* 2006-06-19, KHY: Include TCOR correction flag in diagnostic word */
	  if (got_hkpacket_for_timestamp) time_quality = ddsp_quality(hkpacket_for_timestamp);
	  else time_quality = ddsp_quality(hkpacket_for_block);
      switch (time_quality & 3) {
	    case ddsp_quality_actual:	  break;
	    case ddsp_quality_extrapolated:	  dsd_items->dw |= TED_DSD_DW_HKEXTRAPOLATED; break;
	    case ddsp_quality_contingency:	  dsd_items->dw |= TED_DSD_DW_HKCONTINGENCY; break;
	  }
	  if (time_quality & 4) dsd_items->dw |= TED_DSD_DW_HKTCOR_CORRECTED;

      calculate_sprt(hkpacket_for_timestamp, hkpacket_for_block, s_mcfreq, dsd_items);
    }
  else
    {
      dsd_items->dw |= TED_DSD_DW_NOHKTIME;
      set_zero_timestamp(dsd_items); /* this isn't actually defined in TN-2 remember! */
    }

  if (got_hkpacket_for_block)
    {
      *hkblock = ddsp_data(hkpacket_for_block);

	  /* 2006-06-19, KHY: Set HKTIM mode in diagnostic word */
	  dsd_items->dw |= TED_DSD_DW_HKTIM_MODE_0 * hkblock_EW5SSMOD(*hkblock);

      memcpy(last_corresponding_hk_timestamp, hkpacket_for_block, DDSP_TIMESTAMP_SIZE);
#ifndef NODEBUG
      if (debug)
	show_hkblock_stuff("unpack.timestamp.block.pars", *hkblock);
#endif
    }
  else
    {
      dsd_items->dw |= TED_DSD_DW_NOHK;
      /* Following is not defined in TN-2 */
      memset(last_corresponding_hk_timestamp, 0, DDSP_TIMESTAMP_SIZE);
    }
 
  /* Calculate EW5PRCTLNONZERO and EW5SSOFFTEST Diagnostic Flags */
  /*
  if ((got_hkpacket_for_timestamp == 0) && got_hkpacket_for_block)
    {
      if (hkblock_EW5PRCTL(*hkblock) != 0)   dsd_items->dw |= TED_DSD_DW_EW5PRCTLNONZERO;
    if (hkblock_EW5SSOFF(*hkblock) > 800)    dsd_items->dw |= TED_DSD_DW_EW5SSOFFTEST;
  }
  */
}

/* }}} */

/* }}} */
/* {{{ EXPERIMENT PACKET */

/* {{{ static void process_exppacket(char *,packet_info *,TED_DSD_ITEMS *,int *,int *,char **,char **) */

static
void
process_exppacket(
BYTE *exppacket,
packet_info *info,
TED_DSD_ITEMS *items,
int   *gotscience,
int   *again,
BYTE **sciencedata,
BYTE **hkblock
)
{
  static int wanted_again = 0;	/* 1 iff the previous call returned again=1 */
  static int block_number = 0;	/* number of the block we're on */

  static int number_of_blocks = 0;
  static int block_length = 0;

  /* Following are indexed by DataFileType (NSD or BSD) */
  static int had_packet[2];	/* TRUE iff we've seen a packet from this file */
  static TED_TIME previous_timestamp[2]; /* Timestamp of previous packet if above is TRUE */

  int block_again;		/* 1 iff process_spacecraft_block wants block again */

  /* Resetting code */

  if (reset_info.process_exppacket)
    {
#ifndef NODEBUG
      debug_printf("unpack.exppacket.reset\n");
#endif
      wanted_again = 0;
      had_packet[0] = had_packet[1] = 0;
      reset_info.process_exppacket = FALSE;
    }

  /* If we're starting on a new packet, reset our counter and calculate block details etc */
  if (!wanted_again)
    {
      int timesame;
      block_number = 0;
      number_of_blocks = BlocksInMode[info->acquisition_mode];
      block_length = BlockSizeForSequence[info->acquisition_sequence];

      timesame = (had_packet[info->data_type] && (compare_times(PacketToCCSDS(exppacket), &previous_timestamp[info->data_type]) == 0));
    if (timesame) DIAGNOSTIC(TED_DIAG_DDSTIMESAME, exppacket);
    if (ddsp_length(exppacket) != number_of_blocks * block_length)
      {
	DIAGNOSTIC(TED_DIAG_DATALENGTHWRONG, exppacket);
#ifndef NODEBUG
	debug_printf("unpack.exppacket.discard\n");
#endif
	*gotscience = 0;
	*again = 0;
#ifndef NODEBUG
	debug_printf("unpack.exppacket.return gotscience=%d again=%d\n", *gotscience, *again);
#endif
	return;
      }

#ifndef NODEBUG
    debug_printf("unpack.exppacket.packet %s\n", CCSDSToString(PacketToCCSDS(exppacket)));
    /* debug_printf("unpack.exppacket.info block_length=%d number_of_blocks=%d\n", block_length, number_of_blocks); */
#endif
    }
  
  do
    {
      BYTE *block = ddsp_data(exppacket) + block_number * block_length;
#ifndef NODEBUG
      debug_printf("unpack.exppacket.block %d\n", block_number);
#endif
      process_spacecraft_block(block, block_length, items, gotscience, &block_again, sciencedata);
      if (block_again == 0)
	block_number++;
    }
  while (!(*gotscience == 1 || block_number == number_of_blocks));

  if (*gotscience == 1)
    {
      items->spacecraft = ddsp_spacecraft(exppacket);
      items->ground = ddsp_ground(exppacket);
      generate_dsd_version_items(items);
      generate_dsd_timestamp_items(items, hkblock);
      *again = (block_number < number_of_blocks);
    }
  else
    *again = 0;			/* See "irrespective" in TN-3 2.0 "*again" definition */

  if (!wanted_again)
    {
      had_packet[info->data_type] = TRUE;
      previous_timestamp[info->data_type] = *PacketToCCSDS(exppacket);
    }

  wanted_again = *again;

#ifndef NODEBUG
  debug_printf("unpack.exppacket.return gotscience=%d again=%d\n", *gotscience, *again);
#endif

  return;
}

/* }}} */

/* }}} */
/* {{{ LV0 */

/* {{{ static void show_lv0_info(char *,packet_info *,char *,packet_info *,double *) */

static
void
show_lv0_info(
BYTE * hkpacket,
packet_info *hkinfo,
BYTE *exppacket,
packet_info *expinfo,
double *previous_time
)
{
  BYTE *packet;
  double current_time;

  if (hkpacket == NULL && exppacket == NULL)
    return;

  /* We can assume timestamps, spacecraft and ground are same for hk/exp, so use, say, hkpacket
     if both are available, and whatever's available otherwise */

  packet = (hkpacket ? hkpacket : exppacket);

  debug_printf("unpack.decom.LV0 %s ",CCSDSToString(PacketToCCSDS(packet)));
  
  current_time = CCSDSToSeconds(PacketToCCSDS(packet));
  if (*previous_time) {
    double interval = current_time - *previous_time;
    if (interval < 100.0)
      fprintf(stderr, "%9.6f ", interval);
    else
      fprintf(stderr, "**.****** ", interval);
  } else
    fprintf(stderr, "--.------ ");

  *previous_time = current_time;

  debug_printf("%d %s * ", ddsp_spacecraft(packet),
	       text_ddsp_ground(ddsp_ground(packet)));
  if (hkpacket)
    fprintf(stderr, "%s %s %s %s * ",
	    text_DataFileType[hkinfo->data_type],
	    text_TransmissionMode[hkinfo->transmission_mode],
	    text_VirtualChannel[hkinfo->virtual_channel],
	    text_AcquisitionSequence[hkinfo->acquisition_sequence]
	    );
  else
    fprintf(stderr, "-------------- * ");
  if (exppacket)
    fprintf(stderr, "%s %s %s %s",
	    text_DataFileType[expinfo->data_type],
	    text_TransmissionMode[expinfo->transmission_mode],
	    text_VirtualChannel[expinfo->virtual_channel],
	    text_AcquisitionSequence[expinfo->acquisition_sequence]
	    );
  else
    fprintf(stderr, "--------------");
  fprintf(stderr, "\n");
}

/* }}} */

/* }}} */
/* {{{ LV1 */

/* {{{ static void show_lv1_info(TED_DSD_ITEMS *,double *) */

static void show_lv1_info(TED_DSD_ITEMS *items, double *previous_time)
{
  debug_printf("unpack.decom.LV1 ");

  if (items->dw & TED_DSD_DW_NOHKTIME)
    fprintf(stderr, "--------------------------- ");
  else
    fprintf(stderr, "%04d-%02d-%02dT%02d:%02d:%02d.%03d%03dZ ",
	    items->year + 1900, items->month,
	    items->day, items->hour,
	    items->min, items->sec,
	    items->msec, items->usec);
  if (!(items->dw & TED_DSD_DW_NOHKTIME)) {
    double current_time = DSDToSeconds(items);
    if (*previous_time) {
      double interval = current_time - *previous_time;
      if (interval < 100.0)
	fprintf(stderr, "%9.6f ", interval);
      else
	fprintf(stderr, "**.****** ", interval);
    } else
      fprintf(stderr, "--.------ ");
    *previous_time = current_time;
  } else {
    *previous_time = 0;
    fprintf(stderr, "--.------ ");
  }
  fprintf(stderr, "%-9s %3d %4d %d %d %d ",
	  text_minipd_source(items->source),
	  items->length, items->dwpcount, items->obdhmod8, items->udef0, items->udef1);
  if (items->dw & TED_DSD_DW_NOHK) fprintf(stderr, "NOHK ");
  if (items->dw & TED_DSD_DW_TRUNC) fprintf(stderr, "TRUNC ");
  if (items->dw & TED_DSD_DW_HKTIMESAME) fprintf(stderr, "HKTIMESAME ");
  if (items->dw & TED_DSD_DW_HKEXTRAPOLATED) fprintf(stderr, "HKEXTRAPOLATED ");
  if (items->dw & TED_DSD_DW_HKCONTINGENCY) fprintf(stderr, "HKCONTINGENCY ");
  if (items->dw & TED_DSD_DW_NOHKTIME) fprintf(stderr, "NOHKTIME ");
  fprintf(stderr, "\n");
}

/* }}} */

/* }}} */

/* {{{ static int wants_this_instrument(int source) */

static int wants_this_instrument(int source)
{
  int mask;

  switch (source) {
  case minipd_source_efw:      mask = INSTRUMENT_EFW;      break;
  case minipd_source_staffsa:  mask = INSTRUMENT_STAFFSA;  break;
  case minipd_source_staffmwf: mask = INSTRUMENT_STAFFMWF; break;
  case minipd_source_whisper:  mask = INSTRUMENT_WHISPER;  break;
  case minipd_source_dwp:      mask = INSTRUMENT_DWP;      break;
  case minipd_source_wbd:      mask = INSTRUMENT_WBD;      break;
  }
  return (init_options.instruments & mask);
}

/* }}} */

/* {{{ TED_API */

/* 2006-06-19, KHY: New API function to set TCOR path */
TED_STATUS
ted_unpack_hktcor(
char *path
)
{
  hk_tcor_path (s_hktcor, path);
  return (TED_OK);
}


/* {{{ TED_STATUS           ted_unpack_hktim(TED_HKTIM_MODE) */

TED_STATUS
ted_unpack_hktim(
TED_HKTIM_MODE mode
)
{
  static double s_EW5MCFRQ[] = { 900.0 ,900.021 ,900.013 ,900.006 ,900.006 };

  if (debug) fprintf (stderr, "ted_unpack_hktim: %08x %d\n", m_hktim, mode);

  hktim_set_mode(m_hktim,mode);
  hktim_set_mode(NULL,mode);

  switch (mode) {
	case HKTIM_MODE_RUNNING_AVERAGE:
	case HKTIM_MODE_EXTRAPOLATION:
	hktim_set_mcfreq (m_hktim, s_EW5MCFRQ[init_options.spacecraft]);
  }

  return (TED_OK);
}

/* }}} */
/* {{{ TED_STATUS           ted_unpack_debug(int) */

TED_STATUS ted_unpack_debug(int debug_level)
{
  debug = debug_level;
  return TED_OK;
}

/* }}} */
/* {{{ TED_STATUS            ted_unpack_init(TED_SPACECRAFT,int,int,int []) */

TED_STATUS ted_unpack_init(TED_SPACECRAFT spacecraft,int instruments,int ccs_converted,int model_tag_table[])
{
  int i;

  /* show the options if we're debugging */

  if (debug) {
    fprintf(stderr,
	    "unpack.init.options %04X %04X %04X %04X %s ",
	    model_tag_table[1],
	    model_tag_table[2],
	    model_tag_table[3],
	    model_tag_table[4],
	    spacecraft == 0 ? "CLUSTER_X" :
	    text_ddsp_spacecraft(spacecraft));
    if (instruments == 0) fprintf(stderr, "(no instruments)");
    if (instruments & INSTRUMENT_EFW) fprintf(stderr, "EFW ");
    if (instruments & INSTRUMENT_STAFFSA) fprintf(stderr, "STAFFSA ");
    if (instruments & INSTRUMENT_STAFFMWF) fprintf(stderr, "STAFFMWF ");
    if (instruments & INSTRUMENT_WHISPER) fprintf(stderr, "WHISPER ");
    if (instruments & INSTRUMENT_DWP) fprintf(stderr, "DWP ");
    if (instruments & INSTRUMENT_WBD) fprintf(stderr, "WBD ");
    fprintf(stderr, "\n");
  }

  /* copy the options into init_options */

  init_options.spacecraft = spacecraft;
  init_options.instruments = instruments;
  init_options.ccs_converted = ccs_converted;

#ifdef CL_WEC_TED_147
  init_options.model_tag_table[1] = 0xcd09;
  init_options.model_tag_table[2] = 0xcd06;
  init_options.model_tag_table[3] = 0xcd07;
  init_options.model_tag_table[4] = 0xcd08;
#else
  init_options.model_tag_table[1] = 0xcd01;
  init_options.model_tag_table[2] = 0xcd02;
  init_options.model_tag_table[3] = 0xcd03;
  init_options.model_tag_table[4] = 0xcd04;
#endif

  for (i = 1; i <= 4; i++)
    if (model_tag_table[i] != 0x0000)
      init_options.model_tag_table[i] = model_tag_table[i];

#ifndef NODEBUG
  debug_printf("unpack.init.model_tag_table %04X %04X %04X %04X\n",
	       init_options.model_tag_table[1], init_options.model_tag_table[2],
	       init_options.model_tag_table[3], init_options.model_tag_table[4]);

#endif


  /* allocate memory for the science buffer if we haven't already */

  if (!module_initialised) {
    if ((science.packet = (BYTE *) malloc(SCIENCEPACKET_MAX*2)) == NULL)
      return TED_MEMORY;
  }

  /* fetch and record the version number from VERSION */
  ted_version_ted(&ted_version.version, &ted_version.revision,
		  &ted_version.patch, &ted_version.userpatch);

  /* reset some external variables */

  last_hk_time.day = last_hk_time.ms = last_hk_time.us = 0;
  last_experiment_time.day = last_experiment_time.ms = last_experiment_time.us = 0;
  queue.front = queue.rear = NULL;
  /* Following is not defined in TN-2 */
  memset(last_corresponding_hk_timestamp, 0, DDSP_TIMESTAMP_SIZE);

  /* let some modules know they need to reset themselves */

  reset_info.append_science = TRUE;
  reset_info.ted_unpack_decom = TRUE;
  reset_info.process_packet_header = TRUE;
  reset_info.process_spacecraft_block = TRUE;
  reset_info.process_exppacket = TRUE;
  reset_info.process_hkpacket = TRUE;

  reset_hktable();
  purge_queue();

  /* Initialise HKTIM, make running average the default mode */
  if (m_hktim != NULL) hktim_final(&m_hktim);
  hktim_init(&m_hktim);
  ted_unpack_hktim(HKTIM_MODE_RUNNING_AVERAGE);
  if (debug) hktim_show(m_hktim, stderr);

  /* 2006-06-19, KHY: Initialise HK_TCOR module */
  if (module_initialised) hk_tcor_fini (s_hktcor);
  hk_tcor_init (s_hktcor);
  if (debug) hk_tcor_show (s_hktcor, stderr);

  module_initialised = TRUE;
  return TED_OK;
}

/* }}} */
/* {{{ TED_STATUS           ted_unpack_decom(char *,char *,int *,TED_DSD_ITEMS **,char **,char **,int *) */

TED_STATUS
ted_unpack_decom(
char *exppacket,
char *hkpacket,
int *gotscience,
TED_DSD_ITEMS **items,
char **hkblock,
char **sciencedata,
int *again
)
{

  typedef
    enum PRCESS_STATE
    {
      PROCESS_NEXT,
      FLUSH_THEN_BUFFER_EXPPACKET,
      FLUSH_THEN_INSERT_HKPACKET,
      FLUSH_THEN_DO_EXPPACKET,
      FLUSH_THEN_RETURN
    } PROCESS_STATE;

  static TED_DSD_ITEMS dsd_items; /* The DSD Header Items we'll pass back */

  static packet_info expinfo;	/* Information on NSD/BSD packet */
  static packet_info hkinfo;	/* Information on HKD packet */

  static int         have_buffered      = FALSE; /* TRUE iff we have a buffered experiment packet */
  static BYTE *      buffered_exppacket = NULL; /* Buffered experiment packet iff have_buffered=TRUE */
  static packet_info buffered_expinfo; /* Buffered experiment packet info iff have_buffered=TRUE */


  static PROCESS_STATE state;		/* What we should do next */


  static TED_TIME previous_time;        /* Time of previous reset pulse, or 0 if there was none */
  static double   previous_lv0_time;    /* Similar to above but for debugging and represented in seconds */
  static int      previous_count;	/* Reset pulse count of previous pulse, or -1 if there was none */
  static double   previous_lv1_time[6]; /* Timestamp (us) from previous LV1 packet per instrument, or 0 if none  */

  TED_STATUS result;


  static char *state_strings[] = {
    "PROCESS_NEXT",
    "FLUSH_THEN_BUFFER_EXPPACKET",
    "FLUSH_THEN_INSERT_HKPACKET",
    "FLUSH_THEN_DO_EXPPACKET",
    "FLUSH_THEN_RETURN"
  };

  if (!module_initialised) return TED_NOTINITIALISED;

  if (reset_info.ted_unpack_decom)
    /* {{{ reset */

  {
    int i;
    state = PROCESS_NEXT;

    if (have_buffered) free(buffered_exppacket);
    have_buffered = FALSE;

    previous_time.day = 0;
    previous_time.ms  = 0;
    previous_time.us  = 0;
    previous_lv0_time = 0;

    for (i = 5; i; i--) { previous_lv1_time[i] = 0; }
    previous_count = -1;
    reset_info.ted_unpack_decom = FALSE;
  }

/* }}} */

  if (result = (TED_STATUS) setjmp(return_from_ted_unpack_decom))
  {
    debug_printf("unpack.decom.longjump\n");
    return result;
  }

  *gotscience = 0;
  *again = 0;

 start:

  if (state != PROCESS_NEXT && have_buffered) {
    if (process_packet_header(buffered_exppacket, &buffered_expinfo) != 0)
      return TED_UNPACK_FATAL;
    process_exppacket( buffered_exppacket,
		      &buffered_expinfo,
		      &dsd_items,
		       gotscience,
		       again,
		      (BYTE **) sciencedata,
		      (BYTE **) hkblock);
    if (*gotscience) {
      *items = &dsd_items;
      if (*again) {
	goto out;
      }
      else {
	free(buffered_exppacket);
	have_buffered = FALSE;
      }
    }
    else {
      free(buffered_exppacket);
      have_buffered = FALSE;
    }
  }

#ifndef NODEBUG
  debug_printf("unpack.decom.state %s\n", state_strings[state]);
#endif

  switch (state) {

  case FLUSH_THEN_INSERT_HKPACKET:
    process_hkpacket((BYTE *) hkpacket, &hkinfo);
    state = PROCESS_NEXT;
    goto do_exppacket;

  case FLUSH_THEN_DO_EXPPACKET:
    state = PROCESS_NEXT;
    goto do_exppacket;

  case FLUSH_THEN_BUFFER_EXPPACKET: {
    int packet_length = DDSP_HEADER_SIZE + ddsp_length(exppacket);
    buffered_exppacket = (BYTE *) must_malloc(packet_length);
    memcpy(buffered_exppacket, exppacket, packet_length);
    buffered_expinfo = expinfo;
    have_buffered = TRUE;
    state = PROCESS_NEXT;
    goto out;
  }

  case FLUSH_THEN_RETURN:
    state = PROCESS_NEXT;
    goto out;

  case PROCESS_NEXT:
    goto do_packets;

  }

 do_packets:

#ifndef NODEBUG
  debug(unpack.decom.packets);
#endif

  if (hkpacket)
  {
    if (process_packet_header((BYTE *) hkpacket, &hkinfo) != 0)      return TED_UNPACK_FATAL;
    if (hkinfo.data_type != HKD)
    {
      DIAGNOSTIC(TED_DIAG_DDSDATATYPEWRONG, (BYTE *) hkpacket);
      return TED_UNPACK_FATAL;
    }
  }

  if (exppacket)
  {
    if (process_packet_header((BYTE *) exppacket, &expinfo) != 0)      return TED_UNPACK_FATAL;
    if (!(expinfo.data_type == NSD || expinfo.data_type == BSD))
    {
      DIAGNOSTIC(TED_DIAG_DDSDATATYPEWRONG, (BYTE *) exppacket);
      return TED_UNPACK_FATAL;
    }
  }

#ifndef NODEBUG
  if (debug)
    show_lv0_info(hkpacket, &hkinfo, exppacket, &expinfo, &previous_lv0_time);
#endif

 do_hkpacket:

#ifndef NODEBUG
  debug(unpack.decom.hkpacket);
#endif

  if (hkpacket)
  {
    TED_TIME *current_time;
    int current_count;
    int pulses;

    last_hk_time = *PacketToCCSDS((BYTE *)hkpacket);

    current_time = PacketToCCSDS((BYTE *) hkpacket);
    current_count = hkblock_EW5RSCNT(ddsp_data(hkpacket));

	/*
    pulses = estimate_pulses_from_timestamps_and_counts(&previous_time, current_time,
							previous_count, current_count);
	*/
	/* 2006-06-23, KHY: The estimate_pulses_from_timestamps code appears dubious, so we use counts only. */
    pulses = estimate_pulses_from_counts(previous_count, current_count);
    previous_time = *current_time;

#ifndef NODEBUG
    debug_printf("unpack.decom.hkpacket.estimate %d\n", pulses);
#endif

    if (pulses > 1)
    {
      fix_hktable(previous_count, pulses-1);
      previous_count = current_count;
      state = FLUSH_THEN_INSERT_HKPACKET;
      goto start;
    }
    else
    {
      process_hkpacket((BYTE *) hkpacket, &hkinfo);
      previous_count = current_count;
      state = FLUSH_THEN_DO_EXPPACKET;
      goto start;
    }
  }

  /**************************/
  /* Process any EXP Packet */
  /**************************/

 do_exppacket:

#ifndef NODEBUG
  debug(unpack.decom.exppacket);
#endif

  if (exppacket)
  {
    TED_TIME *current_time;

    last_experiment_time = *PacketToCCSDS((BYTE *) exppacket);
    current_time = PacketToCCSDS((BYTE *)exppacket);

    if (!hkpacket)
    {
      int pulses;
      pulses = estimate_pulses_from_timestamps(&previous_time, current_time);
      previous_time = *current_time;
#ifndef NODEBUG
      debug_printf("unpack.decom.exppacket.estimate %d\n", pulses);
#endif
      if (pulses > 1)
	fix_hktable(previous_count, pulses-1);
      state = FLUSH_THEN_BUFFER_EXPPACKET;
      goto start;
    }
    else
    {
      int packet_length = DDSP_HEADER_SIZE + ddsp_length(exppacket);
      buffered_exppacket = (BYTE *) must_malloc(packet_length);
      memcpy(buffered_exppacket, exppacket, packet_length);
      buffered_expinfo = expinfo;
      have_buffered = TRUE;
      goto out;
    }
  }

  /* For NULL, NULL */

#ifndef NODEBUG
  debug_printf("unpack.decom.end\n");
#endif

  state = FLUSH_THEN_RETURN;
  goto start;

 out:

#ifndef NODEBUG
  if ((*gotscience) && !wants_this_instrument(dsd_items.source))
    debug_printf("unpack.decom.notwanted\n");
#endif

  /* Clear the "gotscience" flag if data from this instrument is not wanted */
  *gotscience = *gotscience && wants_this_instrument(dsd_items.source);

#ifndef NODEBUG
  if (*gotscience && debug)
    show_lv1_info(&dsd_items, &previous_lv1_time[dsd_items.source]);
  debug_printf("unpack.decom.return gotscience=%d again=%d\n", *gotscience, *again);
#endif

  return TED_OK;
}

/* }}} */
/* {{{ TED_STATUS      ted_unpack_diagnostic(TED_UNPACK_DIAG **) */

TED_STATUS
ted_unpack_diagnostic(
TED_UNPACK_DIAG **diagnostic
)
{
  static int             the_diagnostic_details = 0;
  static TED_UNPACK_DIAG the_diagnostic = { { 0,0,0 },0,NULL} ;
  queue_element *temp;

  if (queue.front == NULL)
  {
    return TED_UNPACK_DIAGEND;
  }

  if ((char *) public_Diagnostic_getDetails(&the_diagnostic) != NULL)
  {
    public_Diagnostic_final(&the_diagnostic);
  }

  temp = queue.front;
  public_Diagnostic_dup(&(temp->diagnostic),&the_diagnostic);
  queue.front = queue.front->next;
  public_QueueElement_final(&temp);

  *diagnostic = &the_diagnostic;
  return TED_OK;
}

/* }}} */
/* {{{ TED_STATUS    ted_unpack_hk_timestamp(char **) */

TED_STATUS ted_unpack_hk_timestamp(char **timestamp)
{
  /* NB this just copies addresses not the actual value */
  *timestamp = last_corresponding_hk_timestamp;

  return TED_OK;
}

/* }}} */
/* {{{ TED_STATUS ted_unpack_diagcode2string(TED_UNPACK_DIAGCODE,char **) */

TED_STATUS ted_unpack_diagcode2string(TED_UNPACK_DIAGCODE diagcode,char **string)
{
    *string = static_Diagnostic_codeString(diagcode);
    return TED_OK;
}

/* }}} */
/* {{{ TED_STATUS     ted_unpack_diag2string(TED_UNPACK_DIAG,char **) */

TED_STATUS ted_unpack_diag2string(TED_UNPACK_DIAG diag,char **string)
{
  static char mystring[100];
  char *diag_string;

  sprintf(mystring,"%s",public_Diagnostic_toString(&diag));
  *string = mystring;
  return TED_OK;
}

/* }}} */

/* }}} */

/* Local variables: */
/* folded-file: t */
/* end: */
