 
/**
 * $Id: read.c,v 1.1 2001/09/26 14:07:03 simond Exp $
 */

#define _read_read_c
#include <tedsys.h>
#include <read.h>
#include <xmacros.h>
#include <show.h>

int ted_read_file_no;

#define DDSP_HEADER_SIZE 15
#define MAX_PACKET_NSD   (DDSP_HEADER_SIZE + 10 * 336)
#define MAX_PACKET_BSD   (DDSP_HEADER_SIZE + 62 * 948)
#define MAX_PACKET_HKD   (DDSP_HEADER_SIZE + 192)

static int debug = 0;
static struct {
  int ted_read_packet;
} reset_info;

typedef enum {
  nsd = 1,
  bsd = 2,
  hkd = 3
} filenumber;

typedef struct READ_RECORD {
  int max;			/* Size of largest possible packet, in bytes */
  FILE *stream;			/* Stream */
  char *packet;			/* Packet */
  int next;			/* (See algorithm documentation) */
  int needs_sucking;		/* TRUE iff this file needs sucking */
  int been_read;		/* TRUE iff a packet has been read from this file */
} READ_RECORD;

static READ_RECORD table[4] = {
  { 0,             NULL,NULL,0,TRUE,FALSE },
  { MAX_PACKET_NSD,NULL,NULL,0,TRUE,FALSE },
  { MAX_PACKET_BSD,NULL,NULL,0,TRUE,FALSE },
  { MAX_PACKET_HKD,NULL,NULL,0,TRUE,FALSE }
};

static char *nsd_packet;
static char *bsd_packet;
static char *hkd_packet;

static int status;

/* {{{ static void        debug_printf(char message[], ...) */

static void debug_printf(char message[], ...)
{
  va_list args;

  va_start(args, message);
  if (debug)
    vfprintf(stderr, message, args);
  va_end(args);
}

/* }}} */

/* {{{ static long           remaining(FILE *) */

static long remaining(FILE *stream)
{
  long bytes, pos;

  pos = ftell(stream);
  if (fseek(stream, 0L, 2) == -1)
    return -1L;
  if ((bytes = ftell(stream)) == -1)
    return -1L;
  if (fseek(stream, pos, 0) == -1)
    return -1L;

#ifndef NODEBUG
  debug_printf("remaining=%ld\n", bytes - pos);
#endif

  return (bytes - pos);
}

/* }}} */
/* {{{ static TED_STATUS   read_header(filenumber) */

static TED_STATUS read_header(filenumber filenum)
{
  int rem;

  /* the amount of data remain to be read is undefined */
  if ((rem = remaining(table[filenum].stream)) == -1L)
    {
      ted_read_file_no = filenum;
      return TED_READ_FILE;
    }

#ifndef NODEBUG
  debug_printf("r(%d)=%d\n", filenum, rem);
#endif

  /* if bytes remaining < size of a packet header */
  if (rem < DDSP_HEADER_SIZE)  {
      return TED_READ_EOF;
  }
  else {
    int nitems = fread(table[filenum].packet, 1, DDSP_HEADER_SIZE, table[filenum].stream);
    if (nitems != DDSP_HEADER_SIZE)
      {
	ted_read_file_no = filenum;
	return TED_READ_FILE;
      }
  }
  return TED_OK;
}

/* }}} */
/* {{{ static TED_STATUS     read_data(filenumber) */

static TED_STATUS read_data(filenumber filenum)
{
  int rem;

  /* if the packet is bugger that maximum length allowed */
  if (ddsp_length(table[filenum].packet) + 15 > table[filenum].max)
    {
      return TED_READ_SIZE;
    }

  /* if the packet is zero */
  if (ddsp_length(table[filenum].packet) == 0)
    {
      return TED_READ_SIZE;
    }

  /* if the No. of bytes remaining in the file is undefined? */
  if ((rem = remaining(table[filenum].stream)) == -1L)
    {
      ted_read_file_no = filenum;
      return TED_READ_FILE;
    }

#ifndef NODEBUG
  debug_printf("r=%d\n", rem);
#endif

  /* if the No. of bytes remain in the file is less
   * than the No.of bytes to be read
   */
  if (rem < ddsp_length(table[filenum].packet))
    return TED_READ_EOF;

#ifndef NODEBUG
  debug_printf("length=%d\n",ddsp_length(table[filenum].packet));
#endif

  /* OK! So read the data! */
  if (fread(ddsp_data(table[filenum].packet), 1, ddsp_length(table[filenum].packet), table[filenum].stream) == 0) {
    ted_read_file_no = filenum;
    return TED_READ_FILE;
  }

  return TED_OK;
}

/* }}} */
/* {{{ static TED_STATUS   read_packet(filenumber) */

static TED_STATUS read_packet(filenumber filenum)
{
  if ((status = read_header(filenum)) == TED_OK)
    status = read_data(filenum);

  return status;
}

/* }}} */

/* {{{ static int        compare_times(char *packet1,char *packet2) */

static int compare_times(char *packet1,char *packet2)
{
  int result;
  result = (ddsp_day(packet1) - ddsp_day(packet2));
  if (result != 0) return result;
  result = (ddsp_msec(packet1) - ddsp_msec(packet2));
  if (result != 0) return result;
  return (ddsp_usec(packet1) - ddsp_usec(packet2));
}

/* }}} */
/* {{{ void                 show_table(void) */

void show_table(void)
{
  int filenum;

#ifdef DEBUG_READ
  debug_printf("TABLE:\n");
  for (filenum = table[0].next; filenum != 0; filenum = table[filenum].next)
    debug_printf("table[%d] %s -> %d\n", filenum,
	   CCSDSToString(PacketToCCSDS(table[filenum].packet)),
	   table[filenum].next);
#endif
}

/* }}} */

/* {{{ static TED_STATUS        insert(filenumber filenum) */

static TED_STATUS insert(filenumber filenum)
{
  int where;
  int previous;

#ifndef NODEBUG
  debug_printf("insert(%d)\n", filenum);
#endif

  where = table[0].next;
  previous = 0;

  while (where != 0 && compare_times(table[filenum].packet, table[where].packet) >= 0) {
    previous = where;
    where = table[where].next;
  }

  table[filenum].next = where;
  table[previous].next = filenum;
  show_table();

  return TED_OK;
}

/* }}} */
/* {{{ static TED_STATUS          suck(filenumber filenum) */

static TED_STATUS suck(filenumber filenum)
{
  /* If been_read == TRUE for file, following is first 8 bytes
     of most recent header in this file */
  char recent_scet[8];

#ifndef NODEBUG
  debug_printf("suck(%d)\n", filenum);
#endif

  /* if >= 1 packet has been read since initilization */
  if (table[filenum].been_read) {
    /* copy the time stamp into the working buffer */
    (void) memcpy(recent_scet, table[filenum].packet, 8);
  }

  switch (status = read_packet(filenum)) {
  case TED_OK:
    if (table[filenum].been_read && compare_times(table[filenum].packet, recent_scet) < 0) {
#ifndef NODEBUG
      debug_printf("TIMESTAMP IS < PREVIOUS: %s\n",CCSDSToString(PacketToCCSDS(table[filenum].packet)));
#endif
      return TED_READ_EARLIER;
    }

    if (table[filenum].been_read && compare_times(table[filenum].packet, recent_scet) == 0) {
#ifndef NODEBUG
      debug_printf("PACKET HAS SAME TIME AS PREVIOUS: %s\n", CCSDSToString(PacketToCCSDS(table[filenum].packet)));
#endif
    }
    table[filenum].been_read = TRUE;
    (void) insert(filenum);
    return TED_OK;
  case TED_READ_SIZE:    return TED_READ_SIZE;
  case TED_READ_FILE:    return TED_READ_FILE;
  case TED_READ_EOF:
    /* don't insert it -- nothing interesting left in the file */
#ifndef NODEBUG
    debug_printf("EOF on %d\n", filenum);
#endif
    return TED_OK;
  }
  /* NOTREACHED */
  
}

/* }}} */
/* {{{ static TED_STATUS     read_next(filenumber *) */

static TED_STATUS read_next(filenumber *filenump)
{
  int filenum = 0;
  int n;

  /* if a packet need to be loaded
   * make it so!
   */
  for (n = 1 ; n < 4 ; n++) {
    if (table[n].needs_sucking) {
      switch (status = suck(n)) {
      case TED_READ_EOF:	return TED_OK;
      case TED_OK:		break;
      default:			return status;
      }
      table[n].needs_sucking = FALSE;
    }
  }

#ifndef NODEBUG
  debug_printf("before: \n");show_table();
#endif

  /* Find out which file has the earliest packet in */

  filenum = table[0].next;

#ifndef NODEBUG
  debug_printf("filenum=%d\n", filenum);
#endif

  if (filenum == 0)
    return TED_READ_EOF;

  table[0].next = table[filenum].next;

  /* ddsp_show_header(table[filenum].packet); */

#ifndef NODEBUG
  debug_printf("after: \n");show_table();
#endif
  
  table[filenum].needs_sucking = TRUE;

  *filenump = filenum;

  return TED_OK;
}

/* }}} */

/* {{{ TED_STATUS ted_read_debug (int) */

TED_STATUS ted_read_debug(int debug_level)
{
  debug = debug_level;
  return TED_OK;
}

/* }}} */
/* {{{ TED_STATUS ted_read_initf (FILE * ,FILE * ,FILE * ) */

TED_STATUS ted_read_initf(FILE *nsd_stream,FILE *bsd_stream,FILE *hkd_stream)
{
  static int initialised = FALSE; /* TRUE iff we've initialised once already */

  if (!initialised) {
    if ((nsd_packet = (char *) malloc(MAX_PACKET_NSD)) == NULL)      return TED_MEMORY;
    if ((bsd_packet = (char *) malloc(MAX_PACKET_BSD)) == NULL)      return TED_MEMORY;
    if ((hkd_packet = (char *) malloc(MAX_PACKET_HKD)) == NULL)      return TED_MEMORY;
    initialised = TRUE;
  }

  /* Reset some external variables */

  table[nsd].been_read = FALSE;
  table[bsd].been_read = FALSE;
  table[hkd].been_read = FALSE;

  /* Let some modules know they need to reset themselves */

  reset_info.ted_read_packet = TRUE;

  table[nsd].packet = nsd_packet;
  table[bsd].packet = bsd_packet;
  table[hkd].packet = hkd_packet;

  table[nsd].stream = nsd_stream;
  table[bsd].stream = bsd_stream;
  table[hkd].stream = hkd_stream;

  table[0].next = 0;
  
  table[nsd].needs_sucking = TRUE;
  table[bsd].needs_sucking = TRUE;
  table[hkd].needs_sucking = TRUE;

  return TED_OK;
}

/* }}} */
/* {{{ TED_STATUS ted_read_init  (int    ,int    ,int    ) */

#ifndef TED_NO_FD
TED_STATUS
ted_read_init(
int nsd_fd,
int bsd_fd,
int hkd_fd
)
{
  FILE *nsd_stream;
  FILE *bsd_stream;
  FILE *hkd_stream;

  nsd_stream = (FILE *) fdopen(nsd_fd, "r");
  if (nsd_stream == NULL)
  {
    ted_read_file_no = 1;
    return TED_READ_FILE;
  }
  bsd_stream = (FILE *) fdopen(bsd_fd, "r");
  if (bsd_stream == NULL)
  {
    ted_read_file_no = 2;
    return TED_READ_FILE;
  }
  hkd_stream = (FILE *) fdopen(hkd_fd, "r");
  if (hkd_stream == NULL)
  {
    ted_read_file_no = 3;
    return TED_READ_FILE;
  }
  return ted_read_initf(nsd_stream, bsd_stream, hkd_stream);
}
#endif

/* }}} */
/* {{{ TED_STATUS ted_read_packet(char **,char **,char **) */

TED_STATUS ted_read_packet(char **nsd_packetp,char **bsd_packetp,char **hkd_packetp)
{
  static int initialised = FALSE; /* TRUE iff we've initialised once already */

  static char *mypacket[3];
  static int  gotpacket[3];

  static filenumber lookahead = 0;

  static int last = -1;

#ifndef NODEBUG
  debug_printf("ted_read_packet()\n");
#endif

  if (((remaining(table[1].stream) < DDSP_HEADER_SIZE) &&
       (remaining(table[2].stream) < DDSP_HEADER_SIZE) &&
       (remaining(table[3].stream) < DDSP_HEADER_SIZE)) ||
      (lookahead < 0)) {
    return TED_READ_EOF;
  }

  if (!initialised) {
    if ((mypacket[0] = (char *) malloc(MAX_PACKET_NSD)) == NULL)      return TED_MEMORY;
    if ((mypacket[1] = (char *) malloc(MAX_PACKET_BSD)) == NULL)      return TED_MEMORY;
    if ((mypacket[2] = (char *) malloc(MAX_PACKET_HKD)) == NULL)      return TED_MEMORY;
    initialised = TRUE;
  }

  /* {{{ RESETING */

  if (reset_info.ted_read_packet) {
#ifndef NODEBUG
    debug_printf("ted_read_packet(): resetting\n");
#endif
    lookahead = 0;
    reset_info.ted_read_packet = FALSE;
  }  

/* }}} */

  gotpacket[0] = FALSE;
  gotpacket[1] = FALSE;
  gotpacket[2] = FALSE;

  if (lookahead == 0)
    if ((status = read_next(&lookahead)) != TED_OK)
      return status;

  do {

#ifndef NODEBUG
    debug_printf("lookahead = %d\n", lookahead);
    debug_printf("PACKET GOING IN IS from file %d with time %s\n",lookahead,CCSDSToString(PacketToCCSDS(table[lookahead].packet)));
#endif

    (void) memcpy(mypacket[lookahead - 1],table[lookahead].packet,15 + ddsp_length(table[lookahead].packet));
    gotpacket[lookahead - 1] = TRUE;
    last = lookahead - 1;
    
    if ((status = read_next(&lookahead)) != TED_OK)
      lookahead = -status;

  } while ((status == TED_OK) &&
	   (gotpacket[lookahead - 1] == FALSE) &&
	   (compare_times(table[lookahead].packet, mypacket[last]) == 0));

#ifndef NODEBUG
  debug_printf("NO MORE PACKETS GOING IN\n");
#endif

  if (gotpacket[0]) *nsd_packetp = mypacket[0]; else *nsd_packetp = NULL;
  if (gotpacket[1]) *bsd_packetp = mypacket[1]; else *bsd_packetp = NULL;
  if (gotpacket[2]) *hkd_packetp = mypacket[2]; else *hkd_packetp = NULL;

#ifndef NODEBUG
  debug_printf("GOTPACKETS: %d %d %d\n", gotpacket[0], gotpacket[1], gotpacket[2]);
#endif
  
  return TED_OK;
}

/* }}} */

/* Local variables: */
/* folded-file: t */
/* end: */
