/* HK_TCOR module */
/* Keith Yearby, 2006 May 5 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "hk_tcor.h"

extern time_t timezone;
extern int daylight;

static void PutScet (DDShead, scet)
unsigned char DDShead[];
double scet;
{
unsigned int days, micros;
long millis;

days = (int) (scet / 86400.0);
scet -= days * 86400.0;
millis = (long) (scet * 1000.0);
scet -= millis / 1000.0;
micros = (int) (scet * 1000000.0);
days += 4383; /* Adjust from 1970 to 1958 epoch. */

DDShead[0] = days >> 8;
DDShead[1] = days;
DDShead[2] = millis >> 24;
DDShead[3] = millis >> 16;
DDShead[4] = millis >> 8;
DDShead[5] = millis;
DDShead[6] = micros >> 8;
DDShead[7] = micros;
}

static void GetScet (DDShead, scetp)
unsigned char DDShead[];
double *scetp;
{
	*scetp = (DDShead[0] * 256.0 + DDShead[1] - 4383) * 86400.0 +
             DDShead[2] * 16777.216 + DDShead[3] * 65.536 +
             DDShead[4] * 0.256 + DDShead[5] * 0.001 +
             DDShead[6] * 0.000256 + DDShead[7] * 0.000001;
}


static long CCSDS_time (time_str)
char *time_str;
{
struct tm timestruc;

if (sscanf (time_str, "%d-%d-%dT%d:%d:%d",
            &timestruc.tm_year, &timestruc.tm_mon, &timestruc.tm_mday,
            &timestruc.tm_hour, &timestruc.tm_min, &timestruc.tm_sec) != 6)
            return (0);

timestruc.tm_year -= 1900; /* Years measured from 1900 */
timestruc.tm_mon  -= 1;    /* Months start at zero */
timestruc.tm_isdst = 0;    /* No daylight saving */

#ifdef _MSC_VER
    _timezone = 0;
	_daylight = 0;
#else
    timezone = 0;
	daylight = 0;
#endif

return (mktime (&timestruc));
}

static char *CCSDS_str (long ltime)
{
struct tm *ptimestruc;
static char time_str[20];

ptimestruc = gmtime (&ltime);
sprintf (time_str, "%04d-%02d-%02dT%02d:%02d:%02d",
        ptimestruc->tm_year + 1900, ptimestruc->tm_mon +1,
        ptimestruc->tm_mday,
        ptimestruc->tm_hour, ptimestruc->tm_min, ptimestruc->tm_sec);

return (time_str);
}

static int read_tcor_rec (struct tcor_struct *tcp)
{
	char buffer[256];
	char scet_asc[32];
	double offset, diff;
	/* static double last_end_time; */

	if (tcp->tcor_fp == NULL) return (tcp->status = TCOR_STAT_NOFILE);

	/* Read lines from TCOR file until we get non-fill values, or EOF. */
	do {
		if (fgets (buffer, 256, tcp->tcor_fp) == NULL) return tcp->status = TCOR_STAT_EOF;
		if (sscanf (buffer, "%[^,],%lf,%lf", scet_asc, &offset, &diff) != 3) return tcp->status = TCOR_STAT_FORMAT;
	} while (offset == -1.0e31 || diff == -1.0e31);

	/* Save first point */
	tcp->scet_1 = (double)(CCSDS_time (scet_asc));
	tcp->offset = offset;
	tcp->diff_1 = diff;

	/* Get second point */
	if (fgets (buffer, 256, tcp->tcor_fp) == NULL) return tcp->status = TCOR_STAT_EOF;
	if (sscanf (buffer, "%[^,],%lf,%lf", scet_asc, &offset, &diff) != 3) return tcp->status = TCOR_STAT_FORMAT;

	/* Check and save second point */
	if (offset == -1.0e31 || diff == -1.0e31) return tcp->status = TCOR_STAT_FORMAT;
	if (tcp->offset != offset) return tcp->status = TCOR_STAT_FORMAT;
	tcp->scet_2 = (double)(CCSDS_time (scet_asc));
	tcp->diff_2 = diff;
	if (tcp->scet_2 < tcp->scet_1) return tcp->status = TCOR_STAT_FORMAT;

	return tcp->status = 0;
}

static int open_tcor_file (struct tcor_struct *tcp)
{
	char buffer[256];
	char tcor_id_str[16];
	int got_tcor_id;

	if (tcp->tcor_name == NULL) return tcp->status = TCOR_STAT_NOFILE;

	/* Open the file */
	if (tcp->tcor_fp != NULL) fclose (tcp->tcor_fp);
	sprintf (buffer, "%s/%s", tcp->tcor_path, tcp->tcor_name);
	tcp->tcor_fp = fopen (buffer, "r");
	if (tcp->tcor_fp == NULL) return tcp->status = TCOR_STAT_NOFILE;

	/* Form the TCOR ID string, including spacecraft number */
	got_tcor_id = 0;
	sprintf (tcor_id_str, "C%d_CP_DWP_TCOR", tcp->index_scid);

	/* Skip meta-data */
	do {
		if (fgets (buffer, 256, tcp->tcor_fp) == NULL) {
			fclose (tcp->tcor_fp);
			return tcp->status = TCOR_STAT_EOF;
		}
		if (!got_tcor_id) if (strstr (buffer, tcor_id_str)) got_tcor_id = 1;

	} while (strncmp (buffer, "DATA_UNTIL", 10) != 0);

	if (!got_tcor_id) {
		fclose (tcp->tcor_fp);
		return tcp->status = TCOR_STAT_NOID;
	}

	return read_tcor_rec (tcp);
}

static int interpol_tcor (struct tcor_struct *tcp, double scet, double *offsetp, double *diffp)
/* If scet is after TCOR end time, reads TCOR records until its not.
   Then checks that scet is after TCOR start time. If it is not return failure (1).
   Else do interpolation to get DIFF at scet, set DIFF and OFFSET, and return 0. */
{
	if (tcp->tcor_fp == NULL) return tcp->status = TCOR_STAT_NOFILE;
	if (feof (tcp->tcor_fp))  return tcp->status = TCOR_STAT_EOF;

	if (scet > tcp->scet_2) {
		while (scet > tcp->scet_2 && read_tcor_rec (tcp) == 0);
	}

	if (scet < tcp->scet_1) return tcp->status = TCOR_STAT_BEFORE;
	if (scet > tcp->scet_2) return tcp->status = TCOR_STAT_AFTER;

	*diffp = (tcp->diff_1 + (scet - tcp->scet_1) * (tcp->diff_2 - tcp->diff_1) / 
						                    (tcp->scet_2 - tcp->scet_1)) * 1.0e-6;
	*offsetp = tcp->offset * 1.0e-6;

	return tcp->status = 0;
}

static int open_index_file (struct tcor_struct *tcp, int scid)
/* Open the TCOR index file */
{
	char *tcor_path;
	char index_name[_MAX_PATH];

	/* Close any previously open index file */
	if (tcp->index_fp != NULL) {
		fclose (tcp->index_fp);
		tcp->index_fp = NULL;
	}

	/* Save the spacecraft ID */
	tcp->index_scid = scid;

	/* Get the path to the TCOR files */
	if (!tcp->path_specified) {
		tcor_path = getenv ("TCOR_PATH");
		if (tcor_path == NULL) return tcp->status = TCOR_STAT_NOPATH;
		strncpy (tcp->tcor_path, tcor_path, _MAX_PATH);
		tcp->path_specified = 1;
	}
	if (tcp->tcor_path[0] == 0) return tcp->status = TCOR_STAT_NOPATH;

	/* Open the index file */
	sprintf (index_name, "%s/TCOR_INDEX_%d.txt", tcp->tcor_path, scid);
	tcp->index_fp = fopen (index_name, "r");
	if (tcp->index_fp == NULL) return tcp->status = TCOR_STAT_NOINDEX;

	return tcp->status = 0;
}

static int search_index_file (struct tcor_struct *tcp, double scet)
/* Read TCOR index file until the first record that ends after the specified time,
   then opens the selected TCOR file. */
{
	char buffer[256];
	char scet_asc_1[32];
	char scet_asc_2[32];

	if (tcp->index_fp == NULL) return tcp->status = TCOR_STAT_NOINDEX;
	
	/* Close any open TCOR file */
	if (tcp->tcor_fp) {
		fclose (tcp->tcor_fp);
		tcp->tcor_fp = NULL;
	}
	tcp->tcor_name[0] = 0;

	while (fgets (buffer, 256, tcp->index_fp)) {
		scet_asc_2[0] = 0;
		sscanf (buffer, "%s %[^/]/%s", tcp->tcor_name, scet_asc_1, scet_asc_2);
		/* printf ("search_index: %s %s %s\n", tcp->tcor_name, scet_asc_1, scet_asc_2); */
		if ((double)(CCSDS_time (scet_asc_2)) > scet) return open_tcor_file (tcp);
		tcp->tcor_name[0] = 0;
	}

	return tcp->status = TCOR_STAT_ENDIDX;

}

char *hk_tcor_statstr (struct tcor_struct *tcp)
{
	switch (tcp->status) {
	case 0:					return "OK";
	case TCOR_STAT_NOPATH:  return "TCOR_STAT_NOPATH";     
	case TCOR_STAT_EOF:     return "TCOR_STAT_EOF";     
	case TCOR_STAT_FORMAT:  return "TCOR_STAT_FORMAT";  
	case TCOR_STAT_NOFILE:  return "TCOR_STAT_NOFILE";  
	case TCOR_STAT_NOINDEX: return "TCOR_STAT_NOINDEX"; 
	case TCOR_STAT_BEFORE:  return "TCOR_STAT_BEFORE";  
	case TCOR_STAT_AFTER:   return "TCOR_STAT_AFTER";   
	case TCOR_STAT_PRECOR:  return "TCOR_STAT_PRECOR";  
	case TCOR_STAT_ENDIDX:  return "TCOR_STAT_ENDIDX";  
	case TCOR_STAT_NOID:    return "TCOR_STAT_NOID";
	}
	return "UNKNOWN_ERROR";
}

void hk_tcor_init (struct tcor_struct *tcp)
{
	memset (tcp, 0, sizeof (struct tcor_struct));
}

void hk_tcor_path (struct tcor_struct *tcp, char *tcor_path)
{
	if (tcor_path != NULL) {
		strncpy (tcp->tcor_path, tcor_path, _MAX_PATH);
		tcp->path_specified = 1;
	}

	else {
		tcp->tcor_path[0] = 0;
		tcp->path_specified = 0;
	}
}

void hk_tcor_fini (struct tcor_struct *tcp)
{
	if (tcp->index_fp != NULL) {
		fclose (tcp->index_fp);
		tcp->index_fp = NULL;
	}
	if (tcp->tcor_fp != NULL) {
		fclose (tcp->tcor_fp);
		tcp->tcor_fp = NULL;
	}
}

void hk_tcor_show (struct tcor_struct *tcp, FILE *fp)
{
	fprintf (fp, "TCOR_STAT = %d (%s), scid = %d\n", tcp->status, hk_tcor_statstr (tcp), tcp->index_scid);
	fprintf (fp, "TCOR_PATH = %s\n", tcp->tcor_path);
	fprintf (fp, "TCOR_NAME = %s\n", tcp->tcor_name);
	fprintf (fp, "LAST_SCET = %s\n", CCSDS_str ((long)tcp->last_scet));
	fprintf (fp, "TCOR_REC1 = %s %9.6f %9.6f\n", CCSDS_str ((long)tcp->scet_1), tcp->offset, tcp->diff_1);
	fprintf (fp, "TCOR_REC2 = %s %9.6f %9.6f\n", CCSDS_str ((long)tcp->scet_2), tcp->offset, tcp->diff_2);
}

int hk_tcor_appl (struct tcor_struct *tcp, unsigned char *ddshead)
/* Gets the offset and diff values for specified scid and scet.
   Returns 0 on success, non-zero on failure. */
{
	int stat, scid;
	double scet, offset, diff;

	/* Check if time already corrected */
	if (ddshead[14] >= 64) return tcp->status = TCOR_STAT_PRECOR;

	/* Get spacecraft identifier and event time (scid and scet) */
	scid = ddshead[12] >> 4;
	GetScet (ddshead, &scet);
	offset = diff = 0.0;

	/* If change of spacecraft or earlier time, then re-open index file */
	if (scid != tcp->index_scid || scet < tcp->last_scet) {
		open_index_file (tcp, scid);
		if (tcp->status == 0) search_index_file (tcp, scet);
	}
	tcp->last_scet = scet;

	/* Check for fatal errors - no point in retrying */
	switch (tcp->status) {
	case TCOR_STAT_NOPATH:
	case TCOR_STAT_NOINDEX:
	case TCOR_STAT_ENDIDX:
		return tcp->status;
	}

	/* While index file valid, keep trying to read TCOR */
	tcp->status = 0;
	while (tcp->status == 0) {
		
		stat = interpol_tcor (tcp, scet, &offset, &diff);
		if (stat == 0) {
			scet += offset + diff;           /* Adjust SCET by offset and diff */
			PutScet (ddshead, scet);
			ddshead[14] += 64;               /* Flag change in 'Time Quality' */
			return 0;
		}

		if (stat == TCOR_STAT_BEFORE) return stat;

		search_index_file (tcp, scet);
	}

	return tcp->status;
}


	
