 
/**
 * $Id: ddsp.c,v 1.1 2001/09/26 14:06:48 simond Exp $
 */

#define  _ddsp_c
#include <stdio.h>
#include <ddsp.h>
#include <tedsys.h>
#include <ted.h>
#include <field_info.h>
#include <field.h>

static int c_show_head = TRUE;
static int c_show_body = FALSE;
static int c_debug_head = 0;
static int c_debug_body = 0;

/* {{{ STATUS */

/* {{{ field_info    ddsp_status_info[] = { */

field_info ddsp_status_info[] = {
DDSP_STATUS_OK,		"DDSP_STATUS_OK",
DDSP_STATUS_READ,	"DDSP_STATUS_READ",
DDSP_STATUS_WRITE,	"DDSP_STATUS_WRITE",
DDSP_STATUS_MEMORY,	"DDSP_STATUS_MEMORY",
DDSP_STATUS_TYPE,	"DDSP_STATUS_TYPE",
DDSP_STATUS_GROUND,	"DDSP_STATUS_GROUND",
DDSP_STATUS_SOURCE,	"DDSP_STATUS_SOURCE",
DDSP_STATUS_QUALITY,	"DDSP_STATUS_QUALITY",
DDSP_STATUS_ADID,	"DDSP_STATUS_ADID",
DDSP_STATUS_INIT,	"DDSP_STATUS_INIT",
DDSP_STATUS_UNDEFINED,	"DDSP_STATUS_UNDEFINED",
0,NULL	
};

/* }}} */
/* {{{ char *        ddsp_status_text(int) */

char *
ddsp_status_text(
int p_int
)
{
  extern field_info ddsp_status_info[];
  return get_desc(ddsp_status_info,p_int);
}

/* }}} */
/* {{{ int           ddsp_status_check(int) */

int
ddsp_status_check(
int p_int
)
{
  extern field_info ddsp_status_info[];
  return search_enumeration(ddsp_status_info,p_int);
}

/* }}} */

/* }}} */

/* {{{ unsigned char * ddsp_get_header(DDSP *) */

unsigned char *
ddsp_get_header(
DDSP *this
)
{
  return this->header;
}

/* }}} */

/* {{{ SCET */

/* {{{ char *          ddsp_day_text(DDSP_DAY) */

char * ddsp_day_text(DDSP_DAY p_day) {
  static char l_char[1024];
  sprintf(l_char,"<scet-day day=\"%ld\"/>",p_day);
  return l_char;
}

/* }}} */
/* {{{ int       check_ddsp_day(DDSP_DAY) */

int
check_ddsp_day(
DDSP_DAY p_day
)
{
  if (p_day > 0)
    return 1;
  return 0;
}

/* }}} */
/* {{{ DDSP_DAY        ddsp_get_day(DDSP *) */

DDSP_DAY
ddsp_get_day(
DDSP *this
)
{
  return (DDSP_DAY) field_get((unsigned char *)ddsp_get_header(this),0,16);
}

/* }}} */
/* {{{ void            ddsp_set_day(DDSP *,DDSP_DAY) */

void
ddsp_set_day(
DDSP *this,
DDSP_DAY p_day
)
{
  field_set((unsigned char *)ddsp_get_header(this),0,16,(unsigned long)p_day);
}

/* }}} */
/* {{{ char *          ddsp_ms_text(DDSP_MS) */

char *
ddsp_ms_text(
DDSP_MS p_ms
)
{
  static char l_char[1024];
  sprintf(l_char,"<scet-ms ms=\"%ld\"/>",p_ms);
  return l_char;
}

/* }}} */
/* {{{ int       check_ddsp_ms(DDSP_MS) */

int
check_ddsp_ms(
DDSP_MS p_ms
)
{
  return ((p_ms > 0) ? 1 : 0);
}

/* }}} */
/* {{{ DDSP_MS         ddsp_get_ms(DDSP *) */

DDSP_MS
ddsp_get_ms(
DDSP *this
)
{
  return (DDSP_MS) field_get((unsigned char *)ddsp_get_header(this),16,32);
}

/* }}} */
/* {{{ void            ddsp_set_ms(DDSP *,DDSP_MS) */

void
ddsp_set_ms(
DDSP *this,
DDSP_MS p_ms
)
{
  field_set((unsigned char *)ddsp_get_header(this),16,32,(unsigned long)p_ms);
}

/* }}} */
/* {{{ char *          ddsp_us_text(DDSP_US) */

char *
ddsp_us_text(
DDSP_US p_us
)
{
  static char l_char[1024];
  sprintf(l_char,"<scet-us uss=\"%ld\"/>",p_us);
  return l_char;
}

/* }}} */
/* {{{ int       check_ddsp_us(DDSP_US) */

int
check_ddsp_us(
DDSP_US p_us
)
{
  if (p_us > 0) return 1;
  return 0;
}

/* }}} */
/* {{{ DDSP_US         ddsp_get_us(DDSP *) */

DDSP_US
ddsp_get_us(
DDSP *this
)
{
  return (DDSP_US) field_get((unsigned char *)ddsp_get_header(this),48,16);
}

/* }}} */
/* {{{ void            ddsp_set_us(DDSP *,DDSP_US) */

void
ddsp_set_us(
DDSP *this,
DDSP_US p_us
)
{
  field_set((unsigned char *)ddsp_get_header(this),48,16,(unsigned long)p_us);
}

/* }}} */

/* {{{ DDSP_STATUS     scet_rationalize(SCET *) */

DDSP_STATUS
scet_rationalize(
SCET *this
)
{
  if (this == NULL) return (DDSP_STATUS_INIT);
  while (this->us < 0)   { this->ms -= 1; this->us += 1000; }
  while (this->us > 999) { this->ms += 1; this->us -= 1000; }
  while (this->ms < 0)                { this->day -= 1; this->ms += SCET_MS_PER_DAY; }
  while (this->ms > (SCET_MS_PER_DAY - 1)) { this->day += 1; this->ms -= SCET_MS_PER_DAY; }
}

/* }}} */
/* {{{ time_t *        ddsp_scet_get_unix(SCET *) */

time_t *ddsp_scet_get_unix(SCET *this)
{
  static time_t seconds;

  seconds = this->day * 86400;
  seconds += this->ms / 1000;

  seconds -= 378691200;	/* Epoch 1/1/70 */

  return &seconds;
}

/* }}} */
/* {{{ DDSP_STATUS     scet_set_unix     (SCET *,time_t *   ,long *,long *) */

DDSP_STATUS
scet_set_unix(
SCET *this,
time_t *p_time_t,
long   *p_ms,
long   *p_us
)
{
  time_t seconds;
  long ms = this->ms;
  long us = this->us;
  seconds  = this->day * 86400;
  while (us < 0) { ms -= 1; us += 1000; }
  while (ms < 0) { seconds -= 1; ms += 1000; }
  
  seconds += ms / 1000;
  seconds -= 378691200;	/* Epoch 1/1/70 */

  *p_time_t = seconds;
  *p_ms     = (ms - (1000 *(ms /1000)));
  *p_us     =  us;
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_get_unix     (SCET *,time_t     ,long  ,long  ) */

DDSP_STATUS
scet_get_unix(
SCET  *this,
time_t p_time_t,
long   p_ms,
long   p_us
)
{
  time_t seconds;
  seconds = p_time_t + 378691200;

  this->day = (seconds + (p_ms / 1000) + (p_us / 1000000)) / SCET_SEC_PER_DAY ;
  this->ms  = ((seconds * 1000) + (p_ms % 1000) + (p_us / 1000)) % SCET_MS_PER_DAY  ;
  this->us  =  p_us % 1000;
  scet_rationalize(this);
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_get_tm       (SCET *,struct tm  ,long  ,long  ) */

DDSP_STATUS
scet_get_tm(
SCET *this,
struct tm p_tm,
long p_ms,
long p_us
)
{
  time_t seconds;
  if (this == NULL) return (DDSP_STATUS_MEMORY);

  seconds  = mktime(&p_tm);
  seconds += 378691200;	/* Epoch 1/1/70 */

  this->day = seconds / (24 * 60 * 60);
  this->ms  = ((seconds / (24 * 60 * 60)) * 1000) + (p_ms % 1000);
  this->us  = (p_us % 1000);
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_set_tm       (SCET *,struct tm *,long *,long *) */

DDSP_STATUS
scet_set_tm(
SCET      * this,
struct tm * p_tm,
long      * p_ms,
long      * p_us
)
{
  time_t seconds;
  if (this == NULL) return (DDSP_STATUS_MEMORY);
  if (p_tm == NULL) return (DDSP_STATUS_MEMORY);
  if (p_ms == NULL) return (DDSP_STATUS_MEMORY);
  if (p_us == NULL) return (DDSP_STATUS_MEMORY);
  
  scet_set_unix(this,&seconds,p_ms,p_us);
  *p_tm = *gmtime(&seconds);
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_text         (SCET *,char [50]) */

DDSP_STATUS
scet_text(
SCET * this,
char p_char[50]
)
{
  time_t l_time_t;
  long ms;
  long us;
  struct tm tm ;
  if (this   == NULL) return (DDSP_STATUS_MEMORY);
  if (p_char == NULL) return (DDSP_STATUS_MEMORY);

  scet_set_unix(this,&l_time_t,&ms,&us);
  while (us < 0)   { ms -= 1; us += 1000; }
  while (us > 999) { ms += 1; us -= 1000; }
  while (ms < 0)   { l_time_t -= 1; ms += 1000; }
  while (ms > 999) { l_time_t += 1; ms -= 1000; }

  tm = *gmtime(&l_time_t);

  strftime(p_char, 50, "%Y-%m-%dT%H:%M:%S.", gmtime(&l_time_t));
  sprintf(p_char + strlen(p_char), "%03d%03dZ",ms,us);

  return (DDSP_STATUS_OK);
}

/* }}} */

/* {{{ char *          ddsp_scet_text    (SCET * ) */

char *ddsp_scet_text(SCET *this)
{
  static char string[50];
  scet_text(this,string);
  return string;
}

/* }}} */
/* {{{ DDSP_STATUS     scet_set          (SCET *  ,long,long,long) */

DDSP_STATUS
scet_set(
SCET *this,
long p_day,
long p_ms,
long p_us
)
{
  if (this == NULL) return (DDSP_STATUS_INIT);
  this->day = p_day;
  this->ms  = p_ms;
  this->us  = p_us;
  scet_rationalize(this);
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_cpy          (SCET *  ,SCET *) */

DDSP_STATUS
scet_cpy(
SCET *this,
SCET *that
)
{
  if (this == NULL) return (DDSP_STATUS_MEMORY);
  if (that == NULL) return (DDSP_STATUS_MEMORY);
  this->day = that->day;
  this->ms  = that->ms;
  this->us  = that->us;
  scet_rationalize(this);
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_init         (SCET **) */

DDSP_STATUS scet_init(SCET **p_this)
{
  if (p_this == NULL) return (DDSP_STATUS_MEMORY);
  {
    SCET *this = NULL;
    this = malloc(sizeof(SCET));
    if (this == NULL) {
		*p_this = NULL;
		return (DDSP_STATUS_INIT);
	}
    this->day = 0L;
    this->ms  = 0L;
    this->us  = 0L;
    *p_this = this;
    return (DDSP_STATUS_OK);
  }
}

/* }}} */
/* {{{ DDSP_STATUS     scet_clone        (SCET ** ,SCET *) */

DDSP_STATUS
scet_clone(
SCET **p_this,
SCET *that
)
{
  SCET *this = NULL;
  if (p_this == NULL) return (DDSP_STATUS_MEMORY);
  if (that   == NULL) return (DDSP_STATUS_MEMORY);
  scet_init(&this);
  scet_cpy(this,that);
  *p_this = this;
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_init_text    (SCET **,char *) */

DDSP_STATUS
scet_init_text(
SCET **p_pscet,
const char *p_char
)
{
  long Y,m,d,H,M,S;
  long ms,us;

  if (p_pscet == NULL) return (DDSP_STATUS_MEMORY);
  if (p_char  == NULL) return (DDSP_STATUS_MEMORY);

  if (sscanf(p_char,"%d-%d-%dT%d:%d:%d.%03d%03dZ",&Y,&m,&d,&H,&M,&S,&ms,&us) == 7)
    {
      char l_char[50];
      struct tm tm;
      SCET *l_scet = NULL;
      sprintf(l_char,"%d-%d-%dT%d:%d:%d",Y,m,d,H,M,S);
      strptime(l_char,"%Y-%m-%dT%H:%M:%S",&tm);
      scet_init(&l_scet);
      scet_get_tm(l_scet,tm,ms,us);
      *p_pscet = l_scet;
      return (DDSP_STATUS_OK);
    }
  return (DDSP_STATUS_INIT);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_final        (SCET **) */

DDSP_STATUS scet_final(SCET **p_this)
{
  if (p_this == NULL) return (DDSP_STATUS_MEMORY);
  {
    SCET *this = *p_this;
    if (this != NULL)
      free(this);
    *p_this = NULL;
    return (DDSP_STATUS_OK);
  }
}

/* }}} */
/* {{{ void            scet_show         (SCET * ,FILE *) */

void scet_show(SCET *this,FILE *p_FILE)
{

  if (this != NULL)
    if (p_FILE != NULL)
      {
	char l_char[50];
	scet_text(this,l_char);
	fprintf(p_FILE,
	      "\n<scet day=\"%011d\" ms=\"%011d\" us=\"%05d\" text=\"%s\"/>",
	      this->day,
	      this->ms,
	      this->us,
	      l_char);
      }
}

/* }}} */
/* {{{ DDSP_STATUS     scet_diff         (SCET * ,SCET *,SCET *) */

DDSP_STATUS
scet_diff(
SCET *a,
SCET *b,
SCET *d
)
{
  if (a == NULL) return (DDSP_STATUS_MEMORY);
  if (b == NULL) return (DDSP_STATUS_MEMORY);
  if (d == NULL) return (DDSP_STATUS_MEMORY);

  d->us  = a->us - b->us;
  d->ms  = a->ms  - b->ms;
  d->day = a->day - b->day;

  while (d->us >  1000) { d->ms++; d->us -= 1000; }
  while (d->us <  1000) { d->ms--; d->us += 1000; }

  while (d->ms >  (60 * 60 * 24 *  1000)) { d->day++; d->ms -= (60 * 60 * 24 * 1000); }
  while (d->ms <  (60 * 60 * 24 * -1000)) { d->day--; d->ms += (60 * 60 * 24 * 1000); }

  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ DDSP_STATUS     scet_add          (SCET * ,SCET *) */

DDSP_STATUS
scet_add(
SCET *a,
SCET *b
)
{
  if (a == NULL) return (DDSP_STATUS_MEMORY);
  if (b == NULL) return (DDSP_STATUS_MEMORY);

  a->us  += b->us;
  a->ms  += b->ms;
  a->day += b->day;
  scet_rationalize(a);
  return (DDSP_STATUS_OK);
}

/* }}} */

void scet_set_day(SCET *this,long p_long) { this->day = ((p_long > 0) ? p_long : 0) ; }
void scet_set_ms (SCET *this,long p_long) { this->ms  = ((p_long > 0) ? p_long : 0) ; }
void scet_set_us (SCET *this,long p_long) { this->us  = ((p_long > 0) ? p_long : 0) ; }

long scet_get_day(SCET *this) { return ((this->day > 0) ? this->day : 0) ; }
long scet_get_ms (SCET *this) { return ((this->ms  > 0) ? this->ms  : 0) ; }
long scet_get_us (SCET *this) { return ((this->us  > 0) ? this->us  : 0) ; }

/* {{{ long        scet_compare(SCET *,SCET *) */

long scet_compare(SCET *a,SCET *b)
{
  if ((a == NULL) && (b == NULL)) return  0;
  if (a == NULL) return -1;
  if (b == NULL) return  1;

  if (a->day != b->day) return (a->day - b->day);
  if (a->ms  != b->ms ) return (a->ms  - b->ms );
  if (a->us  != b->us ) return (a->us  - b->us );
  return 0L;
}

/* }}} */
/* {{{ DDSP_STATUS scet_get_ddsp(SCET *,DDSP *) */

DDSP_STATUS
scet_get_ddsp(
SCET *this,
DDSP *p_DDSP
)
{
  if (this   == NULL) return (DDSP_STATUS_MEMORY);
  if (p_DDSP == NULL) return (DDSP_STATUS_MEMORY);

  this->day = ddsp_get_day(p_DDSP);
  this->ms  = ddsp_get_ms (p_DDSP);
  this->us  = ddsp_get_us (p_DDSP);
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ int         ddsp_compare(DDSP *,DDSP *) */

int
ddsp_compare(
DDSP *a,
DDSP *b
)
{
  if (ddsp_get_day(a) != ddsp_get_day(b)) return (ddsp_get_day(a) - ddsp_get_day(b));
  if (ddsp_get_ms(a)  != ddsp_get_ms (b)) return (ddsp_get_ms(a)  - ddsp_get_ms(b));
  if (ddsp_get_us(a)  != ddsp_get_us (b)) return (ddsp_get_us(a)  - ddsp_get_us(b));
  return 0;
}

/* }}} */

/* }}} */
/* {{{ TRANSMISSION_MODE */

static field_info ddsp_transmission_mode_info[] = {
  DDSP_TRANSMISSION_REALTIME,		"RT",
  DDSP_TRANSMISSION_PLAYBACK,		"PB",
  DDSP_TRANSMISSION_RECALL,		"RE",
  DDSP_TRANSMISSION_RECALLPLAYBACK,	"RP",
  0,					NULL
};
/* {{{ char * ddsp_transmission_text(int) */

char *
ddsp_transmission_text(
int p_int
)
{
  return get_desc(ddsp_transmission_mode_info,p_int);
}

/* }}} */
/* {{{ int    ddsp_transmission_check(int) */

int
ddsp_transmission_check(
int p_int
)
{
  return search_enumeration(ddsp_transmission_mode_info,p_int);
}

/* }}} */

/* }}} */
/* {{{ VIRTUAL CHANNEL */

static field_info ddsp_vc_info[] = {
  DDSP_VC0,"VC0",
  DDSP_VC2,"VC2",
  DDSP_VC3,"VC3",
  0,       NULL
};

/* {{{ char * ddsp_vc_text(int) */

char *
ddsp_vc_text(
int p_int
)
{
  return get_desc(ddsp_vc_info,p_int);
}

/* }}} */
/* {{{ int    ddsp_vc_check(int) */

int
ddsp_vc_check(
int p_int
)
{
  return search_enumeration(ddsp_vc_info,p_int);
}

/* }}} */

/* }}} */
/* {{{ SOURCE */

/* {{{ field_info ddsp_source_info[] = { */

static field_info ddsp_source_info[] = {
  13, "HKWEC DES",
  35, "WEC NSD on Cluster 1",
  42, "WEC BSD on Cluster 1",
  49, "WEC HKD on Cluster 1",
  75, "WEC NSD on Cluster 2",
  82, "WEC BSD on Cluster 2",
  89, "WEC HKD on Cluster 2",
  115, "WEC NSD on Cluster 3",
  122, "WEC BSD on Cluster 3",
  129, "WEC HKD on Cluster 3",
  155, "WEC NSD on Cluster 4",
  162, "WEC BSD on Cluster 4",
  169, "WEC HKD on Cluster 4",
  0, NULL
};

/* }}} */
/* {{{ int             ddsp_source_type_spacecraft(int,int *,int *) */

static int ddsp_source_type_spacecraft(int source, int *data_type, int *spacecraft)
{
  switch (source) {
  case ddsp_source_wecnsd1: *data_type = DDSP_TYPE_NSD; *spacecraft = 1; break;
  case ddsp_source_wecbsd1: *data_type = DDSP_TYPE_BSD; *spacecraft = 1; break;
  case ddsp_source_wechkd1: *data_type = DDSP_TYPE_HKD; *spacecraft = 1; break;
  case ddsp_source_wecnsd2: *data_type = DDSP_TYPE_NSD; *spacecraft = 2; break;
  case ddsp_source_wecbsd2: *data_type = DDSP_TYPE_BSD; *spacecraft = 2; break;
  case ddsp_source_wechkd2: *data_type = DDSP_TYPE_HKD; *spacecraft = 2; break;
  case ddsp_source_wecnsd3: *data_type = DDSP_TYPE_NSD; *spacecraft = 3; break;
  case ddsp_source_wecbsd3: *data_type = DDSP_TYPE_BSD; *spacecraft = 3; break;
  case ddsp_source_wechkd3: *data_type = DDSP_TYPE_HKD; *spacecraft = 3; break;
  case ddsp_source_wecnsd4: *data_type = DDSP_TYPE_NSD; *spacecraft = 4; break;
  case ddsp_source_wecbsd4: *data_type = DDSP_TYPE_BSD; *spacecraft = 4; break;
  case ddsp_source_wechkd4: *data_type = DDSP_TYPE_HKD; *spacecraft = 4; break;
  default: return 1;
  }
  return 0;
}

/* }}} */
/* {{{ char *          ddsp_source_text(int) */

char *
ddsp_source_text(
int p_source
)
{
  static char l_char[1024];
  sprintf(l_char,
	  "%s",
	  get_desc(ddsp_source_info,p_source));
  return l_char;
}

/* }}} */
/* {{{ int             ddsp_source_check(int) */

int
ddsp_source_check(
int p_int
)
{
  switch(p_int)
    {
    case DDSP_ADID_ECLUD001:	 /* Cluster CD-ROM directory tree listing */
    case DDSP_ADID_ECLUD002:	 /* Cluster CD-ROM cumulative index file format */
    case DDSP_ADID_ECLUD003:	 /* Catalogue format for Cluster off-line delivery */
    case DDSP_ADID_ECLUD004:	 /* Catalogue format for Cluster on-line delivery */
    case DDSP_ADID_ECLUD005:	 /* Acknowledgement format for Cluster off-line delivery */
    case DDSP_ADID_ECLUD006:	 /* Volume production information format for Cluster off-line delivery */
    case DDSP_ADID_ECLUD007:	 /* Mast catalogue of all Cluster data available */
    case DDSP_ADID_ECLUA001:	 /* Long Term Orbit File */
    case DDSP_ADID_ECLUA002:	 /* Long Term Event File */
    case DDSP_ADID_ECLUA003:	 /* Short Term Orbit File */
    case DDSP_ADID_ECLUA004:	 /* Short Term Event File */
    case DDSP_ADID_ECLUA005:	 /* Spacecraft Altitude and Spin Rate Entry */
    case DDSP_ADID_ECLUA006:	 /* Time Calibration File Entry */
    case DDSP_ADID_ECLUA007:	 /* Command History File Entry */
    case DDSP_ADID_ECLUP101:	 /* Housekeeping Parameter Definition File for EDI on CLUSTER 1 */
    case DDSP_ADID_ECLUP102:	 /* Housekeeping Parameter Definition File for FGM on CLUSTER 1 */
    case DDSP_ADID_ECLUP103:	 /* Housekeeping Parameter Definition File for CIS on CLUSTER 1 */
    case DDSP_ADID_ECLUP104:	 /* Housekeeping Parameter Definition File for PEACE on CLUSTER 1 */
    case DDSP_ADID_ECLUP105:	 /* Housekeeping Parameter Definition File for RAPID on CLUSTER 1 */
    case DDSP_ADID_ECLUP106:	 /* Housekeeping Parameter Definition File for WEC on CLUSTER 1 */
    case DDSP_ADID_ECLUP107:	 /* Housekeeping Parameter Definition File for ASPOC on CLUSTER 1 */
    case DDSP_ADID_ECLUP108:	 /* Housekeeping Parameter Definition File for SC on CLUSTER 1 */
    case DDSP_ADID_ECLUP201:	 /* Housekeeping Parameter Definition File for EDI on CLUSTER 2 */
    case DDSP_ADID_ECLUP202:	 /* Housekeeping Parameter Definition File for FGM on CLUSTER 2 */
    case DDSP_ADID_ECLUP203:	 /* Housekeeping Parameter Definition File for CIS on CLUSTER 2 */
    case DDSP_ADID_ECLUP204:	 /* Housekeeping Parameter Definition File for PEACE on CLUSTER 2 */
    case DDSP_ADID_ECLUP205:	 /* Housekeeping Parameter Definition File for RAPID on CLUSTER 2 */
    case DDSP_ADID_ECLUP206:	 /* Housekeeping Parameter Definition File for WEC on CLUSTER 2 */
    case DDSP_ADID_ECLUP207:	 /* Housekeeping Parameter Definition File for ASPOC on CLUSTER 2 */
    case DDSP_ADID_ECLUP208:	 /* Housekeeping Parameter Definition File for SC on CLUSTER 2 */
    case DDSP_ADID_ECLUP301:	 /* Housekeeping Parameter Definition File for EDI on CLUSTER 3 */
    case DDSP_ADID_ECLUP302:	 /* Housekeeping Parameter Definition File for FGM on CLUSTER 3 */
    case DDSP_ADID_ECLUP303:	 /* Housekeeping Parameter Definition File for CIS on CLUSTER 3 */
    case DDSP_ADID_ECLUP304:	 /* Housekeeping Parameter Definition File for PEACE on CLUSTER 3 */
    case DDSP_ADID_ECLUP305:	 /* Housekeeping Parameter Definition File for RAPID on CLUSTER 3 */
    case DDSP_ADID_ECLUP306:	 /* Housekeeping Parameter Definition File for WEC on CLUSTER 3 */
    case DDSP_ADID_ECLUP307:	 /* Housekeeping Parameter Definition File for ASPOC on CLUSTER 3 */
    case DDSP_ADID_ECLUP308:	 /* Housekeeping Parameter Definition File for SC on CLUSTER 3 */
    case DDSP_ADID_ECLUP401:	 /* Housekeeping Parameter Definition File for EDI on CLUSTER 4 */
    case DDSP_ADID_ECLUP402:	 /* Housekeeping Parameter Definition File for FGM on CLUSTER 4 */
    case DDSP_ADID_ECLUP403:	 /* Housekeeping Parameter Definition File for CIS on CLUSTER 4 */
    case DDSP_ADID_ECLUP404:	 /* Housekeeping Parameter Definition File for PEACE on CLUSTER 4 */
    case DDSP_ADID_ECLUP405:	 /* Housekeeping Parameter Definition File for RAPID on CLUSTER 4 */
    case DDSP_ADID_ECLUP406:	 /* Housekeeping Parameter Definition File for WEC on CLUSTER 4 */
    case DDSP_ADID_ECLUP407:	 /* Housekeeping Parameter Definition File for ASPOC on CLUSTER 4 */
    case DDSP_ADID_ECLUP408:	 /* Housekeeping Parameter Definition File for SC on CLUSTER 4 */
    case DDSP_ADID_ECLUN101:	 /* Normal science data for EDI on Cluster 1 */
    case DDSP_ADID_ECLUN102:	 /* Normal science data for FGM on Cluster 1 */
    case DDSP_ADID_ECLUN103:	 /* Normal science data for CIS on Cluster 1 */
    case DDSP_ADID_ECLUN104:	 /* Normal science data for PEACE on Cluster 1 */
    case DDSP_ADID_ECLUN105:	 /* Normal science data for RAPID on Cluster 1 */
    case DDSP_ADID_ECLUN106:	 /* Normal science data for WEC on Cluster 1 */
    case DDSP_ADID_ECLUB101:	 /* Burst science data for EDI on Cluster 1 */
    case DDSP_ADID_ECLUB102:	 /* Burst science data for FGM on Cluster 1 */
    case DDSP_ADID_ECLUB103:	 /* Burst science data for CIS on Cluster 1 */
    case DDSP_ADID_ECLUB104:	 /* Burst science data for PEACE on Cluster 1 */
    case DDSP_ADID_ECLUB105:	 /* Burst science data for RAPID on Cluster 1 */
    case DDSP_ADID_ECLUB106:	 /* Burst science data for WEC on Cluster 1 */
    case DDSP_ADID_ECLUH101:	 /* Housekeeping data for EDI on Cluster 1 */
    case DDSP_ADID_ECLUH102:	 /* Housekeeping data for FGM on Cluster 1 */
    case DDSP_ADID_ECLUH103:	 /* Housekeeping data for CIS on Cluster 1 */
    case DDSP_ADID_ECLUH104:	 /* Housekeeping data for PEACE on Cluster 1 */
    case DDSP_ADID_ECLUH105:	 /* Housekeeping data for RAPID on Cluster 1 */
    case DDSP_ADID_ECLUH106:	 /* Housekeeping data for WEC on Cluster 1 */
    case DDSP_ADID_ECLUH107:	 /* Housekeeping data for ASPOC on Cluster 1 */
    case DDSP_ADID_ECLUH108:	 /* Housekeeping data for Cluster 1 Platform */
    case DDSP_ADID_ECLUN201:	 /* Normal science data for EDI on Cluster 2 */
    case DDSP_ADID_ECLUN202:	 /* Normal science data for FGM on Cluster 2 */
    case DDSP_ADID_ECLUN203:	 /* Normal science data for CIS on Cluster 2 */
    case DDSP_ADID_ECLUN204:	 /* Normal science data for PEACE on Cluster 2 */
    case DDSP_ADID_ECLUN205:	 /* Normal science data for RAPID on Cluster 2 */
    case DDSP_ADID_ECLUN206:	 /* Normal science data for WEC on Cluster 2 */
    case DDSP_ADID_ECLUB201:	 /* Burst science data for EDI on Cluster 2 */
    case DDSP_ADID_ECLUB202:	 /* Burst science data for FGM on Cluster 2 */
    case DDSP_ADID_ECLUB203:	 /* Burst science data for CIS on Cluster 2 */
    case DDSP_ADID_ECLUB204:	 /* Burst science data for PEACE on Cluster 2 */
    case DDSP_ADID_ECLUB205:	 /* Burst science data for RAPID on Cluster 2 */
    case DDSP_ADID_ECLUB206:	 /* Burst science data for WEC on Cluster 2 */
    case DDSP_ADID_ECLUH201:	 /* Housekeeping data for EDI on Cluster 2 */
    case DDSP_ADID_ECLUH202:	 /* Housekeeping data for FGM on Cluster 2 */
    case DDSP_ADID_ECLUH203:	 /* Housekeeping data for CIS on Cluster 2 */
    case DDSP_ADID_ECLUH204:	 /* Housekeeping data for PEACE on Cluster 2 */
    case DDSP_ADID_ECLUH205:	 /* Housekeeping data for RAPID on Cluster 2 */
    case DDSP_ADID_ECLUH206:	 /* Housekeeping data for WEC on Cluster 2 */
    case DDSP_ADID_ECLUH207:	 /* Housekeeping data for ASPOC on Cluster 2 */
    case DDSP_ADID_ECLUH208:	 /* Housekeeping data for Cluster 2 Platform */
    case DDSP_ADID_ECLUN301:	 /* Normal science data for EDI on Cluster 3 */
    case DDSP_ADID_ECLUN302:	 /* Normal science data for FGM on Cluster 3 */
    case DDSP_ADID_ECLUN303:	 /* Normal science data for CIS on Cluster 3 */
    case DDSP_ADID_ECLUN304:	 /* Normal science data for PEACE on Cluster 3 */
    case DDSP_ADID_ECLUN305:	 /* Normal science data for RAPID on Cluster 3 */
    case DDSP_ADID_ECLUN306:	 /* Normal science data for WEC on Cluster 3 */
    case DDSP_ADID_ECLUB301:	 /* Burst science data for EDI on Cluster 3 */
    case DDSP_ADID_ECLUB302:	 /* Burst science data for FGM on Cluster 3 */
    case DDSP_ADID_ECLUB303:	 /* Burst science data for CIS on Cluster 3 */
    case DDSP_ADID_ECLUB304:	 /* Burst science data for PEACE on Cluster 3 */
    case DDSP_ADID_ECLUB305:	 /* Burst science data for RAPID on Cluster 3 */
    case DDSP_ADID_ECLUB306:	 /* Burst science data for WEC on Cluster 3 */
    case DDSP_ADID_ECLUH301:	 /* Housekeeping data for EDI on Cluster 3 */
    case DDSP_ADID_ECLUH302:	 /* Housekeeping data for FGM on Cluster 3 */
    case DDSP_ADID_ECLUH303:	 /* Housekeeping data for CIS on Cluster 3 */
    case DDSP_ADID_ECLUH304:	 /* Housekeeping data for PEACE on Cluster 3 */
    case DDSP_ADID_ECLUH305:	 /* Housekeeping data for RAPID on Cluster 3 */
    case DDSP_ADID_ECLUH306:	 /* Housekeeping data for WEC on Cluster 3 */
    case DDSP_ADID_ECLUH307:	 /* Housekeeping data for ASPOC on Cluster 3 */
    case DDSP_ADID_ECLUH308:	 /* Housekeeping data for Cluster 3 Platform */
    case DDSP_ADID_ECLUN401:	 /* Normal science data for EDI on Cluster 4 */
    case DDSP_ADID_ECLUN402:	 /* Normal science data for FGM on Cluster 4 */
    case DDSP_ADID_ECLUN403:	 /* Normal science data for CIS on Cluster 4 */
    case DDSP_ADID_ECLUN404:	 /* Normal science data for PEACE on Cluster 4 */
    case DDSP_ADID_ECLUN405:	 /* Normal science data for RAPID on Cluster 4 */
    case DDSP_ADID_ECLUN406:	 /* Normal science data for WEC on Cluster 4 */
    case DDSP_ADID_ECLUB401:	 /* Burst science data for EDI on Cluster 4 */
    case DDSP_ADID_ECLUB402:	 /* Burst science data for FGM on Cluster 4 */
    case DDSP_ADID_ECLUB403:	 /* Burst science data for CIS on Cluster 4 */
    case DDSP_ADID_ECLUB404:	 /* Burst science data for PEACE on Cluster 4 */
    case DDSP_ADID_ECLUB405:	 /* Burst science data for RAPID on Cluster 4 */
    case DDSP_ADID_ECLUB406:	 /* Burst science data for WEC on Cluster 4 */
    case DDSP_ADID_ECLUH401:	 /* Housekeeping data for EDI on Cluster 4 */
    case DDSP_ADID_ECLUH402:	 /* Housekeeping data for FGM on Cluster 4 */
    case DDSP_ADID_ECLUH403:	 /* Housekeeping data for CIS on Cluster 4 */
    case DDSP_ADID_ECLUH404:	 /* Housekeeping data for PEACE on Cluster 4 */
    case DDSP_ADID_ECLUH405:	 /* Housekeeping data for RAPID on Cluster 4 */
    case DDSP_ADID_ECLUH406:	 /* Housekeeping data for WEC on Cluster 4 */
    case DDSP_ADID_ECLUH407:	 /* Housekeeping data for ASPOC on Cluster 4 */
    case DDSP_ADID_ECLUH408:	 /* Housekeeping data for Cluster 4 Platform */
      return DDSP_STATUS_OK;
    default:
      return DDSP_STATUS_ADID;
    }
}

/* }}} */
/* {{{ DDSP_SOURCE     ddsp_get_source(DDSP *) */

DDSP_SOURCE
ddsp_get_source(
DDSP *this
)
{
  return (DDSP_SOURCE) field_get((unsigned char *)ddsp_get_header(this),64,8);
}

/* }}} */
/* {{{ void            ddsp_set_source(DDSP *,DDSP_SOURCE) */

void
ddsp_set_source(
DDSP *this,
DDSP_SOURCE p_source
)
{
  field_set((unsigned char *)ddsp_get_header(this),64,8,(unsigned long)p_source);
}

/* }}} */

/* }}} */
/* {{{ TYPE */

static field_info ddsp_type_info[] = {
  DDSP_TYPE_NSD,"NSD",
  DDSP_TYPE_BSD,"BSD",
  DDSP_TYPE_HKD,"HKD",
  0            ,NULL
};

/* {{{ char *    ddsp_type_text(int) */

char *
ddsp_type_text(
int p_int
)
{
  return get_desc(ddsp_type_info,p_int);
}

/* }}} */
/* {{{ int       ddsp_type_check(int) */

int
ddsp_type_check(
int p_int
)
{
  return search_enumeration(ddsp_type_info,p_int);
}

/* }}} */
/* {{{ DDSP_TYPE ddsp_get_type(DDSP *) */

DDSP_TYPE
ddsp_get_type(
DDSP *this
)
{
  switch (ddsp_get_source(this))
    {
    case DDSP_ADID_ECLUN101:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN102:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN103:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN104:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN105:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN106:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUB101:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB102:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB103:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB104:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB105:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB106:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUH101:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH102:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH103:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH104:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH105:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH106:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH107:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH108:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUN201:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN202:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN203:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN204:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN205:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN206:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUB201:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB202:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB203:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB204:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB205:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB206:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUH201:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH202:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH203:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH204:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH205:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH206:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH207:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH208:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUN301:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN302:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN303:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN304:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN305:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN306:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUB301:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB302:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB303:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB304:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB305:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB306:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUH301:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH302:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH303:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH304:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH305:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH306:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH307:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH308:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUN401:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN402:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN403:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN404:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN405:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUN406:	return (DDSP_TYPE_NSD);
    case DDSP_ADID_ECLUB401:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB402:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB403:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB404:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB405:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUB406:	return (DDSP_TYPE_BSD);
    case DDSP_ADID_ECLUH401:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH402:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH403:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH404:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH405:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH406:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH407:	return (DDSP_TYPE_HKD);
    case DDSP_ADID_ECLUH408:	return (DDSP_TYPE_HKD);
    default:
      return (DDSP_TYPE_UNK);
    }
}

/* }}} */

/* }}} */
/* {{{ LENGTH */

/* {{{ char *          ddsp_length_text(DDSP_LENGTH) */

char *
ddsp_length_text(
DDSP_LENGTH p_length
)
{
  static char l_char[1024];
  sprintf(l_char,"%u",(unsigned long) p_length);
  return l_char;
}

/* }}} */
/* {{{ int             ddsp_length_check(int) */

int
ddsp_length_check(
int p_int
)
{
  if ((p_int >= 0) && (p_int <= 0xffffff))
    return 1;
  return 0;
}

/* }}} */
/* {{{ DDSP_LENGTH     ddsp_get_length(DDSP *) */

DDSP_LENGTH
ddsp_get_length(
DDSP *this
)
{
  return (DDSP_LENGTH) field_get((unsigned char *)ddsp_get_header(this),72,24);
}

/* }}} */
/* {{{ void            ddsp_set_length(DDSP *,DDSP_LENGTH) */

void
ddsp_set_length(
DDSP *this,
DDSP_LENGTH p_length
)
{
  field_set((unsigned char *)ddsp_get_header(this),72,24,(unsigned long)p_length);
}

/* }}} */

/* }}} */
/* {{{ SPACECRAFT */

/* {{{ field_info      ddsp_spacecraft_info[] = { */

field_info ddsp_spacecraft_info[] = {
  1, "CLUSTER_1",
  2, "CLUSTER_2",
  3, "CLUSTER_3",
  4, "CLUSTER_4",
  0, "X",
  0, NULL
};

/* }}} */
/* {{{ char *          ddsp_spacecraft_text(int) */

char *
ddsp_spacecraft_text(
int p_int
)
{
  extern field_info ddsp_spacecraft_info[];
  return get_desc  (ddsp_spacecraft_info,p_int);
}

/* }}} */
/* {{{ int             ddsp_spacecraft_check(int) */

int
ddsp_spacecraft_check(
int p_int
)
{
  extern field_info ddsp_spacecraft_info[];
  return search_enumeration(ddsp_spacecraft_info,p_int);
}

/* }}} */
/* {{{ DDSP_SPACECRAFT ddsp_get_spacecraft(DDSP *) */

DDSP_SPACECRAFT
ddsp_get_spacecraft(
DDSP *this
)
{
  return (DDSP_SPACECRAFT) field_get((unsigned char *)ddsp_get_header(this),96,4);
}

/* }}} */
/* {{{ void            ddsp_set_spacecraft(DDSP *,DDSP_SPACECRAFT) */

void
ddsp_set_spacecraft(
DDSP *this,
DDSP_SPACECRAFT p_spacecraft
)
{
  field_set((unsigned char *)ddsp_get_header(this),96,4,(unsigned long)p_spacecraft);
}

/* }}} */

/* }}} */
/* {{{ GROUND */

/* {{{ field_info info_ddsp_ground[] = { */

field_info ddsp_ground_info[] = {
  1, "Odenwald",
  2, "Redu",
  3, "Kourou",
  4, "Perth",
  5, "Malindi",
  6, "Canberra",
  7, "Goldstone",
  15, "N/A",
  0, NULL
};

/* }}} */
/* {{{ char *          ddsp_ground_text(int) */

char *
ddsp_ground_text(
int p_int
)
{
  extern field_info ddsp_ground_info[];
  return get_desc(ddsp_ground_info,p_int);
}

/* }}} */
/* {{{ int             ddsp_ground_check(int) */

int
ddsp_ground_check(
int p_int
)
{
  extern field_info ddsp_ground_info[];
  return search_enumeration(ddsp_ground_info,p_int);
}

/* }}} */
/* {{{ DDSP_GROUND     ddsp_get_ground(DDSP *) */

DDSP_GROUND
ddsp_get_ground(
DDSP *this
)
{
  return (DDSP_GROUND) field_get((unsigned char *)ddsp_get_header(this),100,4);
}

/* }}} */
/* {{{ void            ddsp_set_ground(DDSP *,DDSP_GROUND) */

void
ddsp_set_ground(
DDSP *this,
DDSP_GROUND p_ground
)
{
  field_set((unsigned char *)ddsp_get_header(this),100,4,(unsigned long)p_ground);
}

/* }}} */

/* }}} */
/* {{{ STREAM */

/* {{{ field_info ddsp_stream_info[] = { */

field_info ddsp_stream_info[] = {
  0, "RT VC0",
  2, "RT VC2",
  3, "RT VC3",
  64, "PB VC0",
  66, "PB VC2",
  67, "PB VC3",
  240, "RE VC0",
  242, "RE VC2",
  243, "RE VC3",
  79, "RP VC0",
  226, "RP VC2",
  227, "RP VC3",
  255, "N/A",
  0, NULL
};

/* }}} */
/* {{{ char *          ddsp_stream_text(DDSP_STREAM) */

char *
ddsp_stream_text(
DDSP_STREAM p_stream
)
{
  static char l_char[1024];
  sprintf(l_char,
	  "%s",
	  get_desc(ddsp_stream_info,p_stream));
  return l_char;
}

/* }}} */
/* {{{ int             ddsp_stream_check(int) */

int
check_ddsp_stream(
int p_int
)
{
  switch (p_int)
    {
    case DDSP_STREAM_RTVC0:
    case DDSP_STREAM_RTVC2:
    case DDSP_STREAM_RTVC3:
    case DDSP_STREAM_PBVC0:
    case DDSP_STREAM_PBVC2:
    case DDSP_STREAM_PBVC3:
    case DDSP_STREAM_REVC0:
    case DDSP_STREAM_REVC2:
    case DDSP_STREAM_REVC3:
    case DDSP_STREAM_RPVC0:
    case DDSP_STREAM_RPVC2:
    case DDSP_STREAM_RPVC3:
      return 1;
    default:
      return 0;
    }
}

/* }}} */
/* {{{ DDSP_STREAM     ddsp_get_stream(DDSP *) */

DDSP_STREAM
ddsp_get_stream(
DDSP *this
)
{
  return (DDSP_STREAM) field_get((unsigned char *)ddsp_get_header(this),104,8);
}

/* }}} */
/* {{{ void            ddsp_set_stream(DDSP *,DDSP_STREAM) */

void
ddsp_set_stream(
DDSP *this,
DDSP_STREAM p_stream
)
{
  field_set((unsigned char *)ddsp_get_header(this),104,8,(unsigned long)p_stream);
}

/* }}} */
/* {{{ int             ddsp_stream_mode_channel(int,int *,int *) */

int ddsp_stream_mode_channel(int stream, int *mode, int *channel)
{
  switch (stream) {
  case DDSP_STREAM_RTVC0: *mode = DDSP_TRANSMISSION_REALTIME;       *channel = DDSP_VC0; break;
  case DDSP_STREAM_RTVC2: *mode = DDSP_TRANSMISSION_REALTIME;       *channel = DDSP_VC2; break;
  case DDSP_STREAM_RTVC3: *mode = DDSP_TRANSMISSION_REALTIME;       *channel = DDSP_VC3; break;
  case DDSP_STREAM_PBVC0: *mode = DDSP_TRANSMISSION_PLAYBACK;       *channel = DDSP_VC0; break;
  case DDSP_STREAM_PBVC2: *mode = DDSP_TRANSMISSION_PLAYBACK;       *channel = DDSP_VC2; break;
  case DDSP_STREAM_PBVC3: *mode = DDSP_TRANSMISSION_PLAYBACK;       *channel = DDSP_VC3; break;
  case DDSP_STREAM_REVC0: *mode = DDSP_TRANSMISSION_RECALL;         *channel = DDSP_VC0; break;
  case DDSP_STREAM_REVC2: *mode = DDSP_TRANSMISSION_RECALL;         *channel = DDSP_VC2; break;
  case DDSP_STREAM_REVC3: *mode = DDSP_TRANSMISSION_RECALL;         *channel = DDSP_VC3; break;
  case DDSP_STREAM_RPVC0: *mode = DDSP_TRANSMISSION_RECALLPLAYBACK; *channel = DDSP_VC0; break;
  case DDSP_STREAM_RPVC2: *mode = DDSP_TRANSMISSION_RECALLPLAYBACK; *channel = DDSP_VC2; break;
  case DDSP_STREAM_RPVC3: *mode = DDSP_TRANSMISSION_RECALLPLAYBACK; *channel = DDSP_VC3; break;
  default: return 1;
  }
  return 0;
}

/* }}} */

/* }}} */
/* {{{ QUALITY */

/* {{{ field_info ddsp_quality_info[] = { */

field_info ddsp_quality_info[] = {
  0, "actual time",
  1, "extrapolated time",
  2, "contingency time",
  0, NULL
};

/* }}} */
/* {{{ char *          ddsp_quality_text(DDSP_QUALITY) */

char *
ddsp_quality_text(
DDSP_QUALITY p_quality
)
{
  static char l_char[1024];
  sprintf(l_char,
	  "%s",
	  get_desc(ddsp_quality_info,p_quality));
  return l_char;
}

/* }}} */
/* {{{ int             ddsp_quality_check(int) */

int
check_ddsp_quality(
int p_int
)
{
  switch (p_int)
    {
    case 0:
    case 1:
    case 2:
      return 1;
    default:
      return 0;
    }
}

/* }}} */
/* {{{ unsigned int    ddsp_get_quality(DDSP *) */

unsigned int
ddsp_get_quality(
DDSP *this
)
{
  return (unsigned int) field_get((unsigned char *)ddsp_get_header(this),112,4);
}

/* }}} */
/* {{{ void            ddsp_set_quality(DDSP *,DDSP_QUALITY) */

void
ddsp_set_quality(
DDSP *this,
DDSP_QUALITY p_quality
)
{
  field_set((unsigned char *)ddsp_get_header(this),112,4,(unsigned long)p_quality);
}

/* }}} */

/* }}} */
/* {{{ SEQUENCE */

/* {{{ field_info ddsp_sequence_info[] = { */

field_info ddsp_sequence_info[] = {
  0x1, "ASID_1",
  0x2, "ASID_2",
  0x3, "ASID_3",
  0x4, "ASID_4",
  0x5, "ASID_5",
  0x6, "ASID_6",
  0x7, "ASID_7",
  0x8, "ASID_8",
  0x9, "ASID_9",
  0xa, "ASID_A",
  0xb, "ASID_B",
  0xc, "ASID_C",
  0xd, "ASID_D",
  0xe, "ASID_E",
  0xf, "ASID_F",
  0x0, NULL
};

/* }}} */
/* {{{ char *          ddsp_sequence_text(DDSP_SEQUENCE) */

char *
ddsp_sequence_text(
DDSP_SEQUENCE p_sequence
)
{
  static char l_char[1024];
  sprintf(l_char,"%s",get_desc(ddsp_sequence_info,p_sequence));
  return l_char;
}

/* }}} */
/* {{{ int             ddsp_sequence_check(unsigned int) */

int
ddsp_sequence_check(
unsigned int p_uint
)
{
  if ((p_uint > 0) && (p_uint <= 0xf)) return 1;
  return 0;
}

/* }}} */
/* {{{ unsigned int    ddsp_get_sequence(DDSP *) */

unsigned int
ddsp_get_sequence(
DDSP *this
)
{
  return field_get((unsigned char *)ddsp_get_header(this),116,4);
}

/* }}} */
/* {{{ void            ddsp_set_sequence(DDSP *,DDSP_SEQUENCE) */

void
ddsp_set_sequence(
DDSP *this,
DDSP_SEQUENCE p_sequence
)
{
  field_set((unsigned char *)ddsp_get_header(this),116,4,(unsigned long)p_sequence);
}

/* }}} */

/* }}} */

/* DATA */
/* {{{ unsigned char * ddsp_get_data(DDSP *) */

unsigned char *
ddsp_get_data(
DDSP *this
)
{
  if (this == NULL) return NULL;
  return this->data;
}

/* }}} */
/* {{{ void            ddsp_set_data(DDSP *,DDSP_DATA,DDSP_LENGTH) */

void
ddsp_set_data(
DDSP *this,
DDSP_DATA p_data,
DDSP_LENGTH p_length
)
{
  /* return this->data; */
}

/* }}} */

/* {{{ int             ddsp_header_get_length(DDSP *) */

int
ddsp_header_get_length(
DDSP *this
)
{
  unsigned char *p = this->header;
  return (p[11]& 0xff) + (0x100 * (p[10] & 0xff)) + (0x10000 * (p[9] & 0xff));
}

/* }}} */
/* {{{ void            ddsp_header_set_length(DDSP *,unsigned int) */

void
ddsp_header_set_length(
DDSP *this,
unsigned int p_uint
)
{
  unsigned char *p = this->header;

   p[11] = (p_uint & 0xff);
   p[10] = ((p_uint / 0x100) & 0xff);
   p[9]  = ((p_uint / 0x10000) & 0xff);
}

/* }}} */

/* {{{ int             ddsp_get_show         (DDSP *,int *,int *) */

int
ddsp_get_show(
DDSP *this,
int *p_head,
int *p_body
)
{
  if (p_head == NULL) return (DDSP_STATUS_MEMORY);
  if (p_body == NULL) return (DDSP_STATUS_MEMORY);
  if (this  != NULL)
    {
      *p_head = this->show_head;
      *p_body = this->show_body;
    }
  else
    {
      *p_head = c_show_head;
      *p_body = c_show_body;
    }
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ int             ddsp_set_show         (DDSP *,int  ,int  ) */

int
ddsp_set_show(
DDSP *this,
int p_head,
int p_body
)
{
  if (this  != NULL)
    {
      this->show_head = p_head;
      this->show_body - p_body;
    }
  else
    {
      c_show_head = p_head;
      c_show_body = p_body;
    }
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ int             ddsp_get_debug         (DDSP *,int *,int *) */

int
ddsp_get_debug(
DDSP *this,
int *p_head,
int *p_body
)
{
  if (p_head == NULL) return (DDSP_STATUS_MEMORY);
  if (p_body == NULL) return (DDSP_STATUS_MEMORY);
  if (this  != NULL)
    {
      *p_head = this->debug_head;
      *p_body = this->debug_body;
    }
  else
    {
      *p_head = c_debug_head;
      *p_body = c_debug_body;
    }
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ int             ddsp_set_debug         (DDSP *,int  ,int  ) */

int
ddsp_set_debug(
DDSP *this,
int p_head,
int p_body
)
{
  if (this  != NULL)
    {
      this->debug_head = p_head;
      this->debug_body - p_body;
    }
  else
    {
      c_debug_head = p_head;
      c_debug_body = p_body;
    }
  return (DDSP_STATUS_OK);
}

/* }}} */

/* {{{ int             ddsp_write_header     (DDSP *,FILE *) */

int
ddsp_write_header(
DDSP *this,
FILE *p_FILE
)
{
  long l_tell = ftell(p_FILE);
  if (fwrite(this->header,1,DDSP_HEADER_SIZE,p_FILE) == DDSP_HEADER_SIZE)
    return DDSP_STATUS_OK;
  fseek(p_FILE,l_tell,SEEK_SET);
  return DDSP_STATUS_WRITE;
}

/* }}} */
/* {{{ int             ddsp_write_data       (DDSP *,FILE *) */

int
ddsp_write_data(
DDSP *this,
FILE *p_FILE
)
{
  long l_tell = ftell(p_FILE);
  if (fwrite(this->data,1,ddsp_header_get_length(this),p_FILE) == ddsp_header_get_length(this))
    return DDSP_STATUS_OK;
  fseek(p_FILE,l_tell,SEEK_SET);
  return DDSP_STATUS_WRITE;
}

/* }}} */
/* {{{ int             ddsp_write            (DDSP *,FILE *) */

int ddsp_write(DDSP *this,FILE *p_FILE)
{
  long l_tell = ftell(p_FILE);
  if (ddsp_write_header(this,p_FILE) == DDSP_STATUS_OK)
    {
      if (ddsp_write_data(this,p_FILE) == DDSP_STATUS_OK)
	return DDSP_STATUS_OK;
    }
  fseek(p_FILE,l_tell,SEEK_SET);
  return DDSP_STATUS_WRITE;
}

/* }}} */
/* {{{ int             ddsp_read_header      (DDSP *,FILE *) */

int
ddsp_read_header(
DDSP *this,
FILE *p_FILE
)
{
  long l_tell = ftell(p_FILE);

  if (fread(this->header,1,DDSP_HEADER_SIZE,p_FILE) == DDSP_HEADER_SIZE)
    return DDSP_STATUS_OK;

  /* fseek(p_FILE,l_tell,SEEK_SET); */
  return DDSP_STATUS_READ;
}

/* }}} */
/* {{{ int             ddsp_read_data        (DDSP *,FILE *) */

int
ddsp_read_data(
DDSP *this,
FILE *p_FILE
)
{
  long l_tell = ftell(p_FILE);
  unsigned char *l_data;
  if (this->local == TRUE)
    {
      if (this->size < ddsp_header_get_length(this))
	{
	  l_data = (unsigned char *) malloc(ddsp_header_get_length(this));
	  if (l_data == NULL) return DDSP_STATUS_MEMORY;
	  if (this->data != NULL) free(this->data);
	  this->data = l_data;
	  this->size = ddsp_header_get_length(this);
	  this->len = 0;
	}
    }
  if (this->local == FALSE)
    {
      l_data = (unsigned char *) malloc(ddsp_header_get_length(this));
      if (l_data == NULL) return DDSP_STATUS_MEMORY;
      this->data = l_data;
      this->size = ddsp_header_get_length(this);
      this->len = 0;
      this->local = TRUE;
    }
  
  if (fread(this->data,1,ddsp_header_get_length(this),p_FILE) != ddsp_header_get_length(this))
    {
      /* fseek(p_FILE,l_tell,SEEK_SET); */
      return DDSP_STATUS_READ;
    }
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ int             ddsp_read             (DDSP *,FILE *) */

int ddsp_read(DDSP *this,FILE *p_FILE)
{
  DDSP_STATUS status;
  long l_tell = ftell(p_FILE);
  
  status = ddsp_read_header(this,p_FILE);
  if (status == DDSP_STATUS_OK)
    {
      return ddsp_read_data(this,p_FILE);
    }
  /* fseek(p_FILE,l_tell,SEEK_SET); */
  return DDSP_STATUS_READ;
}

/* }}} */
/* {{{ int             ddsp_read_header_mem  (DDSP *,unsigned char *) */

int
ddsp_read_header_mem(
DDSP *this,
unsigned char *p_uchar
)
{
  memcpy(this->header,p_uchar,DDSP_HEADER_SIZE);
  return 1;
}

/* }}} */
/* {{{ int             ddsp_read_data_mem    (DDSP *,unsigned char *) */

int
ddsp_read_data_mem(
DDSP *this,
unsigned char *p_uchar
)
{
  if (this->size < ddsp_header_get_length(this))
    {
      if (this->local == TRUE)
	{
	  if (this->data != NULL) free(this->data);
	  this->data = NULL;
	  this->size = 0;
	}
      else
	{
	  this->data = NULL;
	  this->size = 0;
	}
    }

  this->len = 0;

  if (this->local == TRUE)
    {
      this->data = (unsigned char *) malloc(ddsp_header_get_length(this));
      memcpy(this->data,p_uchar,ddsp_header_get_length(this));
      this->size = ddsp_header_get_length(this);
    }
  else
    {
      this->data = p_uchar;
      this->size = ddsp_header_get_length(this);
    }
  return 1;
}

/* }}} */
/* {{{ int             ddsp_read_mem         (DDSP *,unsigned char *) */

int
ddsp_read_mem(
DDSP *this,
unsigned char *p_uchar
)
{
  /* ddsp_final(this); */
  this->local = FALSE;
  if (ddsp_read_header_mem(this,(unsigned char *)p_uchar))
    return ddsp_read_data_mem(this,(unsigned char *) p_uchar + DDSP_HEADER_SIZE);
}

/* }}} */
/* {{{ int             ddsp_write_header_mem (DDSP *,unsigned char *) */

int
ddsp_write_header_mem(
DDSP *this,
unsigned char *p_uchar
)
{
  memcpy(p_uchar,this->header,DDSP_HEADER_SIZE);
  return 1;
}

/* }}} */
/* {{{ int             ddsp_write_data_mem   (DDSP *,unsigned char *) */

int
ddsp_write_data_mem(
DDSP *this,
unsigned char *p_uchar
)
{
  memcpy(p_uchar,(unsigned char *)this->data,ddsp_header_get_length(this));
  return 1;
}

/* }}} */
/* {{{ int             ddsp_write_mem        (DDSP *,unsigned char *) */

int
ddsp_write_mem(
DDSP *this,
unsigned char *p_uchar
)
{
  if (ddsp_write_header_mem(this,p_uchar))
    return ddsp_write_data_mem(this,p_uchar + DDSP_HEADER_SIZE);
}

/* }}} */
/* {{{ void            ddsp_data_show        (DDSP *,FILE *) */

void
ddsp_data_show(
DDSP *this,
FILE *p_FILE
)
{
  fprintf(p_FILE,"\n<ddsp-data raw=\"0x%x\" length=\"%d\">",ddsp_get_length(this),ddsp_get_length(this));
  if (this->data != NULL) dpb(p_FILE,ddsp_get_data(this),ddsp_get_length(this));
  fprintf(p_FILE,"\n</ddsp-data>");
}

/* }}} */
/* {{{ void            ddsp_header_show      (DDSP *,FILE *) */

void
ddsp_header_show(
DDSP *this,
FILE *p_FILE
)
{
  fprintf(p_FILE,"\n<ddsp-header>");
  /* dpb(p_FILE,ddsp_get_header(this),15); */
  {
    SCET *l_scet = NULL;
    char l_char[50];
    scet_init(&l_scet);
    scet_get_ddsp(l_scet,this);
    scet_text(l_scet,l_char);
    fprintf(p_FILE,
	    "\n<ddsp-scet day=\"%011d\" ms=\"%011d\" us=\"%05d\" text=\"%s\"/>",
	    scet_get_day(l_scet),
	    scet_get_ms(l_scet),
	    scet_get_us(l_scet),
	    l_char);
    scet_final(&l_scet);
  }
  /*
    fprintf(p_FILE,"\n%s",ddsp_day_text(ddsp_get_day(this)));
    fprintf(p_FILE,"\n%s",ddsp_ms_text(ddsp_get_ms(this)));
    fprintf(p_FILE,"\n%s",ddsp_us_text(ddsp_get_us(this)));
  */
  fprintf(p_FILE,
	  "\n<ddsp-type dec=\"%d\" text=\"%s\"/>",
	  ddsp_get_type(this),
	  ddsp_type_text(ddsp_get_type(this)));
  fprintf(p_FILE,"\n<ddsp-length length=\"%s\"/>",ddsp_length_text(ddsp_get_length(this)));
  fprintf(p_FILE,
	  "\n<ddsp-source dec=\"%d\" text=\"%s\"/>",
	  ddsp_get_source(this),
	  ddsp_source_text(ddsp_get_source(this)));
  fprintf(p_FILE,"\n<ddsp-length length=\"%s\"/>",ddsp_length_text(ddsp_get_length(this)));
  fprintf(p_FILE,
	  "\n<ddsp-spacecraft dec=\"%d\" text=\"%s\"/>",
          ddsp_get_spacecraft(this),
          ddsp_spacecraft_text(ddsp_get_spacecraft(this)));
  fprintf(p_FILE,
	  "\n<ddsp-ground dec=\"%d\" text=\"%s\"/>",
          ddsp_get_ground(this),
          ddsp_ground_text(ddsp_get_ground(this)));
  fprintf(p_FILE,
	  "\n<ddsp-stream dec=\"%d\" text=\"%s\"/>",
          ddsp_get_stream(this),
          ddsp_stream_text(ddsp_get_stream(this)));
  fprintf(p_FILE,
	  "\n<ddsp-quality dec=\"%d\" text=\"%s\"/>",
          ddsp_get_quality(this),
          ddsp_quality_text(ddsp_get_quality(this)));
  fprintf(p_FILE,
	  "\n<ddsp-sequence dec=\"%d\" text=\"%s\"/>",
          ddsp_get_sequence(this),
          ddsp_sequence_text(ddsp_get_sequence(this)));
  fprintf(p_FILE,"\n</ddsp-header>");
}

/* }}} */
/* {{{ void            ddsp_show             (DDSP *,FILE *) */

void
ddsp_show(
DDSP *this,
FILE *p_FILE
)
{
  fprintf(p_FILE,"\n<ddsp>");
  if (this->show_head == TRUE) ddsp_header_show(this,p_FILE);
  if (this->show_body == TRUE) ddsp_data_show(this,p_FILE);
  fprintf(p_FILE,"\n</ddsp>");
}

/* }}} */
/* {{{ int             ddsp_init             (DDSP **,int) */

int
ddsp_init(
DDSP **p_this,
int p_int
)
{
  DDSP *this = NULL;
  this = (DDSP *) malloc(sizeof(DDSP));
  if (this == NULL) {
	  *p_this = NULL;
	  return (DDSP_STATUS_MEMORY);
  }

  memset(this->header,0,DDSP_HEADER_SIZE);
  this->data = NULL;
  this->size = 0;
  this->len  = 0;
  this->local = TRUE;
  this->show_head = c_show_head;
  this->show_body = c_show_body;
  this->debug_head = c_debug_head;
  this->debug_body = c_debug_body;
  if (p_int > 0)
    {
      unsigned char *l_data = NULL;
      l_data = (unsigned char *) malloc(p_int);
      if (l_data == NULL) return (DDSP_STATUS_MEMORY);
      this->data = l_data;
      this->size = p_int;
      this->len  = 0;
    }

  memset(this->data,0,this->size);
  *p_this = this;
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ int             ddsp_init_mem         (DDSP **,int,unsigned char *) */

int
ddsp_init_mem(
DDSP **p_this,
int p_int,
unsigned char *p_uchar
)
{
  int status;
  DDSP *this = NULL;
  status = ddsp_init(&this,p_int);
  if (status == DDSP_STATUS_OK)
    {
      unsigned char * l_data = ddsp_get_data(this);
      memcpy(l_data,p_uchar,p_int);
    }
  if (this != NULL) *p_this = this;
  return (status);
}

/* }}} */
/* {{{ int             ddsp_final            (DDSP **) */

int
ddsp_final(
DDSP **p_this
)
{
  DDSP *this = *p_this;
  if (p_this == NULL) return (DDSP_STATUS_MEMORY);
  if (this   == NULL) return (DDSP_STATUS_MEMORY);
  
  if (this->local == TRUE)
    {
      if (this->data != NULL) free(this->data);
      this->data = NULL;
      this->size = 0;
    }
  free(this);
  *p_this = NULL;
  return (DDSP_STATUS_OK);
}

/* }}} */
/* {{{ int             ddsp_clone            (DDSP **,DDSP *) */

int ddsp_clone(DDSP **p_this,DDSP *src)
{
  int status;
  DDSP *this = NULL;
  status = ddsp_init_mem(&this,ddsp_get_length(src),ddsp_get_data(src));
  if (this != NULL)
    {
      memcpy(this->header,src->header,DDSP_HEADER_SIZE);
      *p_this = this;
    }
  return (status);
}

/* }}} */
/* {{{ DDSP_STATUS ddsp_cpy(DDSP *,DDSP *) */

DDSP_STATUS
ddsp_cpy(
DDSP *p_dst,
DDSP *p_src
)
{
  unsigned char *l_uchar = NULL;
  int l_int = 0;

  if (p_dst == NULL) return (DDSP_STATUS_MEMORY);
  if (p_src == NULL) return (DDSP_STATUS_MEMORY);

  l_uchar = (unsigned char *) malloc(p_src->size);
  if (l_uchar == NULL) return(DDSP_STATUS_MEMORY);

  memcpy(p_src->data,l_uchar,p_src->size);
  if (p_dst->local == TRUE)
    {
      if (p_dst->data != NULL)
	free(p_dst->data);
    }
  p_dst->data = l_uchar;
  p_dst->local == TRUE;
  memcpy(p_dst->header,p_src->header,DDSP_HEADER_SIZE);
  p_dst->size       = p_src->size;
  p_dst->len        = p_src->len;
  p_dst->local      = p_src->local;
  p_dst->show_head  = p_src->show_head;
  p_dst->show_body  = p_src->show_body;
  p_dst->debug_head = p_src->debug_head;
  p_dst->debug_body = p_src->debug_body;
  return (DDSP_STATUS_OK);
}

DDSP_STATUS
scet_to_double(
SCET *this,
double *p_double
)
{
  if (this == NULL) return (DDSP_STATUS_MEMORY);

  *p_double = this->day * (60 * 60 * 24);
  *p_double += this->ms / 1000;
  *p_double += (this->ms % 1000) / 1000.0;
  *p_double += this->us / 1000000.0;
  return (DDSP_STATUS_OK);
}

/* }}} */

/* Local variables: */
/* folded-file: t */
/* end: */
