 
/**
 * $Id: main.c,v 1.1 2001/09/26 14:06:43 simond Exp $
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <ted.h>
#include <version.h>
#include <read.h>
#include <unpack.h>
#include <hktim.h>
#include <dsd.h>
#include <toolbox.h>

#define REQUIRED_LIB_VERSION  4
#define EXPECTED_LIB_REVISION 4

static int debug = 0;

#define USAGE "\
Usage:  levelone  -v\n\
        levelone  -n <NSD file>\n\
                  -b <BSD file>\n\
                  -h <HKD file>\n\
                 [-c]\n\
                 [-d <debug level>]\n\
                 [-m <hktim mode>]\n\
                 [-s <spacecraft ID>]\n\
                 [-T <tcor path>]\n\
				 [-D <diagnostic filter>]\n\
                  -e <diagnostics output file>\n\
                 [-t <tag specifiers>\n\
                 [-f <definition set>]\n\
                 [-0 <LV1 file for EFW>]\n\
                 [-1 <LV1 file for STAFFSA>]\n\
                 [-2 <LV1 file for STAFFMWF>]\n\
                 [-3 <LV1 file for WHISPER>]\n\
                 [-4 <LV1 file for WBD>]\n\
                 [-5 <LV1 file for DWP>]\n\
                 [-N <NSD offset>]\n\
                 [-B <BSD offset>]\n\
                 [-H <HKD offset>]\n\
\n\
        Tag specifiers are tag<N>=<V> where 1<=N<=4 and 1<=V<=ffff\n"

#define MAXPATH 1024

typedef enum {
  EXIT_OK = 0,
  EXIT_USAGE = 10,
  EXIT_UNIMPLEMENTED,
  EXIT_EXISTS,
  EXIT_FILE,
  EXIT_FATAL,
  EXIT_MEMORY,
  EXIT_INTERNAL,
  EXIT_VERSION,
  EXIT_OFFSET
} EXIT_STATUS;

struct {
  int version;			/* TRUE iff -v option used */
  char nsd_path[MAXPATH];	/* Path of NSD file */
  char bsd_path[MAXPATH];	/* Path of BSD file */
  char hkd_path[MAXPATH];	/* Path of HKD file */
  char diag_path[MAXPATH];	/* Path of Diagnostics file */
  char tcor_path[MAXPATH];	/* Path of TCOR files */
  char lv1_path[6][MAXPATH];	/* Path of LV1 files */
  int spacecraft;		/* Spacecraft ID */
  int debug_level;		/* Debugging level */
  int diag_filter_mask, diag_filter_comp;  /* Diagnostic filter */
  int ccs_converted;		/* TRUE iff data is CCS converted */
  TED_HKTIM_MODE hktim_mode;    /* HKTIM mode */
  int model_tag[5];		/* Value to use if 1..4 */
  int definition_set;		/* Header definition set to use or -1 for default */
  long hkd_offset;		/* Offset at which to start in HKD file or -1 if none */
  long nsd_offset;		/* Offset at which to start in NSD file or -1 if none */
  long bsd_offset;		/* Offset at which to start in BSD file or -1 if none */
  int tcor_specified;

} options;

struct {
  FILE *nsd_stream;		/* NSD stream */
  FILE *bsd_stream;		/* BSD stream */
  FILE *hkd_stream;		/* HKD stream */
  FILE *diag_stream;		/* Diagnostics stream */
  FILE *lv1_stream[6];		/* LV1 streams */
} streams;

void show_options(void);
EXIT_STATUS get_options(int argc, char **argv);
int which_instruments(void);
EXIT_STATUS process_files(void);
TED_UNPACK_DIAG *flush_diagnostics(void);
EXIT_STATUS handle_error(TED_STATUS status);
void report_file_error(char *message);

/* {{{ static char *DiagnosticToString(TED_UNPACK_DIAG *diagnostic) */

static char *DiagnosticToString(TED_UNPACK_DIAG *diagnostic) {
  char *string;

  (void) ted_unpack_diag2string(*diagnostic, &string);
  return string;
}

/* }}} */
/* {{{ static char *VersionToString(int version, int revision, int patch, int userpatch) */

static char *VersionToString(int version, int revision, int patch, int userpatch) {
  static char string[40];

  sprintf(string, "Version %d.%d", version, revision);
  if (patch) sprintf(string + strlen(string), " Patch %d", patch);
  if (userpatch) sprintf(string + strlen(string), " (User Patch %d)", userpatch);
  return string;
}

/* }}} */
/* {{{ EXIT_STATUS  get_options(int, char *[]) */

EXIT_STATUS
get_options(
int argc,
char *argv[]
)
{
  char c;			/* For getopt() */
  extern char *optarg;		/* For getopt() */
  extern int optind;		/* For getopt() */
  extern int opterr;		/* For getopt() */
  char *suboptions, *value;	/* For sub options */

  int i;			/* Loops */

  char *tag_tokens[] = { "tag1", "tag2", "tag3", "tag4", (char *) NULL};

  options.version = FALSE;
  options.nsd_path[0] = '\0';
  options.bsd_path[0] = '\0';
  options.hkd_path[0] = '\0';
  options.diag_path[0] = '\0';
  options.spacecraft = 0;
  options.debug_level = 0;
  options.diag_filter_mask = 0;
  options.diag_filter_comp = 0;
  options.ccs_converted = 0;
  options.definition_set = -1;
  options.hkd_offset = -1;
  options.nsd_offset = -1;
  options.bsd_offset = -1;
  options.hktim_mode = HKTIM_MODE_RUNNING_AVERAGE;
  options.tcor_specified = 0;
  options.tcor_path[0] = 0;
  for (i = 0; i <= 5; i++) options.lv1_path[i][0] = '\0';
  for (i = 1; i <= 4; i++) options.model_tag[i] = 0;

  opterr = 0;

  while ((c = (char)getopt(argc, argv, "vh:n:b:s:e:cd:t:f:0:1:2:3:4:5:N:B:H:m:T:D:")) != -1) {

    switch (c) {

    case 'v':
      options.version = TRUE;
      return EXIT_OK;
    case 'm':
      /* {{{ HKTIM_MODE */
      if (sscanf(optarg, "%d", &options.hktim_mode) != 1) return EXIT_USAGE;
      if (options.hktim_mode < 0 || options.hktim_mode > 3) return EXIT_USAGE;
      /* }}} */
      break;
	case 'T':
	  /* TCOR path */
      strcpy(options.tcor_path, optarg);
	  options.tcor_specified = 1;
	  break;
	case 'D':
	  if (sscanf(optarg, "%x:%x", &options.diag_filter_mask, &options.diag_filter_comp) != 2) return EXIT_USAGE;
	  break;
    case 'n':
      /* {{{ NSD */
      if (options.nsd_path[0] != '\0')
	return EXIT_USAGE;	/* Option already used */
      strcpy(options.nsd_path, optarg);
      /* }}} */
      break;
    case 'b':
      /* {{{ BSD */
      if (options.bsd_path[0] != '\0')
	return EXIT_USAGE;	/* Option already used */
      strcpy(options.bsd_path, optarg);
      /* }}} */
      break;
    case 'h':
      /* {{{ HKD */
      if (options.hkd_path[0] != '\0')
	return EXIT_USAGE;	/* Option already used */
      strcpy(options.hkd_path, optarg);
      /* }}} */
      break;
    case 's':
      /* {{{ SPACECRAFT */

      if (options.spacecraft)
	return EXIT_USAGE;	/* Option already used */
      if (sscanf(optarg, "%d", &options.spacecraft) != 1)
	return EXIT_USAGE;	/* Can't parse the integer */
      if (options.spacecraft < 1 || options.spacecraft > 4)
	return EXIT_USAGE;	/* Out of range */

/* }}} */
      break;
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
      /* {{{ O/P Files */
      i = c - '0';
      if (options.lv1_path[i][0] != '\0') return EXIT_USAGE;
      strcpy(options.lv1_path[i], optarg);
      /* }}} */
      break;
    case 'e':
      /* {{{ DIAG FILE */

      if (options.diag_path[0] != '\0')	return EXIT_USAGE;
      strcpy(options.diag_path, optarg);

      /* }}} */
      break;
    case 'd':
      /* {{{ DEBUG FLAG */
      if (options.debug_level)	return EXIT_USAGE;
      if (sscanf(optarg, "%d", &options.debug_level) != 1) return EXIT_USAGE;
      if (options.debug_level < 0 || options.spacecraft > 2) return EXIT_USAGE;
      /* }}} */
      break;
    case 'c':
      /* {{{ CCS CONVERTED DATA */
      options.ccs_converted = 1;
      /* }}} */
      break;
    case 't':
      /* {{{ MODEL TAGS */
      {
	int spacecraft;
	suboptions = optarg;
	while (*suboptions != '\0')
	  {
	    spacecraft = getsubopt(&suboptions, tag_tokens, &value) + 1;
	         if (spacecraft < 1 || spacecraft > 4) return EXIT_USAGE;
	    else if (options.model_tag[spacecraft])    return EXIT_USAGE;
	    else if (value == NULL)                    return EXIT_USAGE;
	    else
	      {
		int model_tag;
		if ((model_tag = strtol(value, (char **) NULL, 16)) == 0) return EXIT_USAGE;
		options.model_tag[spacecraft] = model_tag;
	      }
	  }
      }
      /* }}} */
      break;
    case 'f':
      /* {{{ DEFINITION SET */
      if (options.definition_set != -1)	                      return EXIT_USAGE;
      if (sscanf(optarg, "%d", &options.definition_set) != 1) return EXIT_USAGE;
      /* }}} */
      break;
    case 'N':
      /* {{{ NSD OFFSET */
      if (options.nsd_offset != -1)                        return EXIT_USAGE;
      if (sscanf(optarg, "%ld", &options.nsd_offset) != 1) return EXIT_USAGE;
      if (options.nsd_offset < 0)                          return EXIT_USAGE;
      /* }}} */
      break;
    case 'B':
      /* {{{ BSD OFFSET */
      if (options.bsd_offset != -1)                        return EXIT_USAGE;
      if (sscanf(optarg, "%ld", &options.bsd_offset) != 1) return EXIT_USAGE;
      if (options.bsd_offset < 0)                          return EXIT_USAGE;
      /* }}} */
      break;
    case 'H':
      /* {{{ HKD */
      if (options.hkd_offset != -1)                        return EXIT_USAGE;
      if (sscanf(optarg, "%ld", &options.hkd_offset) != 1) return EXIT_USAGE;
      if (options.hkd_offset < 0)                          return EXIT_USAGE;
      /* }}} */
      break;
    case '?':
      return EXIT_USAGE;
    }
  }

  /*
   * At this stage we know that any options that were used were okay.
   * It remains to check that all the required options were used.
   */

  if (options.nsd_path[0] == '\0') return EXIT_USAGE; /* -n option omitted */
  if (options.bsd_path[0] == '\0') return EXIT_USAGE; /* -s option omitted */
  if (options.hkd_path[0] == '\0') return EXIT_USAGE; /* -h option omitted */
  if (options.diag_path[0] == '\0') return EXIT_USAGE; /* -e option omitted */

  /* We're fine! */

  return EXIT_OK;
}

/* }}} */
/* {{{ void        show_options(void) */

void show_options(void) {
  int i;

  fprintf(stderr, "nsd_path = %s\n", options.nsd_path);
  fprintf(stderr, "bsd_path = %s\n", options.bsd_path);
  fprintf(stderr, "hkd_path = %s\n", options.hkd_path);
  fprintf(stderr, "diag_path = %s\n", options.diag_path);
  fprintf(stderr, "spacecraft = %d\n", options.spacecraft);
  for (i = 0; i <= 5; i++)
    if (options.lv1_path[i][0] != '\0')
      fprintf(stderr, "lv1_path[%d] = %s\n", i, options.lv1_path[i]);
  fprintf(stderr, "debug_level = %d\n", options.debug_level);
  fprintf(stderr, "ccs_converted = %d\n", options.ccs_converted);
  for (i = 1; i <= 4; i++)
    fprintf(stderr, "model_tag[%d] = %x\n", i, options.model_tag[i]);
  fprintf(stderr, "definition_set = %d\n", options.definition_set);
  fprintf(stderr, "options.nsd_offset = %d\n", options.nsd_offset);
  fprintf(stderr, "options.bsd_offset = %d\n", options.bsd_offset);
  fprintf(stderr, "options.hkd_offset = %d\n", options.hkd_offset);

  fprintf(stderr, "hktim_mode = %d\n", options.hktim_mode);
  if (options.tcor_specified) fprintf(stderr, "tcor_path = %s\n", options.tcor_path);
  if (options.diag_filter_mask) fprintf (stderr, "diag_filter = %04x:%04x\n", 
	                               options.diag_filter_mask, options.diag_filter_comp);
}

/* }}} */
/* {{{ EXIT_STATUS seek_to_start(FILE *tmstream, char tmpath[], long tmoffset) */

EXIT_STATUS seek_to_start(FILE *tmstream, char tmpath[], long tmoffset) {
  long bytes;

  /* Seek to the end of the file */
  if (fseek(tmstream, 0L, SEEK_END) == -1) {
    report_file_error(tmpath);
    return EXIT_FILE;
  }
  /* Note how large the file is */
  if ((bytes = ftell(tmstream)) == -1) {
    report_file_error(tmpath);
    return EXIT_FILE;
  }
  /* Report an error if the specified offset extends beyond the end of file */
  if (tmoffset > bytes) {
    fprintf(stderr, "Specified offset extends beyond the end of file \"%s\"\n", tmpath);
    return EXIT_OFFSET;
  }
  /* Now seek to the specified offset */
  if (fseek(tmstream, tmoffset, SEEK_SET) == -1) {
    report_file_error(tmpath);
    return EXIT_FILE;
  }
  return EXIT_OK;
}

/* }}} */
/* {{{ EXIT_STATUS open_files(void) */

EXIT_STATUS
open_files(
void
)
{
  int i;
  EXIT_STATUS status;
  /* {{{ NSD */

  streams.nsd_stream = fopen(options.nsd_path, "r");
  if (streams.nsd_stream == NULL)
    {
      report_file_error(options.nsd_path);
      return EXIT_FILE;
    }
  if (options.nsd_offset != -1)
    {
      status = seek_to_start(streams.nsd_stream, options.nsd_path, options.nsd_offset);
      if (status != EXIT_OK) return status;
    }

  /* }}} */
  /* {{{ BSD */
  streams.bsd_stream = fopen(options.bsd_path, "r");
  if (streams.bsd_stream == NULL)
    {
      report_file_error(options.bsd_path);
      return EXIT_FILE;
    }
  if (options.bsd_offset != -1)
    {
      status = seek_to_start(streams.bsd_stream, options.bsd_path, options.bsd_offset);
      if (status !=  EXIT_OK) return status;
    }
  /* }}} */
  /* {{{ HKD */
  streams.hkd_stream = fopen(options.hkd_path, "r");
  if (streams.hkd_stream == NULL)
  {
    report_file_error(options.hkd_path);
    return EXIT_FILE;
  }
  if (options.hkd_offset != -1)
  {
    status = seek_to_start(streams.hkd_stream, options.hkd_path, options.hkd_offset);
    if (status != EXIT_OK) return status;
  }
  /* }}} */
  /* {{{ DIAG */
  streams.diag_stream = fopen(options.diag_path, "w+");
  if (streams.diag_stream == NULL)
  {
    report_file_error(options.diag_path);
    return EXIT_FILE;
  }
  /* }}} */
  /* {{{ 0 */
  if (options.lv1_path[0][0] != '\0')
  {
    streams.lv1_stream[0] = fopen(options.lv1_path[0], "w+");
    if (streams.lv1_stream[0] == NULL)
    {
      report_file_error(options.lv1_path[0]);
      return EXIT_FILE;
    }
  }
  /* }}} */
  /* {{{ 1 */
  if (options.lv1_path[1][0] != '\0')
  {
    streams.lv1_stream[1] = fopen(options.lv1_path[1], "w+");
    if (streams.lv1_stream[1] == NULL)
    {
      report_file_error(options.lv1_path[1]);
      return EXIT_FILE;
    }
  }
  /* }}} */
  /* {{{ 2 */
  if (options.lv1_path[2][0] != '\0')
  {
    streams.lv1_stream[2] = fopen(options.lv1_path[2], "w+");
    if (streams.lv1_stream[2] == NULL)
    {
      report_file_error(options.lv1_path[2]);
      return EXIT_FILE;
    }
  }
  /* }}} */
  /* {{{ 3 */
  if (options.lv1_path[3][0] != '\0')
  {
    streams.lv1_stream[3] = fopen(options.lv1_path[3], "w+");
    if (streams.lv1_stream[3] == NULL)
    {
      report_file_error(options.lv1_path[3]);
      return EXIT_FILE;
    }
  }
  /* }}} */
  /* {{{ 4 */

  if (options.lv1_path[4][0] != '\0')
  {
    streams.lv1_stream[4] = fopen(options.lv1_path[4], "w+");
    if (streams.lv1_stream[4] == NULL)
    {
      report_file_error(options.lv1_path[4]);
      return EXIT_FILE;
    }
  }

  /* }}} */
  /* {{{ 5 */

  if (options.lv1_path[5][0] != '\0')
  {
    streams.lv1_stream[5] = fopen(options.lv1_path[5], "w+");
    if (streams.lv1_stream[5] == NULL)
    {
      report_file_error(options.lv1_path[5]);
      return EXIT_FILE;
    }
  }

  /* }}} */
  return EXIT_OK;
}

/* }}} */
/* {{{ int which_instruments(void) */

int which_instruments(void) {
  int instruments;

  instruments = 0;

  if (options.lv1_path[0][0] != '\0') instruments |= INSTRUMENT_EFW;
  if (options.lv1_path[1][0] != '\0') instruments |= INSTRUMENT_STAFFSA;
  if (options.lv1_path[2][0] != '\0') instruments |= INSTRUMENT_STAFFMWF;
  if (options.lv1_path[3][0] != '\0') instruments |= INSTRUMENT_WHISPER;
  if (options.lv1_path[4][0] != '\0') instruments |= INSTRUMENT_WBD;
  if (options.lv1_path[5][0] != '\0') instruments |= INSTRUMENT_DWP;

  return instruments;
}

/* }}} */
/* {{{ EXIT_STATUS process_files(void) */

EXIT_STATUS process_files(void) {
  TED_STATUS read_status;
  TED_STATUS unpack_status;
  TED_STATUS dsd_status;
  
  int again;
  int gotscience;
  
  char *nsd_packet;
  char *bsd_packet;
  
  char *exppacket;
  char *hkpacket;
  char *sciencedata;
  TED_DSD_ITEMS *items;
  char *hkblock;
  char *dsdheader;
  char *dsddata;
  int headerlength;
  TED_SPACECRAFT spacecraft;
  int flushing = FALSE;
  int definition_set;

  /* Initialize READ module */
  if (options.debug_level > 1)
    (void) ted_read_debug(1);
  
  read_status = ted_read_initf(streams.nsd_stream, streams.bsd_stream, streams.hkd_stream);
  if (read_status != TED_OK)
    return handle_error(read_status);

  /* Initialize DSD module */
  if (options.definition_set == -1)
    definition_set = 0;
  else
    definition_set = options.definition_set;
  dsd_status = ted_dsd_init(definition_set);

  if (dsd_status != TED_OK)
    return handle_error(dsd_status);

  /* Initialize UNPACK module */
  (void) ted_unpack_debug(options.debug_level);
  switch (options.spacecraft) {
  case 0: spacecraft = CLUSTER_X; break;
  case 1: spacecraft = CLUSTER_1; break;
  case 2: spacecraft = CLUSTER_2; break;
  case 3: spacecraft = CLUSTER_3; break;
  case 4: spacecraft = CLUSTER_4; break;
  }
  unpack_status = ted_unpack_init(spacecraft, which_instruments(), options.ccs_converted,options.model_tag);
  if (unpack_status != TED_OK) return handle_error(unpack_status);

  ted_unpack_hktim(options.hktim_mode);
  if (options.tcor_specified) ted_unpack_hktcor (options.tcor_path);

  read_status = ted_read_packet(&nsd_packet, &bsd_packet, &hkpacket);
  while (read_status == TED_OK || flushing) {
    if (flushing) {
      if (debug) fprintf(stderr, "Flushing ted_unpack_decom()\n");
      hkpacket = NULL;
      exppacket = NULL;
    }
    else {
      if (nsd_packet && bsd_packet) {
	fprintf(stderr, "Fatal error in telemetry:\nGot NSD and BSD for same reset pulse\n");
	return EXIT_FATAL;
      }
      else if (nsd_packet)  exppacket = nsd_packet;
      else if (bsd_packet)  exppacket = bsd_packet;
      else                  exppacket = NULL;
    }
    do {
      unpack_status = ted_unpack_decom(exppacket, hkpacket, &gotscience,&items, &hkblock, &sciencedata, &again);
      if (unpack_status != TED_OK) goto finish;

      (void) flush_diagnostics();

      if (debug) fprintf(stderr, "in main.c gotscience %d again %d\n", gotscience, again);


      if (gotscience == 1) {

	    if (debug) fprintf(stderr, "** GOT SCIENCE for instrument %d\n", items->source);
	    (void) ted_dsd_build(items, hkblock, sciencedata, &dsdheader,&headerlength, &dsddata);

		if ((items->dw & options.diag_filter_mask) != (options.diag_filter_mask & options.diag_filter_comp)) {
		  if (debug) fprintf(stderr, "Diagnostic filter rejecting packet for instrument %d\n", items->source);
		  continue;
		}

	    if (options.lv1_path[items->source][0] != '\0') {
	      if (fwrite(dsdheader, 1, headerlength * 2, streams.lv1_stream[items->source]) != headerlength * 2) {
	        report_file_error(options.lv1_path[items->source]);
	        return EXIT_FILE;
		  }
	      if (fwrite(dsddata, 1, items->length * 2, streams.lv1_stream[items->source]) != items->length * 2) {
	        report_file_error(options.lv1_path[items->source]);
	        return EXIT_FILE;
		  }
		} else if (debug) fprintf(stderr, "Not writing LV1 as corresponding option not used\n");
      } 
    } while (unpack_status == TED_OK && again);

    /* If we're flushing, then don't do another ted_read_packet() */
    if (flushing) {
       flushing = FALSE;
       continue;
     }

    /* Read data for next reset pulse */
    read_status = ted_read_packet(&nsd_packet, &bsd_packet, &hkpacket);
    if (read_status != TED_OK)
      flushing = TRUE;
  }

 finish:

  switch (read_status) {
  case TED_READ_EOF:
    return EXIT_OK;
  case TED_READ_FILE:
    {
      char *path;
      switch (ted_read_file_no) {
      case 1: path = options.nsd_path; break;
      case 2: path = options.bsd_path; break;
      case 3: path = options.hkd_path; break;
      }
      report_file_error(path);
      return EXIT_FILE;
    }
  case TED_READ_SIZE:
    fprintf(stderr, "Fatal error in telemetry:\n");
    fprintf(stderr, "Packet is too large for this data type, or is apparently zero length\n");
    return EXIT_FATAL;
  case TED_READ_EARLIER:
    fprintf(stderr, "Fatal error in telemetry:\n");
    fprintf(stderr, "Timestamp is earlier than that of the previous packet in the same file\n");
    return EXIT_FATAL;
  default:
    if (unpack_status == TED_UNPACK_FATAL) {
      fprintf(stderr, "Fatal error in telemetry:\n%s\n",
	      DiagnosticToString(flush_diagnostics()));
      return EXIT_FATAL;
    }
    return handle_error(read_status);
  }
}

/* }}} */
/* {{{ TED_UNPACK_DIAG *flush_diagnostics(void) */

TED_UNPACK_DIAG *flush_diagnostics(void) {
  TED_UNPACK_DIAG *diagnostic;
  static TED_UNPACK_DIAG last_diagnostic;
  TED_UNPACK_DIAG *result = NULL;

  while (ted_unpack_diagnostic(&diagnostic) == TED_OK) {
    fprintf(streams.diag_stream, "%s\n", DiagnosticToString(diagnostic));
    if (debug)
      fprintf(stderr, "<%s>\n", DiagnosticToString(diagnostic));
    last_diagnostic = *diagnostic;
    result = &last_diagnostic;
  }
  return result;
}

/* }}} */
/* {{{ EXIT_STATUS handle_error(TED_STATUS status) */

EXIT_STATUS handle_error(TED_STATUS status) {
  switch(status) {
  case TED_UNIMPLEMENTED:
    fprintf(stderr, "levelone: Unimplemented feature requested\n");
    return EXIT_UNIMPLEMENTED;
  case TED_MEMORY:
    fprintf(stderr, "levelone: Can't allocate memory: %s\n", strerror(errno));
    return EXIT_MEMORY;
  case TED_DSD_DEFINITIONSETINVALID:
    fprintf(stderr, "levelone: Invalid DSD Header definition set number\n");
    return EXIT_USAGE;
  default:
    {
      char *status_string;
      (void) ted_toolbox_status2string(status, &status_string);
      fprintf(stderr, "levelone: Error from TED Library: %s\n", status_string);
      return EXIT_INTERNAL;
    }
  }
}

/* }}} */
/* {{{ void report_file_error(char *path) */

void report_file_error(char *path) {
  fprintf(stderr, "levelone: %s: %s\n", path, strerror(errno));
}

/* }}} */

/* {{{ int main(int argc, char *argv[]) */

int main(int argc, char *argv[]) {
  int status;
  int ted_version, ted_revision, ted_patch, ted_userpatch;
  int lib_version, lib_revision, lib_patch, lib_userpatch;

  if (get_options(argc,argv)) {
    fprintf(stderr, USAGE);
    return EXIT_USAGE;
  }

  debug = options.debug_level;

  (void) ted_version_ted(&ted_version, &ted_revision,
			    &ted_patch, &ted_userpatch);
  printf("TED Package %s\n",
	 VersionToString(ted_version, ted_revision, ted_patch,
			 ted_userpatch));

  (void) ted_version_lib(&lib_version, &lib_revision,
			    &lib_patch, &lib_userpatch);

  if (lib_version == REQUIRED_LIB_VERSION && lib_revision >= EXPECTED_LIB_REVISION)
    printf("Using TED Library %s\n",
	   VersionToString(lib_version, lib_revision, lib_patch, lib_userpatch));
  else if (lib_version == REQUIRED_LIB_VERSION)
    fprintf(stderr, "Warning: TED Library %s is older than expected %d.%d\n",
	   VersionToString(lib_version, lib_revision, lib_patch, lib_userpatch),
	   REQUIRED_LIB_VERSION, EXPECTED_LIB_REVISION);
  else {
    fprintf(stderr, "Requires TED Library Version %d.x found %s\n", REQUIRED_LIB_VERSION,
	   VersionToString(lib_version, lib_revision, lib_patch, lib_userpatch));
    return EXIT_VERSION;
  }

  fflush(stdout);

  if (options.version)
    return EXIT_OK;

  if (debug)
    show_options();

  (void) putenv("TZ=GMT");

  if (status = open_files())
    return status;

  if (status = process_files())
    return status;

  printf("Okay\n");

  return 0;
}

/* }}} */

/* Local variables: */
/* folded-file: t */
/* end: */
