; = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
;+
; NAME:
;     READ_ASC_HDF
;
; IDENT:
;     $Id: read_asc_hdf.pro,v 1.1 2003/02/14 20:19:48 glennh Exp $
;
; PURPOSE:
;     read in a range of ASC (hdfgen'ed) HDF files
;
; AUTHOR:
;      Ed Santiago
;
; CALLING SEQUENCE:
;     x = READ_ASC_HDF(filelist, basename, fields)
;
; INPUTS:
;     filelist     - array of input files
;     basename     - "hdfgen" name of vdata/sdata, eg, "swepam_dswi"
;     fields       - list of fields to read; if empty all fields are read
;
; OUTPUTS:
;     extracted data is in an array of structures
;
; SIDE EFFECTS:
;
; EXAMPLE:
;
;-
FUNCTION read_asc_hdf, files, basename, fields, Quiet=quiet, Debug=debug

  IF N_Elements(quiet) EQ 0 THEN quiet = 0
  IF N_Elements(debug) EQ 0 THEN debug = 0

  ; Error conditions returned when things don't work out right with one HDF
  ERR_NO_SUCH_VD = 'IDL_M_HDF_VS_SET_FIELDS'
  ERR_NO_SUCH_SD = 'IDL_M_HDF_SD_SELECT'

  nfiles = N_Elements(files)

  FOR i=0,nfiles-1 DO BEGIN
      IF NOT quiet THEN BEGIN
          status = string(format='("-> ",A)', files[i])
          IF nfiles GT 1 THEN $
               status = status + string(format='(" (",I0," of ",I0,")")',$
					i+1, nfiles)
          print, status
      ENDIF

      ; Initialize the HDF thingies
      fid   = HDF_Open(files[i])
      IF fid LE 0 THEN BEGIN
          MESSAGE, 'HDF_Open(' + files[i] + ') failed!', /Continue
          RETURN, 0
      ENDIF

      v     = HDF_VD_Find(fid,basename)
      IF v LE 0 THEN BEGIN
        MESSAGE, 'HDF_Find('+basename+') in '+files[i]+' failed!', /Continue
        HDF_Close, fid
        RETURN, 0
      ENDIF

      vdata = HDF_VD_Attach(fid, v)
      IF vdata LE 0 THEN BEGIN
        HDF_Close, fid
        MESSAGE, 'HDF_VD_Attach('+files[i]+') failed!'
      ENDIF

      ; First time around, figure out what the data types are
      IF N_Elements(fieldinfo) EQ 0 THEN BEGIN
          fieldinfo   = get_fieldinfo(files[i], basename, vdata, fields)
	  fieldstruct = get_fieldstruct(fieldinfo)
      ENDIF

      ; read the file contents, and extract fields
      HDF_VD_Get, vdata, count=nr
      IF nr LE 0 THEN GOTO, skipThisFile

      today = REPLICATE(fieldstruct, nr)

      ; Loop over each field, reading it in using the appropriate HDF interface
      FOR j=0,N_Elements(fieldinfo)-1 DO BEGIN
          ; If field is a multimensional array, get it from the SD interface
          IF fieldinfo[j].ndims GT 1 THEN BEGIN
              sd = -1
	      Catch, err_sts
	      IF err_sts EQ 0 THEN BEGIN
                  sd  = HDF_SD_Start(files[i])
	          sdi = HDF_SD_NameToIndex(sd, basename+'_'+fieldinfo[j].name)
	          sds = HDF_SD_Select(sd, sdi)
	          IF debug THEN print,'Reading SD: '+fieldinfo[j].name+'..'
	          HDF_SD_GetData, sds, data
                  IF debug THEN print,'Copying...'
                  today.(j+1) = temporary(data)
	          IF debug THEN print,'done!'
                  HDF_SD_EndAccess, sds
                  HDF_SD_End, sd
	          sd = -1
	      ENDIF ELSE IF !Error_State.Name EQ ERR_NO_SUCH_SD THEN BEGIN
	          print,'['+files[i]+': no such SD: ' + fieldinfo[j].name + ']'
	      ENDIF ELSE BEGIN
help,!error_state,/st
	          Catch, /Cancel
	          print, 'hdf_sd_getdata(): err=',err_sts
		  HDF_Close, fid
	          Message, !ERR_STRING
	      ENDELSE
	      Catch, /Cancel
	      IF sd GE 0 THEN HDF_SD_End, sd
          ENDIF ELSE BEGIN
	      ; Field is scalar or vector; read it using the VDATA interface
	      Catch, err_sts
	      IF err_sts EQ 0 THEN BEGIN
                  HDF_VD_Seek, vdata, 0
                  status = HDF_VD_Read(vdata, data, fields=fieldinfo[j].name)
	          IF status NE nr THEN MESSAGE, 'mismatch'
                  today.(j+1) = temporary(data)
	      ENDIF ELSE IF !Error_State.Name EQ ERR_NO_SUCH_VD THEN BEGIN
	          print,'['+files[i]+': no such VD: ' + fieldinfo[j].name + ']'
	      ENDIF ELSE BEGIN
	          Catch, /Cancel
	          print, 'hdf_vd_read(): err=',err_sts
	          Message, !ERR_STRING
	      ENDELSE
	      Catch, /Cancel
          ENDELSE
      ENDFOR

      ; Done reading all fields.  Make a list of all structure elements.
      IF N_Elements(matchlist) EQ 0 THEN BEGIN
          matchlist = today
      ENDIF ELSE BEGIN
          matchlist = [ matchlist, today ]
      ENDELSE

      ; done.
skipThisFile:
      HDF_VD_Detach, vdata
      HDF_Close, fid
  ENDFOR

  IF N_Elements(matchlist) EQ 0 THEN BEGIN
      IF NOT quiet THEN MESSAGE, 'Could not find any data in this range',/Cont
      return, 0
  ENDIF

  RETURN, matchlist
END

