;+
; NAME:
;     READ_DATA
;
; PURPOSE:
;	Universal IDL reader for ASCII data files created in the Magnetopause 
;	Crossing Database (MPC). It uses the data file header to recognize the file
;	format and size. Replaces the IDL NaN symbol by zero. Should work with all
;	MPC data files created after November 1, 2008.
;
; CATEGORY:
;	Data Processing
;
; CALLING SEQUENCE:
;	read_data, probe, fname, err, datstr
;
; INPUTS:
;	probe: letter code of THEMIS probe (string). Valid entries: 'a','b','c','d','e' 
;		(they correspond to numeric notations 5,1,2,3,4, respectively)
;	fname: name of the file to read (string)
;
; KEYWORDS: none
;
; PARAMETERS: none
;
; OUTPUTS:
;	err: 1 if requested file is found, 0 otherwise (integer)
;	datstr: data structure with the following fields:
;		.pr: probe letter (string)
;		.time: time grid array for data in THEMIS format yyyy-mm-dd/hh:mm:ss.mil
;			(string array)
;		.val: data values; this is array, which dimensions and size depend on
;			the type of retrieved data. In general, first dimension corresponds
;			to the time array and the second gives the rows from the data file
;			in the order from left to right.
;
; DEPENDENCIES: standalone.
;
; MODIFICATION HISTORY:
;     Written on November 1, 2008 by Vladimir Kondratovich, SP Systems, Inc
;-
;
; THE CODE BEGINS:

pro read_data,probe,fname,err,datstr
!quiet=1
yes=file_test(fname)
if not yes then begin
   err=0
   return
endif else err=1

ephnam1='sc_coords_*'
ephnam2='sc_vels_*'
log1=1-strmatch(fname,ephnam1)
log2=1-strmatch(fname,ephnam2)

if log1 and log2 then begin
   openr,unit,fname,/get_lun
   ;this reads a string without data, if present:
   dummy = ' '
   readf,unit, dummy
   readf,unit, dummy
   ;third line gives the data format:
   form = ' '
   readf,unit, form
   ;fourth line gives the number of lines:
   readf,unit, format='(I10)',numlines

   ;create input arrays:
   timestr=strarr(numlines)
   res=strsplit(form,',',/extract)
   numvar=fix(strmid(res(1),0,2))
   fl='*_flag_*'
   isflag=strmatch(fname,fl)
   if isflag then xarr=strarr(numlines,numvar) else xarr=dblarr(numlines,numvar)


   i=0L
   while(not eof(unit)) do begin
      t=''
      x=strarr(numvar)
      readf,unit,FORMAT=form,t,x
      timestr(i)=t
      if isflag then xarr(i,*)=x else xarr(i,*)=float(x)
      i=i+1
   endwhile

   close,unit
   free_lun,unit

   ;read check:
   if i lt 1 then begin
      err=0
      return
   endif

   timestr=timestr(0:i-1)
   xarr=xarr(0:i-1,*)

   datstr={pr:probe,time:timestr,val:xarr}
endif else begin
   numlines=file_lines(fname)
   openr,unit,fname,/get_lun
   form = '(A23,G15,2G13)'
   ;create input arrays:
   timestr=strarr(numlines)
   numvar=3
   xarr=dblarr(numlines,numvar)
   i=0L
   while(not eof(unit)) do begin
      t=''
      x=fltarr(numvar)
      readf,unit,FORMAT=form,t,x
      timestr(i)=t
      xarr(i,*)=x
      i=i+1
   endwhile
   close,unit
   free_lun,unit

   ;read check:
   if i lt 1 then begin
      err=0
      return
   endif
   timestr=timestr(0:i-1)
   xarr=xarr(0:i-1,*)
   datstr={pr:probe,time:timestr,val:xarr}
endelse

return
end

