;+
; NAME:
;  PLOT_SEE_SPWX
;
; PURPOSE:
;  Display TIMED-SEE space weather data in a plot
;
; CATEGORY:
;  Analysis
;
; CALLING SEQUENCE:
;  plot_see_spwx, /help, index=index, path=path, structure=structure, $
;   xrange=xrange, yrange=yrange, /ylog, /oplot, /noplot, color=color, /error 
;   data=data, att=att, /tsimage, /one_xaxis
;
; INPUTS:
;  No inputs are required.
;
; OPTIONAL INPUTS:
;  index: a number designating which irradiance to use
;    1 - (default) Soft X-Rays from 0.1-7 nm
;    2 - EUV band from 27-34 nm
;    3 - 30.4 nm He II
;    4 - 33.5 nm Fe XVI
;    5 - 36.8 nm Mg IX
;    6 - 121.5 nm H I
;    7 - 133.5 nm C II
;    8 - FUV band from 145-165 nm
;  path: string pointing to the fully qualified path to the data files
;  xrange: 2 element vector same as PLOT parameter
;  yrange: 2 element vector same as PLOT parameter
;  color: scalar argument passed to PLOT (device dependent)
;
; KEYWORD PARAMETERS:
;  /help: display usage information
;  /ylog: use logarithmic scale in plotting
;  /oplot: overplot (oplot) instead of drawing a new plot
;  /noplot: prevent all potting (useful for retrieving data)
;  /error: plot uncertainties instead of irradiance
;  /tsimage: make an image of all indices with irradiance shown as a color
;    try with /ylog and without to adjust color scale
;    (for truecolor devices, I first setup using device,decomposed=0
;     then I loadct,39)
;  /one_xaxis: prevent top axis from being included
;
; OUTPUTS:
;  No required outputs.
;
; OPTIONAL OUTPUTS:
;  structure: an array of data structures merged from all of the files read
;  data: double array of [3,n] containing julian date, irradiance, and uncertainty
;  att: string array of attributes information from the top of the data file
;
; COMMON BLOCKS:
;  plot_see_spwx_reserved, filedata
;   filedata: the array of merged structures read from the data files
;
; SIDE EFFECTS:
;  None
;
; RESTRICTIONS:
;  Data files are required. Download them from http://lasp.colorado.edu/see
;
; PROCEDURE:
; 1) Set path, read and merge all data files that can be found
; 2) Set return structure
; 3) Set return data array, and use internally
; 4) If /noplot set, return
; 5) Set xrange
; 6) Set label_date string for x-axis  
; 7) Set yrange (either irradiance or uncertainty)
; 8) Do the plot or oplot using color
; 9) Return
;
; EXAMPLES:
;  plot_see_spwx,/help ;displays usage
;  plot_see_spwx,i=8,str=all_data,yrange=[0,1] ;display FUV band with large range
;  plot_see_spwx,i=8,/error ;display associated uncertainty in percent
;  plot_see_spwx,/tsimage,xrange=[2003250,2003365],/ylog
;
; ROUTINES NEEDED:
;  These procedures and functions are available on the TIMED-SEE website,
;   http://lasp.colorado.edu/see
;
;  READ_DAT.pro
;  YD_TO_YFRAC.pro
;  YMD_TO_YD.pro
;  LEAP_YEAR.pro
;
; MODIFICATION HISTORY:
;  8/23/05  DLW  Original file creation, v0.1
;  9/13/05  DLW  Add !fancy=4 to look like plot_see.pro, added second x-axis
;  9/26/05  DLW  Updated usage info, fixed ylabel bug for tsimage, no longer
;                printing usage if index is missing but tsimage is set,
;                added att keyword to return attributes
;  11/16/05 DLW  Added latest_TIMED_SEE_obs.dat to file list, 
;                added sort/uniq filtering code
;
;$Id: plot_see_spwx.pro,v 8.2 2005/11/16 16:27:52 see_sw Exp $
;-

pro plot_see_spwx, index, $
  data=data,help=help,path=path,structure=structure,att=att, $
  xrange=xrange,yrange=yrange,ylog=ylog,oplot=oplot,noplot=noplot, $
  color=color,error=error,tsimage=tsimage,one_xaxis=one_xaxis ;,planet=planet

common plot_see_spwx_reserved, filedata, attributes

if keyword_set(help) or  $
  ((size(index,/type) eq 0) AND (not keyword_set(tsimage))) then begin
do_help:
    print,''
    print,'USAGE: plot_see_spwx [, index][,/help][, path=path][, data=data][, structure=structure][, att=att][, xrange=xrange][, yrange=yrange][, /ylog][, /oplot][, color=color][, /error][,/noplot][,/tsimage]'
    print,''
    print,' plot_see_spwx makes a plot of the integrated solar irradiances'
    print,' All arguments are optional. Many are similar to PLOT arguments.'
    print,''
    print,' ARGUMENT:'
    print,'  index: option to select solar irradiance index'
    print,'      = 1 - (default) Soft X-Rays from 0.1-7 nm (broadband)'
    print,'      = 2 - EUV band from 27.0-34.0 nm'
    print,'      = 3 - EUV from 30.2-30.5 nm (30.4 nm - He II)'
    print,'      = 4 - EUV from 33.3-33.7 nm (33.5 nm - Fe XVI)'
    print,'      = 5 - EUV from 36.6-37.0 nm (36.8 nm - Mg IX)'
    print,'      = 6 - FUV from 121.3-121.8 nm (121.5 nm - H I)'
    print,'      = 7 - FUV from 133.3-133.7 nm (133.5 nm - C II)'
    print,'      = 8 - FUV band from 145.0-165.0 nm'
    print,''
    print,' KEYWORDS:'
    print,'  /help: display this help message'
    print,'  path: the fully qualified path to the TIMED-SEE SWX file(s)'
    print,'  data: option to return the 3d data array [jd,E,sigma]'
    print,'  structure: option to return the entire data structure'
    print,'  att: option to return a string array of structure attributes'
    print,'  xrange: 2 element array passed to PLOT (YYYYDDD format)'
    print,'  yrange: 2 element array passed to PLOT'
    print,'  /ylog: set to use log scale for y axis'
    print,'  /oplot: set to prevent erasing the current plot'
    print,'  color: set to a color value to pass to oplot (with /oplot only)'
    print,'  /error: plot uncertainty instead of irradiance'
    print,'  /noplot: suppresses all plotting, useful for data retrieval only'
    print,' /tsimage: set to make a pretty image of all indices'
    ;print,' planet= 1-9 for Mercury, Venus, Earth, Mars, Jupiter, Saturn, Uranus, Neptune, Pluto'
    if keyword_set(help) then return
endif

!fancy=4
maxstrings=80L

;
; 1) Set path, read and merge all data files that can be found
;
if size(path,/type) ne 7 then path=''

if size(filedata,/type) eq 0 then begin
    ;read in and merge all the files that are available
    ;filelist=file_search(path+'20??_TIMED_SEE_obs.dat',count=count)
    filelist=file_search(path+'*_TIMED_SEE_obs.dat',count=count)
    if count eq 0 then begin
        ;search the SEE path
        path=getenv('see_data_merged')+'/'
        ;filelist=file_search(path+'20??_TIMED_SEE_obs.dat',count=count)
        filelist=file_search(path+'*_TIMED_SEE_obs.dat',count=count)
    endif
    if count eq 0 then begin
        print,'***'
        print,'*** PLOT_SEE_SPWX - ERROR: no files found'
        print,'***'
        goto,do_help
    endif else begin
        print,'found '+strtrim(count,2)+' TIMED-SEE SWx files to read'
        tmp=read_dat(filelist[0],status=status)
        if status eq 0 then begin
            filedata=tmp
        endif else begin
            print,'PLOT_SEE_SPWX - ERROR reading file '+filelist[0]
            print,'skipping'
        endelse
        if count gt 1 then begin
            for i=1,count-1 do begin
                tmp=read_dat(filelist[i],status=status)
                if status eq 0 then begin
                    if size(filedata,/type) eq 0 then filedata=tmp else $
                      filedata=[filedata,tmp]

                    ;read attributes from comments at the top of the file
                    openr,lun,filelist[i],/get_lun
                    attributes=strarr(maxstrings)
                    getthenextline=1
                    attcount=0L
                    str=''
                    while getthenextline eq 1 do begin
                        readf,lun,str
                        if eof(lun) or $
                          (strmid(str,0,1) ne ';' ) then getthenextline=0 $
                          else begin
                            attributes[attcount]=str
                            attcount=attcount+1L
                        endelse
                    endwhile
                    close,lun
                    free_lun,lun
                    if attcount gt 0 then attributes=attributes[0:attcount-1]

                endif else begin
                    print,'PLOT_SEE_SPWX - ERROR reading file '+filelist[i]
                    print,'skipping'
                endelse
            endfor
        endif
    endelse
    ;sort and make uniq list
    filedata = filedata[ uniq( filedata.jd, sort( filedata.jd ) ) ]
    ;stop
    ;convert to mW/m^2
    filedata.e1_7     = filedata.e1_7 * 1000.
    filedata.e27_34   = filedata.e27_34 * 1000.
    filedata.e304     = filedata.e304 * 1000.
    filedata.e335     = filedata.e335 * 1000.
    filedata.e368     = filedata.e368 * 1000.
    filedata.e121     = filedata.e121 * 1000.
    filedata.e1335    = filedata.e1335 * 1000.
    filedata.e145_165 = filedata.e145_165 * 1000.
endif

if size(filedata,/type) eq 0 then begin
    print,'Plot_See_SpWx - ERROR of unknown type, no data available to display'
    goto,do_help
endif

;
; 2) Set return structure
;
structure=filedata
att=attributes

;
; 3) Set return data array, and use internally
;
if size(index,/type) eq 0 then index=1
jd=filedata.jd

case index of
    2: begin
        tmp=filedata.E27_34
        utmp=filedata.U27_34
        title='TIMED-SEE Irradiance: 27-34 nm'
    end
    3: begin
        tmp=filedata.E304
        utmp=filedata.U304
        title='TIMED-SEE Irradiance: 30.4 nm (HeII)'
    end
    4: begin
        tmp=filedata.E335
        utmp=filedata.U335
        title='TIMED-SEE Irradiance: 33.5 nm (FeXVI)'
    end
    5: begin
        tmp=filedata.E368
        utmp=filedata.U368
        title='TIMED-SEE Irradiance: 36.8 nm (MgIX)'
    end
    6: begin
        tmp=filedata.E121
        utmp=filedata.U121
        title='TIMED-SEE Irradiance: 121.5 nm (HI)'
    end
    7: begin
        tmp=filedata.E1335
        utmp=filedata.U1335
        title='TIMED-SEE Irradiance: 133.5 nm (CII)'
    end
    8: begin
        tmp=filedata.E145_165
        utmp=filedata.U145_165
        title='TIMED-SEE Irradiance: 145-165 nm'
    end
    else: begin
        tmp=filedata.E1_7
        utmp=filedata.U1_7
        index=1
        title='TIMED-SEE Irradiance: 0.1-7 nm'
    end
end

data=dblarr(3,n_elements(jd))
data[0,*]=jd
data[1,*]=tmp
data[2,*]=utmp

planetstr=''
if size(planet,/type) ne 0 then begin
    planetnum=planet
    if (planetnum lt 1) or (planetnum gt 9) then planetnum = 3 ; Earth
    lastplanet = planetnum
    case planetnum of
      1: planetstr = 'Mercury'
      2: planetstr = 'Venus'
      3: planetstr = 'Earth'
      4: planetstr = 'Mars'
      5: planetstr = 'Jupiter'
      6: planetstr = 'Saturn'
      7: planetstr = 'Uranus'
      8: planetstr = 'Neptune'
      9: planetstr = 'Pluto'
      else: planetstr = '?'
    endcase
    planetstr=planetstr+' '
    ndates = n_elements(jd)
    planetdist2 = fltarr(ndates)
    planetshift = fltarr(ndates,2,2)
    for k=0,ndates-1 do begin
      earth2planet, planetnum, jd[k], pdist, pshift
      ; division correction is R^2
      planetdist2[k] = pdist^2.    
      ;  force time shifts to be integers
      pshift[0,0] = round(pshift[0,0])
      pshift[1,0] = round(pshift[1,0])
      planetshift[k,*,*] = pshift
    endfor
    ;
    ;  fix end points so shifting will not crash program
    ;
    shiftidx=27.0/median(data[0,*]-shift(data[0,*],1),/even)
    ;for k=0,27 do begin
    for k=0,shiftidx-1 do begin
      if (planetshift[k,0,0] lt (-1*k)) then planetshift[k,0,0] = -1*k
      if (planetshift[ndates-1-k,0,0] gt k) then planetshift[ndates-1-k,0,0] = k
      if (planetshift[k,1,0] lt (-1*k)) then planetshift[k,1,0] = -1*k
      if (planetshift[ndates-1-k,1,0] gt k) then planetshift[ndates-1-k,1,0] = k
    endfor

    factor = 1./planetdist2
    spflux2 = (data[1,long(planetshift[*,0,0])]*planetshift[*,0,1] + $
               data[1,long(planetshift[*,1,0])]*planetshift[*,1,1]) * factor
    data[1,*]=data[1,*] + spflux2
    ;stop
endif ;else use 1-AU earth reference

;
; 4) If /noplot set, return
;
if keyword_set(noplot) then return

;data is now ready for plotting/returning
;
; 5) Set xrange
;
if size(xrange,/type) eq 0 then begin
    xrange=[min(data[0,*]),max(data[0,*])] 
endif else begin
    ;convert year.fraction to julian
    ;limit max xrange value
    xmin=(xrange[0]<xrange[1])>2002022
    xmax=(xrange[0]>xrange[1])>(xmin+(97./1440.)*3.)
    xrange=[xmin,xmax]
    ;xrange=(xrange<2020365L)>2002022L
    xrange=yd_to_jd(xrange)
    xrange=(xrange<max(data[0,*]))>min(data[0,*]) ;enforce data limits
; xrange is input as year fraction
;    xrange=yd_to_jd(yfrac_to_yd(xrange))
endelse

;override second x-axis to prevent overlapping numbers
if xrange[1]-xrange[0] le 11 then one_xaxis=1


;
; 6) Set label_date string for x-axis
;
xrangesize=xrange[1]-xrange[0]
if xrangesize gt 365 then begin
    ; show date as mm/year
    dummy=label_date(DATE_FORMAT=['%N/%Y'])
;    print,'gt 365'
endif else if xrangesize gt 30 then begin
    ; show months
    dummy=label_date(DATE_FORMAT=['%N/%D!C%Y'])
;    print,'gt 30'
endif else if xrangesize gt 3 then begin
    ; show days of month
    dummy=label_date(DATE_FORMAT=['%N/%D!C%Y'])
;    print,'gt 3'
endif else if xrangesize gt 1 then begin
    ; show hours of day
    dummy=label_date(DATE_FORMAT=['%H:%I UT!C%N/%D/%Y'])
;    print,'gt 1'
endif else begin
    ; show hours and minutes
    dummy=label_date(DATE_FORMAT=['%H:%I UT!C%N/%D/%Y'])
;    print,'lt 1'
endelse

;
; 7) Set yrange (either irradiance or uncertainty)
;
if size(yrange,/type) eq 0 then begin
    yrange=fltarr(2)
    idx=where(data[0,*] ge xrange[0] and data[0,*] le xrange[1] and $
              data[2,*] lt 100.,n_idx)
    if n_idx eq 0 then idx=lindgen(n_elements(data[0,*]))
    if keyword_set(error) then yrange[1]=max(data[2,idx],min=minval) $
    else yrange[1]=max(data[1,idx],min=minval)
    yrange[0]=minval
    ;try to ignore negative values
    ;tmp=where(data[1,idx] gt 0,n_tmp)
    ;try also to ignore very large errors
    tmp=where(data[1,idx] gt 0 and data[2,idx] lt 100,n_tmp)
    if n_tmp gt 0 then begin
        if keyword_set(error) then yrange[0]=min(data[2,idx[tmp]]) else $
          yrange[0]=min(data[1,idx[tmp]])
    endif
endif

if keyword_set(tsimage) then begin
    ;filter data to remove negative values
    xtime=where(filedata.e1_7 gt 0 and $
               filedata.u1_7 lt 100 and $
               filedata.e27_34 gt 0 and $
               filedata.u27_34 lt 100 and $
               filedata.e304 gt 0 and $
               filedata.u304 lt 100 and $
               filedata.e335 gt 0 and $
               filedata.u335 lt 100 and $
               filedata.e368 gt 0 and $
               filedata.u368 lt 100 and $
               filedata.e121 gt 0 and $
               filedata.u121 lt 100 and $
               filedata.e1335 gt 0 and $
               filedata.u1335 lt 100 and $
               filedata.e145_165 gt 0 and $
               filedata.u145_165 lt 100 $
               ,n_xtime)
    ;dims are 8 by Value by N_xtime
    rawimg=dblarr(n_elements(filedata.e1_7),8)

    t=filedata[xtime].e1_7
    if keyword_set(ylog) then t=alog(t)
    tmp=t-min(t)
    rawimg[xtime,0] = tmp / max(tmp)

    t=filedata[xtime].e27_34
    if keyword_set(ylog) then t=alog(t)
    tmp=t-min(t)
    rawimg[xtime,1] = tmp / max(tmp)

    t=filedata[xtime].e304
    if keyword_set(ylog) then t=alog(t)
    tmp=t-min(t)
    rawimg[xtime,2] = tmp / max(tmp)

    t=filedata[xtime].e335
    if keyword_set(ylog) then t=alog(t)
    tmp=t-min(t)
    rawimg[xtime,3] = tmp / max(tmp)

    t=filedata[xtime].e368
    if keyword_set(ylog) then t=alog(t)
    tmp=t-min(t)
    rawimg[xtime,4] = tmp / max(tmp)

    t=filedata[xtime].e121
    if keyword_set(ylog) then t=alog(t)
    tmp=t-min(t)
    rawimg[xtime,5] = tmp / max(tmp)

    t=filedata[xtime].e1335
    if keyword_set(ylog) then t=alog(t)
    tmp=t-min(t)
    rawimg[xtime,6] = tmp / max(tmp)

    t=filedata[xtime].e145_165
    if keyword_set(ylog) then t=alog(t)
    tmp=t-min(t)
    rawimg[xtime,7] = tmp / max(tmp)

    ;img=congrid(byte(rawimg*255),512,256)
    img=byte(rawimg*255)
    ;create the window with no data first
    ynames=['0.1-7','27-34','30.4','33.5','36.8','121.5','133.5','145-165']
    ;force yrange to fixed value
    yrange=[0,n_elements(ynames)]
    ylabelpos=findgen(n_elements(ynames))+.5
    plot,data[0,*],lindgen(n_elements(data[0,*])), /nodata, $
         ys=1, yr=yrange, $;ylog=ylog,$
         xs=5,xr=xrange, xtickunits=['Time'],xtickformat='LABEL_DATE',ps=10, $
         title=planetstr+'TIMED-SEE SWx Irradiance', $
         ytickv=ylabelpos,yticks=7,ytickname=ynames
    px=!X.window * !d.x_vsize ; size of plot window in device pixels
    py=!Y.window * !d.y_vsize
    sx=px[1]-px[0]+1
    sy=py[1]-py[0]+1
    ;scale using only the portion of the image within the xrange
    idx=where(data[0,*] ge xrange[0] and data[0,*] le xrange[1])
    ; create image adjused to plot size
    timg=congrid(img[idx,*],sx,8)
    ; rescale max per row back up to 255
    for i=0,7 do timg[*,i]=bytscl(timg[*,i]) 
    ;or byte(ftimg[*,i]*255.d0/max(ftimg[*,i]))
    ttimg=congrid(timg,sx,sy)
    tvscl,ttimg,px[0],py[0]
    ; show x-axis properly
    axt=[6,5,4,3,2]
    xticks=where(((xrange[1]-xrange[0])*24. mod axt-1) eq 0,nxticks)
    if nxticks eq 0 then xticks=2
    plot,data[0,*],data[1,*],/noerase ,/nodata, $
         ys=4+1, yr=yrange, xticks=(axt)[xticks[0]], $
         xs=1,xr=xrange, xtickunits=['Time'],xtickformat='LABEL_DATE'

    ; do not continue after drawing the image
    goto,thereturn
endif

;
; 8) Do the plot or oplot using color
;
if keyword_set(oplot) then begin
    ;
    if keyword_set(error) then begin
        ;uncertainty
        oplot,data[0,*],data[2,*],color=color
    endif else begin
        ;irradiance
        oplot,data[0,*],data[1,*],color=color
        ;stop
    endelse
endif else begin
    if keyword_set(error) then begin
        ;uncertainty
        plot,data[0,*],data[2,*],ys=1,yr=yrange,ylog=ylog,$
             xs=1,xr=xrange, xtickunits=['Time'],xtickformat='LABEL_DATE',ps=10, $
             title=planetstr+title,ytit='Relative Accuracy (%)'
    endif else begin
        ;irradiance
        if keyword_set(one_xaxis) then xs=1 else xs=8
        plot,data[0,*],data[1,*],/ynozero,yr=yrange,ylog=ylog,$
             xs=xs,xr=xrange, xtickunits=['Time'],xtickformat='LABEL_DATE',ps=10, $
             ytit='Irradiance [mW/m!U2!N/nm]', title=planetstr+title+'!C', $
             ymargin=[4,3.8],xmargin=[10,5.5],xcharsize=0.9
        if not keyword_set(one_xaxis) then $
          axis, xaxis=1,xcharsize=0.9,xrange=yd_to_yfrac(jd_to_yd(!x.crange))
    endelse
endelse

;
; 9) Return
;
thereturn:
;convert time format from julian to year fraction
;df is day fraction
df=(((filedata.hh*60.d0)+filedata.mm)*60.d0+filedata.ss)/86400.d0
;yd is 7-digit year and day of year (long integer)
ymd_to_yd, long(filedata.year), long(filedata.mo), long(filedata.dd), yd
;yd_to_yfrac converts it to a year and fractional year (accounts for leap)
data[0,*]=yd_to_yfrac(double(yd)+df)
;stop
return
end
