;+
; NAME:
;	earth2planet
;
; PURPOSE:
;	Provide time shift and solar distance information so that Earth-based
;	solar irradiance measurements can be estimated at another planet
;
; CATEGORY:
;	Library procedure
;
; CALLING SEQUENCE:  
;	earth2planet, planet, jdtime, solar_distance, time_shift_days
;
; INPUTS:
;	planet			Planet name or number, e.g.  1 = "Mercury"
;	jdtime			Time for calculation given in Julian Date
;
; OUTPUTS:  
;	solar_distance	Distance from planet to sun in A.U.
;	time_shift_days	Amount to shift Earth-based irradiance in days
;	                2 x 2 array:  tsd[0,*] = + / - shift days
;                                 tsd[1,*] = weights in making average
;
; COMMON BLOCKS:
;	none
;
; PROCEDURE:
;	1.  Check input parameters
;	2.  Calculate position of Earth and planet using ephem.pro in heliocentric frame
;	    and set "solar_distance" to the planet's distance in A.U.
;	3.  Calculate solar rotation angle between Earth and given planet
;	    and determine optimal number of days for time shifting 
;	    Earth-based irradiance to the planet
;
; MODIFICATION HISTORY:
;	26 May 2005    Tom Woods     Original Creation
;
;
;+

pro earth2planet, planet, jdtime, solar_distance, time_shift_days, debug=debug

;
;	1.  Check input parameters - can interactive get input too
;
;	set default output values
;
solar_distance = 1.0
time_shift_days = 0.0

if (n_params() lt 2) then begin
  print, 'USAGE:  earth2planet, planet, jdtime, solar_distance, time_shift_days'
  print, ' '
  print, '  INPUT:  planet    name or number, e.g. 1 = "Mercury"'
  print, '          jdtime    time in Julian date'
  print, ' '
  print, '  OUTPUT: solar_distance    planet distance in A.U.'
  print, '          time_shift_days   time to shift Earth-based irradiance in days'
  print, ' '
endif

;
;	Check (or interactively get) the input "planet" parameter
;
if (n_params() lt 1) then begin
  planet = 0
  print, '  1   Mercury'
  print, '  2   Venus'
  print, '  3   Earth'
  print, '  4   Mars'
  print, '  5   Jupiter'
  print, '  6   Saturn'
  print, '  7   Uranus'
  print, '  8   Neptune'
  print, '  9   Pluto'
  print, ' '
  read, 'ENTER PLANET NUMBER (1-9) ? ', planet
endif
planetnum = 0
pinput = planet[0]
psize = size(pinput)
if (psize[1] eq 7) then begin
  ;  Figure out which Planet string was given
  pstr = strupcase(strmid(pinput,0,3))
  pall = [ "SUN", "MER", "VEN", "EAR", "MAR", "JUP", "SAT", "URA", "NEP", "PLU" ]
  planetnum = where( pall eq pstr )
endif else planetnum = pinput
if (planetnum lt 1) or (planetnum gt 9) then begin
  print, 'EARTH2PLANET ERROR:  "planet" is not a valid planet name or number (1-9)"
  return
endif

;
;	Check (or interactively get) the input "jdtime" parameter
;
if (n_params() lt 2) then begin
  jdtime = 0.D0
  print, ' '
  read, 'ENTER TIME (in Julian date) ? ', jdtime
endif
jdinput = jdtime[0]
if (jdinput lt 2415000L) then begin
  print, 'EARTH2PLANET ERROR:  "jdtime" is not valid Julian date for year 1900 or later"
  return
endif

;
;	2.  Calculate heliocentric position of Earth and planet using helio.pro
;
;	HELIO USAGE:  HELIO, JD, PLANET_NUMBER, HRAD, HLONG, HLAT, [/RADIAN]
;         where output of HRAD is in A.U.
;                         HLONG and HLAT are in degrees
;
;	Get Earth's position in HELIOSPHERIC frame
;
helio, jdinput, 3, erad, elong, elat
;
;	Get Planet's position in HELIOSPHERIC frame
;
helio, jdinput, planetnum, prad, plong, plat
;
;	Set "solar_distance" to the planet's distance in A.U.
;	
earth_distance = erad
solar_distance = prad

;
;	3.  Calculate solar rotation angle between Earth and given planet
;	    and determine optimal number of days for time shifting 
;	    Earth-based irradiance to the planet
;
angle = elong - plong   ; in degrees
if (angle gt 180.) then angle = angle - 360.
if (angle lt -180.) then angle = angle + 360.
;
;	Assumes 27-days for solar rotation (360 degrees)
;
days_rotate = 27.0
flt_shift = angle * days_rotate / 360.
;
;	Make time_shift_days (tsd) a 2 x 2 array
;	    tsd[0,*] = + / - shift days
;       tsd[1,*] = weights in making average
;
;	Weight 7-13.5 days but force weight of 1.0 for days less than 7 days
;
shift1 = flt_shift
if (shift1 lt 0) then shift2 = days_rotate + shift1 else shift2 = shift1 - days_rotate
wt1 = (0.5 + (days_rotate/2. - abs(shift1)) * 0.077) < 1.0
wt2 = 1. - wt1
time_shift_days = [ [shift1, shift2], [wt1, wt2] ]

;
;	print results if all output parameters are not given
;
if ((n_params() lt 3) or keyword_set(debug)) then begin
  print, ' '
  print, 'INPUT:  Planet ', strtrim(planetnum,2), '  Time = ', strtrim(jdinput,2)
  print, 'PLANET SOLAR DISTANCE = ', strtrim(solar_distance,2), ' A.U.'
  print, 'TIME SHIFT FROM EARTH = ', strtrim(time_shift_days[0],2), ' days'
  if (keyword_set(debug)) then begin
     print, ' '
     print, 'Earth at  ', erad, elong, elat
     print, 'Planet at ', prad, plong, plat
     print, 'Time_Shift_Days ARRAY = ', time_shift_days, $
         format='(A,F8.3,"  [",F6.3,"]  &",F8.3,"  [",F6.3,"]")'
  endif
endif

return
end
