;+
; Procedure: thm_write_probe_timing_tables.pro
;
; Purpose:  This routine takes two string file names as input arguments, reads all 'phases_???.txt' files present in the WD,
;           determines the primary (statistically significant) apid sampling times in terms on one-second-mark 'ticks' and spin 'sectors',
;           writes 'ticks' and 'sectors' found to probe timing table files, and generates png images of the timelines per apid. 
; 
; Inputs:   Parameters: ticks_table_file, sectors_table_file (to be generated by the routine)
;           Non-Parameters: phases_xxx.txt files in PWD (ASCII files containing apid:xxx header times, one second mark phases, spin phases, and delta_phi timeline)
;             These ASCII files need to be generated by write_probe_timing_phases.bash  
;                    
; Keywords:
;           NO_PNG: If set, png images are not produced
;           TEST:   If set, ALL tick counts are recorded and written to file (except for apid:410)
;                   Otherwise, only tick counts = 1,2,8,16 and 32 are recorded and written to file
;              
; Outputs:  ticks_table_file, sectors_table_file and png images of phases per apid (default)
;              
; Examples:
;           thm_write_probe_timing_tables,'ticks_table.txt','sectors_table.txt'
;           thm_write_probe_timing_tables,'ticks_table.txt','sectors_table.txt',/no_png
;           thm_write_probe_timing_tables,'ticks_table.txt','sectors_table.txt',/no_png,/test
;
; See Also:
;           ${TMTOOLS_ROOT}/src/scripts/write_probe_timing_phases.bash
;
; $LastChangedBy: $
; $LastChangedDate: $
; $LastChangedRevision: $
; $URL: $
;-

Pro thm_write_probe_timing_tables, ticks_table_file, sectors_table_file, TEST=TEST, NO_PNG=NO_PNG

if (N_Params() lt 2) then message,'Usage: thm_write_probe_timing_tables, ticks_table_file, sectors_table_file, [/TEST], [/NO_PNG]'

;;; Set plotting colors
!p.color = 0              ;;; foreground = black
!p.background = 16777215  ;;; background = white

;;; Search and retrieve phase_xxx.txt files produced by thm_write_probe_timing_tables.bash
phases_files = file_Search('phases_*.txt')
if keyword_set(test) then begin
  p410 = where(phases_files eq 'phases_410.txt')
  n = n_elements(phases_files)-1
  phases_files = [ phases_files[0:p410 - 1], phases_files[p410 + 1:n] ]
endif  

;;; Allocate and initialize ticks and sectors string-arrays
ticks = strarr(256)
for i = 0 , 255 do ticks[i] = 'Tick'+(string(i))+':'+string(9B)
sectors = strarr(n_elements(phases_files))

;;; Loop over phase_files, determine 1secmark-tick and spin-sector counts per apid 
for j = 0,n_elements(phases_files)-1 do begin
  phases_file = phases_files[j]
  phases = read_ascii(phases_file)
  apid = strmid(phases_file,7,3)
  sectors[j] = 'Apid '+ apid +':'+string(9B)

  ;;; 1secmark phase tick count portion
  onesecmark = round(256d*phases.field1[1,*])
  h_mark = histogram(onesecmark,min=0,max=255)
  ticks_found = where(abs(h_mark-mean(h_mark)) gt 1.*stddev(h_mark))
  if (min(ticks_found) eq -1) then n = 0 else n = n_elements(ticks_found) 
  if keyword_set(test) then begin
    for i = 0,(n-1) do ticks(ticks_found[i]) = ticks(ticks_found[i]) + apid + ' '
  endif else begin
    if max(n eq [1,2,8,16,32]) then begin
      for i = 0,(n-1) do ticks(ticks_found[i]) = ticks(ticks_found[i]) + apid + ' '
    endif
  endelse

  ;;; Spin phase sector count portion
  spin = 32d*phases.field1[2,*]
  h_spin = histogram(spin,min=0,max=31)
  sectors_found = where(h_spin gt 2.*stddev(h_spin))
  n = n_elements(sectors_found)
  sectors_found = strtrim(string(sectors_found),2)
  if (n lt 20) then begin
    for i = 0,(n-1) do sectors[j] = sectors[j] + ' ' + sectors_found[i]
  endif
  
  ;;; Check if user requests generation of png images
  if keyword_set(no_png) then begin dprint, 'Apid: ',strtrim(apid,2),', Ticks: ',strtrim(n_elements(ticks_found),2),', Sectors: ',strtrim(n_elements(sectors_found),2)
  endif else begin
    ;;; store timelines as tplot variables, plot them, and save them as png images
    time_array = reform(double(phases.field1[0,*]))
    store_data,'apid_'+apid+'_1secmark_ticks',data={x:time_array,y:reform(double(round(256d*phases.field1[1,*])))}
    store_data,'apid_'+apid+'_spin_phase_sectors',data={x:time_array,y:reform(double(32d*phases.field1[2,*]))}
    store_data,'apid_'+apid+'_delta_phi',data={x:time_array,y:reform(double(phases.field1[3,*]))}  
    tplot_options,title='APID: '+apid+' Phases'
    options,'apid_'+apid+'_1secmark_ticks',yrange=[0d,256d],psym=3,ytitle='OneSecMark Ticks [0,255]'
    options,'apid_'+apid+'_spin_phase_sectors',yrange=[0d,32d],psym=3,ytitle='Spin Phase Sectors [0,31]'
    options,'apid_'+apid+'_delta_phi',psym=3,yrange=[-360d,360d],ytitle='Delta Phi [degrees]'
    tplot,['apid_'+apid+'_1secmark_ticks','apid_'+apid+'_spin_phase_sectors','apid_'+apid+'_delta_phi']
    makepng,'phases_'+apid
    store_data,['apid_'+apid+'_1secmark_ticks','apid_'+apid+'_spin_phase_sectors','apid_'+apid+'_delta_phi'],/delete
  endelse

endfor    ;;; end of loop over phase_files

;;; Write ticks and sectors arrays to table files
openw,1,ticks_table_file
for i = 0, 255 do printf,1,ticks[i]
close,1

openw,2,sectors_table_file
for i = 0, (n_elements(phases_files)-1) do printf,2,sectors[i]
close,2

end 
