	PRO READ_VAX_BLOCK,UNIT,BYTE_ARRAY,END_OF_FILE,ERROR_FLAG,	$
		EXTEND=EXTEND,TARGET_ARCH=TARGET
;+
; NAME:
;	READ_VAX_BLOCK
; PURPOSE:
;	Reads one block from a VAX Fortran unformatted file.
; CALLING SEQUENCE:
;	READ_VAX_BLOCK, UNIT, BYTE_ARRAY, END_OF_FILE  [, ERROR_FLAG ]
; INPUTS:
;	UNIT        = Logical unit number of VAX Fortran unformated file.
; OUTPUTS:
;	BYTE_ARRAY  = Data in block returned as byte array.
;	END_OF_FILE = Switch containing whether end-of-file was reached before
;		      data could be read.
; OPTIONAL OUTPUT PARAMETER:
;	ERROR_FLAG  = Switch containing whether or not an error occurred during
;		      processing.
; OPTIONAL KEYWORD PARAMETER:
;	EXTEND      = If passed, then block length checking is disabled.
;		      Allows blocks larger than 2048 bytes to be read in.
;	TARGET_ARCH = Name (string) of desired target architecture if using
;		      this function on a VAX.  Otherwise !VERSION.ARCH is used
;		      to determine the conversion.
; SUBROUTINES CALLED:
;	CONV_VAX_UNIX
; RESTRICTIONS:
;	The VAX file must be copied over as binary.  The file must be opened
;	*WITHOUT* the /F77_UNFORMATTED switch.
; PROCEDURE:
;	The block header is read, and the number of bytes, and a code word, are
;	read in and checked.  The data is then read in as a byte array.  If
;	necessary, an extra byte is read in to word align.
; MODIFICATION HISTORY:
;	William Thompson, January, 1990.
;			  December, 1990, added TARGET_ARCH keyword.
;-
;
	ON_IOERROR, ERROR
	ERROR_FLAG = 0
	EXTEND = KEYWORD_SET(EXTEND)
	IF N_ELEMENTS(TARGET) EQ 1 THEN ARCH = TARGET ELSE ARCH = !VERSION.ARCH
;
;  Check the number of parameters passed.
;
	IF N_PARAMS(0) LT 3 THEN BEGIN
		PRINT,'*** READ_VAX_BLOCK must be called with 3-4 parameters:'
		PRINT,'      UNIT, BYTE_ARRAY, END_OF_FILE  [, ERROR_FLAG ]'
		PRINT,"  Optional keywords:  EXTEND, TARGET_ARCH"
		RETALL
	ENDIF
;
;  First read in the block descriptor.
;
READ_HEADER:
	IF EOF(UNIT) THEN GOTO, ENDFILE
	N_BYTES = 0
	CODE = 0
	FORRD,UNIT,N_BYTES,CODE
	N_BYTES = CONV_VAX_UNIX(N_BYTES,TARGET_ARCH=ARCH) - 2
	CODE = CONV_VAX_UNIX(CODE,TARGET_ARCH=ARCH)
;
;  If N_BYTES is zero or negative and the end of the file is reached, then skip
;  forward to signal done.
;
	IF (N_BYTES LT 1) AND EOF(UNIT) THEN GOTO, ENDFILE
;
;  Check the number of bytes and the code.  The number of bytes should be
;  between 1 and 2048, and CODE should be between 0 and 3 (?).
;
	IF ((N_BYTES LT 1) OR (N_BYTES GT 2048)) AND (NOT EXTEND) THEN BEGIN
		PRINT,"*** Unreasonable number of bytes, value= " + $
			STRTRIM(N_BYTES) + ", routine READ_VAX_BLOCK."
		GOTO, ERROR
	END ELSE IF (CODE LT 0) OR (CODE GT 3) THEN BEGIN
		PRINT,"*** Unreasonable code, value= " + STRTRIM(CODE) + $
			", routine READ_VAX_BLOCK."
		GOTO, ERROR
	ENDIF
;
;  If CODE is odd, then initialize the output array.  Otherwise, just add to
;  existing array.
;
	IF (CODE MOD 2) EQ 1 THEN BEGIN
		BYTE_ARRAY = BYTARR(N_BYTES)
		FORRD,UNIT,BYTE_ARRAY
	END ELSE BEGIN
		B = BYTARR(N_BYTES)
		FORRD,UNIT,B
		BYTE_ARRAY = [BYTE_ARRAY,B]
	ENDELSE
;
;  If CODE is 0 or 1, we're not done yet.
;
	IF CODE LT 2 THEN GOTO, READ_HEADER
;
;  If N_BYTES is odd, then read in an extra byte.  This byte should be zero.
;
	IF (N_BYTES MOD 2) NE 0 THEN BEGIN
		B = 0B
		FORRD,UNIT,B
		IF B NE 0 THEN BEGIN
			PRINT,"*** Nozero filler byte, value= " + $
				STRTRIM(B) + ", routine READ_VAX_BLOCK."
			GOTO, ERROR
		ENDIF
	ENDIF
	END_OF_FILE = 0
	RETURN
;
;  End-of-file was reached.
;
ENDFILE:
	END_OF_FILE = 1
	RETURN
;
;  Error point.
;
ERROR:
	PRINT,"*** Error occurred reading file, routine READ_VAX_BLOCK."
	ERROR_FLAG = 1
	IF N_PARAMS(0) EQ 4 THEN RETURN ELSE RETALL
	END
