function get_clut, dummy, for_gm=for_gm, debug=debug ; count lookup table
; return lookup table for converting from telemetry bytes to measured counts 
;   on HAPI and LAPI
; Bobby Candey, Atlantic Research
; fixed errors Bobby Candey, Hughes STX 1992 September 18
; added check for divide by zero; Bobby Candey, NASA GSFC Code 632; 1995 Jan 11
 
telemetered = bindgen(256)
d = telemetered and 15b ; 4 penultimate most significant bits (MSB-1:MSB-5)
s = fix(telemetered / 16) ; inverted number of shifts until MSB shifts into 
				; shift register
delta = (2.^(s-2) - 1.) / 2. ; averaged roundoff (halfway to next level)
w2 = where(s le 2)
delta(w2) = 0. ; no roundoff error on small counts
realcount = float(d+16)*2.^(s-2) + delta
; where 16 is thrown-out MSB
wEven = where((s eq 0) and (d mod 2 eq 0))
wOdd = where((s le 1) and (d mod 2 ne 0))
realcount(wEven) = d(wEven) / 2. ; special case for S=0
realcount = realcount - 1. ; subtract extra accumulator-close gate pulse
if not keyword_set(for_gm) then begin
  realcount(0) = -1.0 ; fill values for "impossible" telemetered values
  realcount(wOdd) = -1.0 ; fill values for "impossible" telemetered values
endif ; not for_gm
if keyword_set(debug) then begin
  print, 'Index         shift S   data D    realCounts'
  for i=0,255 do print,i,'   ',s(i),'   ',d(i),'   ',realcount(i)
endif
return, realcount
end ; get_clut
function do_flux, sensors, inputCounts, factor
common fluxCom, count_lookup, delE_LAPI, delE_HAPI, flux_opt1, satID, filename
  Geometric_5x5    = 15.88235
;                 divide by 15.88235 for the 5 x 5 deg detectors (field-aligned)
  flux = count_lookup( inputCounts )   ; Calculate Actual Counts from TM counts
; flux = fltarr( nsens, hlapi.nsteps, nsweeps )
  if ( flux_opt1 ne 1 ) then begin ; not counts only
    fluxSize = size(flux)
    case fluxSize(0) of
	2: nsweeps = 1
	3: nsweeps = fluxSize(3)
	else: message, 'Do_flux: requires inputCounts of dimension 2 or 3'
    endcase
    nsens = fluxSize(1)
    nsteps = fluxSize(2)
    for k = 0L, nsweeps-1 do begin
;     if (nsweeps le 1) then sensorsK = sensors(*,k) else sensorsK = sensors(*,k)
      sensorsK = sensors(*,k)
;     if (nsens ne n_elements(sensorsK)) then print, 'Wrong # of sensors at ', k
      if ( satID eq 'L' ) then begin
        gfactor = 1.0 / delE_LAPI(sensorsK)
        field_aligned = where (( (sensorsK ge  0) and (sensorsK le  3) ) or   $
                               ( (sensorsK ge 26) and (sensorsK le 29) ), wc)
        if (wc gt 0) then gfactor(field_aligned) = gfactor(field_aligned) * $
  					Geometric_5x5
;        if (wc le 0) then print, 'No field-aligned detectors at sweep ', k
      endif else begin
        gfactor = 1.0 / delE_HAPI(sensorsK)
      endelse
      if (nsweeps le 1) then begin
        for j=0L,nsteps-1 do if (factor(j) eq 0.0) then flux(*,j) = 0.0 else $
			flux(*,j) = (flux(*,j) * gfactor) / factor(j)
      endif else begin
        for j=0L,nsteps-1 do if (factor(j) eq 0.0) then flux(*,j,k) = 0.0 else $
			flux(*,j,k) = (flux(*,j,k) * gfactor) / factor(j)
      endelse
    endfor ; k sweeps
  endif ; flux_opt gt 1
  return, flux
end ; do_flux

pro calc_flux, HLAPI, EA, flux_option, flux_type, energy, flux_elec, $
                 flux_ions, fluxOne_elec, fluxOne_ions, $
		 geiger_0, geiger_90, GM=GM, only=only
common fluxCom, count_lookup, delE_LAPI, delE_HAPI, flux_opt1, satID, filename
if keyword_set(only) then onlyKey = strUpCase(strMid(only,0,1)) else onlyKey=' '
;if (n_elements(filename) eq 0) then filename = ' '
;if (((onlyKeys eq 'E') or (onlyKeys eq 'I')) and (filename ne HLAPI.SATM_file)) $
;  then print, 'Please rerun CALC_FLUX first' else filename = HLAPI.SATM_file
;#### check for change in step info to re-run tables
filename = HLAPI.SATM_file
satID = hlapi.satID
flux_opt1 = flux_option
; GM is optional structure of Geiger-Muller data
; only = 'Electrons', 'Ions', 'GM' to only calculate these values
;
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
; calculates fluxes for LAPI and HAPI
; changed to have minimum flux of zero and use true del_E values
; 1992 April 15 Bobby Candey and Norman Reese, Hughes STX
; 1992 September 18 BC updated and checked
; 1993 Feb 18, BC, changed deltaE_E values to use response curve values
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;
;build table to convert telemetered counts to actual counts
if (n_elements(count_lookup) eq 0) then begin ; count_lookup not defined
  count_lookup = get_clut(0)
endif ; count_lookup
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;
;Calculate Geiger-Muller Counts Also
;
;Calculate Actual Counts from TM counts and convert to Flux (*517.2)
if ((satID eq 'L') and keyword_set(GM)) then begin
    count_lookup_gm = get_clut(0,/for_gm)
    geiger_0  = count_lookup_gm ( GM.gm_0 )  * 517.2 ; # / (cm**2 ster sec)
    geiger_90 = count_lookup_gm ( GM.gm_90 ) * 517.2 ; # / (cm**2 ster sec)
endif ; GM
if (onlyKey eq 'G') then return
;
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

;Energy levels in eV
energies = $
      [ 31143.75,  26993.75,  23381.25,  20250.00,  17531.25,  $
        15212.50,  13206.25,  11425.00,   9900.00,   8581.25,  $
         7425.00,   6465.00,   5568.75,   4831.25,   4187.50,  $
         3625.00,   3121.25,   2701.875,  2338.750,  2025.00,  $
         1753.125,  1520.00,   1319.375,  1141.250,   984.375, $
          853.125,   738.688,   639.563,   553.625,   480.313, $
          416.750,   360.125,   313.269,   271.213,   234.644, $
          203.019,   175.663,   152.244,   132.031,   114.194, $
           98.931,    85.700,    74.188,    64.256,    55.656, $
           48.281,    41.913,    36.306,    31.306,    27.163, $
           23.569,    20.444,    17.763,    15.444,    13.463, $
       	   11.688,    10.156,     8.844,     7.719,     6.706, $
            5.875,     5.138,     4.525,     0.000 ]

;Efficiency factor for electron detectors
elecEfficiency = [ .26453, .28030, .29687, .31418, .33226, .35076,   $
                   .36988, .39015, .41084, .43209, .45416, .47674,   $
                   .49949, .52243, .54578, .56951, .59419, .61792,   $
                   .64148, .66468, .68747, .70946, .73061, .75147,   $
                   .77179, .79045, .80815, .82472, .84014, .85414,   $
                   .86697, .87897, .88931, .89889, .90742, .91488,   $
                   .92133, .92678, .93138, .93531, .93852, .94118,   $
                   .94337, .94514, .94658, .94774, .94868, .94945,   $
                   .95009, .95059, .95100, .95133, .95159, .95181,   $
                   .95199, .95214, .95227, .95237, .95245, .95252,   $
                   .95258, .95263, .95267, .0 ]

;Efficiency factor for ion detectors
ionsEfficiency = 0.65

;Geometric Factor for all HAPI and LAPI 5x20 degree detectors
geomFactor = 2.16e-4 ; (cm**2 sr)
geomFactor_mks = 2.16e-8 ; (m**2 sr)
;geomFactor = 2.13e-4 ; (cm**2 sr) ; value in blue book
;geomFactor_mks = 2.13e-8 ; (m**2 sr)
Geometric_5x5    = 15.88235
;                 divide by 15.88235 for the 5 x 5 deg detectors (field-aligned)

delE = 0.32 ; Energy width, standard
;delE_LAPI = replicate(0.32, 32) ; old values
; taken from instrument response curves median values
;### ions may be too low due to 0.65 correction
delE_LAPI = [0.32, 0.26, 0.32, 0.23, 0.33, 0.19, 0.33, 0.20, $
	     0.34, 0.23, 0.34, 0.27, 0.34, 0.21, 0.33, 0.24, $
	     0.31, 0.25, 0.33, 0.22, 0.32, 0.26, 0.34, 0.24, $
	     0.39, 0.25, 0.32, 0.20, 0.35, 0.25, -0.32, -0.32]
delE_HAPI = [0.32, 0.27, 0.30, 0.22, 0.34, 0.23, 0.32, 0.22, $
	     0.30, 0.23, -0.32, -0.32]
;;delE_LAPI = [0.33, 0.33, 0.32, 0.32, 0.33, 0.33, 0.33, 0.33, $
;;	     0.34, 0.34, 0.34, 0.34, 0.34, 0.34, 0.33, 0.33, $
;;	     0.33, 0.33, 0.33, 0.33, 0.32, 0.32, 0.34, 0.34, $
;;	     0.39, 0.39, 0.32, 0.32, 0.35, 0.35, -0.32, -0.32]
;;delE_HAPI = [0.32, 0.32, 0.32, 0.32, 0.32, 0.32, 0.32, 0.32, $
;;	     0.32, 0.32, -0.32] ; ##### exact values unknown 0.32+/-0.02

;accumulation interval
case hlapi.steps_per_second of
  16:	del_T = 5.96e-2 ; sec (16 steps/sec)
  32:	del_T = 2.83e-2 ; sec (32 steps/sec)
  else:	del_T = 1.27e-2 ; sec (64 steps/sec)
endcase

ergs_eV = 1.6021892e-12 ; ergs / eV
eV2J = 1.6021892e-19 ; Joules
emass = 9.107e-31 ; kg
imass = 1.672e-27 ; kg
;A1 = geomFactor * delE * del_T

;Build array of energy steps present for this SATM range (assumes both power 
;	supplies using same stepping #####)
if (satID eq 'H') then begin
  energy_step0 = hlapi.start_pps + (hlapi.skip_pps + 1) * INDGEN(hlapi.nsteps)
endif else begin
  energy_step0 = hlapi.start_pps1 + (hlapi.skip_pps1 + 1) * INDGEN(hlapi.nsteps)
endelse
; test match to pps
energy_step1 = hlapi.pps
w = where((energy_step0 - energy_step1) ne 0, wc)
if ((n_elements(energy_step0) ne n_elements(energy_step1)) or (wc gt 0)) then begin
  print, 'Conflict between PPS levels and Energy_step0'
  print,'This is the correct calc_flux'
  print, energy_step1, format='(32I3)'
  print, energy_step0, format='(32I3)'
;  energy_step = energy_step1
  if (satID eq 'H') then energy_step = energy_step1 else $
					energy_step = energy_step0
endif else begin
  energy_step = energy_step0
endelse
Energy = energies(energy_step)

;Build Flux conversion factors for type of flux requested
case flux_option of
  2: begin ; Differential number flux (#/(cm**2 sr eV sec))
	Ions_Factor = energy * ionsEfficiency * geomFactor * del_T
	Elec_Factor = energy * elecEfficiency(energy_step) * geomFactor * del_T
        flux_type   = "Differential Number Flux (# / (cm**2 sr eV sec) )"
    end
  3: begin ; Differential energy flux ( ergs / (cm**2 sr eV sec) )
	Ions_Factor = replicate( (ionsEfficiency * geomFactor*del_T / ergs_eV),$
					hlapi.nsteps )
	Elec_Factor = elecEfficiency(energy_step) * geomFactor * del_T / ergs_eV
        flux_type   = "Differential Energy Flux ( ergs / (cm**2 sr eV sec) )"
    end
  4: begin ; Distribution function, s3/m6 (* 10e-18?)
	Ions_Factor = energy^2 * (ionsEfficiency * 2.0 * del_T * $
			geomFactor_mks * (eV2J / imass)^2)
	Elec_Factor = energy^2 * elecEfficiency(energy_step) * (2.0 * del_T * $
			geomFactor_mks * (eV2J / emass)^2)
;	Ions_Factor=energy^2*ionsEfficiency * geomFactor*del_T / 5.448e5
;	Elec_Factor=energy^2*elecEfficiency(energy_step)*geomFactor*del_T/0.1616
        flux_type   = "Distribution Function ( s**3 / m**6 )"
    end
  else: begin ; counts only
	Ions_Factor = replicate( 1., hlapi.nsteps )
	Elec_Factor = Ions_Factor
        flux_type   = "Counts Only"
        flux_option    = 1
    end
endcase

if (onlyKey ne 'I') then begin ; electrons
  flux_elec = do_flux(EA.sensors_elec, EA.Counts_elec, elec_factor)
  if (onlyKey eq 'E') then return ; finished with electrons, skip ions
endif

flux_ions = do_flux(EA.sensors_ions, EA.Counts_ions, ions_factor)
if (onlyKey eq 'I') then return ; finished with ions, skip fluxOne

;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
; Calculates the electron flux sensitivity limit for plotting
;    sensitivity curves at one-count level (TM of 4 equals 1 real count)
fluxOne_elec = do_flux([4,0], replicate(4b,2,hlapi.nsteps,1), elec_factor)
fluxOne_ions = do_flux([9,1], replicate(4b,2,hlapi.nsteps,1), ions_factor)
;### choose sensors with del_E near 0.32 and small enough for HAPI

return
end ; calc_flux
