/******************************************************************************
                          qstat.cpp  -  description
                             -------------------
    begin                : Fri Jul 20 15:20:00 CET 2001

    author               : Hans Vaith
    email                : hav@mpe.mpg.de
    copyright            : (C) 2001 by Max-Planck-Institut fr extra-
                           terrestrische Physik, D-85740 Garching

    Cluster-II EDI: extract WW pacmo1/5 quality and display hourly statistics

    modifications
******************************************************************************
$Id: qstat.cpp,v 1.3 2004/08/31 11:27:42 hav Exp $
$Log: qstat.cpp,v $
Revision 1.3  2004/08/31 11:27:42  hav
bug fix in cdds.cpp (check for zero file size)
more dev in corb

Revision 1.2  2004/06/25 12:47:17  hav
added overall version number (to be also used in CVS tag) for
version control in Cluster Active Archive Project

Revision 1.1.1.1  2002/10/14 17:08:19  hav
c/c++ sources

Revision 1.12  2001/11/12 17:31:55  hav
added RDM file support

Revision 1.11  2001/09/10 10:48:26  hav
added support for CDDS files

Revision 1.10  2001/08/10 11:12:37  hav
ficxed [3~[3~[3~ODODOCOCOCOBOA
fixed error with ignoring submodes 2,3 , 6 (NM2,NM3,BM2)

Revision 1.9  2001/08/02 11:58:03  hav
fixed bug with filling of uncovered hours; better debugging info

Revision 1.8  2001/08/01 15:14:19  hav
bug fixes, more elaborate use of new timelib functions

Revision 1.7  2001/07/27 11:28:09  hav
output for MSF input files is now also generated if there are no WW data at all

Revision 1.6  2001/07/27 09:04:15  hav
strip off path from input file name for use in output file name

Revision 1.5  2001/07/26 16:02:37  hav
many changes: new flags, 10 minute interval statistics

Revision 1.4  2001/07/25 16:41:51  hav
fixed missing initialization of construct_ofn

Revision 1.3  2001/07/25 16:28:06  hav
fixed missing initialization of counters in case no WW data at all have been found

Revision 1.2  2001/07/25 15:02:27  hav
fixed missing $-sign behind $Id

******************************************************************************/


/******************** INCLUDE FILES ******************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "openfile.h"
#include "cl2tm.h"
#include "dealias.h"
#include "timelib.h"

/******************** CONSTANTS AND MACROS **********************************/

#define MAXLEN   256
#define SPIN_PERIOD  4 // seconds
#define IVSPH        6 // intervals per hour (10-minute intervals)
#define IVLEN       10 // minutes per interval

/******************** TYPE DEFINITIONS **************************************/


/******************** FUNCTION PROTOTYPES ***********************************/

static void Usage (const char *exename);
static void CreateEmptyLine(FILE *ofp, scet thishour);

/******************** GLOBAL STATIC DATA ************************************/


/******************** LOCAL STATIC DATA *************************************/

static char rcsid[] = "$Id: qstat.cpp,v 1.3 2004/08/31 11:27:42 hav Exp $";
static char version[] = "V1.2";

/******************** MAIN PROGRAM ******************************************/

// ===========================================================================
int
main (int argc, char *argv[])
// ===========================================================================
{
   const long double tmrsec = 5.152221145;

   FILE *ofp = stdout;
   char outfn[MAXLEN];

   tmpacket tmpkt;                   // generic packet structure (header,data,
                                     // output file pointer,packet number)
   scet *tp = &(tmpkt.hdr.sc_time);  // packet (TMR) time stamp from header
   tmpkt.pktno = 0;                  // initialize packet number!
   scet st_time;                     // time of current PACMO structure

   scet current,                     // current acquisition interval time
                                     // (year,month,day,hour needed only)
        last;                        // time of last acquisition interval
   scet msf_date;

   int cur_iv;                       // index of current acquisition interval
                                     // within current hour

   long double spinstart_ct = -1;    // -1 means no WW spin found so far

   short scimo, submo, pacmo,       // EDI science mode, submode, packing mode
         numst,                     // number of PACMO structures in current TMR
         stlen,                     // number of tm words per PACMO structure
         delta_tt;                  // data_timetag - TMR_timetag
   short q1, q2;                    // qualities of detector 1 and 2
   short fr_per_st;                 // GDU frames per PACMO structure

   int minq = 2, minbeams = 4;      // defaults for spin quality statistics
   int allspins=0, goodspins=0;     // counters of all / good WW spins
                                    // 'good' means the number of beams per spin
                                    // with quality >= minq is greater than
                                    // or equal to minbeams

   int allspinarr[IVSPH],           // counters for subintervals of a full hour
       goodspinarr[IVSPH];

   for (int i=0; i<IVSPH; ++i) {    // initialize !
      allspinarr[i] = 0;
      goodspinarr[i] = 0;
   }

   int beamcnt=0;                   // counter of good beam (q >= minq) in
                                    // current spin

   int o_is_dir=0;                  // some more flags
   int force_all = 0;
   int skip_empty = 0;
   int debug = 0;

   if (0) printf("%s\n", rcsid); // dummy instruction

   // Process command line arguments and options
   // ------------------------------------------
   int ch;
   while ((ch = getopt (argc, argv, "hdq:n:so:f")) != -1) {

      switch (ch) {
      case '?':
      case 'h':
         Usage (argv[0]);
         exit (1);

      case 'q':
         if (sscanf(optarg, "%d", &minq) != 1) {
            fprintf(stderr, "Error interpreting %s as an int\n", optarg);
            Usage(argv[0]);
            exit(1);
         }
         break;

      case 'n':
         if (sscanf(optarg, "%d", &minbeams) != 1) {
            fprintf(stderr, "Error interpreting %s as an int\n", optarg);
            Usage(argv[0]);
            exit(1);
         }
         break;

      case 'f':
         force_all = 1;
         break;

      case 's':
         skip_empty = 1;
         break;

      case 'd':
         debug = 1;
         break;

      case 'o':
         ofp = NULL; // flag for later: means do output to file
         // see if there is an argument for -o in optarg
         // --------------------------------------------
         if (optarg==NULL || optarg[0]=='-') {
            fprintf(stderr, "option -o: missing argument\n");
            exit(1);
         }

         outfn[0] = '\0'; // prepare for appending of strings

         if (strlen(optarg) >= MAXLEN-1) {
            fprintf(stderr, "-o argument too long : %s\n", optarg);
            exit(1);
         }

         strcat(outfn, optarg);


         // see if it is a directory
         if (access(optarg, F_OK)==0) {
            struct stat mystat;
            if (stat(optarg, &mystat) != 0) {
               perror(optarg);
               exit(1);
            }
            if (mystat.st_mode & S_IFDIR) {
               if (strlen(outfn) > MAXLEN-2) {
                  fprintf(stderr, "output file name too long\n");
                  exit(1);
               }
               strcat(outfn, "/");
               o_is_dir=1;
            }
         } // endif directory check

         break;

      } // endswitch

   } // endwhile


   // check for existence of input file name argument
   // -----------------------------------------------
   if (argc <= optind) {
      fprintf(stderr, "missing input file name\n");
      Usage(argv[0]);
      exit(1);
   }

   // construct default output file name if -o argument was a directory
   // -----------------------------------------------------------------
   if (o_is_dir) {

      // first get input file name with path stripped off
      char *fns = strrchr(argv[optind], '/'); // get position of last '/'
                                              // in input file name
      if (fns==NULL) fns = argv[optind];
      else           ++fns;

      if ( (strlen(outfn)+strlen(fns)) > MAXLEN-1 ) {
         fprintf(stderr, "output file name too long\n");
         exit(1);
      }
      strcat(outfn, fns);
      char *p = strrchr(outfn, '.');
      if (p==NULL) {
         fprintf(stderr, "Could not find '.' in %s\n", outfn);
         exit(1);
      }

      if (*(p-1) == 'q') {
         fprintf(stderr, "Cannot construct default output filename: not unique\n");
         exit(1);
      }
      *(p-1) = 'q';
   }

//   printf("output file name is <%s>\n", outfn);
//   printf("input file name is <%s>\n", argv[optind]);

   // Open output file
   // ----------------
   if (ofp==NULL) {
      ofp = OpenFile(outfn, OPF_WT, OPF_EXIT);
   }

   // Open the data file
   // ------------------
   CL2TM_OpenFile (argv[optind]);

   // For MSF files read day/month/year from the file name
   int filetype = CL2TM_GetFileType();
   if (filetype == FILE_TYPE_MSF) {
      char *pdot = strrchr(argv[optind], '.');
      if (pdot==NULL) {
         fprintf(stderr, "Could not find dot in filename <%s>\n",
                         argv[optind]);
         exit(1);
      }
      char *pfn = strrchr(argv[optind], '/');
      if (pfn==NULL) pfn = argv[optind];
      else           ++pfn;

      if (pdot-pfn != 8) {
         fprintf(stderr, "Invalid length of MSF file name : %d\n",
                         (int)(pdot-pfn) );
         exit(1);
      }

      if (sscanf(pfn+4, "%2hd", &msf_date.day) != 1   ||
          sscanf(pfn+2, "%2hd", &msf_date.month) != 1 ||
          sscanf(pfn, "%2hd", &msf_date.year) != 1) {
         fprintf(stderr, "Error reading date from MSF file name : %s\n",
                  pfn);
         exit(1);
      }


      TL_Set(&msf_date, msf_date.year+2000, msf_date.month, msf_date.day,
             0, 0, 0,
             0, 0, 0); // calculates also .doy and .ct


      TL_Add(&msf_date, TL_HOUR, 0) ; // dummy to set ctime of msf_date

      last = msf_date;
      current = msf_date; // this is to ensure that 'current' is initialized
                          // for the case where no WW data are found at all;
   }
   else if (filetype == FILE_TYPE_CDDS || filetype == FILE_TYPE_RDM) {
      force_all = 1;

      char *pfn = strrchr(argv[optind], '/');
      if (pfn==NULL) pfn = argv[optind];
      else           ++pfn;

      if (sscanf(pfn+4, "%4hd", &msf_date.year) != 1   ||
          sscanf(pfn+9, "%2hd", &msf_date.month) != 1 ||
          sscanf(pfn+12, "%2hd", &msf_date.day) != 1) {
         fprintf(stderr, "Error reading date from CDDS file name : %s\n",
                  pfn);
         exit(1);
      }


      TL_Set(&msf_date, msf_date.year, msf_date.month, msf_date.day,
             0, 0, 0,
             0, 0, 0); // calculates also .doy and .ct


      TL_Add(&msf_date, TL_HOUR, 0) ; // dummy to set ctime of msf_date

      last = msf_date;
      current = msf_date; // this is to ensure that 'current' is initialized
                          // for the case where no WW data are found at all;

   }
   else { // for non-MSF files
      fprintf(stderr, "not an MSF or CDDS file\n");
      exit(1);
   }

   // write setup options to output file
   // ----------------------------------
      fprintf(ofp, "; min_q: %4d     minbeams per spin: %4d\n",
                  (int)minq, (int)minbeams);



   // read all data packets (TMRs) in a loop
   // --------------------------------------
   while (CL2TM_ReadPacket (&tmpkt) != -1) {

      // make sure it's EDI science data
      if ((tmpkt.hdr.data_source != CL2TM_DSRC_EDI) ||
          (tmpkt.hdr.data_type != CL2TM_NSD_DATA &&
           tmpkt.hdr.data_type != CL2TM_BSD_DATA)) {
         ++tmpkt.pktno;
         continue;
      }


      // extract mode information
      scimo = (tmpkt.data[0] >> 3) & 0x1f;
      submo = tmpkt.data[0] & 0x7;
      pacmo = (tmpkt.data[2] >> 8) & 0xf;


      // see if it is WW and correct submo / pacmo
      if (scimo != 5) {
         ++tmpkt.pktno;
         continue;
      }

      int badmode=0;
      if (submo==1 || submo==2 || submo==3 || submo==6) {
         if (pacmo==1)       fr_per_st = 256;
         else if (pacmo==5)  fr_per_st = 512;
         else                badmode = 1;
      } else if (submo==5) {
         if (pacmo==1) fr_per_st = 64;
         else          badmode = 1;
      } else
         badmode = 1;

      if (badmode) {
         ++tmpkt.pktno;
         continue;
      }

      // de-alias time tag delta and skip packet if invalid
      delta_tt=DeAlias (tmpkt.data[4], tmpkt.data[20], submo, pacmo);
      if (delta_tt == INVALID_TT) {
         ++tmpkt.pktno;
         continue;
      }


      // read number of PACMO 1/5 data structures in this (TMR) packet
      // -------------------------------------------------------------
      numst = tmpkt.data[21];
      if (numst <= 0) {
         ++tmpkt.pktno;
         continue;
      }

      // set number of words per PACMO structure and set time of first structure
      // --------------------------------------------------------------------
      if (submo == 5 || pacmo == 5) stlen = 10;
      else                          stlen = 5;

      TL_SetCt(&st_time, tp->ct - tmrsec + delta_tt/4096.0L);

      // Finally we've got something to do. Initialize
      // ---------------------------------------------
      if (spinstart_ct < 0) {
         spinstart_ct = st_time.ct;
         current = st_time;
         cur_iv = st_time.min/IVLEN;
      }


      // Extract and process quality information for all
      // WW PACMO1/5 structures in this TMR
      // -----------------------------------------------
      for (int j = 0; j < numst; ++j, TL_AddCt(&st_time, fr_per_st/4096.0L)) {

         // have we reached the end of a spin ?
         // -----------------------------------
         if (st_time.ct > spinstart_ct+4) {
            if (beamcnt >= minbeams ) {
               ++goodspins;
               ++goodspinarr[cur_iv];
            }
            ++allspins;
            ++allspinarr[cur_iv];
            beamcnt = 0; // reset counter of good beams for next spin

            // have we also reached the end of an hour ?
            // -----------------------------------------
            if (TL_DateCmp(st_time,current) > 0 || st_time.hr > current.hr) {

               if (debug) {
                  fprintf(ofp, "dbg: end of hour.\n"
                               "     current = ");
                  TL_PrintDateTime(&current, ofp);
                  fprintf(ofp, "     st_time = ");
                  TL_PrintDateTime(&st_time, ofp);
               }
               // fill in possibly empty hours
               if (!skip_empty) {
                  if (debug) {
                     fprintf(ofp, "dbg: <loop> filling in uncovered hours up to current.\n"
                                  "            last = ");
                     TL_PrintDateTime(&last, ofp);
                  }
                  while ( (TL_DateCmp(last, msf_date) == 0) &&
                          (TL_DateCmp(last, current) < 0 ||
                           (TL_DateCmp(last, current) == 0 &&
                            last.hr<current.hr)               )             ) {
                     CreateEmptyLine(ofp, last);
                     TL_Add(&last, TL_HOUR, 1);
                  }
                  if (debug) {
                     fprintf(ofp, "     <loop> end.\n"
                                  "            last = ");
                     TL_PrintDateTime(&last, ofp);
                  }
               }

               // write out spin statistics for current hour
               if (TL_DateCmp(current,msf_date)==0 || force_all) {
                  if (debug) {
                     fprintf(ofp, "dbg: <loop> filling in good hour.\n"
                                  "            curr = ");
                     TL_PrintDateTime(&current, ofp);
                  }
                  fprintf(ofp, "%04hd-%02hd-%02hd  %2hd  1  %5d %5d ",
                          current.year, current.month, current.day,
                          current.hr, (int)allspins, (int)goodspins );
                  for (int k=0; k<IVSPH; ++k) {
                     if (allspinarr[k]==0) allspinarr[k] = 1;
                     fprintf(ofp, " %5.1f ", (double)(100.*goodspinarr[k]/allspinarr[k]));
                  }
                  fprintf(ofp, "\n");

                  // remember last good hour
                  last = current;
                  TL_Add(&last, TL_HOUR, 1);
                  if (debug) {
                     fprintf(ofp, "     <loop> end.\n"
                                  "            last = ");
                     TL_PrintDateTime(&last, ofp);
                  }
               }


               // set new acquisition hour
               current = st_time;

               allspins = 0;      // reset spin counters for next hour
               goodspins = 0;
               for (int k=0; k<IVSPH; ++k) {
                  allspinarr[k] = 0;
                  goodspinarr[k] = 0;
               }
            }

            // update interval index
            cur_iv = st_time.min/IVLEN;


            // Set next spin start time.
            // In order to account for the possibility of gaps in the WW data
            // loop until spinstart is closer than 4 seconds to the current
            // time (st_time) (instead of simply adding 4 seconds)
            while (st_time.ct > spinstart_ct+4) spinstart_ct += 4;
         } // endif (end of a spin)

         // extract and evaluate quality
         q1 = (tmpkt.data[22+stlen*j+4] >> 14) & 0x3;
         q2 = (tmpkt.data[22+stlen*j+4] >> 12) & 0x3;
         if (q1 >= minq) ++beamcnt;
         if (q2 >= minq) ++beamcnt;
      } // endfor (loop through PACMO1/5 structures)

      ++tmpkt.pktno; //next packet (TMR)

   } // endwhile packet-loop


   // fill in possibly empty hours for MSF files
   if (!skip_empty) {
      if (debug) {
         fprintf(ofp, "dbg: <final> filling in leading uncovered hours up to current.\n"
                      "             curr = ");
         TL_PrintDateTime(&current, ofp);
         fprintf(ofp, "             last = ");
         TL_PrintDateTime(&last, ofp);
      }
      while ( (TL_DateCmp(last, msf_date) <= 0) &&
              (TL_DateCmp(last, current) < 0 ||
                (TL_DateCmp(last, current) == 0 &&
                 last.hr<current.hr)               )  ) {
         if (TL_DateCmp(last, msf_date) == 0) {
            CreateEmptyLine(ofp, last);
         }
         TL_Add(&last, TL_HOUR, 1);
      }
      if (debug) {
         fprintf(ofp, "     <final> end.\n"
                      "             last = ");
         TL_PrintDateTime(&last, ofp);
      }
   }

   // do we have to report the statistics of the latest hour ?
   // --------------------------------------------------------
   if (allspins && (TL_DateCmp(current, msf_date) == 0 || force_all)) {
      if (debug) {
         fprintf(ofp, "dbg: <final> filling in last good hour\n"
                      "             curr = ");
         TL_PrintDateTime(&current, ofp);
      }
      fprintf(ofp, "%04hd-%02hd-%02hd  %2hd  1  %5d %5d ",
                   current.year, current.month, current.day,
                   current.hr, (int)allspins, (int)goodspins );
      for (int k=0; k<IVSPH; ++k) {
         if (allspinarr[k]==0) allspinarr[k] = 1;
         fprintf(ofp, " %5.1f ", (double)(100.*goodspinarr[k]/allspinarr[k]));
      }
      fprintf(ofp, "\n");
      last = current;
      TL_Add(&last, TL_HOUR, 1);
   }


   // do we have to fill in statistics data for uncovered hours ?
   // -----------------------------------------------------------
   if (!skip_empty) {
      if (debug) {
         fprintf(ofp, "dbg: <final> filling in trailing uncovered hours\n"
                      "             last = ");
         TL_PrintDateTime(&last, ofp);
      }
      while ( TL_DateCmp(last, msf_date) == 0) {
         CreateEmptyLine(ofp, last);
         TL_Add(&last, TL_HOUR, 1);
      }
      if (debug) {
         fprintf(ofp, "     <final> end.\n"
                      "             last = ");
         TL_PrintDateTime(&last, ofp);
      }
   }

   // Close files and be done
   // -----------------------
   CL2TM_CloseFile ();

   if (ofp != stdout) {
      fclose(ofp);
      printf("output written to %s\n", outfn);
   }

   return 0;
}


/******************** FUNCTION DEFINITIONS **********************************/

/*===========================================================================*/
static void
Usage (const char *exename)
/*===========================================================================*/
{
   fprintf (stdout, "%s %s\n", exename, version);
   fprintf (stdout,
            "Usage: %s [-h] [-d] [-q minq] [-n minbeams] [-s] [-o filename] [-f] input-file\n"
            "\n"
            "   input-file is an MSF file\n"
            "\n"
            "   Options:\n"
            "   -h   display this help message\n"
            "   -d   print additional info for debugging\n"
            "   -q   minq = minimum quality for beam selection\n"
            "   -n   minbeams = minmum number of beams per spin for triangulation\n"
            "   -o   write to output file instead of standard output.\n"
            "        If 'filename' is a directory, the file will have a default name\n"
            "        and will be placed into this directory\n"
            "        The default file name is constructed by replacing the last character\n"
            "        of the input file prior to the extension dot by a 'q'.\n"
            "        Example for an MSF file : 'yymmddem.xxx' --> 'yymmddeq.xxx' \n"
            "   -f   force reporting of periods outside current day for MSF files\n"
            "   -s   skip (=don't report) hours without WW data\n"
            "\n", exename);

}


// ============================================================================
   static void CreateEmptyLine(FILE *ofp, scet thishour)
// ============================================================================
{
   int idummy = 0;

   fprintf(ofp, "%04hd-%02hd-%02hd  %2hd  0  %5d %5d ",
                thishour.year, thishour.month, thishour.day,
                thishour.hr, idummy, idummy);

   for (int l=0; l<IVSPH; ++l)
      fprintf(ofp, " %5.1f ", (double)idummy);

   fprintf(ofp, "\n");
}
