/******************************************************************************
                           cdds.cpp  -  description
                             -------------------                              
    begin                : Thu Sep 28 12:09:00 CET 2000
                                           
    author               : Hans Vaith
    email                : hav@mpe.mpg.de
    copyright            : (C) 2000 by Max-Planck-Institut fr extra-
                           terrestrische Physik, D-85740 Garching

    Library for reading of Cluster-II CDDS and data extraction

        a CDDS file is constructed as follows:
        =====================================
     
        [20 bytes SFDU header] [file data field]

        where [file data field] is:

        [20 bytes acknowledge header][acknowledge data field]
        [20 bytes catalogue header][catalogue data field]
        [20 bytes telemetry header][requested data field]

        each of the 20 byte headers contains in the last 8 bytes the size
        (number of bytes) of the following data field in ASCII notation
        (to be interpreted as a decimal number)

   The [requested data field] consists of a variable number of
   data structures. Each data structure consists of a 15 byte CDDS header
   and a data packet. The size of the data packet is encoded in the
   accompanying CDDS header. See function ProcessHeader() for details.


    Last modification :
    08-Feb-2001  hav  removed EGSE TM file support; renamed to cdds.cpp
                      added function to check integrity of requested data
*****************************************************************************
$Id: cdds.cpp,v 1.7 2004/08/31 11:27:42 hav Exp $
*****************************************************************************
$Log: cdds.cpp,v $
Revision 1.7  2004/08/31 11:27:42  hav
bug fix in cdds.cpp (check for zero file size)
more dev in corb

Revision 1.6  2003/06/18 15:58:37  hav
more development

Revision 1.5  2003/04/02 13:40:04  hav
bug fix: added data source ids for SC in ProcessCDDSheader

Revision 1.4  2003/03/31 18:55:34  hav
*** empty log message ***

Revision 1.3  2003/03/31 18:52:06  hav
*** empty log message ***

Revision 1.2  2003/03/31 18:41:44  hav
support for RAPID

Revision 1.1.1.1  2002/10/14 17:08:19  hav
c/c++ sources

Revision 1.14  2001/11/13 19:22:44  hav
bug fix: SequentialOnly=1 for RDM files

Revision 1.13  2001/11/12 17:30:48  hav
added RDM file type detection

Revision 1.12  2001/10/26 10:52:58  hav
SequentialOnly=1 for BSD files (because different sizes for BM1/2/3 packets)

Revision 1.11  2001/08/10 11:33:10  hav
add function name in diagnostic error message

Revision 1.10  2001/08/01 15:15:03  hav
ProcessCDDSheader makes use of new timelib functions

Revision 1.9  2001/07/26 16:01:02  hav
mover file type constants to cl2tm.h

Revision 1.8  2001/07/25 15:20:14  hav
added rcsid

Revision 1.7  2001/07/25 14:54:34  hav
extended CDDS_OpenFile to handle also MSF files

Revision 1.6  2001/04/02 13:00:35  hav
make compatible with DEC ALPHA platform

Revision 1.5  2001/02/21 22:55:02  hav
complete rewrite of cdds library to support binary transfer of data

Revision 1.4  2001/02/20 15:45:14  hav
added filename as first line of req data ASCII output

Revision 1.3  2001/02/14 12:32:30  hav
common rcs format

******************************************************************************/

/******************** INCLUDE FILES ******************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "timelib.h"
#include "cdds.h"
#include "openfile.h"
#include "filesize.h"

/******************** CONSTANTS AND MACROS ***********************************/

#define CDDS_HDR_SZ  15

/******************** TYPE DEFINITIONS **************************************/

typedef struct
{                               // validity
   char FileName[MAXLEN];       // all
   FILE *Fp;                    // all
   long FileSize;               // all
   int FileType;                // all (CDDS/RDM/MSF-type)
   long ReqOffset;              // all
   int ScId;                    // all
   int SequentialOnly;          // all (1 in case of MSF-type or in case of
                                //      CDDS-type/RDM-type cmd-log file)
   long AckOffset, CatOffset;      // CDDS
   long AckSize, CatSize, ReqSize; // CDDS
   long PktLen;                    // CDDS/RDM
   int DataSource, DataType;       // CDDS/RDM
}
cdds_t;


/******************** FUNCTION PROTOTYPES ***********************************/

static void SetData (long *offsp, long *sizep, long minsize);
static void ProcessCDDSheader (char *buffer, tmheader * hp);
static void PrintCDDSheader (tmheader hdr);
static void SwapBytes (char *buf, long i, long j);

// ****************** GLOBAL STATIC DATA *************************************

// ****************** LOCAL STATIC DATA **************************************

static cdds_t cdds;
static char *DataSourceStr[] = { "???", "SC_", "EDI", "CMD", "FGM", "MSF", "RAP" };
static char *DataTypeStr[]   = { "???", "NSD", "BSD", "HKD", "AUX", "MSF" };
static char Dummy[MAXLEN];

static char rcsid[] = "$Id: cdds.cpp,v 1.7 2004/08/31 11:27:42 hav Exp $";

/******************** FUNCTION DEFINITIONS ***********************************/


// ===========================================================================
void
CDDS_OpenFile (const char *fn)
// ===========================================================================
// open file and set all data elements in static variable 'cdds'
// ***************************************************************************
{
   if (0) printf("%s\n", rcsid); // dummy instruction

   char *fname = "CDDS_OpenFile()";
   char hbuf[CDDS_HDR_SZ];
   tmheader hdr;

   if (cdds.Fp != NULL) {
      fprintf (stderr, "%s: file already open\n", fname);
      exit (1);
   }

   // Open File, check file type and data offset
   // ==========================================
   if (strlen(fn) > MAXLEN-1) {
     fprintf(stderr, "%s: filename too long (%s)\n", fname, fn);
     exit(1);
   }
   strcpy (cdds.FileName, fn);

   cdds.Fp = OpenFile (fn, OPF_RB, OPF_EXIT);

   // determine file type
   // for now there is no RDM determination
   fscanf (cdds.Fp, "%4s", Dummy);
   if (strcmp (Dummy, "CCSD") == 0) {
      cdds.FileType = FILE_TYPE_CDDS;
   }
   else {
      char *p;
      if ( (p=strrchr(fn, '.')) != NULL && strcmp(p,".rdm")==0 ) {
         cdds.FileType = FILE_TYPE_RDM;
      } else {
         cdds.FileType = FILE_TYPE_MSF;
      }
   }

   // determine file size
   cdds.FileSize = FileSize (cdds.Fp);
   
   if (cdds.FileSize == 0) {
      fprintf(stderr, "Empty input file! Exiting...\n");
      exit(1);
   }

   // set offsets and sizes of data fields
   if (cdds.FileType == FILE_TYPE_CDDS) {
      SetData (&cdds.AckOffset, &cdds.AckSize, 40);
      SetData (&cdds.CatOffset, &cdds.CatSize, 60 + cdds.AckSize);
      SetData (&cdds.ReqOffset, &cdds.ReqSize,
               80 + cdds.AckSize + cdds.CatSize);
      if (cdds.ReqSize == 0) {
         fprintf(stderr, "no data in CDDS file\n");
         exit(1);
      }
   }
   else {
      cdds.ReqOffset = 0;
      cdds.ReqSize = cdds.FileSize;
   }


   if (cdds.FileType == FILE_TYPE_CDDS && cdds.ReqSize == 0)
      return;                   // nothing else to do


   // Read first header and determine SC id, data source and type
   // -----------------------------------------------------------
   if (fseek (cdds.Fp, cdds.ReqOffset, SEEK_SET)) {
      perror("fseek to position on first header in CDDS_OpenFile");
      exit(1);
   }
   if (fread (hbuf, 1, CDDS_HDR_SZ, cdds.Fp) != CDDS_HDR_SZ) {
      fprintf(stderr, "fread to read first header in CDDS_OpenFile: wrong number of bytes returned");
      exit(1);
   }   
   ProcessCDDSheader (hbuf, &hdr);
   cdds.ScId = (int)hdr.sc_id;
   if (cdds.FileType == FILE_TYPE_CDDS || cdds.FileType == FILE_TYPE_RDM) {
      cdds.DataSource = (int)hdr.data_source;
      cdds.DataType = (int)hdr.data_type;
      cdds.PktLen = hdr.packet_length + CDDS_HDR_SZ;
//      cdds.MaxPacket = cdds.FileSize / cdds.PktLen;
   } else {
      cdds.DataSource = CL2TM_DSRC_MSF;
      cdds.DataType   = CL2TM_MSF_DATA;
   }


   if (cdds.FileType == FILE_TYPE_CDDS &&           // EDI HKD and NSD files from
       cdds.DataSource == CL2TM_DSRC_EDI &&         // the CDDS have fixed sizes
           (cdds.DataType == CL2TM_NSD_DATA ||
            cdds.DataType == CL2TM_HKD_DATA)   ) {
      cdds.SequentialOnly=0;
   } else {
      cdds.SequentialOnly=1;
   }


   // debugging information
/*
   printf("cdds.ScId           : %ld\n"
          "cdds.FileType       : %ld\n"
          "cdds.DataSource     : %ld\n"
          "cdds.DataType       : %ld\n"
          "cdds.PktLen         : %ld\n"
          "cdds.SequentialOnly : %ld\n"
          "cdds.ReqSize        : %ld\n",
          (long)cdds.ScId,
          (long)cdds.FileType,
          (long)cdds.DataSource,
          (long)cdds.DataType,
          (long)cdds.PktLen,
          (long)cdds.SequentialOnly,
          (long)cdds.ReqSize
          );
*/
}


// ===========================================================================
void
CDDS_CloseFile (void)
// ===========================================================================
{
   if (cdds.Fp != NULL) {
      fclose (cdds.Fp);
      cdds.Fp = NULL;
   }
   cdds.FileType = 0;
}


// ===========================================================================
int
CDDS_GetScId (void)
// ===========================================================================
// ***************************************************************************
{
   if (!cdds.Fp) {
      fprintf (stderr, "CDDS_GetScId(): no file open yet!\n");
      exit (1);
   }

   return cdds.ScId;
}


// ===========================================================================
int
CDDS_GetDataSource (void)
// ===========================================================================
// ***************************************************************************
{
   if (!cdds.Fp) {
      fprintf (stderr, "CDDS_GetDataSource(): no file open yet!\n");
      exit (1);
   }

   return cdds.DataSource;
}

// ===========================================================================
int
CDDS_GetDataType (void)
// ===========================================================================
// ***************************************************************************
{
   if (!cdds.Fp) {
      fprintf (stderr, "CDDS_GetDataType(): no file open yet!\n");
      exit (1);
   }

   return cdds.DataType;
}

// ===========================================================================
int 
CDDS_GetFileType (void)
// ===========================================================================
{
  return cdds.FileType;
}


// ===========================================================================
void
CDDS_ReadAck (FILE * ofp)
// ===========================================================================
// Write Acknowledge Data to ofp (must be open for writing)
// ***************************************************************************
{
   if (!cdds.Fp) {
      fprintf (stderr, "CDDS_ReadAck(): no file open yet!\n");
      exit (1);
   }
   else if (cdds.AckOffset == 0) {
      fprintf (stderr, "CDDS_ReadAck(): no Ack Data\n");
      return;
   }

   fseek (cdds.Fp, cdds.AckOffset, SEEK_SET);   // point to acknowledge data
   char buf[cdds.AckSize + 1];
   fread (buf, cdds.AckSize, 1, cdds.Fp);
   buf[cdds.AckSize] = '\0';
   fprintf (ofp, "%s", buf);

}


// ===========================================================================
void
CDDS_ReadCat (FILE * ofp)
// ===========================================================================
// Write Catalogue Data to ofp (must be open for writing)
// ***************************************************************************
{
   char *fname = "CDDS_ReadCat()";
   //   char tbuf[MAXLEN];

   if (!cdds.Fp) {
      fprintf (stderr, "%s: no file open yet!\n", fname);
      exit (1);
   }
   else if (cdds.CatOffset == 0) {
      fprintf (stderr, "%s: no catalogue data.\n", fname);
      return;
   }

   fseek (cdds.Fp, cdds.CatOffset, SEEK_SET);   // point to catalogue data
   char buf[cdds.CatSize + 1];
   fread (buf, cdds.CatSize, 1, cdds.Fp);
   buf[cdds.CatSize] = '\0';
   fprintf (ofp, "%s", buf);

}

// ===========================================================================
void
CDDS_ReadReq (FILE * ofp)
// ===========================================================================
// Write requested data to ofp (must be open for writing)
// ***************************************************************************
{
   char *fname = "CDDS_ReadReq()";
   tmpacket tmpkt;

   // Check if file has been opened
   // =============================
   if (!cdds.Fp) {
      fprintf (stderr, "%s: no file open yet!\n", fname);
      exit (1);
   }

   // Read if there are data
   // ======================
   if (cdds.ReqSize == 0) {

      fprintf (stderr, "%s: no Req Data\n", fname);
      return;
   }

   // check if output is defined
   // ==========================
   if (ofp == NULL) {
      fprintf (stderr, "%s: ofp is NULL\n", fname);
      return;
   }

   fprintf (ofp, "%s\n", cdds.FileName);

   fseek (cdds.Fp, cdds.ReqOffset, SEEK_SET);
   tmpkt.ofp = ofp;
   tmpkt.pktno = -1;            // -1 is a flag to inhibit file pointer positioning
   // in CDDS_ReadPacket()

   while (1) {

      // Read a packet
      if (CDDS_ReadPacket (&tmpkt) == -1)
         break;                 // end of file

      // Write data to tmpkt.ofp
      CDDS_WritePacket (tmpkt);

   }

}


// ===========================================================================
int
CDDS_ReadPacket (tmpacket * tmptr)
// ===========================================================================
// returns -1 if EOF
// ***************************************************************************
{
   char hdrbuf[CDDS_HDR_SZ];
   static char tmbuf[BM3_SIZE + 1];
   char *fname = "CDDS_ReadPacket()";
   static long old_pktno = -1;

   tmptr->data = (short *) tmbuf;

   // position file pointer
   if ((tmptr->pktno > 0 && !cdds.SequentialOnly) || tmptr->pktno == 0) {
      if (fseek
          (cdds.Fp, cdds.ReqOffset + (tmptr->pktno) * (cdds.PktLen),
           SEEK_SET) != 0) {
         fprintf (stderr, "%s: fseek() error\n", fname);
         perror ("");
         exit (1);
      }
   }
   else if (tmptr->pktno > 0 && cdds.SequentialOnly
            && tmptr->pktno != (old_pktno + 1)) {
      fprintf (stderr,
               "random access not implemented for SequentialOnly files\n"
               "requested packet : %ld    previous packet: %ld\n",
               tmptr->pktno, old_pktno);
      return -1;                // treat like EOF
   }

   //  Read and process CDDS header
   if (fread (hdrbuf, CDDS_HDR_SZ, 1, cdds.Fp) != 1) {
      if (feof (cdds.Fp))
         return -1;
      else {
         fprintf (stderr, "%s: CDDS header fread() error\n", fname);
         exit (1);
      }
   }


   ProcessCDDSheader (hdrbuf, &(tmptr->hdr));

   // Check size of following data packet
   if (tmptr->hdr.packet_length >= (long) sizeof (tmbuf)) {
      fprintf (stderr, "%s: current packet (%ld) is too big: %ld bytes.\n"
               "File pointer position after CDDS header: %ld\n"
               "CDDS packet length (hdr+data) : %ld\n",
               fname, tmptr->pktno, (long) (tmptr->hdr.packet_length),
               (long) ftell (cdds.Fp), cdds.PktLen);
      PrintCDDSheader (tmptr->hdr);
      if (cdds.FileType == FILE_TYPE_CDDS) {
         fprintf(stderr, "Offset Ack/Cat/Req : %6ld / %6ld / %6ld\n"
                         "Size   Ack/Cat/Req : %6ld / %6ld / %6ld\n",
                         cdds.AckOffset, cdds.CatOffset, cdds.ReqOffset,
                         cdds.AckSize, cdds.CatSize, cdds.ReqSize);
      }
      exit (1);
   }

   // Check data packet size consistency
   if (!cdds.SequentialOnly &&
       tmptr->hdr.packet_length != (cdds.PktLen - CDDS_HDR_SZ)) {
      fprintf (stderr, "%s: unexpected data packet length: %ld\n"
               "packet number                : %ld\n",
               fname, tmptr->hdr.packet_length, tmptr->pktno);
      PrintCDDSheader (tmptr->hdr);
      exit (1);
   }

   // Read data packet
   if (fread (tmbuf, tmptr->hdr.packet_length, 1, cdds.Fp) != 1) {
      if (feof (cdds.Fp))
         return -1;
      else {
         fprintf (stderr, "%s: CDDS data fread() error\n", fname);
         exit (1);
      }
   }

   // Terminate string in case of command log data
   if (cdds.DataSource == CL2TM_DSRC_CMDH)
      tmbuf[tmptr->hdr.packet_length] = '\0';

   // Byte swapping if no command log data
   if (cdds.DataSource != CL2TM_DSRC_CMDH) {
      for (int i = 0; i < tmptr->hdr.packet_length; i += 2)
         SwapBytes (tmbuf, i, i + 1);
   }

   old_pktno = tmptr->pktno;
   return 0;

}


// ===========================================================================
void
CDDS_WritePacket (tmpacket tmpkt)
// ===========================================================================
{
   scet *tp = &(tmpkt.hdr.sc_time);
   short *sp = tmpkt.data;

   if (tmpkt.ofp == NULL) {
      fprintf (stderr, "CDDS_WritePacket(): tmpkt.ofp is NULL\n");
      return;
   }

   if (tmpkt.pktno == 0) {
      fprintf (tmpkt.ofp, "%s\n", cdds.FileName);
   }

   // Write out packet time and data packet size (in bytes)
   if (cdds.DataSource != CL2TM_DSRC_CMDH) {
      fprintf(tmpkt.ofp, "%s%s%d "
                         "%04hd.%03hd.%02hd.%02hd.%02hd.%03hd.%03hd "
                         "%8ld\n",
                         DataSourceStr[tmpkt.hdr.data_source],
                         DataTypeStr[tmpkt.hdr.data_type],
                         tmpkt.hdr.sc_id,
                         tp->year, tp->doy, tp->hr, tp->min,
                         tp->sec, tp->msec, tp->usec,
                         tmpkt.hdr.packet_length);
   }

   // Write out data
   if (cdds.DataSource == CL2TM_DSRC_CMDH) {
      fprintf (tmpkt.ofp, "%s", (char *) tmpkt.data);
   }
   else {
      for (long i = 0; i < tmpkt.hdr.packet_length / 2; ++i) {
         fprintf (tmpkt.ofp, "%04hx ", sp[i]);
         if (i % 16 == 15)
            fputc ('\n', tmpkt.ofp);
      }
      fprintf (tmpkt.ofp, "\n");
   }

}


// ===========================================================================
int CDDS_toRDM(void)
// ===========================================================================
{
   const char *fname = "CDDS_toRDM()";

   // Check if CDDS file is open
   // ==========================
   if (!cdds.Fp) {
      fprintf (stderr, "%s: no file open yet!\n", fname);
      exit (1);
   }

   // Check if input file is a CDDS file
   // ==================================
   if (cdds.FileType != FILE_TYPE_CDDS) {
      fprintf(stderr, "%s: input file is not a CDDS file\n", fname);
      exit(1);
   }

   // Read if there are data
   // ======================
   if (cdds.ReqSize == 0) {
      fprintf (stderr, "%s: no Req Data in CDDS file\n", fname);
      return 0;
   }


   // create RDM output file name and open file
   // =========================================
   char rdm_fn[256];
   if (strlen(cdds.FileName) + 4  > 255) {
      fprintf(stderr, "%s: file name too long!\n", fname);
      exit(1);
   }
   strcpy(rdm_fn, cdds.FileName);
   strcat(rdm_fn, ".rdm");
   FILE *rdm_fp = OpenFile(rdm_fn, OPF_WB, OPF_EXIT);

   // position input file pointer
   // ===========================
   long stored_pos = ftell(cdds.Fp);
   if (stored_pos < 0) {
      perror("ftell");
      exit(1);
   }

   if (fseek (cdds.Fp, cdds.ReqOffset, SEEK_SET) != 0) {
      perror("fseek");
      exit(1);
   }

   // copy file
   // =========
   int c;
   while ((c=fgetc(cdds.Fp)) != EOF)
      fputc(c,rdm_fp);

   // close RDM file
   // ==============
   fclose(rdm_fp);

   // set CDDS file pointer back to stored position
   // =============================================
   if (fseek(cdds.Fp, stored_pos, SEEK_SET) != 0) {
      perror("Error repositioning file pointer (fseek)");
      exit(1);
   }

   printf("created RDM file %s\n", rdm_fn);

   return 0;
}

// ===========================================================================
   void CDDS_PrintGroundStation(tmheader tmhdr)
// ===========================================================================
{
   switch (tmhdr.gst_id) {
   case 0: printf("unknown (0)\n"); break;
   case 1: printf("Vil-1\n"); break;
   case 2: printf("Kiruna\n"); break;
   case 3: printf("Kourou\n"); break;
   case 4: printf("Perth\n"); break;
   case 5: printf("Malindi\n"); break;
   case 6: printf("Canberra\n"); break;
   case 7: printf("REDU\n"); break;
   case 8: printf("Vil-2\n"); break;
   case 9: printf("Mas Palomas\n"); break;
   default: printf("unknown\n"); break;
   }
}


// ===========================================================================
   char *CDDS_GroundStation(tmheader tmhdr)
// ===========================================================================
{
   switch (tmhdr.gst_id) {
   case 0: return "unknown (0)";
   case 1: return "Vil-1      ";
   case 2: return "Kiruna     ";
   case 3: return "Kourou     ";
   case 4: return "Perth      ";
   case 5: return "Malindi    ";
   case 6: return "Canberra   ";
   case 7: return "REDU       ";
   case 8: return "Vil-2      ";
   case 9: return "Mas Palomas";
   default:return "unknown    ";
   }

}


// *********************** STATIC FUNCTIONS **********************************


// ===========================================================================
static void
SetData (long *offsp, long *sizep, long minsize)
// ===========================================================================
{
   if (cdds.FileSize > minsize) {
      fseek (cdds.Fp, minsize - 8, SEEK_SET);
      fscanf (cdds.Fp, "%8s", Dummy);
      sscanf (Dummy, "%ld", sizep);
      *offsp = minsize;
   }
   else
      *sizep = 0;
}



// ===========================================================================
static void
ProcessCDDSheader (char *buffer, tmheader * hp)
// ===========================================================================
// partial CDDS header description (note the Motorola notation!)
//    byte 0-1  (0=MSB) day since 1 Jan 1958
//    byte 2-5  (2=MSB) ms of day
//    byte 6-7  (6=MSB) us of day
//    byte 9-11 (9=MSB) length of tm data packet (in bytes)
//    byte 12
// ***************************************************************************
{

   char *fname="ProcessCDDSheader(): ";
   scet *tp = &(hp->sc_time);

   // calculate year and day of year
   tp->year = 2000;


   // now the hours, minutes, seconds etc.
   unsigned long msecs_of_day = (buffer[5] & 0xff)
      + (buffer[4] & 0xff) * 256L
      + (buffer[3] & 0xff) * 65536L + (buffer[2] & 0xff) * 16777216L;

   tp->hr = msecs_of_day / (3600 * 1000);
   tp->min = (msecs_of_day % (3600 * 1000)) / (60 * 1000);
   tp->sec = (msecs_of_day % (60 * 1000)) / (1000);
   tp->msec = msecs_of_day % 1000;

   tp->usec = (buffer[7] & 0xff) + (buffer[6] & 0xff) * 256;
   tp->nsec = 0;


   tp->doy = 0;
   TL_Add(tp, TL_DAY, (buffer[1] & 0xff) + (buffer[0] & 0xff) * 256 - 15339);

//   printf("packet time: %04hd.%03hd %02hd:%02hd:%02hd\n", tp->year, tp->doy, tp->hr, tp->min, tp->sec);
/*
//   short day_max;
   tp->doy = (buffer[1] & 0xff) + (buffer[0] & 0xff) * 256 - 15339;
   while (1) {
      day_max = 365;
      if (tp->year % 4 == 0)
         day_max = 366;

      if (tp->doy > day_max) {
         tp->doy -= day_max;
         ++tp->year;
      }
      else
         break;
   }

   // calculate day and month from day of year
   short days_in_month[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
   if (tp->year % 4 == 0)
      days_in_month[1] = 29;

   tp->month = 0;
   short day = tp->doy;
   while (day > days_in_month[tp->month]) {
      day -= days_in_month[tp->month];
      ++tp->month;
      if (tp->month > 11) {
         fprintf (stderr,
                  "ProcessCDDSheader(): date conversion error: month > Dec\n");
         exit (1);
      }
   }
   tp->day = day;
   ++tp->month;                 //  1 - 12
*/

   // finally calculate packet time in seconds since 1.1.1970
//   TimeDate2Ct(tp); // assigns ctime to tp->ct



   hp->packet_length = (buffer[11] & 0xff)
      + (buffer[10] & 0xff) * 256L + (buffer[9] & 0xff) * 65536L;

   // ground station ID
   hp->gst_id = buffer[12] & 0x0f;
//   printf("byte 12 : %02hx\n", (int)buffer[12]);

   // determine data source and type from byte 8 according to
   // CL-ESC-ID-2001 Issue 3 19-May-2000, Appendix A
//   printf("buffer[8] : %04hx\n", (unsigned short int)(buffer[8]));
   unsigned int id = (int)buffer[8] & 0xff;
//   printf("id        : %04x\n", id);
   int id_error = 0;

   if (id==7) {
      hp->sc_id = (buffer[12]>>4)&0xf;
      hp->data_source=CL2TM_DSRC_CMDH;
      hp->data_type=CL2TM_AUX_DATA;
   } else {

      if      (id>=30  && id<=51 ) hp->sc_id = 1;
      else if (id>=70  && id<=91 ) hp->sc_id = 2;
      else if (id>=110 && id<=131) hp->sc_id = 3;
      else if (id>=150 && id<=171) hp->sc_id = 4;
      else {
         id_error = id_error + 1;
      }

      if (id==30 || id==37 || id==44 ||
          id==70 || id==77 || id==84 ||
          id==110 || id==117 || id== 124 ||
          id==150 || id==157 || id==164)

         hp->data_source = CL2TM_DSRC_EDI;

      else if (id==34 || id==41 || id==48 ||
               id==74 || id==81 || id==88 ||
               id==114 || id==121 || id==128 ||
               id==154 || id==161 || id==168)

         hp->data_source = CL2TM_DSRC_RAP;

      else if (id==51 || id==91 || id==131 || id==171)

         hp->data_source = CL2TM_DSRC_SC;

      else {
        id_error = id_error + 2;
      }

      if ((id>=30  && id<=35 ) || (id>=70  && id<=75) ||
          (id>=110 && id<=115) || (id>=150 && id<=155))

          hp->data_type = CL2TM_NSD_DATA;

      else if ((id>=37 && id<=42) || (id>=77 && id<=82) ||
               (id>=117 && id<=122) || (id>=157 && id<=162))

          hp->data_type = CL2TM_BSD_DATA;

      else if ((id>=44 && id<=51) || (id>=84 && id<=91) ||
               (id>=124 && id<=131) || (id>=164 && id<=171))

          hp->data_type = CL2TM_HKD_DATA;

      else {
         id_error = id_error + 4;
      }

   if (id_error != 0) {
      fprintf(stderr, "%s : Error code = %d; Unknown Data source/type id : %u\n",
              fname, id_error, id);
      exit(1);   
   }
}

   
                   
/*      
   switch ((unsigned char)buffer[8]) {
   case 7: hp->sc_id = (buffer[12]>>4)&0xf;
           hp->data_source=CL2TM_DSRC_CMDH;
           hp->data_type=CL2TM_AUX_DATA;
           break;

   case 30: hp->sc_id = 1;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_NSD_DATA;
            break;
   case 34: hp->sc_id = 1;
            hp->data_source=CL2TM_DSRC_RAP;
            hp->data_type=CL2TM_NSD_DATA;
            break;
            
            
   case 37: hp->sc_id = 1;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_BSD_DATA;
            break;
   case 44: hp->sc_id = 1;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_HKD_DATA;
            break;
   case 51: hp->sc_id = 1;
            hp->data_source=CL2TM_DSRC_SC;
            hp->data_type=CL2TM_HKD_DATA;
            break;
   case 70: hp->sc_id = 2;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_NSD_DATA;
            break;
   case 77: hp->sc_id = 2;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_BSD_DATA;
            break;
   case 84: hp->sc_id = 2;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_HKD_DATA;
            break;
   case 91: hp->sc_id = 2;
            hp->data_source=CL2TM_DSRC_SC;
            hp->data_type=CL2TM_HKD_DATA;
            break;
   case 110:hp->sc_id = 3;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_NSD_DATA;
            break;
   case 117:hp->sc_id = 3;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_BSD_DATA;
            break;
   case 124:hp->sc_id = 3;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_HKD_DATA;
            break;
   case 131:hp->sc_id = 3;
            hp->data_source=CL2TM_DSRC_SC;
            hp->data_type=CL2TM_HKD_DATA;
            break;
   case 150:hp->sc_id = 4;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_NSD_DATA;
            break;
   case 157:hp->sc_id = 4;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_BSD_DATA;
            break;
   case 164:hp->sc_id = 4;
            hp->data_source=CL2TM_DSRC_EDI;
            hp->data_type=CL2TM_HKD_DATA;
            break;
   case 171:hp->sc_id = 4;
            hp->data_source=CL2TM_DSRC_SC;
            hp->data_type=CL2TM_HKD_DATA;
            break;
   default:
      fprintf(stderr, "%s : unknown Data source/type id : %d\n",
              fname, (unsigned int)buffer[8]);
      exit(1);
   }
*/

}

// ============================================================================
static void
PrintCDDSheader (tmheader hdr)
// ============================================================================
{
   scet *sctp = &hdr.sc_time;
   fprintf (stderr, "partial CDDS header dump\n");
   TL_PrintDateTime(sctp, stderr);
   fprintf(stderr, "packet length: %ld\n", hdr.packet_length);
}


// ===========================================================================
static void
SwapBytes (char *buf, long i, long j)
// ===========================================================================
{
   char savebyte = buf[i];
   buf[i] = buf[j];
   buf[j] = savebyte;
}


