; +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; $Id: magutil.pro,v 1.4 2004/03/30 13:20:02 hav Exp $
; +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

@clutime.pro
@myfread.pro

; ============================================================================
  PRO MagConv, fn, hr=hr, csys=csys, quiet=quiet
; ============================================================================
; read magnetic field data from a file (format of Edita's scripts)
; ****************************************************************************

@paths.inc
@vector.inc

   if keyword_set(quiet) then quiet=1 else quiet=0

   if not keyword_set(csys) then csys = 'SCS' $
   else if strupcase(csys) ne 'SCS' and csys ne 'GSE' then begin
      message, 'invalid coordinate system: ' + csys
   endif
   csys = strupcase(csys)

   if not keyword_set(hr) then begin
      highrate=0
      filter='*sr_' + csys + '.mag'
   endif else begin
      highrate = 1
      filter='*hr_' + csys + '.mag'
   endelse

   if n_params() eq 0 then $
      fn = dialog_pickfile(path=fgm_dir, filter=filter)

   if fn(0) eq '' then begin
      print, 'B data file not read.'
      return
   endif


   ; read the data
   ; -------------
   bdata = myfread(fn, '', {FVEC3_TT_T}, comment = '#', quiet=quiet)
   totnst = n_elements(bdata)

   if not quiet then begin
       print, CluTimeVal2Str(bdata[0].ct)
       print, CluTimeVal2Str(bdata[totnst-1].ct)
   endif

   ; create output file name, write data to binary files
   ; ---------------------------------------------------
   outfn = fn + '_bin'

   openw, fp, outfn, /get_lun
   writeu, fp, n_elements(bdata)
   writeu, fp, bdata
   free_lun, fp

   if not quiet then print, 'output to ' + outfn

END


; ============================================================================
  FUNCTION ReadBinaryMagData, fn, quiet=quiet
; ============================================================================
; reads binary magnetic field data from file 'fn'
; ****************************************************************************

@vector.inc

  if not keyword_set(quiet) then print, 'reading ' + fn

  openr, fp, fn, /get_lun
  nst = 0L & readu, fp, nst
  st = replicate({FVEC3_TT_T}, nst)
  for i=0L,nst-1 do begin
      readu, fp, fvec3_tt_t
      st(i) = fvec3_tt_t
  endfor
  free_lun, fp

  return, st

END


; ============================================================================
  PRO RequestAsciiMagData, scid, ymdh, sr=sr, $
                           csys=csys, quiet=quiet, calfile=calfile
; ============================================================================
; Request one hour of high resolution magnetic field data or one day of
; spin resolution data by calling a shell script with appropriate parameters
; via the SPAWN procedure. This script will extract magnetic field
; vectors from the CLUSTER telemetry files via Edita Georgescu's chain
; of programs. The data will be written to an ASCII file. For the
; location of the ASCII file see the shell script.
;
; Care is taken that no data are lost at the boundaries. Therefore, to
; get a full one day spin resolution file, three adjacent dates will
; be passed to the script in an ascii file containing these dates.
;
; Parameters
;   scid  int      SC identifier (1,2,3,4)
;   ymdh  string   yyyy-mm-dd (spin res) or yyyy-mm-ddThh (high res)
;
; Keywords
;   sr      0/1      if set, one day of spin res data will be
;                    requested, otherwise one hour of
;                    high res data will be requested
;   csys    string   coordinate system ('scs' or 'gse'; default is 'scs')
;   quiet   0/1      if set, only error messages will be printed
;   calfile string   optional special calibration file to be used
;                    A value of 'none' causes both calibration AND coordinate
;                    transformation to be skipped
; ****************************************************************************

@paths.inc

  if keyword_set(quiet) then quiet=1 else quiet=0
  if not keyword_set(calfile) then calfile = ''

  if not keyword_set(csys) then begin
     if calfile ne 'none' and not quiet then $
        print, 'keyword csys not set. Assuming default (SCS)'
     csys = 'scs'
  endif else if csys ne 'scs' and csys ne 'gse' then begin
     message, 'invalid coordinate system: ' + csys
  endif

  if not keyword_set(sr) then begin
      high_res = 1
      spin_res = 0
      rate_str = 'hr'
      long_str = 'High'
  endif else begin
      high_res = 0
      spin_res = 1
      rate_str = 'sr'
      long_str = 'Spin'
  endelse


  if not quiet then $
    print, long_str + '-res binary B data not found. Need to create.' + $
    'This may take a while.'

  ymd = strmid(ymdh, 0, 10)

  ct_noon = CluTimeStr2Val(ymd) + 86400./2 ; noon
  ymd_prev = strmid(CluTimeVal2Str(ct_noon-86400.),0,10)
  ymd_next = strmid(CluTimeVal2Str(ct_noon+86400.),0,10)

  if spin_res then begin

      datelist = [ ymd_prev, ymd, ymd_next ] ; three days
      ; these date strings are used to specify
      ; what should actually be in the ASCII data file
      ct0_str = string(format='(F17.6)', CluTimeStr2Val(ymd))
      ct1_str = string(format='(F17.6)', CluTimeStr2Val(ymd_next))

  endif else begin

      hour = fix(strmid(ymdh, 11,2))
      if hour eq 0 then datelist = [ ymd_prev, ymd ] $
      else if hour eq 23 then datelist = [ ymd, ymd_next ] $
      else datelist = [ ymd ]

      ct0 = CluTimeStr2Val(ymdh)
      ymdh_next = strmid(CluTimeVal2Str(ct0+3600.),0,13)
      ct0_str = string(format='(F17.6)', CluTimeStr2Val(ymdh))
      ct1_str = string(format='(F17.6)', CluTimeStr2Val(ymdh_next))

  endelse

  datelist = strmid(datelist, 0, 4) + strmid(datelist, 5, 2) + $
             strmid(datelist, 8, 2)

  ; write list of dates into file
  ; -----------------------------
  date_file_name = '~/datelist.txt'
  openw, fp, date_file_name, /get_lun
  for i=0,n_elements(datelist)-1 do $
    printf, fp, datelist[i]
  free_lun, fp

  ; setup the command string for execution of the script
  ; ----------------------------------------------------
  cmdstr = GETFGM_DIR + 'getfgm2 ' + $
    strtrim(scid,2) + ' ' + $
    date_file_name + ' ' + $
    ct0_str + ' ' + $
    ct1_str + ' ' + $
    csys + ' ' + $
    rate_str

  if calfile ne '' then $
     cmdstr = cmdstr + ' ' + calfile


  ; call the script
  ; ---------------
  if not quiet then print, 'executing ', cmdstr
  spawn, cmdstr, result, err_result, exit_status=exit_status

;  print, result

  ; check exit status
  ; -----------------
  if exit_status ne 0 then begin
      print, result
      print, err_result
      print, 'getfgm2 exited with error code ' + strtrim(exit_status,2)
      retall
  endif

END

; ============================================================================
  PRO DeleteBinMagFiles, sc, ctr, debug=debug
; ============================================================================

@paths.inc

  if not keyword_set(debug) then flag = ' -f ' else flag = ' '

  ct = ctr[0]
  while 1 eq 1 do begin

     date_time = CluTimeVal2Str(ct)
     arr = strsplit(date_time, '-tT:.',/extract)
     filename = 'C' + strtrim(sc,1) + '_' + strjoin(arr[0:2]) + 't' + arr[3] + '0000_hr_SCS.mag_bin'
     spawn, 'rm' + flag + FGM_DIR + filename, stdout, stderr, exit_status=exit_status
     if keyword_set(debug) then begin
        if exit_status ne 0 then begin
           print, stdout
           print, stderr
        endif
     endif

     ct = ct + 3600.
     if ct gt ctr[1] then break

  endwhile

END
