/******************************************************************************
                        hkextract.cpp  -  description
                             -------------------
    begin                : Sat Sep 21 13:12:00 CET 2002

    author               : Hans Vaith
    email                : hav@mpe.mpg.de

    Cluster-II EDI: extract EDI HK parameters from EDI HK telemetry files
                    (SCS, CDDS EDI HK, MSF)
******************************************************************************
$Id: hkextract.cpp,v 1.2 2003/08/27 09:26:12 hav Exp $

******************************************************************************/

static char rcsid[] = "$Id: hkextract.cpp,v 1.2 2003/08/27 09:26:12 hav Exp $";

/******************** INCLUDE FILES ******************************************/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "hkextract.h"

/******************** CONSTANTS AND MACROS **********************************/


/******************** TYPE DEFINITIONS **************************************/

typedef struct {
  char *cnv_tbl_name;
  float cnv_offs;
  float cnv_factor;
} hk_cnv_t;

/******************** FUNCTION PROTOTYPES ***********************************/

static float ConvertHk(const char *cnv_tbl_name, short raw_val);
static void SetRawVal(const char *de_name, short val);

/******************** GLOBAL STATIC DATA ************************************/

static hk_extract_t hk[] = {
  {"FRCNT", 0,  0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"CONPC", 1,  0, 0x0fff,  0, 0.0, "CONPC", "mA", NULL, 0 },
  {"SUBMO", 1, 12, 0x000f,  0, 0.0, NULL, NULL, NULL, 0 },
  {"GD1PC", 2,  0, 0x0fff,  0, 0.0, "GDXPC", "mA", NULL, 0 },
  {"28VOL", 2, 12, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"GD1OL", 2, 13, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"GD2OL", 2, 14, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"5VSOL", 2, 15, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"GD2PC", 3,  0, 0x0fff,  0, 0.0, "GDXPC", "mA", NULL, 0 },
  {"WRTCP", 3, 12, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"WRSCM", 3, 13, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"EEWRP", 3, 14, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"EEPON", 3, 15, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"CRCNT", 4,  8, 0x00ff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"CVCNT", 4,  0, 0x00ff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"LASTC", 5,  0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"MODID", 6, 10, 0x003f,  0, 0.0, NULL, NULL, NULL, 0 },
  {"MOD08", 6, 10, 0x003f,  0, 0.0, "MOD08", NULL, NULL, 0},
  {"HVON1", 6,  9, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HVON2", 6,  8, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HVDIS", 6,  7, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"PW221", 6,  6, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"PW241", 6,  5, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"PW222", 6,  4, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"PW242", 6,  3, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"SUCOC", 6,  0, 0x0007,  0, 0.0, NULL, NULL, NULL, 0 },
  {"SUCB0", 6,  0, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"SUCXX", 6,  0, 0x0003,  0, 0.0, NULL, NULL, NULL, 0 },
  {"CUR5V", 7,  0, 0xffff,  0, 0.0, "CUR5V", "mA", "SUCOC", 0 },
  {"CU12V", 7,  0, 0xffff,  0, 0.0, "CU12V", "mA", "SUCOC", 1 },
  {"CN12V", 7,  0, 0xffff,  0, 0.0, "CU12V", "mA", "SUCOC", 2 },
  {"ACU5V", 7,  0, 0xffff,  0, 0.0, "ACU5V", "mA", "SUCOC", 3 },
  {"VOL5V", 7,  0, 0xffff,  0, 0.0, "VOL5V", "V",  "SUCOC", 4 },
  {"VO12V", 7,  0, 0xffff,  0, 0.0, "VO12V", "V",  "SUCOC", 5 },
  {"VN12V", 7,  0, 0xffff,  0, 0.0, "VO12V", "V",  "SUCOC", 6 },
  {"AVO5V", 7,  0, 0xffff,  0, 0.0, "AVO5V", "V",  "SUCOC", 7 },
  {"GD121", 8,  0, 0xffff,  0, 0.0, NULL, NULL, "SUCB0", 0 },
  {"GD221", 8,  0, 0xffff,  0, 0.0, NULL, NULL, "SUCB0", 1 },
  {"GD122", 9,  0, 0xffff,  0, 0.0, NULL, NULL, "SUCB0", 0 },
  {"GD222", 9,  0, 0xffff,  0, 0.0, NULL, NULL, "SUCB0", 1 },
  {"GD117", 10, 0, 0xffff,  0, 0.0, NULL, NULL, "SUCB0", 0 },
  {"GD217", 10, 0, 0xffff,  0, 0.0, NULL, NULL, "SUCB0", 1 },
  {"EVT11", 11, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"EVT21", 12, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"EVT12", 13, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"EVT22", 14, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"MCAD1", 15,12, 0x000f,  0, 0.0, NULL, NULL, NULL, 0 },
  {"MCRR1", 15, 0, 0x0fff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"MCAD2", 16,12, 0x000f,  0, 0.0, NULL, NULL, NULL, 0 },
  {"MCRR2", 16, 0, 0x0fff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"CHKSU", 17, 8, 0x00ff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"PLLST", 17, 0, 0x00ff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"BMAGN", 18, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"PGACF", 19,14, 0x0003,  0, 0.0, NULL, NULL, NULL, 0 },
  {"GDSET", 19,12, 0x0003,  0, 0.0, NULL, NULL, NULL, 0 },
  {"VASEC", 19,10, 0x0003,  0, 0.0, NULL, NULL, NULL, 0 },
  {"VASEC012",19,10,0x0003, 0, 0.0, "VASEC012", NULL, NULL, 0 },
  {"STPS2", 19, 9, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"STPS1", 19, 8, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"RAMHI", 19, 0, 0x00ff,  0, 0.0, NULL, NULL, "VASEC", 3 },
  {"GDSEL", 19, 7, 0x0001,  0, 0.0, NULL, NULL, "VASEC012", 1 },
  {"GDCHS", 19, 0, 0x003f,  0, 0.0, NULL, NULL, "VASEC012", 1},
  {"RAMAD", 20, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3 },

  {"S1MX1", 22,13, 0x0007,  0, 0.0, NULL, NULL, NULL, 0 },
  {"S1MX2", 22,10, 0x0007,  0, 0.0, NULL, NULL, NULL, 0 },
  {"OVRF1", 22, 9, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"DISA1", 22, 8, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"BASA1", 22, 0, 0x00ff,  0, 0.0, NULL, NULL, "SUCXX", 0 },
  {"ACPO1", 22, 0, 0x00ff,  0, 0.0, NULL, NULL, "SUCXX", 1 },
  {"COPO1", 22, 0, 0x00ff,  0, 0.0, NULL, NULL, "SUCXX", 2 },
  {"PWST1", 22, 7, 0x0001,  0, 0.0, NULL, NULL, "SUCXX", 3 },
  {"DTIM1", 22, 4, 0x0007,  0, 0.0, NULL, NULL, "SUCXX", 3 },
  {"EXIM1", 22, 2, 0x0001,  0, 0.0, NULL, NULL, "SUCXX", 3 },
  {"SWAP1", 22, 1, 0x0001,  0, 0.0, NULL, NULL, "SUCXX", 3 },
  {"TPLS1", 22, 0, 0x0001,  0, 0.0, NULL, NULL, "SUCXX", 3 },

  {"S2MX1", 24,13, 0x0007,  0, 0.0, NULL, NULL, NULL, 0 },
  {"S2MX2", 24,10, 0x0007,  0, 0.0, NULL, NULL, NULL, 0 },
  {"OVRF2", 24, 9, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"DISA2", 24, 8, 0x0001,  0, 0.0, NULL, NULL, NULL, 0 },
  {"BASA2", 24, 0, 0x00ff,  0, 0.0, NULL, NULL, "SUCXX", 0 },
  {"ACPO2", 24, 0, 0x00ff,  0, 0.0, NULL, NULL, "SUCXX", 1 },
  {"COPO2", 24, 0, 0x00ff,  0, 0.0, NULL, NULL, "SUCXX", 2 },
  {"PWST2", 24, 7, 0x0001,  0, 0.0, NULL, NULL, "SUCXX", 3 },
  {"DTIM2", 24, 4, 0x0007,  0, 0.0, NULL, NULL, "SUCXX", 3 },
  {"EXIM2", 24, 2, 0x0001,  0, 0.0, NULL, NULL, "SUCXX", 3 },
  {"SWAP2", 24, 1, 0x0001,  0, 0.0, NULL, NULL, "SUCXX", 3 },
  {"TPLS2", 24, 0, 0x0001,  0, 0.0, NULL, NULL, "SUCXX", 3 },

  {"2.5V1", 21, 8, 0x00ff,  0, 0.0, "2.5V", "V",  "S1MX1", 0},
  {"HVCR1", 21, 8, 0x00ff,  0, 0.0, "HVCR", "mA", "S1MX1", 1},
  {"HVER1", 21, 8, 0x00ff,  0, 0.0, "HVER", "V",  "S1MX1", 2},
  {"HVRF1", 21, 8, 0x00ff,  0, 0.0, "HVRF", "V",  "S1MX1", 3},
  {"PRGN1", 21, 8, 0x00ff,  0, 0.0, "PRGN", "V",  "S1MX1", 4},
  {"N_5V1", 21, 8, 0x00ff,  0, 0.0, "N_5V", "V",  "S1MX1", 5},
  {"P_5V1", 21, 8, 0x00ff,  0, 0.0, "P_5V", "V",  "S1MX1", 6},
  {"STMP1", 21, 8, 0x00ff,  0, 0.0, "STMP", "C",  "S1MX1", 7},

  {"2.5VA", 21, 0, 0x00ff,  0, 0.0, "2.5V", "V",  "S1MX2", 0},
  {"HVCRA", 21, 0, 0x00ff,  0, 0.0, "HVCR", "mA", "S1MX2", 1},
  {"HVERA", 21, 0, 0x00ff,  0, 0.0, "HVER", "V",  "S1MX2", 2},
  {"HVRFA", 21, 0, 0x00ff,  0, 0.0, "HVRF", "V",  "S1MX2", 3},
  {"PRGNA", 21, 0, 0x00ff,  0, 0.0, "PRGN", "V",  "S1MX2", 4},
  {"N_5VA", 21, 0, 0x00ff,  0, 0.0, "N_5V", "V",  "S1MX2", 5},
  {"P_5VA", 21, 0, 0x00ff,  0, 0.0, "P_5V", "V",  "S1MX2", 6},
  {"STMPA", 21, 0, 0x00ff,  0, 0.0, "STMP", "C",  "S1MX2", 7},

  {"2.5V2", 23, 8, 0x00ff,  0, 0.0, "2.5V", "V",  "S2MX1", 0},
  {"HVCR2", 23, 8, 0x00ff,  0, 0.0, "HVCR", "mA", "S2MX1", 1},
  {"HVER2", 23, 8, 0x00ff,  0, 0.0, "HVER", "V",  "S2MX1", 2},
  {"HVRF2", 23, 8, 0x00ff,  0, 0.0, "HVRF", "V",  "S2MX1", 3},
  {"PRGN2", 23, 8, 0x00ff,  0, 0.0, "PRGN", "V",  "S2MX1", 4},
  {"N_5V2", 23, 8, 0x00ff,  0, 0.0, "N_5V", "V",  "S2MX1", 5},
  {"P_5V2", 23, 8, 0x00ff,  0, 0.0, "P_5V", "V",  "S2MX1", 6},
  {"STMP2", 23, 8, 0x00ff,  0, 0.0, "STMP", "C",  "S2MX1", 7},

  {"2.5VB", 23, 0, 0x00ff,  0, 0.0, "2.5V", "V",  "S2MX2", 0},
  {"HVCRB", 23, 0, 0x00ff,  0, 0.0, "HVCR", "mA", "S2MX2", 1},
  {"HVERB", 23, 0, 0x00ff,  0, 0.0, "HVER", "V",  "S2MX2", 2},
  {"HVRFB", 23, 0, 0x00ff,  0, 0.0, "HVRF", "V",  "S2MX2", 3},
  {"PRGNB", 23, 0, 0x00ff,  0, 0.0, "PRGN", "V",  "S2MX2", 4},
  {"N_5VB", 23, 0, 0x00ff,  0, 0.0, "N_5V", "V",  "S2MX2", 5},
  {"P_5VB", 23, 0, 0x00ff,  0, 0.0, "P_5V", "V",  "S2MX2", 6},
  {"STMPB", 23, 0, 0x00ff,  0, 0.0, "STMP", "C",  "S2MX2", 7},

  {"GDUSC",  0, 0, 0xffff,  0, 0.0, "GDUSC", NULL, NULL,   0},

  {"I_D11", 25, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_D21", 26, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_D31", 27, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_D41", 28, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_D51", 29, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_D61", 30, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_D71", 31, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_D81", 32, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_AN1", 33, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_FC1", 34, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_CA1", 35, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_BW1", 36, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_BC1", 37, 0, 0xffff,  0, 0.0, "NEG", "V", "GDUSC", 0},
  {"GNDG1", 38, 0, 0xffff,  0, 0.0, "NEG", "V", "GDUSC", 0},
  {"I_EU1", 39, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},
  {"I_AU1", 40, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 0},

  {"28VS1", 25, 0, 0xffff,  0, 0.0, "28VS", "V",  "GDUSC", 1},
  {"5VUF1", 26, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 1},
  {"P5VC1", 27, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 1},
  {"GNDO1", 28, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},
  {"N5VC1", 29, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 1},
  {"5VIF1", 30, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 1},
  {"5VP11", 31, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 1},
  {"5VP21", 32, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 1},
  {"GPUX1", 33, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},
  {"GNUX1", 34, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},
  {"GPUY1", 35, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},
  {"GNUY1", 36, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},
  {"G_AN1", 37, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},
  {"G_FC1", 38, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},
  {"G_CA1", 39, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},
  {"G_BW1", 40, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 1},

  {"O_EU1", 25, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"O_AU1", 26, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"G_BC1", 27, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"GTMP1", 28, 0, 0xffff,  0, 0.0, "TEMP",    "C", "GDUSC", 2},
  {"REG11", 29, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"REG21", 30, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"N2VR1", 31, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"I_CS1", 32, 0, 0xffff,  0, 0.0, "I_CS",   "mA", "GDUSC", 2},
  {"O_EL1", 33, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"O_AL1", 34, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"O_OA1", 35, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"O_SR1", 36, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"O_SP1", 37, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"O_IA1", 38, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"O_RC1", 39, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},
  {"O_ET1", 40, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 2},

  {"I_EL1", 25, 0, 0xffff,  0, 0.0, "MA",        "mA", "GDUSC", 3},
  {"I_AL1", 26, 0, 0xffff,  0, 0.0, "MA",        "mA", "GDUSC", 3},
  {"I_OA1", 27, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 3},
  {"I_SR1", 28, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 3},
  {"I_SP1", 29, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 3},
  {"I_IA1", 30, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 3},
  {"I_RC1", 31, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 3},
  {"I_ET1", 32, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 3},
  {"CFQ11", 33, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 3},
  {"CFQ21", 34, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 3},
  {"CFQ31", 35, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 3},
  {"CFQ41", 36, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 3},
  {"CFQ51", 37, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 3},
  {"CFQ61", 38, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 3},
  {"CFQ71", 39, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 3},
  {"CFQ81", 40, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 3},

  {"I_D12", 25, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_D22", 26, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_D32", 27, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_D42", 28, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_D52", 29, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_D62", 30, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_D72", 31, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_D82", 32, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_AN2", 33, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_FC2", 34, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_CA2", 35, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_BW2", 36, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_BC2", 37, 0, 0xffff,  0, 0.0, "NEG", "V", "GDUSC", 4},
  {"GNDG2", 38, 0, 0xffff,  0, 0.0, "NEG", "V", "GDUSC", 4},
  {"I_EU2", 39, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},
  {"I_AU2", 40, 0, 0xffff,  0, 0.0, "MA", "mA", "GDUSC", 4},

  {"28VS2", 25, 0, 0xffff,  0, 0.0, "28VS", "V",  "GDUSC", 5},
  {"5VUF2", 26, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 5},
  {"P5VC2", 27, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 5},
  {"GNDO2", 28, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},
  {"N5VC2", 29, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 5},
  {"5VIF2", 30, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 5},
  {"5VP12", 31, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 5},
  {"5VP22", 32, 0, 0xffff,  0, 0.0, "NEG5", "V",  "GDUSC", 5},
  {"GPUX2", 33, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},
  {"GNUX2", 34, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},
  {"GPUY2", 35, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},
  {"GNUY2", 36, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},
  {"G_AN2", 37, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},
  {"G_FC2", 38, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},
  {"G_CA2", 39, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},
  {"G_BW2", 40, 0, 0xffff,  0, 0.0, "NEG",  "V",  "GDUSC", 5},

  {"O_EU2", 25, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"O_AU2", 26, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"G_BC2", 27, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"GTMP2", 28, 0, 0xffff,  0, 0.0, "TEMP",    "C", "GDUSC", 6},
  {"REG12", 29, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"REG22", 30, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"N2VR2", 31, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"I_CS2", 32, 0, 0xffff,  0, 0.0, "I_CS",   "mA", "GDUSC", 6},
  {"O_EL2", 33, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"O_AL2", 34, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"O_OA2", 35, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"O_SR2", 36, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"O_SP2", 37, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"O_IA2", 38, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"O_RC2", 39, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},
  {"O_ET2", 40, 0, 0xffff,  0, 0.0, "NEG",     "V", "GDUSC", 6},

  {"I_EL2", 25, 0, 0xffff,  0, 0.0, "MA",        "mA", "GDUSC", 7},
  {"I_AL2", 26, 0, 0xffff,  0, 0.0, "MA",        "mA", "GDUSC", 7},
  {"I_OA2", 27, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 7},
  {"I_SR2", 28, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 7},
  {"I_SP2", 29, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 7},
  {"I_IA2", 30, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 7},
  {"I_RC2", 31, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 7},
  {"I_ET2", 32, 0, 0xffff,  0, 0.0, "MA1",       "mA", "GDUSC", 7},
  {"CFQ12", 33, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 7},
  {"CFQ22", 34, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 7},
  {"CFQ32", 35, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 7},
  {"CFQ42", 36, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 7},
  {"CFQ52", 37, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 7},
  {"CFQ62", 38, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 7},
  {"CFQ72", 39, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 7},
  {"CFQ82", 40, 0, 0xffff,  0, 0.0, "FREQUENZ",  "Hz", "GDUSC", 7},

  {"RAM00", 25, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM01", 26, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM02", 27, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM03", 28, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM04", 29, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM05", 30, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM06", 31, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM07", 32, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM08", 33, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM09", 34, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM10", 35, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM11", 36, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM12", 37, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM13", 38, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM14", 39, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},
  {"RAM15", 40, 0, 0xffff,  0, 0.0, NULL, NULL, "VASEC", 3},

  {"OLPRM", 43,12, 0x000f,  0, 0.0, NULL, NULL, "MODID", 8},
  {"CONPV", 43, 0, 0x0fff,  0, 0.0, NULL, NULL, NULL,    0},
  {"EECHK", 44,12, 0x000f,  0, 0.0, NULL, NULL, "MODID", 8},
  {"FGCNT", 44, 8, 0x000f,  0, 0.0, NULL, NULL, "MODID", 8},
  {"PACMO", 44, 0, 0x0007,  0, 0.0, NULL, NULL, NULL, 0},
  {"M8STA", 44, 4, 0x0007,  0, 0.0, NULL, NULL, "MODID", 8},
  {"BTYPE", 44, 7, 0x0001,  0, 0.0, NULL, NULL, "MOD08", 0},

  {"HKW00",  0, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW01",  1, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW02",  2, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW03",  3, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW04",  4, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW05",  5, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW06",  6, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW07",  7, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW08",  8, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW09",  9, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW10", 10, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW11", 11, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW12", 12, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW13", 13, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW14", 14, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW15", 15, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW16", 16, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW17", 17, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW18", 18, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW19", 19, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW20", 20, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW21", 21, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW22", 22, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW23", 23, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW24", 24, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW25", 25, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW26", 26, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW27", 27, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW28", 28, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW29", 29, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW30", 30, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW31", 31, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW32", 32, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW33", 33, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW34", 34, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW35", 35, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW36", 36, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW37", 37, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW38", 38, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW39", 39, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW40", 40, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW41", 41, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW42", 42, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW43", 43, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },
  {"HKW44", 44, 0, 0xffff,  0, 0.0, NULL, NULL, NULL, 0 },

  { NULL,    0, 0, 0x0000,  0, 0.0, NULL, NULL, NULL, 0 }
};

static hk_cnv_t cnv[] = {
  { "CONPC", -512.0, 0.4888 },
  { "GDXPC",    0.0, 0.0 },
  { "CUR5V", -512.0, 1.4664 },
  { "CU12V", -512.0, 0.04888 },
  { "ACU5V",    0.0, 0.0 },
  { "VOL5V", -512.0, 0.01191 },
  { "VO12V", -512.0, 0.027696 },
  { "AVO5V", -512.0, 0.018736 },
  { "VASEC012", 0.0, 0.0 },
  { "2.5V", -1127.48, -0.0025 },
  { "HVCR",  -227.5, -0.25 },
  { "HVER",  -227.5, -0.025 },
  { "HVRF",  -227.5, -0.0125 },
  { "PRGN",  -227.5, -0.025 },
  { "N_5V",   372.5, -0.01 },
  { "P_5V",  -627.5, -0.01 },
  { "STMP",  -157.5, -0.5 },
  { "GDUSC",    0.0, 0.0 },
  { "MA",       0.0, 0.625/22 },
  { "NEG",      0.0, 0.000625 },
  { "28VS",     0.0, 0.000625 * 20.6 },
  { "NEG5",     0.0, 0.000625 * 5 },
  { "TEMP",  -436.8, 0.625 },
  { "I_CS",     0.0, 0.625 / 50 },
  { "MA1",      0.0, 0.625 / 56 },
  { "FREQUENZ", 0.0, 32. },
  { "MOD08",    0.0, 0.0 },
  { NULL,       0.0, 0.0 }
};

static hk_val_t *hk_val_p = NULL;


/******************** FUNCTION DEFINITIONS  *********************************/

// ==========================================================================
   size_t HkExtractInitialize(void)
// ==========================================================================
{
  size_t n_entries = sizeof(hk) / sizeof(hk_extract_t);

  hk_val_p = (hk_val_t *)calloc(n_entries, sizeof(hk_val_t));

  if (hk_val_p==NULL) {
     fprintf(stderr, "HkExtractInitialize(): error allocating memory\n");
     exit(1);
  }

  return (n_entries * sizeof(hk_val_t));
}


// ==========================================================================
   void *HkExtractGetAddress(void)
// ==========================================================================
{
  if (hk_val_p==NULL) {
      fprintf(stderr, "HkExtractGetAddress(): memory not yet allocated\n");
      exit(1);
  }

  return hk_val_p;
}



// ==========================================================================
   void hkextract(short *hkd)
// ==========================================================================
// Parameters:
//    short *hkd       pointer to 90 byte EDI HK data buffer
// **************************************************************************
{
  int i;
  hk_extract_t *p;

  if (0) printf("%s\n", rcsid);

  if (hk_val_p==NULL) {
      fprintf(stderr, "hkextract(): memory not yet allocated\n");
      exit(1);
  }

  for (i=0; hk[i].de_name != NULL; ++i) {

     p = &hk[i];

     if (p->subcom_name == NULL || GetRawVal(p->subcom_name)==p->subcom_val) {

        hk_val_p[i].raw_val = (hkd[p->wd] >> p->offs) & p->mask;

        if (p->cnv_tbl_name) {
           hk_val_p[i].cnv_val = ConvertHk(p->cnv_tbl_name, hk_val_p[i].raw_val);
        }

     }
  }

}

// =========================================================================
   static float ConvertHk(const char *cnv_tbl_name, short raw_val)
// ==========================================================================
{
  int i;

  if (strcmp(cnv_tbl_name, "GDXPC") == 0) {

    return 0.4888 * ( raw_val + 0.716*GetCnvVal("CONPC") - 512 );

  } else if (strcmp(cnv_tbl_name, "ACU5V") == 0) {

    //    printf("ACU5V conversion needs to be implemented\n");

  } else if (strcmp(cnv_tbl_name, "VASEC012") == 0) {

    SetRawVal("VASEC012", raw_val==3 ? 0 : 1);
    return 0.0;

  } else if (strcmp(cnv_tbl_name, "GDUSC") == 0) {

    SetRawVal("GDUSC", GetRawVal("VASEC")==0 ?
                       4 * GetRawVal("GDSEL") + GetRawVal("GDSET") : -1);
    return 0.0;

  } else if (strcmp(cnv_tbl_name, "MOD08") == 0) {

    SetRawVal("MOD08", (raw_val == 0 || raw_val == 8) ? 1 : 0);
    return 0.0;

  } else {

    for (i=0; cnv[i].cnv_tbl_name != NULL; ++i) {
      if (strcmp(cnv_tbl_name, cnv[i].cnv_tbl_name) == 0) {
	return (raw_val + cnv[i].cnv_offs) * cnv[i].cnv_factor;
      }
    }

    fprintf(stderr, "conversion table %s not found\n", cnv_tbl_name);
    exit(1);

  }

  return 0.0;

}

// ==========================================================================
   static void SetRawVal(const char *de_name, short val)
// ==========================================================================
{
  int i;

  for (i=0; hk[i].de_name != NULL; ++i) {
    if (strcmp(hk[i].de_name, de_name) == 0) {
      hk_val_p[i].raw_val = val;
      break;
    }
  }

}


// ==========================================================================
   short GetRawVal(const char *de_name)
// ==========================================================================
{
  int i;

  for (i=0; hk[i].de_name != NULL; ++i) {
    if (strcmp(hk[i].de_name, de_name) == 0) {
       return hk_val_p[i].raw_val;
    }
  }

  fprintf(stderr, "GetRawVal(): parameter %s not found\n", de_name);
  exit(1);
}


// ==========================================================================
   float GetCnvVal(const char *de_name)
// ==========================================================================
{
  int i;

  for (i=0; hk[i].de_name != NULL; ++i) {
    if (strcmp(hk[i].de_name, de_name) == 0) {
      return hk_val_p[i].cnv_val;
    }
  }

  fprintf(stderr, "GetRawVal(): parameter %s not found\n", de_name);
  exit(1);
}

