; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; Cluster Active Archive : Program to produce EDI Time-of-Flight data
; ****************************************************************************

@readlib.pro
@runest.pro

; ===========================================================================
  FUNCTION create_header_egd, sc, date, version, gendate, iso_time_range, $
                              program_version
; ===========================================================================
;
; ***************************************************************************

   if n_params() ne 6 then begin
      message, 'Expecting six parameters (sc,date,version,gendate,iso_time_range,program_version)', /cont
      retall
   endif

   name = 'C'+sc+'_CP_EDI_EGD__'+date+'_V'+version


   header_lines = $
      [ 'FILE_NAME="'+name+'.cef"', $
        'FILE_FORMAT_VERSION="CEF-2.0"', $
        '!', $
        'START_META = LOGICAL_FILE_ID', $
        ' ENTRY = "'+name+'"', $
        'END_META = LOGICAL_FILE_ID', $
        '!', $
        'START_META = VERSION_NUMBER', $
        ' ENTRY = "'+version+'"', $
        'END_META = VERSION_NUMBER', $
        '!', $
        'START_META = FILE_TIME_SPAN', $
        ' VALUE_TYPE = ISO_TIME_RANGE', $
        ' ENTRY = '+iso_time_range, $
        'END_META = FILE_TIME_SPAN', $
        '!', $
        'START_META = GENERATION_DATE', $
        ' VALUE_TYPE = ISO_TIME', $
        ' ENTRY = '+gendate, $
        'END_META = GENERATION_DATE', $
        '!', $
        'START_META = FILE_CAVEATS', $
        ' ENTRY = "'+program_version+'"', $
        'END_META = FILE_CAVEATS', $
        '!', $
        '! include EGD (DATASET) HEADER File for Cluster-'+sc, $
        '! with variable definitions, metadata_type and _version', $
        'include="C'+sc+'_CH_EDI_EGD_DATASET.ceh"', $
        '!', $
        'DATA_UNTIL=EOF', $
        '!' ]

   return, header_lines

END


; ============================================================================
  PRO CAA_EDITOF, sc, year, month, day, version, fgmrng=fgmrng, debug=debug
; ============================================================================
; Parameters
;    sc     int   (1,2,3,4)
;    year   int   (2001,...)
;    month  int   (1,2,...,12)
;    day    int   (1,2,...,31)
;
; Keywords
;    fgmrng intarr  an array of fgm ranges for which EDI Time-of-Flight
;                   data shall be produced
;    debug  0/1     print debugging information
; ****************************************************************************

@paths.inc
@vector.inc

  program_version = 'CAA_EDITOF_V1_02  2006-03-01T10:30:00Z'

  if n_params() ne 5 then begin
     message, 'Call syntax: caa_editof, sc, year, month, day, version, fgmrng=fgmrng, debug=debug', /cont
     retall
  endif

  i_sc      = fix(sc)
  i_year    = fix(year)
  i_month   = fix(month)
  i_day     = fix(day)
  i_version = fix(version)

  if i_version lt 0 or i_version gt 99 then begin
     message, 'Error: version must be between 0 and 99', /cont
     retall
  endif

  s_sc      = strtrim(i_sc,2)
  s_year    = strtrim(i_year,2)
  s_month   = string(format='(I2.2)', i_month)
  s_day     = string(format='(I2.2)', i_day)
  s_version = string(format='(I2.2)', i_version)


  if not keyword_set(debug) then debug = 0 else debug = 1

  date     = s_year + s_month + s_day
  iso_date = s_year + '-' + s_month + '-' + s_day
  ctr = CluTimeStr2Val(iso_date) + [16, 24.*3600 - 16]

  if not keyword_set(fgmrng) then fgmrng = [2,3,4]

  ; Read EDI Windshield Wiper mode data
  ; -----------------------------------
  edi_ww = GetWWdata(i_sc, ctr=ctr, debug=debug)
  if edi_ww.hdr.status eq 1 then begin
     message, edi_ww.hdr.msg, /cont
     return
  endif

  print, 'Reading magnetic field data...'
  datelist = '~/datelist.txt'
  spawn, 'echo ' + date + ' >' + datelist

  ct0_str = string(format='(F17.6)', ctr[0]-15)
  ct1_str = string(format='(F17.6)', ctr[1]+15)
  cmd_str = GETFGM_DIR + 'getfgm2 ' + s_sc + ' ' + datelist + ' ' + $
            ct0_str + ' ' + ct1_str + ' scs hr'
  spawn, cmd_str, result, err_result, exit_status=exit_status
  if exit_status ne 0 then begin
     print, 'Reading of magnetic field data failed!'
     print, result
     print, err_result
     retall
  endif

  tempmagfile = '~/temporary.fgm' ; this name must match the name used in script getfgm2
  if debug then print,file_lines(tempmagfile)
  proto = replicate( {VTS_T}, 1 )
  b_fgm = { hdr:getcluhdr(0,'',sc_id=i_sc,frame='SCS',units='nT'), $
            data:myfread(tempmagfile, '', proto) }
  spawn, 'rm -f ' + tempmagfile

  print, 'Looping through FGM ranges...'

  dummy = { EGD_TMP_T, ct:0.0d, str:'' } ; define data structure

  ; Loop: treat different FGM ranges separately
  ; -------------------------------------------
  for idx=0,n_elements(fgmrng)-1 do begin

     range = fgmrng[idx]
     if debug then print, 'FGM range = ', range

     ; Extract EDI Windshield Wiper data for current fgm range
     ; -------------------------------------------------------
     ds = ExtractWWdata(edi_ww, fgmrng=range)
     if ds.hdr.status eq 1 then begin
        message, 'FGM range='+strtrim(range,2)+': '+ds.hdr.msg, /cont
        continue
     endif

     ; Cut EDI data into chunks and do runner order determination
     ; separately for each chunk
     ; ----------------------------------------------------------
     N_TOTAL = n_elements(ds.data)
     N_CHUNK = 60000L
     idx0 = 0L
     first_pass = 1
     data = 0
     while idx0 lt N_TOTAL do begin

        idx1 = min([idx0+N_CHUNK,N_TOTAL])
        if N_TOTAL - idx1 lt N_CHUNK/2 then idx1 = N_TOTAL
        if idx1-idx0 lt 1000 then break
        x = lindgen(idx1-idx0) + idx0
        now = { hdr:ds.hdr, data:ds.data[x] }
        if debug then $
           print, strmid(clutimeval2str(min(now.data.ct)),0,19) + ' -  ' + $
           strmid(clutimeval2str(max(now.data.ct)),0,19) + '  #points : ', n_elements(x)
        re = RunEst(now, b_fgm, smooth=5, /quiet)
        if re.status ne 0 then begin
           if debug then print, re.msg
           idx0 = idx1
           continue
        endif else begin
           if first_pass eq 1 then begin
              data = re.data
              first_pass = 0
           endif else begin
              data = [ data, re.data ]
           endelse
        endelse
        idx0 = idx1
     endwhile


     ; Collect time-of-flight data for current FGM range
     ; -------------------------------------------------
     nrdatapoints=n_elements(data)

     if nrdatapoints le 100 then begin
        if debug then print, 'not enough EGD data --> skipping this range'
        continue
     endif

     xA = where(data.runorder gt 0, cntA) ; find class A beams
     if cntA eq 0 then continue

     classA = data[xA]

     NN = n_elements(out)
     outnew = replicate( {EGD_TMP_T}, NN+cntA )
     if NN gt 0 then outnew[0:NN-1] = out

     outnew[NN:NN+cntA-1].ct = classA.beamtime

     print, 'Writing ' + strtrim(cntA,2) + ' class A beam ToFs...'
     for i=0L,cntA-1 do begin
        this = classA[i]
        if this.gd eq 21 then detector_id = 1 else detector_id = 2
        outnew[NN+i].str = string(format='(A23,A3,"Z, ",F10.3,", ",F10.3,", ",I1,", ",I1)', $
                              strmid(CluTimeVal2Str(this.beamtime),0,23), $
                              strmid(CluTimeVal2Str(this.beamtime),24,3), $
                              this.estof[this.runorder-1], $
                              0.2 * this.tchip, $ ; 1/5 of tChip
                              detector_id, $
                              this.en)
     endfor

     out = outnew

  endfor

  ; Sort data
  ; ---------
  nodata = 0
  if n_elements(out) eq 0 then nodata = 1 $
  else begin
     sortindex = sort(out.ct)
     out_lines = out[sortindex].str
  endelse

  ; Construct file name for output
  ; ------------------------------
  date = s_year + s_month + s_day
  out_filename = 'C'+s_sc+'_CP_EDI_EGD__'+date+'_V'+s_version+'.cef'

  iso_generation_date = strmid(CluTimeVal2Str(systime(1)),0,19) + 'Z'

  iso_this_day = s_year+'-'+s_month+'-'+s_day
  iso_next_day = strmid(CluTimeVal2Str(CluTimeStr2Val(iso_this_day+'t12')+86400.),0,10)
  iso_time_range = iso_this_day+'T00:00:00Z/'+iso_next_day+'T00:00:00Z'
  header_lines = create_header_egd(s_sc,date,s_version,iso_generation_date,$
                                   iso_time_range,program_version)

  ; Write data to output file
  ; -------------------------
  all_lines = header_lines
  if nodata eq 0 then all_lines = [ all_lines, out_lines ]
  MyFwrite, out_filename, all_lines


  if nodata eq 1 then print, 'No EGD data! CEF header data written to ' + out_filename $
  else print, 'EGD data written to ' + out_filename

END

