; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; $Id: axlabel.pro,v 1.8 2004/08/23 12:09:38 hav Exp $
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


@clutime.pro

; ============================================================================
  FUNCTION CalcAxis, ct1, ct2, maxmajorticks, debug=debug
; ============================================================================

COMMON AXLABEL2_BLOCK, ax_mode

  if keyword_set(debug) then debug = 1 else debug = 0

  ret = { status:0, msg:'' }

  mostr = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', $
           'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec' ]

  SECS_PER_MONTH = 86400.d*365./12 ; average number of seconds per month

  n_months = fix( (ct2-ct1)/SECS_PER_MONTH )

  if n_months gt 3 then begin  ; ok, we do it in (non-constant!) increments of months

     ax_mode = 1 ; flag for 'increment in months'

     if debug then begin
        message, 'Time tags cover more than 3 months.', /cont
        message, 'Using first-of-month type tick marks', /cont
     endif

     ct0 = CluTimeStr2Val(strmid(CluTimeVal2Str(ct1),0,10)) ; ctime of midnight

     ; major tick mark spacing in months
     xmajor_arr = [ 1L, 2, 3, 6, 12, 24, 36, 48 ]
     xminor_arr = [ 2,  4, 3, 3,  4,  4,  3,  4 ]

     dummy = where( n_months gt (xmajor_arr*maxmajorticks), idx )
     if idx ge n_elements(xmajor_arr) then begin
        msg = 'first-of-month tick mark calculation: index out of bounds!'
        if debug then message, msg, /cont
        return, { status:1, msg:msg }
     endif

     xmajor = xmajor_arr[idx]
     xminor = xminor_arr[idx]

     if debug then message, 'Tick mark spacing is ' + strtrim(xmajor,2) + ' month(s)', /cont

     ym = fix( strsplit( strmid(CluTimeVal2Str(ct1),0, 7), '-', /extract) )
     year  = ym[0]
     month = ym[1]
     month = month - month mod xmajor + 1 - xmajor


     tickv = 0.d & tickn = ''
     space = 15.d * 86400. * xmajor ; half the (average) major tick mark spacing in seconds

     if debug then begin
        print, 'ct1 = ' + clutimeval2str(ct1)
        print, 'ct2 = ' + clutimeval2str(ct2)
     endif

     repeat begin
        month = month + xmajor
        while month gt 12 do begin
           year = year + 1
           month = month - 12
        endwhile
        ymd = string(format='(I4.4,"-",I2.2,"-01")', year, month)
        ct = CluTimeStr2Val(ymd)
        if ct lt ct1 then continue
        if ct gt ct2 then break

        if ct lt ct1+space or ct gt ct2-space then $
           tickn = [tickn, ' ' ] $
        else $
           tickn = [tickn, ymd ]

        tickv = [tickv, ct-ct0]

        if debug then message, 'Next tick : ' + ymd, /cont

     endrep until 1 eq 0

     nt = n_elements(tickv) - 1      ;
     tickv = tickv[1:nt]
     tickn = tickn[1:nt]

     mydate = ''

     return, { status:0, msg:'Ok', $
               xtickv:tickv, $
               xtickn:tickn, $
               xticks:nt-1, $
               xminor:xminor, $
               xrange:[ct1,ct2]-ct0, $
               datestr:mydate, $
               ct0:ct0 }


  endif else begin

     ax_mode = 0

     ; time axis tick mark spacing
     ; and corresponding number of minor tick marks
     ; --------------------------------------------

     ; days
     xmajor_arr = [ 1L, 2, 5, 10, 20,    30, 50 ]
     xminor_arr = [ 4,  4, 5,  5,  5,     5,  5 ]
     xmajor_arr = xmajor_arr * 24L ; convert from days to hours

     ; hours
     xmajor_arr = [ 1L, 2, 3, 6, 12, xmajor_arr ]
     xminor_arr = [ 6,  4, 6, 6,  6, xminor_arr ]
     xmajor_arr = xmajor_arr * 60L ; convert from hours to minutes

     ; minutes
     xmajor_arr  = [ 1L, 2, 5, 10, 20, 30, xmajor_arr ]
     xminor_arr  = [ 6,  4, 5,  5,  4,  6, xminor_arr ]
     xmajor_arr = xmajor_arr * 60L ; convert from minutes to seconds

     ; seconds
     xmajor_arr = [ 1, 2, 5, 10, 20, 30, xmajor_arr ]
     xminor_arr = [ 6, 4, 5,  5,  4,  6, xminor_arr ]

     ; find suitable tickmark spacing
     ; ------------------------------
     delta_sec = ct2 - ct1  ; interval width in seconds

     cta = ct1 & ctb = ct2

     if delta_sec lt 3 then begin
        print, 'cannot plot ranges less than 3 seconds : ' + strtrim(delta_sec,2)
        print, 'extending plot range to 4 seconds'

        add = 4 - delta_sec
        cta = ct1 - add/2
        ctb = ct2 + add/2
        delta_sec = 4
     endif

     gtspc = where( delta_sec gt (xmajor_arr*maxmajorticks), idx )

     if idx ge n_elements(xmajor_arr) then begin
        print, 'Calculated index out of bounds. Correcting...'
        idx = n_elements(xmajor)-1
     endif

     xmajor = xmajor_arr[idx]
     xminor = xminor_arr[idx]

     ct0 = CluTimeStr2Val(strmid(CluTimeVal2Str(cta),0,10)) ; ctime of midnight
     tsec1 = cta - ct0 ; start time : seconds since midnight of start day
     tsec2 = ctb - ct0 ; end time   : seconds since midnight of start day

     ; set startvalue to an integer multiple of xmajor
     ; -----------------------------------------------
     ;  startval = (long(tsec1) + xmajor)/xmajor * xmajor
     tsec1_long = long(tsec1)
     startval = tsec1_long + xmajor - tsec1_long mod xmajor ; same thing as above but easier to understand...
     i = 0L
     tickv = 0
     tickn = ''
     while tickv(i) lt tsec2 do begin
        newval = startval + xmajor*i
        tickv = [tickv, newval]
        tickhr = newval / 3600
        tickmin = (newval mod 3600) / 60
        ticksec = newval mod 60
        tickn = [tickn, string(format = '(I2.2,":",I2.2,":",I2.2)', $
                               tickhr mod 24, tickmin, ticksec)]
        i = i+1

     endwhile
     nt = n_elements(tickv) - 2      ;
     tickv = tickv[1:nt]
     tickn = tickn[1:nt]

     ymd  = CluTimeVal2Str(ct0+tickv[0])
     ymd2 = CluTimeVal2Str(ct0+tickv[nt-1])

     mydate = strmid(ymd,8,2) + '-' + mostr(fix(strmid(ymd,5,2))-1) + '-' + strmid(ymd,0,4)
     if strmid(ymd,0,10) ne strmid(ymd2,0,10) then begin
        mydate = mydate + '  -  ' + $
           strmid(ymd2,8,2) + '-' + mostr(fix(strmid(ymd2,5,2))-1) + '-' + strmid(ymd2,0,4)
     endif

     if strmid(ymd2,0,10) ne strmid(ymd,0,10) then begin ; 'yyyy-mm-dd' comparison
        for i=0,nt-1 do begin         ; add yyyy-mm-dd to each major tick
           ymd = strmid(CluTimeVal2Str(ct0+tickv[i]),0,10)
           tickn[i] = ymd + '!c' + tickn[i]
        endfor
     endif


     return, { status:0, msg:'Ok', $
               xtickv:tickv, $
               xtickn:tickn, $
               xticks:nt-1, $
               xminor:xminor, $
               xrange:[tsec1,tsec2], $
               datestr:mydate, $
               ct0:ct0 }
  endelse

END




; =======================================================================
  FUNCTION axlabel, ct1, ct2, $
                    suppress=suppress, $
                    show=show, $
                    reset=reset, $
                    maxmajorticks=maxmajorticks, $
                    noxtitle=noxtitle, $
                    debug=debug, $
                    option_string=option_string
; =======================================================================
; Provide nice time axis labelling
; ct1 and ct2 are the begin and end times of the time interval in ctime
; (seconds since midnight of 1-Jan-1970)
; The !x.... variables are set, so that ctime can be used as the
; x-variable in plots (subtract ret.ct0 !!)
; example --> plot, ctime-ret.ct0, mydata
; Use 'dummy=axlabel(/reset)' to reset !x... variables after plotting
; ***********************************************************************

COMMON AXLABEL_BLOCK, ax_ret
COMMON AXLABEL2_BLOCK, ax_mode

  if not keyword_set(debug) then debug = 0 else debug = 1


  ; Reset !X... variables if requested
  ; ----------------------------------
  if keyword_set(reset) then begin
     !x.tickv = 0
     !x.tickname = ''
     !x.ticks = 0
     !x.minor = 0
     !x.range = [0,0]
     !x.title=''

     ax_ret = { status:1, msg:'axlabel state: reset' }
     return, ax_ret
  endif

  ; This affects the maximum number of major tickmarks,
  ; although in a somewhat convoluted way
  ; ----------------------------------------------------
  if not keyword_set(maxmajorticks) then maxmajorticks = 7L $
  else maxmajorticks = long(maxmajorticks)


  ; return structure
  ; ----------------

  case n_params() of
  1 : begin
     if n_elements(ct1) lt 2 then message, 'not enough elements in parameter ct1'
     cta = min(ct1, max=ctb)
     do_calc = 1
  end
  2 : begin
     cta = ct1
     ctb = ct2
     do_calc = 1
  end
  else : begin
     ; no parameters present
     if n_elements(ax_ret) eq 0 then $
        message, 'x-axis not yet established. Need parameters!'
     if ax_ret.status ne 0 then begin
        message, ax_ret.msg
     endif
     do_calc = 0
  end
  endcase

  if do_calc eq 1 then begin
     ax_ret = CalcAxis(cta, ctb, maxmajorticks, debug=debug)
     if ax_ret.status ne 0 then message, ax_ret.msg
  endif



  ; set the stuff directly (use /reset to undo)
  ; ----------------------
  !x.tickv    = ax_ret.xtickv
  !x.tickname = ax_ret.xtickn
  !x.ticks    = ax_ret.xticks
  !x.minor    = ax_ret.xminor
  !x.range    = ax_ret.xrange
  !x.title    = ax_ret.datestr
;  !x.title    = ''

  if keyword_set(noxtitle) then !x.title = ''
  if keyword_set(suppress) then begin
     !x.title = ''
     !x.tickname = strarr(30) + ' '
  endif


  if debug then begin
     print, tag_names(ax_ret)
     print, ax_ret
  endif

  return, ax_ret

end


