; +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; $Id: vtstriang.pro,v 1.6 2004/03/29 14:34:31 hav Exp $
; +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

@vts_util.pro

; =============================================================================
  FUNCTION vDeSpin_, invec, wt,  reverse = reverse
; =============================================================================
; EQ-S / Cluster : despin a 3D vector
;
; Parameters:
;   invec : fltarr(3,NN)
;   wt    : phase array for despinning
;
; Return
;   outvec : fltarr(3,NN)
; *****************************************************************************

@vector.inc

if n_params() ne 2 then begin
   print, 'vDeSpin_() : need 2 parameters (invec, wt)'
   retall
endif

if keyword_set(reverse) then sign = -1. else sign = 1.

invec_info = size(invec)
if invec_info[0] ne 2 or invec_info[1] ne 3 then begin
   message, 'Bad dimensions! Expecting fltarr(3,NN)', /cont
   help, invec
   retall
endif

NN = invec_info[2]

dim_wt = n_elements(wt)

if NN ne dim_wt then begin
   message, 'vDeSpin_() : arrays are incompatible in size', /cont
   help, invec, wt
   retall
endif

sinwt = sin(wt)
coswt = cos(wt)

outvec = invec

outvec[1,*] =        invec[1,*] * coswt - sign * invec[2,*] * sinwt
outvec[2,*] = sign * invec[1,*] * sinwt +        invec[2,*] * coswt

return, outvec

END



; =============================================================================
  FUNCTION vGyroUnitVector_, Bvector
; =============================================================================
; EQ-S / Cluster : calculate a gyro-coordinate unit vector (x,y) from a
; magnetic field vector (both expressed in cartesian coordinates)
; The resulting vector will be returned as [xgyro,ygyro], where
; *****************************************************************************

@vector.inc

  if n_params() ne 1 then begin
     message, 'need exactly one parameter!', /cont
     retall
  endif

  NN = vec_check(Bvector)

  ; use an arbitrary but fixed vector as reference direction
  ; --------------------------------------------------------
  s_vec = fltarr(3,NN)
  s_vec[1,*] = 1.0

  lt0 = where(Bvector[2,*] lt 0) ; really depends on the Bz component of Bgse !!!!
  if lt0[0] ne -1 then s_vec[1,lt0] = -s_vec[1,lt0]

  ; gyro coordinate unit vectors: xgyro = S x B , ygyro = B x xgyro
  ; ---------------------------------------------------------------
  xgyro = vec_normalize( vec_crossp(s_vec,Bvector) )
  ygyro = vec_normalize( vec_crossp(Bvector,xgyro) ) ; normalize should not be necessary here
                                                     ; but do it anyway
  return, [ xgyro, ygyro ]

END


; =============================================================================
  FUNCTION vGyroProjection_, invec, Bvec
; =============================================================================
; EQ-S / Cluster : project a 3D vector into the gyro plane (perp to B)
;
; Parameters
;   invec   fltarr(3,NN)
;   Bvec    fltarr(3,NN)
; *****************************************************************************

@vector.inc

dimv = vec_check(invec)
dimB = vec_check(Bvec)

if dimv ne dimB and dimB ne 1 then begin
   message, 'array dimensions of passed vectors are incompatible', /cont
   help, dimv, dimB
   help, /traceback
   retall
endif

Bmag = vec2mag(Bvec)

v_dot_B = total(invec*Bvec,1) ; invec.x*Bvec.x + invec.y*Bvec.y + invec.z*Bvec.z

factor1 = v_dot_B / Bmag    ; component along B, for diagnostics
factor2 = v_dot_B / Bmag^2

outvec = invec

for i=0,2 do $
   outvec[i,*] = invec[i,*] - factor2 * Bvec[i,*]

return, outvec

END


; =============================================================================
  FUNCTION vGyroComponents_, vector_in, b_in, ctarr=ctarr
; =============================================================================
; EQ-S / Cluster : calculate components of a 3D vector time series vts_in
; along the two components of a gyro-coordinate unit vector that is derived
;  from the magnetic field vector time series b_in
; (all expressed in cartesian coordinates)
;
; Two ways to call this function are supported
; --------------------------------------------
;   Either
;            vector_in and b_in are both of type fltarr(3,NN)
;            in this case the return quantity will be of type fltarr(2,NN)
;
;   Or       vector_in is of type array(VTS_T, NN) and
;            b_in is of type array(VTS_T, MM)
;            in this case the keyword ctarr is allowed to specify
;            an alternate time tag series for output,
;            and the return quantity will be of type array(V2TS_T, LL)
;            where LL is either NN or (if ctarr is specified) n_elements(ctarr)
;
; *****************************************************************************

@vector.inc

  vts_debug, 1

  dim_vec = vec_check(vector_in)
  dim_vts = vts_check(vector_in)

  if dim_vec eq 0 and dim_vts eq 0 then begin
     message, 'passed vector (vector_in) is neither fltarr(3,NN) nor array(VTS_T, NN)', /cont
     help, vector_in
     help, /traceback
     retall
  endif else if dim_vec gt 0 then begin
     if keyword_set(ctarr) then begin
        message, 'keyword ctarr is only allowed with vector time series'
        retall
     endif
     if vec_check(b_in) ne dim_vec then begin
        message, 'vector_in and b_in are not of same type or dimension', /cont
        help, vector_in, b_in
        help, /traceback
        retall
     endif
     in_vec = vector_in
     b_vec = b_in
     NN = dim_vec
  endif else begin
     ; make sure b_in is of type VTS_T
     dim_b = vts_check(b_in)
     if dim_b eq 0 then begin
        message, 'b_in is not of expected type (VTS_T)', /cont
        help, vector_in, b_in
        help, /traceback
        retall
     endif
     ; see if we need to interpolate the input vector time series
     ; ----------------------------------------------------------
     if keyword_set(ctarr) then begin
        in_vec = vts2arr(vts_interpol(vector_in, ctarr))
     endif else begin
        in_vec = vector_in.val
        ctarr  = vector_in.ct
     endelse

     NN = n_elements(ctarr)
     
     ; interpolate magnetic field vector time series
     ; ---------------------------------------------
     if n_elements(b_in) eq 1 then begin
        bts = replicate( {VTS_T}, NN ) ; so that it can be interpolated
        bts.val = b_in.val
        bts.ct  = ctarr
     endif else $
        bts = vts_interpol(b_in, ctarr)

     b_vec = bts.val
  endelse

  ; project vector into the gyro-plane
  ; ----------------------------------
  prj_vec = vGyroProjection_(in_vec, b_vec)

  ; calculate gyro-plane unit vectors (x = towards sun)
  ; ---------------------------------------------------
  unit_vecs = vGyroUnitVector_(b_vec)
  x_gyro = unit_vecs[0:2,*]
  y_gyro = unit_vecs[3:5,*]

  ; calculate components of projected vector along gyro unit vectors
  ; ----------------------------------------------------------------
  outvec_x = vec_dotp(prj_vec, x_gyro)
  outvec_y = vec_dotp(prj_vec, y_gyro)

  if dim_vts gt 0 then begin
     vts_gyro = replicate( {V2TS_T}, NN )
     vts_gyro.ct = ctarr
     vts_gyro.val[0] = outvec_x
     vts_gyro.val[1] = outvec_y

     return, vts_gyro

  endif else begin
      vec_gyro = fltarr(2,NN)
      vec_gyro[0,*] = outvec_x
      vec_gyro[1,*] = outvec_y

      return, vec_gyro
  endelse

END

