; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; Cluster EDI
; $Id: readlib.pro,v 1.37 2006/02/28 21:06:49 hav Exp $
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

; ============================================================================
; READLIB --- functions for reading Cluster data
;
; MPE-Garching, Cluster-II, EDI
; ============================================================================
; GetWWData     read EDI WW data structures from MSF/RDM/CDDS files
; GetShdrData   read EDI second header data from MSF/RDM/CDDS files
; GetSrpData    read sun ref pulse data     from MSF/RDM/CDDS files
; GetHkData     read EDI housekeeping data  from MSF/RDM/CDDS files
;
; GetMagData    read magnetic field data using Edita's scripts
; GetSpinCenterTimes    get spin center times from spin resolution B data
;
;
; ****************************************************************************

@clutime.pro
@wwutil.pro
@magutil.pro
@getcluhdr.pro
@myfread.pro
@getfgmrange.pro
@uncalspinaxis.pro

@getmagdata.pro
@edianomalylist.pro

; ============================================================================
  FUNCTION GetData, datatype, $ ; ww, shdr, hk, srp
                    scid, $
                    ctr, $
                    cdds = cdds, $
                    debug = debug, $
                    quiet = quiet, $
                    file = file, $
                    anomaly_ignore=anomaly_ignore, $
                    anomaly_setquality=anomaly_setquality
; ============================================================================
; Low level function to read data from CLUSTER telemetry data files,
; used by the high level functions GetWWData(), GetAGData(), GetShdrData(),
; GetSrpData() and GetHkData().
;
; Parameters
;    datatype  int         as defined by DATATYPE_... in constants.inc
;    scid      int
;    ctr       dblarr(2)   begin and end time in t70 format
;
; Keywords
;    debug   0/1        print debugging messages
;    quiet   0/1        print only error messages
;    file    string     specify the file name directly. This way other
;                       than MSF files can be accessed.
;    anomaly_ignore       0/1    do not remove data that were gathered during
;                                instrument anomalies
;    anomaly_setquality   0/1    flag quality of anomaly data as -1 instead
;                                of removing them
;
; ****************************************************************************

common getwwdatablock, common_tm_mode
@paths.inc
@datast.inc
@constants.inc

  ag_cor_info = [ $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-10-11t18:18')   ,str2ct('2004-10-15t05:50')],    bitval:0 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-10-24t12:28:13'),str2ct('2004-10-26t21:29:13')], bitval:1 }, $
 ;    { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-10-26t21:29:13'),str2ct('2004-10-29t06:32:13')], bitval:0 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-11-07t18:54:12'),str2ct('2004-11-10t03:54:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-11-10t03:54:11'),str2ct('2004-11-12t12:58:11')], bitval:0 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-11-14t22:05:11'),str2ct('2004-11-17t07:03:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-11-19t16:14:11'),str2ct('2004-11-22t01:15:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-11-22t01:15:11'),str2ct('2004-11-24t10:19:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-11-26t19:27:11'),str2ct('2004-11-29t04:25:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-11-29t04:25:11'),str2ct('2004-12-01t13:28:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-12-03t22:38:11'),str2ct('2004-12-06t07:38:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-12-06t07:38:11'),str2ct('2004-12-08t16:42:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-12-11t01:50:11'),str2ct('2004-12-13t10:47:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-12-13t10:47:11'),str2ct('2004-12-16t04:26:40')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-12-18t05:03:11'),str2ct('2004-12-20t14:04:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-12-20t14:04:11'),str2ct('2004-12-22t23:07:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-12-25t08:15:11'),str2ct('2004-12-27t17:13:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2004-12-27t17:13:11'),str2ct('2004-12-30t02:18:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-01t11:28:11'),str2ct('2005-01-03t20:28:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-03t20:28:11'),str2ct('2005-01-06t05:32:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-08t14:40:11'),str2ct('2005-01-10t23:38:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-10t23:38:11'),str2ct('2005-01-13t08:44:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-15t17:57:11'),str2ct('2005-01-18t02:57:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-18t02:57:11'),str2ct('2005-01-20t12:00:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-22t21:09:11'),str2ct('2005-01-25t06:09:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-25t06:09:11'),str2ct('2005-01-27t15:14:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-01-30t00:25:11'),str2ct('2005-02-01t09:26:01')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-02-01t09:26:11'),str2ct('2005-02-03t18:30:11')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-02-06t03:39:11'),str2ct('2005-02-08t12:39:11')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-02-08t12:39:11'),str2ct('2005-02-10t21:47:12')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-02-13t06:59:15'),str2ct('2005-02-15t15:58:15')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'2',  ctr:[str2ct('2005-02-15t15:58:15'),str2ct('2005-02-16t09:35:00')], bitval:1 }, $

;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2004-10-27t02:00'),   str2ct('2004-10-29t06:00')], bitval:0 }, $
 ;    { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2004-10-27t02:00'),   str2ct('2004-10-29t06:00')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2004-11-10t08:00'),   str2ct('2004-11-12t13:00')], bitval:0 }, $
     { AG_COR_INFO_T, sc_mask:'1', ctr:[str2ct('2004-11-15t02:00'),   str2ct('2004-11-24t14:48:19')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'3', ctr:[str2ct('2004-11-15t02:00'),   str2ct('2004-11-24t15:04:53')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2004-11-19t20:00'),   str2ct('2004-11-21t21:00')], bitval:0 }, $
;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2004-11-29t08:00'),   str2ct('2004-12-01t09:15')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'1', ctr:[str2ct('2004-12-04t03:00'),   str2ct('2004-12-08t21:11:13')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'3', ctr:[str2ct('2004-12-04t03:00'),   str2ct('2004-12-08t21:27:38')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2004-12-13t15:00'),   str2ct('2004-12-15t40:00')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'1', ctr:[str2ct('2004-12-18t09:00'),   str2ct('2004-12-23t03:35:52')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'3', ctr:[str2ct('2004-12-18t09:00'),   str2ct('2004-12-23t03:52:07')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2004-12-27t21:00'),   str2ct('2004-12-29t22:05')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'1', ctr:[str2ct('2005-01-01t15:00'),   str2ct('2005-01-06t10:01:15')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'3', ctr:[str2ct('2005-01-01t15:00'),   str2ct('2005-01-06t10:17:22')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'1', ctr:[str2ct('2005-01-09t03:00'),   str2ct('2005-01-11t04:07:19')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'3', ctr:[str2ct('2005-01-09t03:00'),   str2ct('2005-01-11t04:23:23')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2005-01-11t04:00'),   str2ct('2005-01-13t05:00')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2005-01-15t22:00'),   str2ct('2005-01-18t14:29:00')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2005-01-18t14:00'),   str2ct('2005-01-20t02:02')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'1', ctr:[str2ct('2005-01-23t07:00'),   str2ct('2005-01-25t10:38:21')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'3', ctr:[str2ct('2005-01-23t07:00'),   str2ct('2005-01-25t10:55:15')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2005-01-25t10:00'),   str2ct('2005-01-27t11:02')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'1', ctr:[str2ct('2005-01-30t04:00'),   str2ct('2005-02-01t19:10:00')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'3', ctr:[str2ct('2005-01-30t05:07:01'),   str2ct('2005-02-01t19:10:00')], bitval:1 }, $

;     { AG_COR_INFO_T, sc_mask:'1',  ctr:[str2ct('2005-02-01t19:00'),   str2ct('2005-02-03t09:00')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'3',  ctr:[str2ct('2005-02-01t19:00'),   str2ct('2005-02-02t17:00')], bitval:1 }, $

     { AG_COR_INFO_T, sc_mask:'1', ctr:[str2ct('2005-02-06t12:00'),   str2ct('2005-02-08t17:08:24')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'3', ctr:[str2ct('2005-02-06t12:00'),   str2ct('2005-02-08t17:25:30')], bitval:1 }, $

;     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2005-02-08t17:00'),   str2ct('2005-02-10t18:00')], bitval:1 }, $
     { AG_COR_INFO_T, sc_mask:'13', ctr:[str2ct('2005-02-13t11:00'),   str2ct('2005-02-16t00:39:00')], bitval:1 } $
;     { AG_COR_INFO_T, sc_mask:'1' , ctr:[str2ct('2005-02-16t00:00'),   str2ct('2005-02-16t14:20')], bitval:1 }, $
;     { AG_COR_INFO_T, sc_mask:'3',  ctr:[str2ct('2005-02-16t00:00'),   str2ct('2005-02-16t14:40')], bitval:1 }
       ]


  if not keyword_set(debug)  then debug = 0 else debug = 1
  if not keyword_set(quiet)  then quiet = 0 else quiet = 1

  if n_params() lt 3 then begin
      message, 'Argument List:', /cont
      message, '(datatype,scid,ctr,file=file,debug=debug,quiet=quiet)', /cont
      message, 'insufficient argument list!', /cont
      retall
  endif

  ctr_ext = ctr + [-15, 15]

  if datatype ne DATATYPE_WW then begin
     if keyword_set(anomaly_setquality) then $
        message, 'Warning: keyword <anomaly_setquality> is applicable only to WW data!. Ignoring...', /cont
     if datatype ne DATATYPE_AG then $
        if keyword_set(anomaly_ignore) then $
           message, 'Warning: keyword <anomaly_ignore> is applicable only to WW or AG data! Ignoring...', /cont
  endif


  case datatype of
      DATATYPE_WW : begin
          prototype = replicate( {WW_T}, 1 )
          name = 'EDI WW data'
          exe_str = WWCONV_DIR + 'wwconv -b' + $
            ' -s ' + strmid(CluTimeVal2Str(ctr_ext[0]),0,19) + $
            ' -e ' + strmid(CluTimeVal2Str(ctr_ext[1]),0,19)
          cdds_ext = '.?sd'
          if n_elements(common_tm_mode) ne 0 then begin
             if common_tm_mode ne '' then begin
                exe_str = exe_str + ' -m ' + common_tm_mode
                case common_tm_mode of
                   'nm' : cdds_ext = '.nsd'
                   'bm' : cdds_ext = '.bsd'
                   else : begin
                      msg = 'unknown value for common_tm_mode : ' + common_tm_mode
                      if debug then message, msg, /cont
                      return, { hdr:GetCluHdr(1, msg, sc_id = scid) }
                   end
                endcase
             endif
          endif
          offset = 0
      end
      DATATYPE_AG : begin
          prototype = replicate( {AG_T}, 1 )
          name = 'EDI AG data'
          exe_str = WWCONV_DIR + 'agconv -b' + $
            ' -s ' + strmid(CluTimeVal2Str(ctr_ext[0]),0,19) + $
            ' -e ' + strmid(CluTimeVal2Str(ctr_ext[1]),0,19)
          cdds_ext = '.?sd'
          offset = 0
      end
      DATATYPE_SHDR : begin
          prototype = { SHDR_T, ct:0.0d, data:intarr(16) }
          name = 'EDI second header data'
          exe_str = SHDR_DIR + 'shdr -b'
          cdds_ext = '.?sd'
          offset = 0
      end
      DATATYPE_HK : begin
          ; 45 words + 3 dummies to make it a
          ; multiple of 8 (as double is)
          prototype = { HK_T, ct:0.0d, data:intarr(48) }
          name = 'EDI HK data'
          exe_str = HK_DIR + 'hk -b'
          cdds_ext = '.hkd'
          offset = 0
      end
      DATATYPE_SRP : begin
          prototype = { SRP_T,  ct:0.0d, tspin:0.0d }
          name = 'sun reference pulse data'
          exe_str = TSPIN_DIR + 'tspin'
          offset = 12
          cdds_ext='.shk'
      end
      else : begin
          message, 'bad value for datatype: ' + strtrim(datatype,2), /cont
          retall
      end
  endcase

  if keyword_set(cdds) then begin
     latest = 0
  endif else begin
     cdds_ext = 0
     latest = 1
  endelse

  cd, '~', current=olddir

  ; set time range
  ; --------------
  beg_ymd = strmid(CluTimeVal2Str(ctr_ext[0]),0,10)
  end_ymd = strmid(CluTimeVal2Str(ctr_ext[1]),0,10)


  ; Get data in a loop (one day per file)
  ; -------------------------------------
  i = 0
  nodata = 1
  all_data = replicate(prototype, 1)
  n_passes = 1
  repeat begin

      ; Get filename for current day
      ; ----------------------------
      if keyword_set(file) then begin
          filename = file
      endif else begin
          if n_passes eq 1 then begin
             cur_ymd = strmid(CluTimeVal2Str(ctr_ext[0]+i*86400.d),0,10)
             fn = GetFileName(scid, ymd=cur_ymd, cdds=cdds_ext, latest=latest)
             if fn.status eq 1 then begin
                message, fn.msg, /cont
                i = i + 1
                continue
             endif

             filename = fn.filename ; only one filename if MSF, but can be more for CDDS (nsd+bsd)

             if n_elements(filename) gt 1 then begin
                if keyword_set(cdds) then begin
                   n_passes = 2
                   store_filename = filename[1]
                   filename = filename[0]
                endif else begin
                   message, 'internal error: more than one file name returned for MSF file', /cont
                   retall
                endelse
             endif

          endif else begin
             filename = store_filename
             n_passes = 1
          endelse

          ct_beg_of_day = CluTimeStr2Val(cur_ymd)
          ct_end_of_day = ct_beg_of_day + 86400.d
      endelse


      ; execute program to create binary data file
      ; ------------------------------------------
      if not quiet then message, 'reading ' + name + ' from ' + filename, /cont
      cmd_str = exe_str + ' ' + filename
      if debug then message, 'executing ' + cmd_str, /cont
      spawn, cmd_str, result, err_result, exit_status=exit_status
      if exit_status ne 0 then begin
;          print, 'exit_status = ' + strtrim(exit_status,2)
;          print, 'result      = ' + result
;          print, 'err_result  = ' + err_result
          help, result, err_result
          message, 'exit_status = <'+strtrim(exit_status,2)+'>', /cont
          message, 'result = <'+result+'>', /cont
          message, 'err_result = <' + strjoin(err_result,'; ')+'>', /cont
      endif else begin
         bin_fn = result

         if debug then message, 'binary file is <' + bin_fn + '>', /cont

         ; read data from binary data file
         ; -------------------------------
         dstruc = ReadData(bin_fn, prototype, offset, debug = debug)
         spawn, 'rm -f ' + bin_fn
         if dstruc.status ne 0 then begin
             if debug eq 1 or quiet ne 1 then message, name + ' : ' + dstruc.msg, /cont
         endif else begin
             nodata = 0

             ; cut off data which are not from current day (MSF files...)
             ; ----------------------------------------------------------
             if keyword_set(file) then $
               x = lindgen(n_elements(dstruc.data)) $
             else $
               x = where(dstruc.data.ct ge ct_beg_of_day and $
                         dstruc.data.ct le ct_end_of_day)

             if x[0] ne -1 then all_data = [ all_data, dstruc.data[x] ] $
             else if debug then message, 'no data for current day', /cont
         endelse
      endelse

      if keyword_set(file) then break
      if n_passes eq 1 then i = i + 1

  endrep until cur_ymd eq end_ymd and n_passes eq 1

  if nodata eq 1 then begin
      cd, olddir
      return, { hdr:GetCluHdr(1, name + ': no data', sc_id = scid) }
  endif

  cd, olddir

  ; cut off leading dummy element of the data array
  ; -----------------------------------------------
  NN = n_elements(all_data)
  if NN eq 1 then begin
     return, { hdr:GetCluHdr(1, name + ': no data', sc_id = scid) }
  endif else $
    all_data = all_data[1:NN-1]

  sortindex = sort(all_data.ct)
  all_data = all_data[sortindex]

  ; cut off data outside specified interval
  ; ---------------------------------------
;  if not keyword_set(file) then begin
     x = where(all_data.ct ge ctr[0] and all_data.ct le ctr[1])
     if x[0] eq -1 then begin
        return, { hdr:GetCluHdr(1, name + ': no data', sc_id = scid) }
     endif

     all_data = all_data[x]
;  endif

  ; For Windshield Wiper and ambient GEOS data we need to see
  ; if we need to remove data because of instrument anomalies
  ; ---------------------------------------------------------
  if datatype ne DATATYPE_AG and datatype ne DATATYPE_WW then goto, behind_anomaly_handling

  if debug then message, strtrim(n_elements(all_data),2) + ' data points initially', /cont
  anomaly_list = GetEdiAnomalyList()
  if debug then print, strtrim(n_elements(anomaly_list),2) + ' anomaly interval(s) initially.'

  ; select applicable intervals (correct sc, time interval overlap)
  data_min_ct = min(all_data.ct, max=data_max_ct)
  if datatype eq DATATYPE_WW then $
     x = where(anomaly_list.sc eq scid and $
               anomaly_list.ctr[0] lt data_max_ct and $
               anomaly_list.ctr[1] gt data_min_ct and $
               (anomaly_list.ww_gun ne 0 or anomaly_list.ww_det ne 0), anomaly_cnt) $
  else $ ; DATATYPE_AG
     x = where(anomaly_list.sc eq scid and $
               anomaly_list.ctr[0] lt data_max_ct and $
               anomaly_list.ctr[1] gt data_min_ct and $
               anomaly_list.ag_det ne 0, anomaly_cnt)
  if anomaly_cnt eq 0 then goto, behind_anomaly_handling
  anomaly_list = anomaly_list[x]
  if debug then print, strtrim(n_elements(anomaly_list),2) + ' applicable anomaly interval(s).'

  ; now we know there are intervals to handle
  ; check if the ignore keyword has been specified
  ; the reason we are not doing this up front is that we want to issue an ignore message only
  ; if there would have been data to treat
  if keyword_set(anomaly_ignore) then begin
     message, 'Ignoring instrument anomalies per keyword request. Anomalous data will NOT be removed!', /cont
     goto, behind_anomaly_handling
  endif

  N_ANOMALY_IVS = n_elements(anomaly_list)
  flagarr = intarr(n_elements(all_data)) + 1 ; flag the anomaly data with a negative value

  ; Loop through all remaining anomaly intervals and flag data
  if datatype eq DATATYPE_WW then begin

     for i=0L,N_ANOMALY_IVS-1 do begin
        current = anomaly_list[i]

        ; find indices of data in current anomaly interval
        x_anomaly = where(all_data.ct ge current.ctr[0] and all_data.ct le current.ctr[1], anomaly_cnt)
        if anomaly_cnt eq 0 then continue

        if debug then begin
           message, 'Detected instrument anomaly condition in requested WW data interval', /cont
           message, 'Anomaly details: ' + current.msg, /cont
        endif

        if current.ww_gun eq 3 or current.ww_det eq 3 or current.ww_gun eq current.ww_det then gd = 33 $ ; remove both pairs
        else if current.ww_gun eq 2 or current.ww_det eq 1 then gd = 21 $
        else if current.ww_gun eq 1 or current.ww_det eq 2 then gd = 12

        if gd eq 33 then flagarr[x_anomaly] = -1 $
        else begin
              x = where(all_data[x_anomaly].gd eq gd, cnt_gd)
              if cnt_gd ne 0 then flagarr[x_anomaly[x]] = -1
        endelse
     endfor

     ; identify flagged WW data
     x_in = where(flagarr gt 0, keep_cnt)
     x_out = where(flagarr lt 0, remove_cnt)

     if debug then message, 'Number of identified WW anomaly data: ' + strtrim(remove_cnt,2), /cont

     if keep_cnt eq 0 then begin
        msg = 'no data left after anomaly data removal.'
        if debug then message, msg, /cont
        return, { hdr:GetCluHdr(1, name + ': ' + msg, sc_id = scid) }
     endif

     if keyword_set(anomaly_setquality) then begin
        message, 'setting quality of anomalous WW data to -1', /cont
        all_data[x_out].q = -1
     endif else begin
        message, 'removing anomalous WW data', /cont
        all_data = all_data[x_in]
     endelse

  endif else begin ; DATATYPE_AG

     for i=0L,N_ANOMALY_IVS-1 do begin
        current = anomaly_list[i]

        ; find indices of data in current anomaly interval
        x_anomaly = where(all_data.ct ge current.ctr[0] and all_data.ct le current.ctr[1], anomaly_cnt)
        if anomaly_cnt eq 0 then continue

        if debug then begin
           message, 'Detected instrument anomaly condition in requested AG data interval', /cont
           message, 'Anomaly details: ' + current.msg, /cont
        endif

        if current.ag_det eq 1 or current.ag_det eq 3 then all_data[x_anomaly].cnt1 = -1
        if current.ag_det eq 2 or current.ag_det eq 3 then all_data[x_anomaly].cnt2 = -1
        message, 'Flagging counts of anomalous ambient GEOS data as -1...', /cont
     endfor
  endelse

behind_anomaly_handling:

  ; For ambient GEOS data we need to correct a TM bit
  ; -------------------------------------------------
  if datatype ne DATATYPE_AG then goto, behind_ag_cor

  ; first see if we are in the range of interest...
  ag_cor_ctr = [ Str2Ct('2004-10-11'), Str2Ct('2005-02-17') ]
  xdummy = where(all_data.ct gt ag_cor_ctr[0] and all_data.ct lt ag_cor_ctr[1], ag_cor_cnt)
  if ag_cor_cnt eq 0 then goto, behind_ag_cor

  ; pick the correction time ranges for the correct spacecraft
  x_sc = where( strpos(ag_cor_info.sc_mask, strtrim(scid,2)) ge 0)
  if x_sc[0] eq -1 then goto, behind_ag_cor

  if debug then message, 'applying dirflag bit correction...', /cont

  min_ct = min(all_data.ct, max=max_ct)

  for i=0L,n_elements(x_sc)-1 do begin
     this = ag_cor_info[x_sc[i]]
     msg = 'interval : ' + ct2str(this.ctr[0]) + ' - ' + ct2str(this.ctr[1]) + ' : '
     if this.ctr[1] lt min_ct or this.ctr[0] gt max_ct then begin
        if debug then message, msg + 'not applicable', /cont
        continue
     endif
     x = where(all_data.ct ge this.ctr[0] and $
               all_data.ct le this.ctr[1] and $
               ishft(all_data.dirflg,-1) mod 2 ne this.bitval, cnt)
     if debug then message,  msg + strtrim(cnt,2) + ' hits', /cont
     if cnt gt 0 then begin ; toggle the bit
        if this.bitval eq 1 then begin
           if debug then message, ct2str(this.ctr[0]) + ' - ' + ct2str(this.ctr[1]) + $
                                ' : setting bit in ' + strtrim(cnt,2) + ' records.', /cont
           all_data[x].dirflg = all_data[x].dirflg + 2  ; set bit
        endif else begin
           if debug then message, ct2str(this.ctr[0]) + ' - ' + ct2str(this.ctr[1]) + $
                                ' : clearing bit in ' + strtrim(cnt,2) + ' records.', /cont
           all_data[x].dirflg = all_data[x].dirflg - 2 ; clear bit
        endelse
     endif
  endfor

behind_ag_cor:


  return, { hdr:GetCluHdr(0, 'ok', sc_id=scid), $
            data:all_data }

END

; ============================================================================
  FUNCTION GetWWData, scid, year, month, day, hour, $
                      minute=minute, second=second, $
                      dhour=dhour, dmin=dmin, dsec=dsec, $
                      ctr=ctr, $
                      debug=debug, $
                      quiet=quiet, $
                      file=file, $
                      cdds=cdds, $
                      n_sel=n_sel, $
                      ctype=ctype, $
                      fgmrng=fgmrng, $
                      tmmode=tmmode, $
                      anomaly_ignore=anomaly_ignore, $
                      anomaly_setquality=anomaly_setquality
; ============================================================================
; This function uses the new Windshield Wiper data structure and allows
; data from the one gun/detector pair telemetry mode to be stored
;
; Two ways to specify intervals:
;    ds = GetWWData(scid, ctr=ctr)
;    ds = GetWWData(scid, year, month, day, hour, [ minute=minute,
;                         second=second, dhour=dhour, dmin=dmin,
;                         dsec=dsec ] )
; The keywords n_sel, ctype, fgmrng can be used to select only WW data with
; certain values of correlator n (n_sel), code type (long/short: ctype) and/or
; within a certain FGM range (fgmrng). These keywords may be scalars or arrays.
; Example:
;
;   ds = GetWWData(3, ctr=ctr, n_sel=[1,2], ctype='s', fgmrng=[3,4]
;
;   This selects all WW data within interval ctr, with correlator n values
;   of 1 or 2, long code, and FGM ranges of 3 or 4
;
; ****************************************************************************

common getwwdatablock, common_tm_mode

@constants.inc
@paths.inc


  if not keyword_set(debug) then debug = 0 else debug = 1
  if not keyword_set(quiet) then quiet = 0 else quiet = 1

  if keyword_set(tmmode) then begin ; may be 'nm' or 'bm'
     common_tm_mode = strlowcase(tmmode)
     if common_tm_mode ne 'nm' and common_tm_mode ne 'bm' then begin
        msg = 'Bad value for keyword tmmode: <' + tmmode + '>'
        if not quiet or debug then begin
           message, msg, /cont
        endif
        return, { hdr:GetCluHdr(1, msg, sc_id=scid) }
     endif
  endif else begin
     common_tm_mode = ''
  endelse

  if keyword_set(ctr) then begin
     if n_params() gt 1 then begin
        message, 'keyword ctr: incompatible with parameters year,month,day,hour', /cont
        retall
     endif
     if keyword_set(minute) or $
        keyword_set(second) or $
        keyword_set(dhour) or $
        keyword_set(dmin) or $
        keyword_set(dsec) then begin
           message, 'keyword ctr: incompatible with keywords minute,second,dhour,dmin,dsec', /cont
           retall
        endif
  endif else begin
     if not keyword_set(minute) then minute=0
     if not keyword_set(second) then second=0
     if not keyword_set(dhour) then dhour = 0
     if not keyword_set(dmin) then dmin = 0
     if not keyword_set(dsec) then dsec = 0

     if dhour eq 0 and dmin eq 0 and dsec eq 0 then dhour = 1

     ; calculate end time
     ; ------------------
     delta_sec = 0L
     delta_sec = delta_sec + dhour * 3600L
     delta_sec = delta_sec + dmin * 60L
     delta_sec = delta_sec + dsec


     ctr = dblarr(2)
     fmt = '(I4.4,"-",I2.2,"-",I2.2,"T",I2.2,":",I2.2,":",I2.2)'
     ctr[0] = CluTimeStr2Val(string(format=fmt, year, month, day, $
                                      hour, minute, second))
     ctr[1] = ctr[0] + delta_sec
  endelse

  ds = GetData(DATATYPE_WW, scid, ctr, debug=debug, quiet=quiet, cdds=cdds, file=file, $
               anomaly_ignore=anomaly_ignore, anomaly_setquality=anomaly_setquality)
  if ds.hdr.status ne 0 then begin
     if debug then message, ds.hdr.msg, /cont
     return, ds
  endif


  ; select data with certain correlator n values
  ; --------------------------------------------
  if keyword_set(n_sel) then begin
     if debug then message, 'selecting data according to specified n', /cont
     x_sel = [ -1L ]
     for i = 0,n_elements(n_sel)-1 do begin
        x = where(ds.data.nc eq n_sel[i], cnt)
        if cnt gt 0 then x_sel = [ x_sel, x ]
     endfor
     if n_elements(x_sel) eq 1 then $
        return, { hdr:GetCluHdr(1, 'no EDI WW data left after n selection!', $
                                sc_id=ds.hdr.sc_id) }
     x_sel = x_sel[1:n_elements(x_sel)-1]
     ds = { hdr:ds.hdr, data:ds.data[x_sel] }
  endif

  ; select data with short or long code
  ; -----------------------------------
  if keyword_set(ctype) then begin
     if debug then message, 'selecting data according to specified code type', /cont
     x_sel = [ -1L ]
     if ctype eq 's' then cmpval = 0 else if ctype eq 'l' then cmpval = 1 $
     else begin
        message, 'bad value for keyword ctype (s,l allowed): ' + strtrim(ctype,2), /cont
        retall
     endelse
     x = where(ds.data.ctype eq cmpval, cnt)
     if cnt gt 0 then x_sel = [ x_sel, x ]
     if debug then message, 'found ' + strtrim(cnt,2) + ' beam(s) with ctype = ' + $
                          strtrim(cmpval,2), /cont
     if n_elements(x_sel) eq 1 then $
        return, { hdr:GetCluHdr(1, 'no EDI WW data left after ctype selection!', $
                                sc_id=ds.hdr.sc_id) }
     x_sel = x_sel[1:n_elements(x_sel)-1]
     ds = { hdr:ds.hdr, data:ds.data[x_sel] }
  endif

  ; select data within certain FGM ranges
  ; -------------------------------------
  if keyword_set(fgmrng) then begin
     if debug then message, 'selecting data within specified FGM range(s)', /cont
     x_sel = [ -1L ]
     fgminfo = GetFgmRange(ds.hdr.sc_id, ctr)
     if fgminfo.status eq 1 then begin
        return, { hdr:GetCluHdr(1, fgminfo.msg, sc_id=ds.hdr.sc_id) }
     endif
     for i=0,n_elements(fgmrng)-1 do begin
        x = where(fgminfo.data.range eq fgmrng[i], cnt)
        for j=0,cnt-1 do begin
           xin = where(ds.data.ct ge fgminfo.data[x[j]].ct_beg and $
                       ds.data.ct lt fgminfo.data[x[j]].ct_end, cnt_in)
           if cnt_in gt 0 then x_sel = [ x_sel, xin ]
        endfor
     endfor
     if n_elements(x_sel) eq 1 then $
        return, { hdr:GetCluHdr(1, 'no EDI WW data within specified FGM range(s)!', $
                                sc_id=ds.hdr.sc_id) }
     x_sel = x_sel[1:n_elements(x_sel)-1]
     ds = { hdr:ds.hdr, data:ds.data[x_sel] }
  endif

  return, ds

END

; ============================================================================
  FUNCTION GetAGData, scid, year, month, day, hour, $
                      minute=minute, second=second, $
                      dhour=dhour, dmin=dmin, dsec=dsec, $
                      ctr=ctr, $
                      debug=debug, $
                      quiet=quiet, $
                      file=file, $
                      cdds=cdds, $
                      anomaly_ignore=anomaly_ignore
; ============================================================================
; Read ambient GEOS data
;
; Two ways to specify intervals:
;    ds = GetAGData(scid, ctr=ctr)
;    ds = GetAGData(scid, year, month, day, hour, [ minute=minute,
;                         second=second, dhour=dhour, dmin=dmin,
;                         dsec=dsec ] )
; ****************************************************************************

@constants.inc
@paths.inc


  if not keyword_set(debug) then debug = 0 else debug = 1
  if not keyword_set(quiet) then quiet = 0 else quiet = 1


  if keyword_set(ctr) then begin
     if n_params() gt 1 then begin
        message, 'keyword ctr: incompatible with parameters year,month,day,hour', /cont
        retall
     endif
     if keyword_set(minute) or $
        keyword_set(second) or $
        keyword_set(dhour) or $
        keyword_set(dmin) or $
        keyword_set(dsec) then begin
           message, 'keyword ctr: incompatible with keywords minute,second,dhour,dmin,dsec', /cont
           retall
        endif
  endif else begin
     if not keyword_set(minute) then minute=0
     if not keyword_set(second) then second=0
     if not keyword_set(dhour) then dhour = 0
     if not keyword_set(dmin) then dmin = 0
     if not keyword_set(dsec) then dsec = 0

     if dhour eq 0 and dmin eq 0 and dsec eq 0 then dhour = 1

     ; calculate end time
     ; ------------------
     delta_sec = 0L
     delta_sec = delta_sec + dhour * 3600L
     delta_sec = delta_sec + dmin * 60L
     delta_sec = delta_sec + dsec


     ctr = dblarr(2)
     fmt = '(I4.4,"-",I2.2,"-",I2.2,"T",I2.2,":",I2.2,":",I2.2)'
     ctr[0] = CluTimeStr2Val(string(format=fmt, year, month, day, $
                                      hour, minute, second))
     ctr[1] = ctr[0] + delta_sec
  endelse

  ag = GetData(DATATYPE_AG, scid, ctr, debug=debug, quiet=quiet, cdds=cdds, file=file, $
               anomaly_ignore=anomaly_ignore)

  return, ag

END

; ============================================================================
  FUNCTION GetShdrData,  scid,  year,  month,  day, ctr=ctr, $
                         debug = debug, quiet = quiet, $
                         file = file, cdds=cdds
; ============================================================================
; read EDI science TM second header data
; ****************************************************************************

@constants.inc
@paths.inc

  if not keyword_set(debug)  then debug = 0 else debug = 1
  if not keyword_set(quiet)  then quiet = 0 else quiet = 1
  if not keyword_set(file)   then file = 0

  if not keyword_set(ctr) then begin
      fmt = '(I4.4,"-",I2.2,"-",I2.2)'
      ctr = CluTimeStr2Val(string(format=fmt, year, month, day))
      ctr = ctr + [ 0, 86400.]
  endif

  shdr = GetData(DATATYPE_SHDR, scid, ctr, debug=debug, quiet=quiet, cdds=cdds, file=file)

  return, shdr

END

; ============================================================================
  FUNCTION GetSrpData, scid,  year,  month,  day, ctr=ctr, $
                       debug = debug, $
                       quiet = quiet, $
                       cdds = cdds, $
                       file = file, $
                       diagnose=diagnose, nogap=nogap
; ============================================================================
; read sun reference pulse data (SC HK)
; ****************************************************************************

@constants.inc

  if not keyword_set(debug)  then debug = 0 else debug = 1
  if not keyword_set(quiet)  then quiet = 0 else quiet = 1
  if not keyword_set(diagnose) then diagnose=0 else diagnose=1
  if not keyword_set(nogap) then nogap=0 else nogap=1

  if not keyword_set(ctr) then begin
      fmt = '(I4.4,"-",I2.2,"-",I2.2)'
      ctr = CluTimeStr2Val(string(format=fmt,year,month,day))
      ctr = ctr + [ 0, 86400.]
  endif

  sd = GetData(DATATYPE_SRP, scid,  ctr, $
               debug = debug, quiet = quiet, file=file, cdds=cdds)


  if sd.hdr.status eq 1 then return, sd

  if nogap eq 1 then return, sd

  out = FillGaps(sd, xarr=xarr, debug=debug, quiet=quiet)

  if xarr[0] ne -1 then $
     out.data[xarr].tspin = total(sd.data.tspin)/n_elements(sd.data)

  return, out

END

; ============================================================================
  FUNCTION GetHkData, scid,  year,  month,  day, $
                      ctr=ctr, $
                      debug = debug, quiet = quiet, $
                      cdds=cdds, file=file
; ============================================================================
; read EDI HK TM data
; ****************************************************************************

@constants.inc

  if not keyword_set(debug)  then debug = 0 else debug = 1
  if not keyword_set(quiet)  then quiet = 0 else quiet = 1
  if not keyword_set(ctr)   then begin
     file  = 0
     fmt = '(I4.4,"-",I2.2,"-",I2.2)'
     ctr = CluTimeStr2Val(string(format=fmt, year, month, day))
     ctr = ctr + [ 0, 86400.]
  endif

  return, GetData(DATATYPE_HK, scid,  ctr, debug = debug, quiet = quiet, $
                               file=file, cdds=cdds)

END

; ============================================================================
  FUNCTION GetTspin, sc_id, year, month, day, sd=sd, ctr=ctr, $
                     quiet=quiet, debug=debug
; ============================================================================

  if not keyword_set(debug) then debug = 0 else debug = 1
  if not keyword_set(quiet) then quiet = 0 else quiet = 1

  if not keyword_set(ctr) then begin
      ctr = clutimestr2val(string(format='(I4.4,"-",I2.2,"-",I2.2)', $
                                  year, month, day)) + $
        [0,86400.]
  endif

  if not keyword_set(sd) then $
    sd=getsrpdata(sc_id, year, month, day, ctr=ctr, debug=debug, quiet=quiet)

  if sd.hdr.status ne 0 then message, sd.hdr.msg

  x = where(sd.data.ct ge ctr[0] and sd.data.ct le ctr[1])

  if x[0] eq -1 then message, 'no data within specified interval'

  return, total(sd.data[x].tspin) / n_elements(x)


END




; ============================================================================
  FUNCTION GetSpinCenterTimes, scid, year, month, day, ctr=ctr, quiet=quiet
; ============================================================================
; reads spin averaged magnetic field data and extracts spin center
; times
; ****************************************************************************

  if not keyword_set(quiet) then quiet = 0 else quiet=1

  if n_params() eq 4 then begin
     srb = GetMagData(scid, year, month, day, /sr, quiet=quiet)
  endif else if n_params() eq 1 and keyword_set(ctr) then begin
     srb = GetMagData(scid, ctr=ctr, /sr, quiet=quiet)
  endif else begin
     message, 'GetSpinCenterTimes(): bad argument list'
     retall
  endelse

  if srb.hdr.status ne 0 then begin
      message, srb.hdr.msg, /cont
      retall
  endif


  sct = { hdr:GetCluHdr(0, '', sc_id=scid), $
          data:replicate({ct:0.0d} , n_elements(srb.data)) $
        }

  sct.data.ct = srb.data.ct

  out = FillGaps(sct, quiet=quiet)

  return, out

END




