; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; $Id: getmagdata.pro,v 1.2 2006/02/28 03:16:29 hav Exp $
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

; ============================================================================
  FUNCTION GetMagData, scid, $
                       year, month, day, hour, $
                       ctr=ctr, $
                       ct_arr=ct_arr, $
                       csys = csys, $
                       sr=sr, $
                       quiet=quiet, $
                       calfile=calfile, $
                       delbin=delbin, new=new, $
                       debug=debug, $
                       errexit=errexit
; ============================================================================
; Possible calls for high resolution data
;   bdata = GetMagData(scid, year, month, day, hour)
;   bdata = GetMagData(scid, ctr=ctr)
;   bdata = GetMagData(scid, ct_arr=ct_arr)
;
; Possible calls for spin resolution data
;   bdata = GetMagData(scid, year, month, day, /sr)
;   bdata = GetMagData(scid, ctr=ctr, /sr)
;   bdata = GetMagData(scid, ct_arr=ct_arr, /sr)
;
; Keywords
;   ctr   double or dblarr[2]   One or two times in t70 format
;   ct_arr  dblarr(NN)          Can be used alternatively instead of keyword
;                               ctr; useful for large time intervals (many
;                               hours or even days). Hourly intervals will
;                               be skipped if there are no time tags within.
;   csys  string                'scs' or 'gse' (default is 'scs')
;   sr    int                   If set then get spin resolution data,
;                               otherwise high resolution data
;   quiet int                   If set then print only error messages
;   calfile string              Optional special calibration file. If this
;                               keyword is set, the created binary file will
;                               be deleted after the data have been read
;                               use 'none' to apply no calibration at all;
;                               with calfile='none' data will be in the FS
;                               coordinate system and the keyword csys will be ignored
;   delbin  int   0/1           If set the created binary file will be deleted
;                               after reading the data
;   new     int   0/1           If set recreate data from scratch (ASCII+binary)
; ****************************************************************************

@paths.inc ; defines variable 'hostname'

  if keyword_set(quiet) then quiet= 1 else quiet= 0
  if not keyword_set(calfile) then calfile = '' $
  else if not quiet then print, 'using special calfile ' + calfile

  if keyword_set(new) then new = 1 else new = 0
  if keyword_set(delbin) then delbin = 1 else delbin = 0

  if keyword_set(debug) then debug = 1 else debug = 0


  ; check argument list
  ; -------------------
  if n_params() eq 5 then begin ; scid, year, month, day, hour
                                ; ----------------------------
     if keyword_set(ctr) then MyErrExit, 'superfluous keyword <ctr>'
     if keyword_set(ct_arr) then MyErrExit, 'superfluous keyword <ct_arr>'
     if keyword_set(sr) then MyErrExit, 'superfluous parameter <hour>'
     fmt = '(I4.4,"-",I2.2,"-",I2.2,"T",I2.2)'
     ctrng = CluTimeStr2Val(string(format=fmt,year,month,day,hour))
     ctrng = ctrng + [ 0 , 3599.999999d ]

  endif else if n_params() eq 4 then begin ; scid, year, month, day
                                           ; ----------------------
     if keyword_set(ctr) then MyErrExit, 'superfluous keyword <ctr>'
     if keyword_set(ct_arr) then MyErrExit, 'superfluous keyword <ct_arr>'
     if not keyword_set(sr) then MyErrExit, 'missing parameter <hour> or keyword <sr>'
     fmt = '(I4.4,"-",I2.2,"-",I2.2)'
     ctrng = CluTimeStr2Val(string(format=fmt,year,month,day))
     ctrng = ctrng + [ 0 , 86399.999999d ]

  endif else if n_params() eq 1 then begin ; scid, { ctr=ctr | ct_arr=ct_arr }
     if not keyword_set(ctr) and not keyword_set(ct_arr) then begin
        MyErrExit, 'missing parameters year,month,day or keyword ctr or keyword ct_arr', /cont
     endif

     if keyword_set(ctr) then begin
        if n_elements(ctr) eq 1 then begin
            if not keyword_set(sr) then ctrng = ctr + [ 0, 3599.999999d] $
            else ctrng = ctr + [ 0, 86399.999999d ]
        endif else begin
            ctrng = ctr[0:1]
        endelse
     endif else if keyword_set(ct_arr) then begin
        ctrng = [ min(ct_arr), max(ct_arr) ]
     endif else begin
        MyErrExit, 'neither of keywords ctr,ct_arr is set'
     endelse

  endif else begin
     print, 'bad argument list'
     print, 'allowed : (scid,year,month,day,hour)   one hour of high res data'
     print, '          (scid,year,month,day,/sr)    one day of spin res data'
     print, '          (scid,ctr=ctr)               range of high res data'
     print, '          (scid,ctr=ctr,/sr)           range of spin res data'
     message, ''
  endelse

  ctr_ext = ctrng ; + [ -15, 15 ]

  if not keyword_set(csys) then csys = 'scs' $
  else if csys ne 'scs' and csys ne 'gse' then begin
     MyErrExit, 'invalid coordinate system: ' + csys
  endif
  csys_caps = strupcase(csys)


  ; set up string extraction length and time increment
  ; --------------------------------------------------
  if calfile eq 'none' then begin
         if keyword_set(sr) then begin
            MyErrExit, 'cannot combine spinres with calfile=none'
         endif
         length = 13   ; extract yyyy-mm-ddthh
         inc = 3600.d  ; get one hour per request
         sr=0
         hr=1
         rate_str = 'hr'
         csys_caps = 'FS'
  endif else begin
     if keyword_set(sr) then begin ; spin res data : one day at a time
         length = 10      ; extract yyyy-mm-dd
         inc    = 86400.d ; get 24 hours per request
         sr=1
         hr=0
         rate_str = 'sr'
     endif else begin
         length = 13   ; extract yyyy-mm-ddthh
         inc = 3600.d  ; get one hour per request
         sr=0
         hr=1
         rate_str = 'hr'
     endelse
  endelse


  ; set the begin/end time strings for comparison below
  ; ---------------------------------------------------
  beg_str = strmid(CluTimeVal2Str(ctr_ext[0]), 0, length)
  end_str = strmid(CluTimeVal2Str(ctr_ext[1]), 0, length)

  ; get the data in a loop
  ; ----------------------
  i = 0 & first = 1
  repeat begin
      cur_str = strmid(CluTimeVal2Str(ctr_ext[0] + i * inc), 0, length)
      if keyword_set(ct_arr) then begin
         cta = clutimestr2val(strmid(clutimeval2str(ctr_ext[0] + i*inc),0,length))
         ctb = cta + inc

         x = where(ct_arr ge cta and ct_arr lt ctb)
         if x[0] eq -1 then begin
            i = i + 1
            if debug then $
               print, 'Skipping hour/day ' + strmid(clutimeval2str(cta),0,19)
            continue
         endif
      endif
      yyyymmdd = strmid(cur_str,0,4) + strmid(cur_str,5,2) + $
                 strmid(cur_str,8,2)

      if hr then hh = strmid(cur_str,11,2) else hh = '00'
      asc_fn = fgm_dir + 'C' + strtrim(scid,2) + '_' + yyyymmdd + $
               't' + hh + '0000' + '_' + rate_str + '_' + csys_caps + '.mag'

      bin_fn = asc_fn + '_bin'
      if debug then print, bin_fn
      dummy = file_search(bin_fn, count=cnt)
;help, cnt

      ; if not found then request ASCII data and convert to binary file
      ; ---------------------------------------------------------------
      if cnt eq 0 or calfile ne '' or new eq 1 then begin
          if hostname ne 'ibmcl1' and hostname ne 'lapedicaa' then begin
             MyErrExit, 'Cannot create ASCII mag file. You need to do this on host ibmcl1 or lapedicaa!'
          endif

          spawn, 'rm -f ~/temporary.fgm'
          RequestAsciiMagData, scid, cur_str, sr=sr, csys=csys, $
                   quiet=1-debug, calfile=calfile
          ; make sure the file contains data
          openr, fp, '~/temporary.fgm', /get_lun
          fileinfo = fstat(fp)
          free_lun, fp
          if fileinfo.size eq 0 then begin
             msg='ASCII file has zero file size.'
             if keyword_set(errexit) then MyErrExit, msg
             if not quiet then print, msg
             return, { hdr:GetCluHdr(1, msg, sc_id = scid) }
          endif
          spawn, 'mv ~/temporary.fgm ' + asc_fn, result, err_result, exit_status=exit_status
          if exit_status ne 0 then $
              MyErrExit, [ result, err_result, 'mv exited with status ' + strtrim(exit_status,2) ]

          MagConv, asc_fn, hr=hr, quiet=quiet
          spawn, 'rm -f ' + asc_fn
      endif

      dummy = file_search(bin_fn, count=cnt)
      if cnt eq 0 then $
          MyErrExit, 'Error: binary B data file does not seem to exist after creation'

      curdata = ReadBinaryMagData(bin_fn, quiet=quiet)

      if keyword_set(calfile) or delbin eq 1 then begin
         if debug then print, 'rm ' + bin_fn
         spawn, 'rm -f ' + bin_fn, result, err_result, exit_status=exit_status
         if exit_status ne 0 then $
            MyErrExit, [ result, err_result, 'failed to remove ' + bin_fn + ' ???' ]
      endif


      if first eq 1 then begin
         bdata = curdata
         first = 0
      endif else $
         bdata = [ bdata, curdata ]

      i = i + 1
  endrep until cur_str eq end_str

  x = where(bdata.ct ge ctrng[0] and bdata.ct le ctrng[1])
  if x[0] eq -1 then begin
     msg = 'no data within selected interval'
     if keyword_set(errexit) then MyErrExit, msg
     message, msg, /cont
     bs = { hdr:GetCluHdr(1,msg) }
     return, bs
  endif

  bdata = bdata[x]

  if keyword_set(sr) then comment = 'FGM spin resolution data' $
  else                    comment = 'FGM high resolution data'

  bdata2 = replicate( {VTS_T}, n_elements(bdata) )
  bdata2.ct = bdata.ct
  bdata2.val[0] = bdata.x
  bdata2.val[1] = bdata.y
  bdata2.val[2] = bdata.z


  bs = { hdr:GetCluHdr(0, 'ok', sc_id = scid, frame=csys), $
             opt:comment, $
             data:bdata2 }

  return, bs

END

