/* search edr for specific entries */
/**************************************************************************
 * TITLE:  vedrdmp.c
 *
 * AUTHOR:         Unknown
 * CREATED:        Sep 8, 1994
 * modified:    08-24-2020 - Ray Bambery
 *              add debug statements for migration to solaris 2.10
 *               on shock3
 *              08-27-2020 - Ray Bambery
 *              add print of arguments for log
 *              03-09-2021 - Ray Bambery
 *              Made program command line driven, got rid of getline routine.
 *              reworked main routine to get rid of interactive modes
 *              10-12-2021 - Ray Bambery 
 *              added prompt for 2nd parameter
 *              print out input[0], corrected lineln  
 *              fixed logic for begin BLK and end BLK for BLK=sblk,eblk
 *              10-25-2021 - Ray Bambery 
 *              BLK changed to SEQ to match listing
 *              Fixed logic for SEQ, MODE, FMT, and TYPE
 *              10-26-2021 - Ray Bambery
 *              Fixed logic for SCID, FMT - working on ERT
 *              10-27-2021 - Ray Bambery
 *              Cant get ERT, FDS or SCET to work right!
 *              11-08-2021 - Ray Bambery
 *              Fixed all logic for SEQ, MODE, FMT and TYPE
 *              Added error messges for SCET, ERT, FDS, SKIP              
 *              12-09-2021 - Ray Bambery
 *              added dictmcnv prototype
 *              detect if datafile is byte swapped, abort if it is
 *
 * COMPILER:    gcc version 4.5.2 (GCC)
 * OS:          Solaris 10.2   
 * COMPILER:    gcc version 7.1
 * OS:          Centos 7
 ************************************************************************
 * program to dump the contents of a Voyager instrument experiment
 * data record (EDR) file. Each instrument has a slightly different
 * structure. For details see  JPL Doc 618-306, Rev D, Change 4, Voyager
 * Experient Data Record Format Specification, 1 May 1986.
 *
 * Each EDR has a common EDR header as specified in Appendix C of that
 * document. Section 3 goes into details about each experiment (type_tbl). 
 * Section 4 is divided into descriptions of the data format for each
 * of the mission phases (modes variable below).
*/ 
#include <stdio.h>                                                              
#include <math.h>                                                               
#include <string.h>                                                             
#include <stdlib.h>                                                             
#include <ctype.h>                                                              
#define  VGR_HDR   0xd4d1e2     /* MJS in EBCDIC/BCD */
#define  BLNK      " "                                                          
#define  MAXLN     72           /* number of bytes in command line */           
#define  HDRSIZE   240          /* EDR header bytes */                                                
#define  MAXCHAR   32           /* data characters per line  */                 
#define  INTCNT     9           /* count of possible parameters in argv[2] */                                                
#define  TWOEXP8   256                     
#define  XSCET(X,Y)  (TWOEXP8*(X) + (Y))                                       
                                                                                
unsigned char            input[MAXLN+1];                                                        
unsigned char            file_name[44];                                                         
unsigned char            datax[11300];                                                          
                    
char            vgrname[4] = { 0xD4,0xD1,0xE2,0x00};     /* MJS\0 in ebcdic */                             
char            rev_vgrname[2] = { 0xD1,0xD4};
long int        seq_bgn = 0 ;                                                          
long int        seq_end = 10 ;                                                         
short int        prfmt = 0;          /* minimum */                                                               
short int       do_type[18];        /* number of DRS DATA TYPES = 17 values - Table C-1 - item 217 word 60 - Bits 31-24     */
short int       ntypes = 0;        /* number of in_type entries(0 is all)   */          
short int       ok_flag= 0;        /* if zero then no legal type found      */            
char            in_mode[5];                                                           
char            in_scid[3] = "al";                                                    
char            fds_bgn[6];                                                           
char            fds_end[6];                                                           
char            cur_fds[6];                                                           
char            tim_bgn[7];                                                           
char            tim_end[7];                                                           
char            ert_bgn[7];                                                           
char            ert_end[7];                                                           
char            cur_ert[7];                                                           
char            sct_bgn[7];                                                           
char            sct_end[7];                                                           
char            cur_sct[7];                                                           
unsigned long int  recnum = 0   ;      /*  running count of records  **/            
unsigned long int  recsiz = 0   ;      /*  length of current record  **/            
unsigned long int  nbyte = 0;                                                       
unsigned short int hdr_ok= 0;                                                       
unsigned short int  tim_typ = 0;    /*  default to SEQ        */                     
unsigned short int hdr_ln; 

//edr                                                         
char            hdr_ebc[4];                                                           
char            sc_id[2];                                                            
unsigned short int mode ;                                                           
unsigned short int tmode ;                                                          
unsigned short int sublen;                                                          
unsigned long  int  new_scet = 0;                                                    
unsigned short int nscet_frc = 0;                                                   
unsigned short int fds_linum = 0;                                                   
char            amode[6];                                                             
char            low_scet[22];                                                         
char            high_scet[22];                                                         
char            low_ert[22];                                                         
char            high_ert[22];                                                         
char            pscet [20];                                                        
char            pertb[20];                                                        
char            perte[20];                                                        
double          dptim[4];                                                          
long int        tmaray[16];                                                        
char            tstrng[22];                                                            
char            rtcode[5];                                                             
int             i,j,k,m,n,dsnum;   
int             maxl = MAXLN; 
long            edrloc;                                                                 
int             itype;                                             
 int            edrerr;                         
int             lineln,c;                   /* lineln is strlen of 2nd parameter */                                                 
int             xlineln,ylineln,inlineln;
short           lcn;                                                                      
short           prln;                                                                     
short           xval;                                                                     
unsigned short  fval;                                                            
char            ltyp[6];                                                               
//618-306 Voyager EDR Format - Rev D - Table C-1 
/* EDR HEADER structure */                                                                                
struct       {                                                                  
  char                  prjebc[3];     //word 1     PRJID (31-8)        word 1      PROJECT IDENTIFICATION (MJS) 
  unsigned int          exp_id:4;      //4-bits     RECID (7-4)         word 1      Record Indentification
  unsigned int          scft_id:4;     //4-bits     SCID (3-0)          word 1      Spacecraft ID     
  unsigned short int    file_sqnc  ;   //16-bits    PHREC (31-16)       word 2      Physical Record Number 
  char                  sc_mode;       //8-bits     DMODE (15-8)        word 2      Data mode 
  unsigned int          ext_eng:2;     //2-bits     EEF (7-6)           word 2      Engineering Extraction Plan
  unsigned int          sc_pb:1;       //1-bita     SCPB 5              word 2      Spaceceaft Playback
  unsigned int          edflags:5;     //5-bits     SPRE                word 2      Spare
  unsigned short int    ert_begn[3];   //3 words    SERTH (31-16)       word 3      Starting ERT Hour
                                       //           SERTS (15-0)        word 3      Seconds of current hour
                                       //           SERTM (31-16)       word 4      Millseconds of current secpnd
  char                  ert_bgnyr;     //8-bits     SERTY (15-8)        word 4      Year (2-digits)
  unsigned int          data_src:2 ;   //2-bits     DSRC (7-6)          word 4      Data Source
  unsigned int          golay:2 ;      //2-bits     GLYF (5-4)          word 4      Golay/Reed Solomon Encoded Flag
  unsigned int          src_seg:4 ;    //4-bits     SEGN (3-0)          word 4      Segment Number
  unsigned short int    ert_end[3];    //3 words    EERTH (31-16)       word 5      Ending ERT Hour
                                       //           EERTS (15-0)        word 5      Seconds of current hour
                                       //           EERTM (31-16)       word 6      Millseconds of current secpnd
  char                  ert_endyr;     //8-bits     EERTY (15-8)        word 6      Year (2-digits)
  char                  sft_versn;     //8-bits     SWVER (7-0)         word 6      Software version   
  unsigned short int    scet[3];       //3 words    SCETH (31-16)       word 7      Starting ERT Hour
                                       //           SCETS (15-0)        word 7      Seconds of current hour
                                       //           SCETM (31-16)       word 8      Millseconds of current secpnd 
  char                  scet_yr;       //8-bits     SCETY (15-8)        word 8      Year (2-digits)
  char                  corr_flg;      //8-bits     SCETF (7-4)         word 8      S/C Event Time Flag
                                       //           FDSCF (3-0)         word 8      FDSC Correction Flag
  unsigned short int    fds_md16;      //8-bits     FDS16 (31-16)       word 9      FDS Mod 2^16 count word
  unsigned char         fds_md60;      //8-bits     FDS60 (15-8)        word 9      FDS Mod 2^60 Count word 
  unsigned char         fds_lncnt[2];  //16-bits    LCW (7-0)(31-10)    words 9/10  MJS Line Count word
  char                  dwn_link;      //8-bits     DLKTR (23-16)       word 10     Tlm rate to earth 
  char                  exp_rate;      //8-bits     EFFR (15-8)         word 10     Effective TLM rate to earth
  char                  eng_fid;       //8-bits     FID (7-0)           word 10     Format ID word for Engr record
  char                  bet_tol;       //8-bits     BET (31-24)         word 11     Bit error Tolerance
  int                   dsn_cnfg:8;    //8-bits     DSNEC (23-16)       word 11     DSN Equipment Configuration
  short                 xcv_agc;       //8-bits     RAGC (15-0)         word 11     Receiver AGC
  unsigned int          dss_id1:8;     //8-bits     DSNN (31-24)        word 12     DSN Stn Number  
  char                  dss_id2:8;     //8-bits     SPRE (23-16)        word 12     Spare
  unsigned short        edber;         //8-bits     EBRC (15-0)         word 12     Estimated BER Count 
  short                 symsnr;        //8-bits     SSNR (31-16)        word 13     Symbol SNR 
  short                 decsnr;        //8-bits     DSNR (15-0)         word 13     Data Decoder SNR  
  unsigned short        typseq;        //8-bits     PRN (31-16)         word 14     Physical Record Number   
  unsigned short        dqsw[10];      //10-bytes   DQSW                words 14-19 Data Quality Status Words (MF-1 to MF-80)
  unsigned short        dpi_gci[81];   //81-bytes   DPI                 words 19-59 Data Presence Indicators   
  char                  data_type;     //8-bits     DRSDT (31-24)       word 60     DRS Data Type
  char                  gol_rate;      //8-bits     GOLR (23-16)        word 60     Golay/Reed-Solomon corr BER count
  unsigned short int    gol_sum;       //8-bits     GOLS (15-0)         word 60     Folay/Reed-Soloman BER Summation
  }  edr_hdr;                                                                   
/*  618-306 - Rev D - Voyager Experiment Data Record Format  Specification  */
/*  Change 4:  1 May 1986                                                   */  
/*  DRS DATA TYPE  - Table C-1 - item 217 word 60 - Bits 31-24              */
/* Hex Values                                                               */
/*  20=IRIS     21=CRS   22=LECP     23=MAG      24=PLS      25=PPS         */
/*  26=PRA      27=PWS   28=UVS      29=ISR      2A=DCOM     2B=MON 5-8     */
/*  2C=ENGS     2D=ENGE  2E=DCMS     2F=ENGC     30=MON 5-9                 */                                                                              
  int ifound;                   /* flag to indicate whether requested record is found */
  int mfound;                   /* flag to indicate MODE found */
  int sfound;                   /* flag to indicate SCID found */
  int ffound;                   /* flag to indicate FMT found */
  int qfound;                   /* flag to indicate SEQ found */
  /* Items to be searched for  */
  /****    modified for VGR edr dump     ****/                                  
char *type_tbl[18]    = {"IRIS ",                                           
     "CRS  ","LECP ","MAG  ","PLS  ",                                           
     "PPS  ","PRA  ","PWS  ",                                                   
     "UVS  ","ISR  ","DCOM ",                                                   
     "MN58 ","ENGS ","ENGE ",                                                   
     "DCMS ","ENGC ","MN59 ","****"}  ;                                         
                                                                                
char *sc_tbl[6] = {"32","31","PT","**","41","42"};                          
                                                                                
 char *pbflg[2] = {"NO ","YES"};                                            

 /* FMT subtype */                                                                                
 char *fmtbl[7] = {"MIN","FMT","HDR","SUB","DATA","ALL"};                   
/*  2nd parameter search_term options  */ 
 char *parmtbl[INTCNT] =                                                    
    {"SEQ ","ERT ","SCET","FDS ","TYPE","SKIP","FMT ","SCID","MODE"};          
 
char mode_ids[18] =                                                        
    {0,1,2,3,4,5,6,7,8,10,12,14,22,23,24,25,26,29};                             

//data modes - currently only CR5A and UV5A are supported 
char *mode_tbl[18] =      {"ENG  ","CR2  ","CR3  ","CR4  ",                     
                           "CR5  ","CR6  ","CR7  ","CR1  ",                     
                           "GS10A","GS3  ","GS7  ","GS6  ",                     
                           "OC2  ","OC1  ","CR5A ","GS10 ",                     
                           "GS8  ","UV5A "};                                    
                                                                                
char *emodes[] =                                                            
  {"LN-10","CR-10","EN-10","TC-10","MC-10","SM-10","CC-10","FD-10",             
   "LN-40","CR-40","EN-40","TC-40","MC-40","SM-40","CC-40","FD-40",             
   "LN-12","CR-12","EN-12","TC-12","MC-12","SM-12","CC-12","FD-12",             
   "LN-72","CR-72","EN-72","TC-72","MC-72","SM-72","CC-72","FD-72",             
   "LA-10","RA-10","EA-10","TA-10","MA-10","SA-10","CA-10","FA-10",             
   "LA-40","RA-40","EA-40","TA-40","MA-40","SA-40","CA-40","FA-40",             
   "LA-12","RA-12","EA-12","TA-12","MA-12","SA-12","CA-12","FA-12",             
   "LA-72","RA-72","EA-72","TA-72","MA-72","SA-72","CA-72","FA-72"};            
                                                                                
static short int reclen[18] [14] =    
/* 18=mode,14=record lengths in bytes       */                                  
/*  IRIS  CRS LECP  MAG   PLS  PPS  PRA  PWS  UVS ISR DCM MON ENGS ENGE */         
  {{7936,2360,6032, 6512, 512, 508,1872, 864,4092,500,716,316,3860,3860},           
   {0000,2360, 912, 6536, 912, 508,1872, 864, 552,000,716,316,3860,3860},             
   {0000,2360,1552, 3704, 704, 508,1872, 504, 784,000,716,316,3860,3860},            
   { 000,2360,1552, 3992,1112, 312,1872, 504, 528,000,716,316,3860,3860},            
   { 000,3400,2192, 5848,1528, 504,1072, 612, 528,000,716,316,3860,3860},            
   {0000,2360,6032, 6512,3256, 512,1072, 864,3452,000,716,316,3860,3860},          
   { 000,1992,1808, 2512,1360, 384, 584, 624, 848,500,716,316,3860,3860},             
   {  00,2360,2672,11280,1672, 508,1872, 864,1312,000,716,316,3860,3860},           
   { 000,0000,0000, 0000, 000,0000,0000,9840,0000,500,716,316,3860,3860},          
   {7936,2360,6032, 6512, 512, 508,1872, 864,3452,500,716,316,3860,3860},           
   {7936,2360,6032, 6512, 512, 508,1872, 864,3452,500,716,316,3860,3860},           
   { 576,2360,6032, 6512, 512, 508,1872, 864,3452,500,716,316,3860,3860},            
   { 576,2360,6032, 6512, 512, 508,1872, 864,0000,500,716,316,3860,3860},            
   { 576,2360,6032, 6512, 512, 508,1872, 864,4092,500,716,316,3860,3860},            
   {0000,3400,0564, 1712, 672,0000, 528, 540, 376,500,716,316,3860,3860},            
   { 000,0000,0000, 0000, 000,0000,0000,2160,0000,500,716,316,3860,3860},          
   {7936,2360,6032, 6512, 512, 508,1872, 864,3452,500,716,316,3860,3860},           
   {0000,3400, 564, 1712, 672,0000, 528, 540,3452,500,716,316,3860,3860}};           
                                                                                
 char *dsn_tbl[] =                                                          
   {"00","00","70","72","03","00","33","00","00","97","53","74","00",          
    "32","16","09","75","00","00","81","00","00","00","00","00","00",          
    "00","00","75","75","75","48","75","00","21","00","00","00","75",          
    "54","00","75","00","75","00","49","00","00","75","00","75","00",          
    "00","75","00","00","75","75","00","00","75","00","00","00","00",          
    "55","22","00","00","00","75","00","00","00","58","00","75","75",          
    "00","00","75","99","00","04","00","05","00","10","00","23",               
    "00","06","00","00","24","07","73","00","00","75","00","86","08",          
    "50","00","75","00","00","30","39","00","00","75","00","00","75",          
    "77","00","00","24","26","00","00","00","29","79","75","20",               
    "00","12","14","15","16","85","00","00","00","13","75","75","00",          
    "00","00","75","40","00","42","43","45","46","00","51","75","00",          
    "83","75","00","00","00","00","00","56","00","00","98","00","00",          
    "52","00","75","34","35","36","37","38","02","00","00","75","75",          
    "00","75","00","25","75","00","00","75","24","78","75","75",               
    "00","00","75","00","75","00","75","96","80","91","92","93",               
    "94","75","95","00","59","00","19","82","75","84","00","60","00",          
    "00","00","17","00","75","00","00","75","75","00","00","00","62",          
    "27","28","57","00","00","00","75","00","75","00","00","00","75",          
    "61","63","00","66","00","00","00","00","00","75","65","00","75"           
  };                                                                            
                                                                                
static short subhdr[14] [18] =                                          
/*  IRIS  CRS LECP  MAG PLS  PPS  PRA PWS  UVS ISR DCM MON ENGS ENGE */         
 {{ 0,  0,  0,  0,   0,   0,   0,   0, 0, 16, 16, 16, 16, 16,  0,  0,  0, 0},                                   
  { 0, 40, 40, 40,  40,  40,  24,  40, 0, 40, 40, 40, 40, 40, 40,  0, 40,40},                         
  { 0, 32, 32, 32,  32,  32,  32,  32, 0, 32, 32, 32, 32, 32,  4,  0, 32, 4},                           
  { 0,264,464,872,1288,3016,1120,1440, 0,272,272,272,272,272,432,272,432},        
  { 0, 64, 64, 72,  88,  88,  64,  64, 0, 80, 80, 80, 80, 80, 32,  0, 80,32},                         
  { 0, 24, 24, 24,  24,  32,  24,  24, 0, 24, 24, 24, 24, 24,  0,  0,  0, 0},                            
  { 0, 32, 32, 32,  32,  32,  32,  32, 0, 32, 32, 32, 32, 32,  8,  0, 32, 8},                           
  { 0,352, 88,160, 240, 352, 192, 352, 0,432,432,432,432,432, 80,  0,416,80},             
  { 0, 32, 32, 32,  32,  32,  32,  32, 0, 12, 12, 12, 12,  0, 16,  0, 12,12},                          
  { 0, 20, 20, 20,  20,  20,  20,  20, 0, 20, 20, 20, 20, 20,  0,  0,  0, 0},                            
  {28, 28, 28, 28,  28,  28,  28,  28,28, 28, 28, 28, 28, 28, 28, 28, 28,28},                      
  { 0, 00, 00, 00,  00,  00,  00,  00,00, 00, 00, 00, 00, 00, 00, 00, 00,00},                       
  {20, 20, 20, 20,  20,  20,  20,  20,20, 20, 20, 20, 20, 20, 20, 20, 20,20},                      
  {20, 20, 20, 20,  20,  20,  20,  20,20, 20, 20, 20, 20, 20, 20, 20, 20,20} };                    


/**************************************************************************/                                                                                
int  main(int, char**);
void dumphex(char ltyp[], short lcn, short);

void seqproc(char input[],int lineln,long *seq_bgn,long *seq_end);
int typproc(char input[],int lineln,char **type_tbl,
                                      short do_type[]);
int skiproc(char input[],int lineln,char **type_tbl,
                                      short do_type[]);
void fdsproc(char input[],int lineln,
                        char fds_bgn[],char fds_end[]);
void timeproc(void);
/* from include/diclkext.h */
extern  void    dictmcnv(long int *,char *,char *,double *,char *,
              char *,char *,char *,char *);

//--------------------------------------------------------------------------//
int  main (int argc,char** argv)
{
FILE  *edrin;
unsigned char input3[4]="\0",inpmode[5]="\0",inpfmt[4];
int  ijj,ikk, ill, arglen, file_seq;
char chr;
unsigned char input6[5],input6a[5]="\0";
/**  PARAMETER PROCESSOR   ***/
    if (argc < 2 ){
      printf("usage:  vedrdmp infile search_term [ > output_file  ]  \n");
      exit(0);


    }
    strcpy (&file_name[0], argv[1]);
     printf(" INPUT FILE= %s  \n\n",file_name);
     edrin = fopen(file_name,"rb");
       if (edrin == NULL)
     {
        printf("vedrcmp:  UNABLE TO OPEN FILE %s FOR INPUT.\n",file_name);
        exit(0);
     }

    if (argc < 3) {
        printf ("Missing 2nd parameter - search_term\n");
        printf ("  SEQ=, ERT=, SCET=, FDS=, TYPE=, SKIP, FMT=, SCID= MODE=\n");
        exit(0);
    } else {
        /* convert to UC */
        lineln = strlen(argv[2]);
        printf ("linel n= %d\n",lineln);
        strcpy(&input[0],argv[2]);
        for (ill=0;ill<lineln;ill++)  { 
            chr = input[ill]; 
            input[ill] = toupper(input[ill]);
//            printf("%c (%c)\n", (chr), input[ill]); 
        } 

    }
    printf ("input[0] = %s\n",&input);
//"SEQ ","ERT ","SCET","FDS ","TYPE","SKIP",
//            "FMT= ","SCID", "MODE"   
//    lineln = strlen(input);         /* added 10-22-2021 */ 
    printf ("lineln = %d\n",lineln);

    strncpy (input3,input,3);
    printf ("input3 = %s\n",input3);

/* see if 2nd parameter is in list  */
    for (i = 0; i < INTCNT ; i++)
       {
        ikk = memcmp(input3,parmtbl[i],3);      /* valid parameter value */
    printf("memcmp value = %d\n",ikk);          
       if (ikk == 0)
          {
            break;
            /* not in list  Added 10-23-2021 */
//            printf(" UNDEFINED 2nd INPUT PARAMETER -- %s \n",input3);
//            printf(" allowed are: SEQ=, ERT=, SCET=, FDS=, TYPE=, SKIP, FMT=, SCID=, MODE=\n");
//            goto EXITS;
          }
       }
    printf ("i = %d\n",i);
  switch(i)                                                                     
     {                                                                          
      case 0:        /*  seq=   formerly blk  */  
        printf ("vedrdmp:  case SEQ  lineln = %d\n",lineln);                                            
        tim_typ=1;                                                              
        seqproc(input,lineln,&seq_bgn,&seq_end);                                
//        xlineln = -1;
         break;                    
                                   
      case 1:      /*  ert=   */
        printf ("vedrdmp: case 1 - ERT=  \n");                                                
        tim_typ=2;                                                              
   /****timeproc(input,lineln,ert_bgn,ert_end);   ****/                         
        timeproc();       /*input,lineln);  */                                  
        memcpy(ert_bgn,tim_bgn,7);                                              
        memcpy(ert_end,tim_end,7);
        printf ("vedrdmp: ert_bgn = %s   ert_end = %s\n",ert_bgn,ert_end);
        printf ("logic for ERT=BGN,END is not implemented...EXIT\n");
        goto EXITS;

        break;                                                                  

      case 2:      /*  scet=  */         
        printf ("vedrdmp: case 2 - SCET=\n");                                        
        tim_typ=3;                                                              
  /**** timeproc(input,lineln,sct_bgn,sct_end);    *****/                       
        printf ("vedrdmp: case SCET - inlineln = %d\n",inlineln);
        timeproc();        /*input,lineln);  */                                 
        memcpy(sct_bgn,tim_bgn,7);                                              
        memcpy(sct_end,tim_end,7);
        printf ("vedrdmp: sct_bgn = %s   sct_end = %s\n",sct_bgn,sct_end); 
        printf ("logic for SCET=BEG,END is not implemented...EXIT\n");
        goto EXITS;
 
        break;                                                                  

      case 3:      /*  fds=   */                                                
        tim_typ=4;
        seq_end = 1000;
        printf ("vedrdmp: case 3 - FDS - inlineln = %d\n",inlineln);                                                              
        fdsproc(input,inlineln,fds_bgn,fds_end);
        printf (" fds_bgn = %s   fds_end = %s\n",fds_bgn,fds_end);
        printf ("logic for FDS=BGN,END is not implemented...EXIT\n");
        goto EXITS;
 
        break;                                                                  

      case 4:      /*  type=  (instrument) */
        printf ("vedrdmp: case 4 - TYPE=    lineln = %d\n",lineln);                                                
        ntypes = typproc(input,lineln,type_tbl,do_type); 
        printf ("case4: TYPE input = %s\n",input);   

        break;
      case 5:      /*  skip=  */
        /* skip=(TYPE in typtbl)          */                                                
        printf ("vedrdmp: case 5 - SKIP= <TYPE>\n");
        ntypes = skiproc(input,lineln,type_tbl,do_type);
        printf ("logic for SKIP=<TYPE> is not implemented...EXIT\n");
        goto EXITS;
        break;                                                                  

      case 6:      /*  fmt =   */
        /* fmt= MODE in fmtbl   */   
        printf ("vedrdmp: case 6 - FMT=\n"); 
        j = 1;
/* got thru prfmts MIN,FMT,HDR,SUB,DATA, ALL */ 
        strcpy(inpfmt,&input[4]);                                           
        for(prfmt=0; prfmt<6; prfmt++)                                          
        { 
            printf ("input[4] = %s\n",&input[4]);                                                                      
           if(strncmp(&input[4],fmtbl[prfmt],3) ==0) {     
           j = 0;             
           break;                          
            }        
 
        }
        printf ("prfmt = %d\n",prfmt);        
        if (j == 1)  {
          printf("ILLEGAL FMT= value\n");       
          printf(" LEGAL FMT= VALUES are:   \n");
          printf("MIN, FMT, HDR, SUB, DATA, ALL\n");            
            goto EXITS; 
           j = 0;
	    }
        break;                                                                  

      case 7:      /*  scid=  */
        printf("vedrdmp: case 7 - SCID=\n");
            strncpy (input6,&input[5],3);
            printf ("input6 = %s\n",input6);
            if (strcmp(input6,"1") == 0) {strcpy(input6a,"31");};
            if (strcmp(input6,"2") == 0) {strcpy(input6a,"32");};
                                                
        memcpy(in_scid,&input6a,2);
        break;                                                                  

      case 8:      /* MODE   */   
        printf("vedrdmp: case 8 - MODE=\n");   
        for (ijj=0;ijj<18;ijj++) 
        {
           printf ("input[4] = %s\n",&input[4]);
            strncpy(inpmode,&input[4],4);
            printf ("inpmode= %s\n",inpmode);
           if(strncmp(inpmode,mode_tbl[ijj],3) ==0) {
           j = 0;
           break;
            }

        }
        if (j == 1)  {
          printf("ILLEGAL MODE= value\n");
          printf(" LEGAL MODE VALUES are:   \n");
          printf(" ENG, CR2, CR3 , CR4, CR5, CR6, CR7, CR1, GS10A,\n");
          printf(" GS3, GS7, GS6, OC2, OC1, CR5A, GS10, GS8, UV5A\n");

            goto EXITS;
           j = 0;
        }

        break;
                                                              
      default:      /*  undefined input   */                                    
        printf(" UNDEFINED INPUT PARAMETER -- %s \n",input);  
        goto EXITS;                  
        break;                                                                

   }          /* end of switch.                  */
//    xlineln = -1;                             
//  }     /*  end of while ...  getline ...        */                             
                                                                                
RUNIT:   ;                                                                      
   printf("\n    END OF INPUT PROCESSING \n");                                          
                                                                                
    printf(" \f");     /* EJECT A PAGE & START DATA PRINT  */                   

/*---------------------------------------------------------------*/ 
/* DATA PROCESSOR **/

    for (;;)          /*   do forever            */                             
    {
    printf ("***************************************************************\n");
    printf ("                  FOREVER LOOP                                *\n"); 
    printf ("***************************************************************\n");
    printf ("           LOOP: fread recnum  (1 less than SEQ)  = %d\n",recnum); 

    if((edrerr=fread(datax,1,HDRSIZE,edrin)) == 0) /* READ EDR HEADER */            
        {
        recnum++;                                                                       
        printf("\n");                                                           
        printf(" END OF INPUT FILE. \n");                                       
	printf("  %d RECORDS READ. ",recnum);
        printf("\n");                                                           
        break;                                                                  
        }        
                                                                  
                                                                                
    hdr_ln = HDRSIZE;       
                                                    
/*** HEADER PROCESSOR ***/                                                                                

    printf ("Check for byte swap %0x %0x\n", datax[0], datax[1]);
    /* check for byte swap  */
    if (memcmp((char *)&datax[0],(char *)rev_vgrname,2) == 0 ) {
        printf ("\nDATA SET is Byte reversed\n");
        printf ("PLEASE run byteswap program\n");
        exit(0);
    }
    /*     code for bad records/ pps block with leading zeroes */               
    for(k=0, nbyte=0; k<hdr_ln; k++)                                            
        { 
           if (k==0) printf("k = %d\n",k);                                                                     
   if(memcmp((char *)&datax[k],(char *)vgrname,3) == 0)   /*   good record  */    
           {                
	       printf ("hdr_ok = 1: datax[0] datax[1] datax[2] = %x%x%x\n",datax[0],datax[1],datax[2]);
           hdr_ok = 1;                                                          
           break;                                                               
           }                                                                    
        else                                                                    
           {
//            printf ("datax[0] datax[1] datax[3] != %x%x%x\n",datax[0],datax[1],datax[2]);                                                                    
           nbyte++;                                                      
           hdr_ok = 0;                                                          
           }                                                                           
                                                                                
    }                       /*  end of for(k=0 ... loop   */                   
    if(hdr_ok == 0) goto ENDLOOP;  /* read more data, no hdr yet */ 

    printf ("here0\n");                                                    
    hdr_ln -= nbyte;     
    memcpy(hdr_ebc,&datax[nbyte],3);
    hdr_ebc[3] = '\0';                                                     

/* if incomplete header */
    if(hdr_ln < HDRSIZE)         /* for partial pps blocks   */                 
      {
       printf ("incomplete header - fread (2) hdr_ln = %d < HDRSIZE %d\n",hdr_ln,HDRSIZE);                                                                         
       fread(&datax[HDRSIZE],1,nbyte,edrin);    
       edrloc = ftell(edrin);                            
       memcpy(&datax[0],&datax[nbyte],HDRSIZE);                         
      }  
                                                                       
    memcpy((char *)&edr_hdr,datax,HDRSIZE); /* copy hdr to structure */                
    recnum++;               /*  TOTAL RECORD COUNT   */                         
                                                                                
                    /*   READ REMAINDER OF RECORD   */                          
/*  20=IRIS     21=CRS   22=LECP     23=MAG      24=PLS      25=PPS         */
/*  26=PRA      27=PWS   28=UVS      29=ISR      2A=DCOM     2B=MON 5-8     */
/*  2C=ENGS     2D=ENGE  2E=DCMS     2F=ENGC     30=MON 5-9                 */
    k = edr_hdr.data_type - 32; /* hex 20 */                                                
/* ntyp*s > 0 if "SKIP" argv[2] */
    printf("k = data_type = %d  ntypes = %d \n",k,ntypes);
    if(ntypes != 0)                                                             
       {                                                                        
       for(n=0; n < 18 ; n++)                                                   
          {                                                                     
          if(k == do_type[n])                                               
             {                                                                  
             ok_flag = 1;       

             break;                                                             
             }              /*  end of if(k  ...     */                         
          ok_flag = 0;  

        }             /*  end of for(n  ...    */                               
    }                 /*  end of if(ntypes ... */                               
    else   /*   come here if ntypes=0   */                                      
        ok_flag = 1;                                                            

    printf ("here1  ok_flag = %d\n",ok_flag);                                                                                
    itype = k;                                                                  
    switch(k)         /* convert odd types to standard type */                  
    {                                                                           
    case 14:          /* DCMS */                                                
      k=10;           /* DCOM */                                                
      break;
                                                                    
    case 15:          /* ENGC */                                                
      k=13;           /* ENGE */                                                
      break;                                                                    

    case 16:         /*  MON 5-9 */                                             
      k=11;          /*  MON 5-8 */                                             
      break;                                                                    

    default:                                                                    
      break;                                                                    
    }                                                                           
/*
char *mode_tbl[18] =      {"ENG  ","CR2  ","CR3  ","CR4  ",
                           "CR5  ","CR6  ","CR7  ","CR1  ",
                           "GS10A","GS3  ","GS7  ","GS6  ",
                           "OC2  ","OC1  ","CR5A ","GS10 ",
                           "GS8  ","UV5A "};
    ONLY NODE 14 (CR5A) is OK   (UV5A = 17)
*/                                                                                
    for (j=0; j<18; j++)                                                        
       {                                                                        
       if (edr_hdr.sc_mode == mode_ids[j] )                                 
          {                                                                     
          mode = j;                                                             
          break;                                                                
          }                                                                     
       }                                                                        
    printf ("here2  mode = %d \n",mode);                                                                                
    n = edr_hdr.scft_id;                                                        
    memcpy(sc_id,sc_tbl[edr_hdr.scft_id],2);
    sfound=0;                               
    printf ("sc_id = %s input6a = %s\n",sc_id,input6a);
    if (strcmp(sc_id, input6a) == 0) {sfound=1;}
    
    /* TIME processing */
    printf ("vedrdmp:   edr_hdr.fds_md16 = %d\n",edr_hdr.fds_md16); 
    printf ("           edr_hdr.fds_md60 = %d\n",edr_hdr.fds_md60);
    fds_linum = XSCET(edr_hdr.fds_lncnt[0],edr_hdr.fds_lncnt[1]);
    printf ("           fds_linum        = %d\n",fds_linum);
    // itoa not supported by solaris compiler - 8-14-2020 //
//    &cur_fds[0] == itoa (&edr_hdr.fds_md16);
     sprintf(cur_fds, "%d",edr_hdr.fds_md16);
     printf ("vedrdmp:   &cur_fds = %s\n",&cur_fds[0]);
     printf ("vedrdmp:   &edr_hdr.ert_bgnyr = %d\n",edr_hdr.ert_bgnyr);                                
     printf ("vedrdmp:   &edr_hdr.edr_begn = %d\n",edr_hdr.ert_begn);                                                                           
    cur_ert[0] = edr_hdr.ert_bgnyr;                                         
    memcpy(&cur_ert[1],edr_hdr.ert_begn,6);                                

     printf ("vedrdmp:   &edr_hdr.scet_yr = %d\n",edr_hdr.scet_yr);
     printf ("vedrdmp:   &edr_hdr.scet = %d\n",edr_hdr.scet);                                                                                         
    cur_sct[0] = edr_hdr.scet_yr;                                           
    memcpy(&cur_sct[1],edr_hdr.scet,6);                                    
    
    /* read remainder of record */                                                                            
    hdr_ln = reclen[mode] [k] - HDRSIZE;                                
  /******   READ REMAINDER OF THIS RECORD, THEN TEST AGAINST INPUTS */          
    printf ("here4\n");
    printf ("fread (3)  remainder of record\n");                                                                                
    n = fread(&datax[HDRSIZE],1,hdr_ln,edrin);                              
    recsiz = hdr_ln + HDRSIZE;                                                  

    printf ("tim_typ = %d  ok_flag =%d\n",tim_typ,ok_flag);                                                                                
    if (ok_flag == 1)      /* if type selected do rest of code  */              
    {                                                                           
    /***    test start/stop conditions here       ***/                          
    switch(tim_typ)                                                             
    {                                                                           
    case 1:         /***   SEQ input   ***/ 
        printf ("tim_typ case 1 (SEQ) input = %s\n",input); 
        printf ("recnum   = %d  seq_bgn = %d\n",recnum,seq_bgn);  
        if(recnum > seq_end) goto EXITS;                               
       if(recnum  <=  seq_bgn) goto ENDLOOP;
        if (strncmp(input,"FMT=",4) == 1) {
            printf ("********  if FMT=XXX only 1st 10 records shown *********\n");
        }
        if (strncmp(input,"TYPE=",5) == 0) {
            printf ("TYPE= \n");
            seq_end=1000;
        }
        if (strncmp(input,"SCID=",5) == 0) {
            printf ("SCID=%s \n",input6);
            if (strcmp(input6,"1") == 0) {strcpy(input6a,"31");}
            if (strcmp(input6,"2") == 0) {strcpy(input6a,"32");}
            seq_end=1000;
        }                    
         printf ("recnum   = %d  seq_end = %d\n",recnum,seq_end);                                                    
//       if(recnum > seq_end) goto EXITS;                                         
       break;
                                                                   
    case 2:         /***   ert input   ***/ 
        printf ("tim_typ case 2 (ERT)\n");
        printf ("cur_ert = %s    ert_bgn = %s  ert_end = %s\n",cur_ert,ert_bgn,ert_end);                                    
       if(memcmp(cur_ert,ert_bgn,7) < 0) goto ENDLOOP;                          
       if(memcmp(cur_ert,ert_end,7) > 0) goto EXITS  ;                          
       break;                                                                   

    case 3:         /***   sct input   ***/    
        printf ("tim_typ case 3 (SCET)\n");
        seq_end=1000;
        printf ("cur_sct =  %s    sct_bgn = %s  sct_end = %s\n",cur_sct,sct_bgn,sct_end);
       if(memcmp(cur_sct,sct_bgn,7) < 0) goto ENDLOOP;                          
       if(memcmp(cur_sct,sct_end,7) > 0) goto EXITS  ;                          
       break;                                                                   

    case 4:         /***   fds input   ***/  
        printf ("tim_typ case 4 (FDS)\n");
        printf ("cur_fds = %s    fds_bgn = %s  fds_end = %s\n",cur_fds,fds_bgn,fds_end); 
        seq_end=1000;
//       if(memcmp(cur_fds,fds_bgn,6) < 0) goto ENDLOOP;                          
//       if(memcmp(cur_fds,fds_end,6) > 0) goto EXITS  ;                          
        if(recnum - 1 <= seq_bgn) goto ENDLOOP;
//        if(recnum   < seq_bgn) goto ENDLOOP;
      break;                                                                   
    default:        /***   no  input   ***/                                     
       break;                                                                   
    }                                                                           
     printf ("here5\n");

/* TMARRAY - pack in time array for time library call */
    tmaray[0] = edr_hdr.ert_bgnyr +1900;                                    
    if (tmaray[0] < 1970) tmaray[0] +=100;  /* after 2000 */            
    tmaray[2] = edr_hdr.ert_begn[0];                                    
    tmaray[3] = edr_hdr.ert_begn[1];                                    
    tmaray[9] = edr_hdr.ert_begn[2];                                    
    dictmcnv(tmaray,NULL,NULL,dptim,NULL,"hoy","doy",high_ert,rtcode);          
    high_ert[4] = high_ert[8] = ':';                                    
    memcpy(pertb,&high_ert[2],20);    /*   begin ert  */                   
    printf("vedrdmp: pertb = %s\n",pertb);                                                                            
    tmaray[0] = edr_hdr.ert_endyr +1900;                                    
    if (tmaray[0] < 1970) tmaray[0] +=100;  /* after 2000 */            
    tmaray[2] = edr_hdr.ert_end[0];                                     
    tmaray[3] = edr_hdr.ert_end[1];                                     
    tmaray[9] = edr_hdr.ert_end[2];                                     
    dictmcnv(tmaray,NULL,NULL,dptim,NULL,"hoy","doy",high_ert,rtcode);          
    high_ert[4] = high_ert[8] = ':';                                    
    memcpy(perte,&high_ert[2],20);   /*   end ert   */                     
                                                                                
    tmaray[0] = edr_hdr.scet_yr +1900;                                      
    if (tmaray[0] < 1970) tmaray[0] +=100;  /* after 2000 */            
    tmaray[2] = edr_hdr.scet[0];                                        
    tmaray[3] = edr_hdr.scet[1];                                        
    tmaray[9] = edr_hdr.scet[2];                                        
    dictmcnv(tmaray,NULL,NULL,dptim,NULL,"hoy","doy",high_scet,rtcode);         
    high_scet[4] = high_scet[8] = ':';                                  
    memcpy(pscet,&high_scet[2],20);                                        
                                                                                
      tmode = edr_hdr.eng_fid;                                                  
      tmode >>= 1;       /*  right shift 1 bit for s/c id   */                  
    if(mode == 0) {      /*  for dcom and eng types   */                        
      strcpy(amode,emodes[tmode]);                                          
      }                                                                         
    else  {                                                                     
      strcpy(amode,mode_tbl[mode]);                                         
      }                                                                         

/* Go find requested parameter in the data */
    printf("here6\n");
  fds_linum = XSCET(edr_hdr.fds_lncnt[0],edr_hdr.fds_lncnt[1]);  

    ifound=0;           /* initialize ifound flag */ 
    mfound=0;
    sfound=0;
    ffound=0;
    qfound=0;
    printf ("sc_id = %s  in_scid = %s\n",sc_id,in_scid);
    if (strncmp(in_scid,sc_id,2) == 0) {
        sfound = 1;
        printf ("sfound = %d\n",sfound);
    } else {
                                
        if((memcmp(in_scid,sc_id,2) != 0)  &&
            (memcmp(in_scid,"al",2) != 0)) { 
            printf ("0) vedrdmp: (x) %s NOT FOUND\n",in_scid);

            goto ENDLOOP;
        }
    }                            
           /****  PRINT MIN FORMAT FROM HEADER  ****/  
   dsnum = edr_hdr.dss_id1;                           
   printf ("dsnum = %d\n",dsnum);

    printf ("amode = %s  inpmode = %s\n",amode,inpmode);
        if (strncmp(amode,inpmode,4) == 0) {mfound=1;}

/*  GET information to publish in recor */
    if (tim_typ == 1) {
        file_seq = edr_hdr.file_sqnc;
        if (file_seq <= seq_end && file_seq >= seq_bgn) {qfound=1;}
        printf ("file_seq %d between seq_bgn = %d and seq_end = %d\n",file_seq,seq_bgn,seq_end);

    }                                                                             
  if(prfmt == 0)   {    /*  prfmt=0  = MIN     */
    printf ("PRINT MIN\n");                                      
    printf("\n SEQ=%6d SC=%2s %s",
             edr_hdr.file_sqnc,sc_id,type_tbl[itype]);  
 
      printf("MODE=%s ",amode);
//        if (strncmp(amode,inpmode,4) == 0) {ifound=1;}
      printf("SEG=%2d ",edr_hdr.src_seg);
      printf("ERT=%s FDS=%.5d.%.2d.%.3d \n",
            pertb,edr_hdr.fds_md16,edr_hdr.fds_md60,fds_linum);
//      printf ("here6a\n");
/*
    printf ("%22s\n = ",BLNK);
    printf ("P/B=%s \n",pbflg[edr_hdr.sc_pb]);
    printf ("DSN=%s  \n",dsn_tbl[dsnum]);
    printf ("SCT=%s\n",pscet);
*/                                                                          
    printf("%22s P/B=%s   DSN=%s SCT=%s\n",BLNK, 
     pbflg[edr_hdr.sc_pb],dsn_tbl[dsnum],pscet);                                        
     ifound = 1;                                                                           
           printf("here7\n"); 
                                                                                
   }         /*      end of fmt=min     */                                      
   else       /*  FULL FMT PRINT BELOW HERE     */                              
   {
    printf("here8\n");
    printf ("prfmt = %d\n",prfmt);
//    printf("PRINT FULL for fmtbl[prfmt] = %s\n",fmtbl[prfmt]);                                                                            
   /******    insert full print of hdr for fmt=hdr,sub,all etc.  ****/          
    printf("\n  %07d  %s  ",                                                   
                   edr_hdr.file_sqnc,type_tbl[itype]);                      
    printf("MODE=%s ",mode_tbl[mode]);                                      
    printf("SEG=%02d ",edr_hdr.src_seg);                                        
    printf("S/C=%2s ",sc_id);                                                   
    printf(" DSN=%s ",dsn_tbl[dsnum]);  /* [edr_hdr.dss_id1]);    */                                                                                
    xval = edr_hdr.xcv_agc;   /*  agc has 9 bits integ & 7 bit frac*/           
    if(xval<0) xval = -xval;  /*  knock off sign bit for shifting  */           
    xval = xval>>7;          /* shift off fraction bits.      */                
    if(edr_hdr.xcv_agc<0) xval = -xval;   /* put sign back    */                
    printf("AGC =%4d.",xval);                                                   
    fval = edr_hdr.xcv_agc<<9;  /*  shift off integer bits  */                  
    fval = fval>>8;          /* shift back 8 for alignment    */                
    fval = (((float)fval)/256.)*100.;                                           
    printf("%02.2uDB  ",fval);                                                  

/* ERT record */                                                                                
    printf("ERTBEG=%s   FDS=%.5d.%.2d.%.3d ",                                  
            pertb,edr_hdr.fds_md16,edr_hdr.fds_md60,fds_linum);                 
                                                                                
    printf("\n%18s",BLNK);                                                      
    if(mode != 0)  strcpy(amode,"N/A  ");                                       
    printf("FID = %s",amode);                                                   
    printf("%6s P/B=%s CFG=%02X ",BLNK,                                         
     pbflg[edr_hdr.sc_pb],edr_hdr.dsn_cnfg);                                
                                                                                
    xval = edr_hdr.symsnr;   /* ssnr has 9 bits integ & 7 bit frac*/            
    if(xval<0) xval = -xval;  /*  knock off sign bit for shifting  */           
    xval = xval>>7;          /* shift off fraction bits.      */                
    if(edr_hdr.symsnr<0) xval = -xval;   /* put sign back    */                 
    printf("SSNR=%4d.",xval);                                                   
    fval = edr_hdr.symsnr<<9;  /*  shift off integer bits  */                   
    fval = fval>>8;          /* shift back 8 for alignment    */                
    fval = (((float)fval)/256.)*100.;                                           
    printf("%02.2uDB  ",fval);                                                  
                                                                                
    printf("ERTEND=%s  ",perte);   /*   end of line 2 */                        
                                                                                
    printf("\n%18s",BLNK);                                                      
    printf("LRECL= %5d",recsiz);                                                
    printf("%21s",BLNK);                                                        
                                                                                
    xval = edr_hdr.decsnr;   /* dsnr has 9 bits integ & 7 bit frac*/            
    if(xval<0) xval = -xval;  /*  knock off sign bit for shifting  */           
    xval = xval>>7;          /* shift off fraction bits.      */                
    if(edr_hdr.decsnr<0) xval = -xval;   /* put sign back    */                 
    printf("DSNR=%4d.",xval);                                                   
    fval = edr_hdr.decsnr<<9;  /*  shift off integer bits  */                   
    fval = fval>>8;          /* shift back 8 for alignment    */                
    fval = (((float)fval)/256.)*100.;                                           
    printf("%02.2uDB ",fval);                                                   
                                                                                
    printf("  SCET =%s  ",pscet);   /*  end of line 3   */                      


/* HAVE the information */
/*  FMT PROCESSING */
/* prfmt    MIN=0, FMT=1, HDR=2, SUB=3, DATA=4, ALL=5    */
    ffound = 0;
    if (strcmp(inpfmt,"MIN") ==0 ) { ffound=1;}
    if (strcmp(inpfmt,"HDR") == 0) {ffound=1;};
    if (strcmp(inpfmt,"SUB") == 0) {ffound=1;};
    if (strncmp(inpfmt,"DAT",3) == 0) {ffound=1;};
    if (strcmp(inpfmt,"ALL") == 0) {ffound=1;};

    if (prfmt == 2 || prfmt == 5)     /* at least HDR print  (2)  */                              
       {
        printf ("\n******* HDR record = \n");                                                                        
       printf("\n");                                                            
       lcn = 0;                                                                 
       prln = HDRSIZE;                                                          
    /*                                */
       dumphex("HDR",lcn,prln);              /*   end of header hex print    */                                    
       }                                                                         
    if (prfmt == 3 || prfmt ==5)     /* at least SUB print   (3)  */                              
       {
        printf ("\n******* SUB record\n");                                                                        
       lcn = prln;                                                              
       sublen = subhdr[k] [mode];                                   
       prln = sublen + HDRSIZE;                                                 
    /*                                         */
       dumphex("SUB ",lcn,prln);                                           
          /*   end of data hex print    */                                      
       }                                                                        
                                                                                
    if (prfmt == 4 || prfmt == 5)     /* print entire record  (4)  */                              
       {
        printf ("\n****** DATA record\n");                                                                        
       lcn = prln;                                                              
       prln = recsiz;                                                           
    /*                                            */
       dumphex("DATA",lcn,prln);                                               
          /*   end of data hex print    */                                      
       } 
        ifound=1;                                                                       
//    }         /*  end of print loop   ********/                                 
  } 
    printf ("here9\n");                                                                            
    printf("\n\n");          /*  space between records   */                     
/*  end of FMT */

/* not for mode  */
/* now report if FOUND or NOT FOUND */

    printf ("  ifound = %d  mfound= %d sfound = %d  ffound = %d qfound = %d\n",ifound,mfound,sfound,ffound,qfound);

    if (ffound == 1) {ifound = 0;}
    if (sfound == 1) {ifound = 0;}
    if (sfound == 0 && mfound == 0 && ifound == 0 && ffound == 1) {
       printf ("vedrdmp: %s FOUND\n",inpfmt);
        ffound=0;
        goto ENDLOOP;
    }

    if (qfound == 1) {
        printf (">>vedrdmp: %s FOUND\n",input);
//        qfound=0;
        if(recnum == seq_end) goto EXITS;
        goto ENDLOOP;
    }

    if (sfound == 1 ) {
        printf ("vedrdmp: %s FOUND\n",input6a);
//        sfound=0;
        goto ENDLOOP;
//    } else {
//        printf ("1) vedrdmp: %s NOT FOUND\n",input6a);
//        goto ENDLOOP;
    }

    if (mfound == 1) {
        printf ("::vedrdmp: %s FOUND\n",inpmode);
        ifound = 0;
        goto ENDLOOP;
    } else {
        printf ("2) vedrdmp: %s NOT FOUND\n",inpmode);
        goto ENDLOOP;
    }
        printf (" ifound = %d\n",ifound); 
        if (ifound == 1) {
            printf ("vedrdmp: %s FOUND\n",input); 
        } else { 
            printf ("3) vedrdmp: %s NOT FOUND\n",input);
        }                                                                          
    }

ENDLOOP:  ; 
    printf ("ENDLOOP\n");
//    if (ifound == 0) printf ("vedrdmp: %s NOT FOUND\n",input);                                                                    
  }         /*  end of main while loop  do forever  */                                    
                                                                                
EXITS:    ;      /*  end time or eof reached - get out   */                     
    printf ("EXITS (EOF)\n");
    if (ifound == 0) printf ("4) vedrdmp: %s NOT FOUND\n",input);
  fclose(edrin);                                                                
                                                                                
  printf("     JOB ENDS    *********  \n");                                     
  fclose(stdout);                                                               
  fclose(stderr);                                                               
                                                                                
 exit(0);     /*****  END OF JOB ___ EXIT TO SYSTEM  *****/                     
                                                                                
}                                                                               

                                                         
 /*****************************************************************/            
 /*********                                             ***********/            
 /*********   BEGINNING OF FUNCTIONS                    ***********/            
 /*********                                             ***********/            
 /*****************************************************************/            
                                                                                
 /*****************************************************************/            
                                                                                
 /****   FUNCTION  DUMPHEX                      ****/                           
 /****   print data record in hex, 32 char per line  ****/                      
                                                                                
/* void dumphex(char datax[],char ltyp[],short lcn,short prln)   */
 void dumphex(char ltyp[],short lcn,short prln)
/*
char	ltyp[];
short	lcn;
short	prln;
*/           
 { 
                                                                             
 int lc,pt,incnt;                                                               
                                                                                
 for (lc=lcn, incnt=0; lc<prln; lc++)     /* process data from  */              
   {                                                                            
   if((incnt%MAXCHAR)==0)      /* start to end.  print 32 char  */              
      {                                                                         
      printf("\n   %04X",lc);     /* per line(modifiable).      */              
     if(incnt != 0) printf("%6s",BLNK);                                         
      else          printf(" %-5s",ltyp);                                       
    }            /*   end of maxchar loop   */                                  
   if((incnt%2) ==0) printf("%s",BLNK);  /* space every 2 bytes */              
    printf("%02X",datax[lc]&0xff);                      
    incnt++;                                                                    
   }       /*  end of for(lc=                               */                  
  }       /*  end of dumphex ---- there is no return value  */                  
                                                                
 /*****************************************************************/            
                                                                                
 /****   FUNCTION  TYPPROC                      ****/                           
 /****   convert type=aaaa,bbb,ccc... to type numbers.  ****/                   
                                                                                
int typproc(input,lineln,type_tbl,do_type) 

char	input[];
int	lineln;
char	**type_tbl;
short	do_type[];
              
 { 
 int i,j,k; 
 char  in_type[5];                                                          
 int ntypes=0;       
  printf ("typproc:  Entry\n");

  for(i=0; i<18; i++)      /*  initialize do_type all off for type  */      
    do_type[i] = 99;
                                        
    j = 0; 
    printf ("lineln+1 = %d\n",lineln+1);                                                                                        

    if (lineln <= 4) {
        printf ("NO instrument TYPE given\n");
        printf (" valid types are: \n");
        printf ("IRIS, CRS, LECP, MAG, PLS, PPS, PRA, PWS,\n");
        printf ("UVS, ISR, DCOM , MN58, ENGS, ENGE, DCMS, ENGC, MN58, '****'\n");
        exit(1);

    }

  for(i=5; i<lineln+1; i++)      /*  scan input line for legal types */          
    {   
    printf ("loop\n");
         /***  comma separates entries, \0 marks end of line ***/               
    if(isalnum(input[i]) == 0)      /*  check for separator */              
       {                              
 
                                                                                
       if(j != 0)        /* check for blank entry   */                          
       {                                                                        
       memcpy(in_type,&input[i-j],j);                                       
       in_type[j] = '\0';

       if(memcmp(in_type,"ALL ",3) == 0)  {
          ntypes = 18;

          for(k=0; k<18; k++)    {
             do_type[k] = k;
              }
       } 
        printf ("typproc: in_type = %s\n",in_type); 
        if(ntypes == 18) goto ENDTYP;
        printf ("typproc: in_type = %s\n",in_type);                                                     
       for(k=0; k<18 ; k++)       /* is name legal?     */                      
          {                                                                     
                                                                                
          if (memcmp(in_type,type_tbl[k],strlen(in_type)) == 0)             
             {                                                                  
             do_type[ntypes] = k;                                           
             ntypes++;   


             j=0;        /*  NOTE:  J IS RESET HERE!!!!!!  */                   
             break;                                                             
             }    /*  end of if(memcmp ...  loop     */                         
                                                                                
          }          /*  end of for(k=0   ...  loop     */                      
          if(j!=0)                                                              
                 {                                                              
                 printf("  ILLEGAL TYPE VALUE 1 \n");    
                 ntypes = 0;             
                 break;                                                       
                 }                                                              
     }    /*   end of if j!= 0 test.    */                                      
                                                                                
    }         /*   end of isalnum loop      */                                  
    else        /*  no separator found yet, keep looking    */                  
       {                                                                        
       j++;                                                                     
       if(j>5)                                                                  
          {                                                                     
          printf("  ILLEGAL TYPE VALUE  \n");  
          ntypes = 0;
          break;

          }                                                                     
     }               /*  end of else  ...                  */       
  

 }                   /*  end of for(i=5 ...  loop          */  
ENDTYP:
    printf ("typproc: return ntypes = %d\n",ntypes);                 
 return ntypes;      /* return number of type names read   */                   
}                                                                               
                                                            
 /*****************************************************************/            
                                                                                
 /****   FUNCTION  SKIPROC                      ****/                           
 /****   convert type=aaaa,bbb,ccc... to type numbers.  ****/                   
                                                                                
int skiproc(input,lineln,type_tbl,do_type) 

char	input[];
int	lineln;
char	**type_tbl;
short	do_type[];
                   
 {                                                                              
 int i,j,k;                                                                     
 char  in_type[5];                                                          
 int ntypes=18;     
     printf ("skipproc: Entry\n");                                                       
  /****   NOTE:   'SKIP=ALL'  is NOT valid.    ****/                                                                                 
 for(i=0; i<18; i++)      /*  initialize do_type all on for skip  */            
          do_type[i] = i;                                                   
                                                                                
                                                                                
 j = 0;                                                                         
 for(i=5; i<lineln+1; i++)      /*  scan input line for legal types */          
    {                                                                           
         /***  comma separates entries, \0 marks end of line ***/               
    if(isalnum(input[i]) == 0)      /*  check for separator */              
       {                                                                        
                                                                                
       if(j != 0)        /* check for blank entry   */                          
       {                                                                        
       memcpy(in_type,&input[i-j],j);                                       
       in_type[j] = '\0';                                                   
       for(k=0; k<18 ; k++)       /* is name legal?     */                      
          {                                                                     
                                                                                
          if (memcmp(in_type,type_tbl[k],strlen(in_type)) == 0)             
             {                                                                  
             do_type[k] = 99;  /*  delete type here  */                     
             ntypes--;                                                          
             j=0;        /*  NOTE:  J IS RESET HERE!!!!!!  */                   
             break;                                                             
             }    /*  end of if(memcmp ...  loop     */                         
                                                                                
          }          /*  end of for(k=0   ...  loop     */                      
          if(j!=0)                                                              
                 {                                                              
                 printf("  ILLEGAL TYPE VALUE  \n");        
                 ntypes = 0;
                 break;
                 }                                                              
     }    /*   end of if j!= 0 test.    */                                      
                                                                                
    }         /*   end of isalnum loop      */                                  
    else        /*  no separator found yet, keep looking    */                  
       {                                                                        
       j++;                                                                     
       if(j>5)                                                                  
          {                                                                     
          printf("  ILLEGAL TYPE VALUE  \n");  
          ntypes = 0;                                 
          break;
          }                                                                     
     }               /*  end of else  ...                  */                   
 }                   /*  end of for(i=5 ...  loop          */                   
 return ntypes;      /* return number of type names read   */                   
}                                                                               
                                                            
 /*****************************************************************/            
                                                                                
 /****   FUNCTION  SEQPROC                      ****/                           
 /****   convert seq=nnn,mmmm from char to int  ****/                           
                         void seqproc(input,lineln,seq_bgn,seq_end)      
char	input[];
int 	lineln;
long	*seq_bgn;
long 	*seq_end;           
 {                                                                              
 int k,ik,ik1,ik2,iblk;
 char  inblk[2] [6];                                                    
 long bgnb,endb;                                                                

    printf ("seqproc: Entry input = %s, lineln = %d\n",input,lineln);                                                                                
 bgnb=0;                                                                        
 endb=0;                                                                        
 iblk = 0;                                                                      
 ik = 0;
 ik1=0;
 ik2 = 0;
    printf ("SEQ      lineln = %d\n",lineln); 
    printf ("input = %s\n",input);                                                                       
 for (k=4; k< lineln; k++)      /* scan line from = on to the end */          
   {                                                                            
   if(input[k] != ',')     /* comma separates begin from end  */            
     { 
     inblk[iblk] [ik] = input[k];  
//    printf ("input[%d] = %c\n",k,input[k]);
//    printf ("inblk[%d] [%d]=%c\n",iblk,ik,inblk[iblk][ik]);                //input[k]);                                 
     ik++;
     if (iblk == 0) { 
        ik1++; 
     } else {
         ik2++;                                                                    
     }
     }
   else          /*   comma found in line   */                                  
     {
     iblk = 1;   /*   switch to end portion of work area    */  
     ik = 0;
     }                                                                          
   }           /*   end of comma parser                   */ 
//    ikk = lineln - ik - 2;
//   printf ("seqproc: ik = %d   iblk = %d\n",ik,iblk);
//    printf ("seqproc:  inblk[%d][%d] = %s\n",iblk,ik,inblk[iblk,ik]);
//     printf ("seqproc: lineln-ik = %d\n",lineln-ik);
    printf ("seqproc: ik1 = %d  ik2 = %d\n",ik1,ik2);
  for (k=0; k < ik1; k++)   /*  translate from char to int */           
  {                                                                             
    bgnb = 10L*bgnb + (inblk[0] [k]-'0');                               
  }                                                                             
/*  par *mode_tbl[18] =      {"ENG  ","CR2  ","CR3  ","CR4  ",
                           "CR5  ","CR6  ","CR7  ","CR1  ",
                           "GS10A","GS3  ","GS7  ","GS6  ",
                           "OC2  ","OC1  ","CR5A ","GS10 ",
                           "GS8  ","UV5A "};
*/
    printf ("seqproc: bgnb = %d\n",bgnb);                                                                              
  for (k=0; k < ik2; k++)   /*  translate from char to int */                    
  {                                                                             
    endb = 10L*endb + (inblk[1] [k]-'0');                               
  }                                                                             
    printf ("seqproc: endb = %d\n",endb);                                                                                
  *seq_bgn = bgnb;                                                              
  *seq_end = endb;   
    printf ("SEQPROC end\n") ;                                                          
  }       /*  end of seq parser - there is no return value  */                  
                                                                
/*****************************************************************/            
                                                                                
 /****   FUNCTION  FDSPROC                      ****/                           
 /****   convert fds=aaaaa.bb.ccc,aaaaa.bb.ccc to internal form  ***/           
                                                                                
void fdsproc(input,lineln, fds_bgn,fds_end)                                     
                                   
char 	input[];
int	lineln;
char	fds_bgn[];
char	fds_end[];        
 {

/*  this transformation seems to convert FDS into SCET  */
/*  e.g., giving FDS range of  37559,39111              */
/*  yields fds_bgn = 224368  fds_end = 224376           */
/*  examining the header shows this to be true          */
/* therefore, Aug 13, 2020 rewrote to give 37559,3911   */
/*  as return values R- Ray Bambery                     */
/*                                                      */
 int k,ik,ib,el1,el2,m;                                                           
 char *kk;  
    char *kk1, *kk2;                                                                      
 unsigned char  inblk[2] [13];                                            
 unsigned char  fds0[6],fds1[13], fds2[13];
  unsigned short wkfds=0;                                                        

    printf ("fdsproc Entry\n");                                                                                
 ib= ik= el1= el2= j=0;                                                             
    printf("HERE lineln = %d\n",lineln);
    kk1 = strtok(input,"=,"); 
    printf ("HERE2  kk1 = %s\n");
    kk = strtok(NULL,"=,");
     printf ("   kk = %s\n",kk);
    kk2 = strtok(NULL,":,");
    printf ("   kk2 = %s\n",kk2);
    el1 = strlen(kk);
    el2 = strlen(kk2);
    goto ENDD;

ENDD:
//     memcpy(fds_bgn,kk,el1);
//     memcpy(fds_end,kk2,el2);
    strcpy(fds_bgn,kk);
    strcpy(fds_end,kk2);
    printf("fdsproc: fds_bgn = %s   fds_end = %s\n",fds_bgn,fds_end);                                   
}          /*  end of fdsproc........... */                                    
                                                             
 /*****************************************************************/            
                                                                                
 /****   FUNCTION  TIMEPROC                     ****/                           
 /****   convert scet/ert=yy:ddd:mm:ss.sss to internal form  ***/               
                                                                                
void timeproc()      /*char input[],int lineln)  */                     
                  /*        char tim_bgn[],char tim_end[]) */           
 {                                                                              
  int i,l,m,n;                                                                  
  int ik=0;                                                                     
  short year_wk, hour_wk, secs_wk, msec_wk;                                     
  char time_str[30];     /* output from dictmcnv   */                       
                                                                                
  i = m = n = 0;                                                                
   printf ("timeproc:  Entry tim_typ = %d\n",tim_typ);
    printf ("timeproc: input = %s\n",input); 
  input[lineln] = ',';         /*  force end of second entry  */            
  for(i=5; i<lineln+1; i++)      /*   scan input line for comma  */             
     {                                                                          
     if(input[i] != ',')                                                    
        {                                                                       
        tstrng[n] = input[i];                                           
        n++;                                                                    
        }                                                                       
     else         /* end of first field(comma found)    */                      
        {                                                                       
        tstrng[2] = tstrng[6] = ' ' ; /* clear colons */                
        printf ("tstrng = %s\n",tstrng);                                                                        
        for(ik=0; ik<16; ik++)  tmaray[ik] = 0;                             
                                                                                
        dictmcnv(tmaray,NULL,NULL,dptim,tstrng,"CHAR","HOY",                   
                 time_str,rtcode);                                              

        year_wk = tmaray[0] - 1900;                                         
        if(year_wk >= 100) year_wk-=100;                                        

        year_wk = year_wk<<8;           /* last 2 digits only */                

        hour_wk = tmaray[2];                /*  hr of yr */                 

        secs_wk = tmaray[3];                 /* sc of hr */                 

        msec_wk = tmaray[9];                 /*  msecs   */                 
    printf ("year_wk = %d  hour_wk = %d  secs_wk = %d msec_wk = %d\n",year_wk,hour_wk,secs_wk,msec_wk);
        if(m == 0)      /*   start time field     */                            
            {                                                                   
            memcpy(&tim_bgn[0],&year_wk,1);                                 
            memcpy(&tim_bgn[1],&hour_wk,2);                                 
            memcpy(&tim_bgn[3],&secs_wk,2);                                 
            memcpy(&tim_bgn[5],&msec_wk,2);                                 
            n = 0;
            printf ("tim_bgn = %s\n",tim_bgn);                                                              
            m++;                                                                
            }           /*   end of start time field   */                       
        else            /*   stop time field           */                       
            {                                                                   
            memcpy(&tim_end[0],&year_wk,1);                                 
            memcpy(&tim_end[1],&hour_wk,2);                                 
            memcpy(&tim_end[3],&secs_wk,2);                                 
            memcpy(&tim_end[5],&msec_wk,2);                                 
            printf ("tim_end = %s\n",tim_end);
            }           /*   end of stop time field   */                        
        }     /*    end of else... tstrng      */                               
        printf ("timeproc: time_beg =%s   time_end = %s\n",tim_bgn,tim_end);

   }         /*    end of for(i=5    ...     */
        printf ("\n");                                 
 }       /*    end of timeproc        */                                        
