/**************************************************************************
 * TITLE:  sfdu2dmp.c
 *
 * AUTHOR:      Ray Bambery
 * CREATED:     Sep 2, 2020
 * modified:    Mar 10, 2021 -  Ray Bambery
 *              code doc, add routines like vedrdmp.c
 *              Nov 10, 2021 - Ray Bambery
 *              removed some debugging statements
 *              Nov 11, 2021 - Ray Bambery
 *              Formatting for BLK parameter
 *              fixed ERT
 *              Dec 09, 2021 - Ray Bambery
 *              at char* value to return in twodate_diff
 *              detect if datafile is byte swapped, abort if it is
 *
 * COMPILER:    gcc version 4.5.2 (GCC)
 * OS:          Solaris 10.2   
 * COMPILER:    gcc version 7.1
 * OS:          Centos7
 *
 *************************************************************************
 * Program to dump the contents of a Voyager telemetry file in
 * SFDU2 format. For details see JPL Doc. 820-013 DSN External
 * Interface Specification (D-16765) "Telemetry Standard Format
 * Data Unit (SFDU) Interface" - Rev A - Aug 29, 2008
 *
 * Section 3 describes the Detailed Interface Description and physical
 * layout of the telemetry SFDU.
 *
 * Also the CHDO descriptions are detailed in JPL Doc 820-013-0172
 * Telecom CHDO Structures - Rev E - Jul 30, 2010
 */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>
#define MAXCHAR 32
#define BLNK   " "
#define HDRSIZE 20       /* in bytes */
#define CHDO_AGGREG_SIZE 4              //CHDO = Compressed Header Data Object
#define CHDO_PRIMARY_SIZE 8
#define CHDO_SECONDARY_SIZE 84
#define TELEMETRY_SIZE 242
#define SFDU2_RECORD_SIZE 358
#define  INTCNT     8           /* count of possible parameters in argv[2] */ 
#define  MAXLN     72           /* number of bytes in command line */      

unsigned char   input[MAXLN+1];
unsigned char   datax[1000];
char            ltyp[20];
short           recno;
int             ifound;                   /* flag to indicate whether requested record is found */
int             lineln; 
unsigned short int  tim_typ = 1;    /*  default to blk        */
char            ert_day;
char            tim_bgn[7];
char            tim_end[7];
char            ert_bgn[7];
char            ert_end[7];
char            cur_ert[7];
char            sct_bgn[7];
char            sct_end[7];
char            cur_sct[7];
long int        blk_bgn;
long int        blk_end;
double          dptim[4];
long int        tmaray[16];
char            tstrng[22];
char            rtcode[5];
char            magic[13] = "NJPL2I00C220";
char            rev_magic[3] = "JN";            /* byte swapped indicator */
char            ert_date[12]="\0";
char            ert_time[20];
int             in_scid;
int             scid;
unsigned long   recseqno;

int day, month, year;
static int days_in_month[] = { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
unsigned short day_counter;
long hr, min, sec;

/*  2nd parameter search_term options  */
 char *parmtbl[INTCNT] =
    {"BLK ","ERT ","SCET","FDS ","TYPE","SKIP","FMT ","SCID"};


//typedef struct sfoc_ert_struct{
unsigned short ert_ds1958;
unsigned long ert_msec;
//} SFOC_ERT_TIME;

/* prototypes */
void dumphex(char ltyp[],short lcn,short prln, short recno);
void timeproc(void);
void blkproc(char input[],int lineln,long *blk_bgn,long *blk_end);
char twodate_diff(int days);
int is_leap(int y);
void next_day(void);
void set_date(int d, int m, int y);
void skip_days(int x);
char print_date(void);
char ms2hmd(long msecs);


/*
 *  Quotes implies ASCII - numbers are binary unsigned integers
 *  In documentation a byte is called an "octet"
 ********************************************* 
 *  each sfdu2 file has 5 labels (headers) 
 *  SFDU Label - 20 bytes
 *  Aggregation CHDO - 4 bytes
 *  Primary CHDO - 8 bytes
 *  Secondary CHDO - 84 bytes
 *  Telemetry Data CHDO - (N-116) bytes
 *
 * *******************************************
 *  SFDU label has 20 bytes
 *  0-3   - Control Authority ID = "NJPL" - 4 bytes
 *  4     - Version ID = "2" - 1 byte
 *  5     - Class ID = "I" - 1 byte
 *  6,7   - Spare = "00" - 2 bytes
 *  8-11  - Data Description ID = "0800" - 4 bytes
 *  12-19 - Length Attribute = (variable - length, in binary, of the remainder, starting with byte 20 of this sfdu) - 8 bytes
 * ******************************************
 * Aggregation CHDO has 4 bytes
 * 0,1    - Type Attribute = 1 (Aggregation CHDO) - 2 bytes
 * 2,3    - Length Attribute = 92 (length of Primary and Secondary CHDOs) - 2 bytes
 * ******************************************
 * Primary CHDO has 8 bytes
 * 0,1   - Type Attribute = 2 (Primary CHDO) - 2 bytes
 * 2-9   - Length Attribute = 4 (4 16-bit words) - 8 bytes
 * 10    - Major Class = 1 - 1 byte
 * 11    - Minor Class = (variable 7-17) - 1 byte
 * 12    - Mission ID = (variable 0-255) - 1 byte
 * 13    - Format Code = 0  (DSS formatted SFDU) - 1 byte
 * ******************************************
 * Secondary CHDO has 84 bytes
 * 0,1   - Type Attribute = 78 (telemetry secondary SFDU) - 2 bytes
 * 2,3   - Length Attribute = 80 (bytes in this CDHO not including Type Attribute) - 2 bytes
 * 4     - Originator ID = 48 (within DSN) - 1 byte
 * 5     - Last Modifier ID = 48 (within DSN) -  byte
 * 6     - Reserved - 1 byte
 * 7     - Spacecraft ID (variable) - 1 byte  (23x or 35d =VGR-2)
 * 8,9   - Pass Number (0-9999) 2 bytes
 * 10    - Data Source ID (0-255) - 1 byte
 * 11    - Arrayed Stations (variable) - 1 byte
 * 12,13 - Flags (variable) - 2 bytes
 * 14-21 -       - Earth Received Time - 8 bytes
 * 14-19 -  SERT_DS1958 = days since 1 Jan 1958
 * 22,21 -  SERT_MSEC = ms of day
 * 22-25 - Record Sequence Number - 4 bytes
 * 26    - U/L Band - 1 byte
 * 27    - D/L Band - 1 byte
 * 28    - PDX mode - 1 byte 
 * 29    - D/L Source - 1 byte
 * 30    - Virtual Stream ID - 1 byte
 * 31    - Virtual Channel ID - 1 byte
 * 32,33 - Telemetry Lock Status - 2 bytes
 * 34-37 - Number of Bits = (Number of Bits contained in Telemetry Data CHDO below) - 4 bytes
 * 38-41 - Measured Bit Rate - 4 bytes
 * 42-45 - System Noise Temperature - 4 bytes
 * 46-49 - Symbol SNR - 4 bytes
 * 50-53 - Receiver Signal Level - 4 bytes
 * 54    - Acquisition BET - 1 byte
 * 55    - Maintenance BET - 1 byte
 * 56    - Verify Count - 1 byte
 * 57    - Flywheel Count - 1 byte
 * 58    - Frame Sync Mode Flags - 1 byte
 * 59    - Sync Status| Bit slip - 1 byte
 * 60    - FS Error Count - 1 byte
 * 61    - FS Buffer Size - 1 byte
 * 62    - RS Decoder Status - 1 byte
 * 63    - RS Symbol Error Count - 1 byte
 * 64    - Turbo Decoder Status - 1 byte
 * 65    - Processor number - 1 byte
 * 66    - Number of Iterations - 1 byte
 * 67    - Reserve - 1 byte
 * 68    - Code Rate Numerator - 1 byte
 * 69    - Code Rate Denominator - 1 byte
 * 70,71 - Frame Data Bits - 2 bytes
 * 72,73 - Decoder Confidence - 2 bytes
 * 74,75 - Telemetry Equipment ID - 2 bytes
 * 76,77 - Telemetry Software ID - 2 bytes
 * 78-83 - Reserved - 6 bytes
 * **********************************************************
 * Telemetry Data CHDO has N-117 bytes
 * Type Attribute = 10 (CHDO contains binary data only) - 2 bytes 
 * Length Attribute = N-117 - 2 bytes
 * Received Telemetry Bits (Can contain telemetry from S/C or
 *           undecoded signals) - see either Turbo Decoder Status
 *           Byte 64 (Bit 8) of Secondary CHDO or
 *           Minor Data Class in Primary CHDO Byte 5 
 *           (7-11 non-turbo encoded), (12-16 turbo encoded data)
 *           (17 for no processing done) 
*/
/**************************************************************************/                                         
int  main (argc, argv)
int   argc;
char *argv[];

{
int             inlineln,tsize,type,recnum,sfduerr;
unsigned long int  ms_ert,    days_ert;
unsigned int    datax14;
char            file_name[104];
char            size_str[3];
char            chr;
FILE *          sfduin;
int             xlineln = 0;
short           lcn;
short           prln;
int             loop,loop_index,inc_zero,i;
int             data_offset;
int             ill;
unsigned char input6[5],input6a[5]="\0",inpmode[5]="\0";

//    printf ("sfdu2dmp: argc = %d\n",argc);

    if (argc < 2 ){
        printf("usage:  sfdu2dmp infile search_term \n");
        exit(0);
    }  

    if (argc >2 ) {
        strcpy (file_name,argv[1]);
        inlineln = strlen(file_name);
//       printf ("argv[1] = %s\n",argv[1]);
//       printf ("argv[2] = %s\n",argv[2]);
//       printf ("inlineln = %d\n",inlineln);
    } else if  (argc < 2 ) {
        printf ("argc < 2\n");
    } else {
//        printf ("argc == 2\n");
//        inlineln = getline(xlineln);
         strcpy (file_name,argv[1]);
        inlineln = strlen(file_name);
        printf ("sfdu2dmp: file_name = %s\n",file_name);
    } 
    sfduin = fopen(file_name,"rb");

     if (sfduin == NULL)
     {
      printf(" UNABLE TO OPEN FILE %s FOR INPUT.\n",file_name);
       exit(8);
     }    
    if (argc < 3) {
        printf ("Missing 2nd parameter - search_term\n");
        printf (" Search terms are: SC_ID=, ERT, and BLK=start,end\n");
        exit(0);
    } else {
        /* convert to UC */
        lineln = strlen(argv[2]);
//        printf ("linel n= %d\n",lineln);
        strcpy(&input[0],argv[2]);
        for (ill=0;ill<lineln;ill++)  { 
            chr = input[ill]; 
            input[ill] = toupper(input[ill]);
//            printf("%c (%c)\n", (chr), input[ill]); 
         } 

     }

//   input =  "BLK ","ERT ","SCET","FDS ","TYPE","SKIP",
//            "FMT ","SCID"                      
    for (i = 0; i < INTCNT ; i++)
    {
    if (memcmp(input,parmtbl[i],3) == 0)
         {
           break;
         }
    }  
  switch(i)
     {
      case 0:        /*  blk=   */
        printf ("sfdu2dmp:  case 0 - BLK  lineln = %d\n",lineln);
//        tim_typ=1;
        blkproc(input,lineln,&blk_bgn,&blk_end);
//        xlineln = -1;
        printf ("recnum   = %d  blk_bgn = %d blk_end = %d\n",recnum,blk_bgn,blk_end);
        if(recnum > blk_end) goto EXITS;
        if(recnum  <=  blk_bgn) goto ENDLOOP;

        break;

      case 1:      /*  ert=   */
        printf ("sfdu2dmp: case 1 - ERT  \n");
//        printf ("         input = %s\n",input);
      break;

      case 7:      /*  scid=  */
        printf ("sfdu2dmp: case 7 - SCID=\n");
            strncpy (input6,&input[5],3);
            printf ("input6 = %s\n",input6);
            if (strcmp(input6,"1") == 0) {strcpy(input6a,"31");};
            if (strcmp(input6,"2") == 0) {strcpy(input6a,"32");};
            printf ("input6a = %s\n",input6a);
        in_scid = atoi(input6a);
      break;

      default:      /*  undefined input   */
        printf(" UNDEFINED INPUT PARAMETER -- %s \n",input);
        goto EXITS;
        break;

   }           /* end of switch.                  */
//
    loop = 0;
    loop_index = loop * SFDU2_RECORD_SIZE;  /* size of data in one record of sfdu */ 
    recnum = 0;

/******************* FOREVER LOOP **********************/
    for (;;) {     /*do forever */
    loop++;
    recnum++;
    printf ("\n **********************************************************************\n");
    printf ("           LOOP %d: fread recnum  = %d\n",loop,recnum);
    printf (" read sfdu2 HEADER\n");
    if (strncmp(input,"BLK",3) == 0) {
        printf ("blk_bgn = %d   blk_end = %d\n",blk_bgn,blk_end); 
        if(recnum > blk_end) goto EXITS;
        if(recnum  <  blk_bgn) goto ENDLOOP;
    }
/****** SFDU2 Header - 20 bytes  *********************************************/

    sfduerr=fread(datax,1,HDRSIZE,sfduin);    /* read 20 bytes */
    if(sfduerr != 0) /* READ SFDU HEADER */
    {
//        printf("\n");
//        printf("  %d RECORDS READ.\n ",recnum);

    } else {
        if(feof(sfduin)) {   //if file pointer reach to end it will break{
            printf ("sfdu2dmp: reached end of file\n");
            exit(0);
        } else {
            printf ("sfdudmp: ERROR on fread of HEADER\n");
            exit(1);
        }
    }   
    if ((strncmp(rev_magic,datax,2)) != 0) {
        printf ("\nDATA SET is Byte reversed\n");
        printf ("PLEASE run byteswap program\n");
        exit(0);
    }
    if ((strncmp(magic,datax,12)) != 0) {
        printf ("file %s is not an sfdu type 2 file\n",file_name);
        exit(0);
    } 
    if (strncmp(input,"BLK",3) == 0) {
        tsize = 256*datax[18]+datax[19];
        printf ("sfdu2dmp: HEADER = %d bytes\n",HDRSIZE);
        printf ("   0000       %c%c%c%c %c%c%c%c %c%c%c%c\n",datax[0],datax[1],datax[2],datax[3],datax[4],datax[5],datax[6],
            datax[7],datax[8],datax[9],datax[10],datax[11]);
        printf ("block length (after SFDU2 HDR) = %d bytes\n",tsize);
        lcn = 0;
        prln = HDRSIZE;
        recno = 0;          /* recno to be printed in dumphex */
        strcpy(ltyp,"SFDU HDR");
        dumphex ("HDR",lcn,prln,recno);     /* dump HDR to screen */
        printf ("\n");
    }
/*********  SFDU2   CHDO_AGGREG_HEADER  4 bytes  ***********************************/
//      printf ("\n===== fread 2\n");
    sfduerr=fread(&datax[HDRSIZE],1,CHDO_AGGREG_SIZE,sfduin);   /* read 4 bytes  after 20 */
    if(sfduerr != 0) // READ CHDO_AGGREG_HDR //
        {
//            printf("\n");
//            printf("  %d RECORDS READ.\n ",recnum);

        } else {
            if(feof(sfduin)) {   //if file pointer reach to end it will break 
                printf ("sfdu2dmp: reached end of file");
                exit(0);
            } else {
                printf ("sfdudmp: ERROR on fread of CHDO_AGGR_HDR\n");
                exit(1);
            }
        }   
    if (strncmp(input,"BLK",3) == 0) {
        type = 256*datax[prln+0]+datax[prln+1];
        tsize = 256*datax[prln+2]+datax[prln+3]&0xff;
        printf (" sfdu2dmp: CHDO_AGGREG_HDR = %d\n",CHDO_AGGREG_SIZE);
        printf ("  type attribute =  %d,  length = %d\n",type,tsize);
//      printf (" %X %X  %X %X\n",datax[0],datax[1],datax[2],datax[3]&0xff);
//      recno = 24; /* 2Cx */ 
        lcn = lcn + HDRSIZE;
        prln = CHDO_AGGREG_SIZE;
        recno = lcn;
        strcpy(ltyp,"CHDO_AGGREG_HDR");
        dumphex (ltyp,lcn,prln,recno);         /* dump CHDO_AGGREG_HDR to secreen */
    }
/*********  SFDU2 PRIMARY_HDR  - 4 bytes ***********************************************/
//      printf ("\n====== fread  3\n");
    sfduerr=fread(&datax[prln+lcn],1,CHDO_PRIMARY_SIZE,sfduin);
    if(sfduerr != 0) /* READ CHDO_PRIMARY_HDR */
    {
         printf("\n");
//         printf("  %d RECORDS READ.\n ",recnum);

    } else {
         printf ("sfdu2dmp: ERROR on fread of CHDO_PRIMARY_HDR\n");
         exit(1);
    }    
    if (strncmp(input,"BLK",3) == 0) {
        type = 256*datax[prln+lcn+0]+datax[prln+lcn+1];
        tsize = 256*datax[prln+lcn+2]+datax[prln+lcn+3];
        printf ("sfdu2dmp: CHDO_PRIMARY_HDR = %d\n",CHDO_PRIMARY_SIZE);
        printf ("  type attribute =  %d,  length = %d\n",type,tsize);
//      recno = 28; /* 30x */
        lcn = lcn + CHDO_AGGREG_SIZE;
        prln = CHDO_PRIMARY_SIZE;
        recno = lcn;
        strcpy(ltyp,"CHDO_PRIMARY_HDR");
//      printf("ltyp = %s\n",ltyp);

        dumphex ("CHDO PRIMARY_HDR",lcn,prln,recno);      /* dump CHDO_PRIMARY_HDR ro screen */  
    }
/*****************  SFDU2 SECONDARY_HDR - 84 bytes ***********************************/
//     printf ("\n========== fread 4\n");
    sfduerr=fread(&datax[prln+lcn],1,CHDO_SECONDARY_SIZE,sfduin);
    if(sfduerr != 0) /* READ CHDO SECONDARY HEADER */
    {
//        printf("\n");
//        printf("  %d RECORDS READ.\n ",recnum);

    } else {
        printf ("sfdu2dmp: ERROR on fread of CHDO_SECONDARY_HDR\n");
        exit(1);
    }   
    if (strncmp(input,"BLK",3) == 0) {
        type = 256*datax[prln+lcn+0]+datax[prln+lcn+1];
        tsize = 256*datax[prln+lcn+2]+datax[prln+lcn+3];
        printf ("sfdu2dmp: CHDO_SECONDARY_HDR = %d\n",CHDO_SECONDARY_SIZE);
        printf ("  type attribute =  %d,  length = %d\n",type,tsize);
//      recno = 56;  /*  40x  */
        lcn = lcn + CHDO_PRIMARY_SIZE;
        prln = CHDO_SECONDARY_SIZE;
        recno = lcn;
        strcpy(ltyp,"CHDO_SECONDARY_HDR");
        dumphex (ltyp,lcn,prln,recno);     /* dump CHDO_SECONDARY_HDR to screen */
        printf ("\n");
//      for (i=0;i<85;i++) {
//        printf ("   datax[%d] = %X\n",i,datax[i]&0xff);
//      }
    } /* end of if (strncmp(input,"BLK",3) == 0  )*/ 


// doy 350 - Dec 15, 2020 - days since 1 Jan 1958 = 22,994
// datax[30] is beginning of SECONDARY_HEADER
    if (strcmp(input,"ERT") == 0) {
//        printf ("desire SERT\n");
        ert_ds1958 = 0;
        ert_ds1958 =  datax[42] << 8;
        ert_ds1958 +=  datax[43];
        ert_msec += datax[44] << 24;
        ert_msec += datax[45] << 16;
        ert_msec += datax[46] << 8;
        ert_msec += datax[47];
        printf ("sfdu2dmp:      ert_ds1958 = %d   ert_msec = %d\n",ert_ds1958,ert_msec);
        *ert_date = twodate_diff(ert_ds1958);
//        printt ("sfdu2dmp:  ert_date = %s\n",ert_date);
        *ert_time = ms2hmd(ert_msec);
        printf ("sfdu2dmp:      ert_time = %s\n",ert_time);

    } 
    if (strncmp(input,"SCID",4) == 0) {
//        printf ("desire SCID\n");
        scid = datax[7];
//        printf ("scid = %d   in_scid = %d\n",scid,in_scid);
        if (in_scid == 31) {in_scid = 63;};
        if (in_scid == 32) {in_scid = 35;};    
//        printf ("sfdu2dmp:  SC_ID = %d\n",scid);
//        printf ("sfdu2dmp:  input SCID = %d\n",in_scid);
        if (scid == in_scid) {
            printf ("SC_ID = %s  FOUND\n",input6a);
        } else {
            printf ("SC_ID = %s  NOT FOUND\n",input6a);
        }
    }
    if (strncmp(input,"BLK",3) == 0) { 
        recseqno = 0L;
        recseqno += datax[22] << 24;
        recseqno += datax[23] << 16;
        recseqno += datax[24] << 8;
        recseqno += datax[25];
//      printf ("sfdu2dmp: recseqno = %ld\n",recseqno);

    }
/***********  SFDU2 TELEMETRY DATA ***********************************************/
//     printf ("\n======== fread 5\n");
     sfduerr=fread(&datax[prln+lcn],1,TELEMETRY_SIZE,sfduin);
     if(sfduerr != 0) /* READ SFDU HEADER */
     {
        printf("\n");
        printf("  %d RECORDS READ.\n ",recnum);

     } else {
        printf ("ERROR on fread \n");
        exit(1);
     }  
     if (strncmp(input,"BLK",3) == 0) {
        type = 256*datax[prln+lcn+0]+datax[prln+lcn+1];
        tsize = 256*datax[prln+lcn+2]+datax[prln+lcn+3];

        printf ("sfdu2dmp: TELEMETRY DATA = %d\n",TELEMETRY_SIZE);
//      printf  (" ONLY first 23 bytes of 242 listed\n");
        printf ("  type attribute =  %d,  length = %d\n",type,tsize);
//      recno = 272;
        lcn = lcn + CHDO_SECONDARY_SIZE;
        prln = TELEMETRY_SIZE;
        recno = lcn;
        strcpy(ltyp,"TELEMETRY");
        dumphex (ltyp,lcn,prln,recno);         /* dump SFDU2 TELEMTRY DATA to screen */
/*
        printf ("\n");
        for (i=0;i<359;i+=2){
            ert_ds1958 = 0;
            ert_ds1958 = datax[i+0] << 8;
            ert_ds1958 += datax[i+1];
            printf ("   datax[i=%d] = %X %X  ert_ds1958 = %d\n",i,
            datax[i+0]&0xff,datax[i+1]&0xff,ert_ds1958);
        }
        printf ("\n");
        for (i=0;i<359;i+=2){
          ert_ds1958 = 0;
          ert_ds1958 = datax[i+1] << 8;
          ert_ds1958 += datax[i+2];
            printf ("   datax[i=%d] = %X %X  ert_ds1958 = %d\n",i,
            datax[i+1]&0xff,datax[i+2]&0xff,ert_ds1958);
        }
*/
// find end of data in telemetry record
        printf ("\n   END OF DATA = %d bytes\n",SFDU2_RECORD_SIZE);
        inc_zero=0;
        for (i=SFDU2_RECORD_SIZE;i>=0;--i) {
            if (datax[i] == 0) {
//               printf ("EOD i = %d\n",i);
                inc_zero++;
            } else { 
//              printf ("HDRSIZE + CHDO_AGGREG_SIZE + CHDO_PRIMARY_SIZE + CHDO_SECONDARY_SIZE = %d\n",
//                HDRSIZE + CHDO_AGGREG_SIZE + CHDO_PRIMARY_SIZE + CHDO_SECONDARY_SIZE);
                data_offset = SFDU2_RECORD_SIZE - (HDRSIZE + CHDO_AGGREG_SIZE + CHDO_PRIMARY_SIZE + CHDO_SECONDARY_SIZE)
                 - (SFDU2_RECORD_SIZE - i);
                printf ("sfdu2dmp: End of data at offset %d in telemetry record = %d\n",
                data_offset,loop);
                break;
            }
        }  
    }

ENDLOOP:    ;
    }  //  end for(;;)
/*
    printf ("\n======== fread 6\n");
    sfduerr=fread(datax,1,HDRSIZE,sfduin);
    if(sfduerr != 0) // READ SFDU HEADER //
    {
        recnum++;
        printf("\n");
        printf("  %d RECORDS READ.\n ",recnum);

    } else {
        printf ("ERROR on fread \n");
        exit(1);
    }
    printf (" *********************************** SECOND **************************************\n");
    tsize = 256*datax[18]+datax[19];
    printf ("sfdu2dmp: HEADER = %d\n",HDRSIZE);
    printf ("%c%c%c%c%c%c%c%c%c%c%c%c\n",datax[0],datax[1],datax[2],datax[3],datax[4],datax[5],datax[6],
        datax[7],datax[8],datax[9],datax[10],datax[11]);
    printf ("%xx %xx %xx %xx %xx %xx  %d\n",datax[12],datax[13],datax[14],datax[15],datax[16],datax[17],tsize);
 */ 

EXITS:    ;      /*  end time or eof reached - get out   */
    printf ("EXITS (EOF)\n");
    if (ifound == 0) printf ("vedrdmp: %s NOT FOUND\n",input);
    fclose(sfduin);
     exit(0);     /*****  END OF JOB ___ EXIT TO SYSTEM  *****/

  
}
/*****************************************************************/
 /*********                                             ***********/
 /*********   BEGINNING OF FUNCTIONS                    ***********/
 /*********                                             ***********/
 /*****************************************************************/

 /*****************************************************************/

 /****   FUNCTION  DUMPHEX                      ****/
 /****   print data record in hex, 32 char per line  ****/

/* void dumphex(char datax[],char ltyp[],short lcn,short prln)   */
 void dumphex(char ltyp[],short lcn,short prln,short recno)
/*
 * char    ltyp[];
 * short   lcn;
 * short   prln;
 * */
 {

 int lc,pt,incnt;
 size_t len;

//  printf ("dumphex: ltyp = %s\n",ltyp);
//printf ("sizeof ltyp = %d\n",sizeof(ltyp) / sizeof(char));
//  printf ("dumphex lcn = %d  prln = %d  prln+lcn = %d\n",lcn,prln,prln+lcn);
  for (lc=lcn, incnt=0; lc<prln+lcn; lc++)     /* process data from  */
  {
   if((incnt%MAXCHAR)==0)      /* start to end.  print 32 char  */
      {
      if (incnt == 0) {
          printf(" %-5s",ltyp);
          printf("\n   %04X",lc);     /* per line(modifiable).      */
          printf("%6s",BLNK);
      } else {
          printf("\n   %04X",lc);     /* per line(modifiable).      */
          printf("%6s",BLNK);
      } 
   }            /*   end of maxchar loop   */
   if((incnt%2) ==0) {
        printf("%s",BLNK);  /* space every 2 bytes */
   }
    printf("%02X",datax[lc]&0xff);
    incnt++;
   }       /*  end of for(lc=                               */
  }       /*  end of dumphex ---- there is no return value  */

 /*****************************************************************/

 /****   FUNCTION  BLKPROC                      ****/
 /****   convert blk=nnn,mmmm from char to int  ****/
                         void blkproc(input,lineln,blk_bgn,blk_end)
char    input[];
int     lineln;
long    *blk_bgn;
long    *blk_end;
 {
 int k,ik,ik1,ik2,iblk;
 char  inblk[2] [6];
 long bgnb,endb;

//    printf ("blkproc: Entry input = %s, lineln = %d\n",input,lineln);
 bgnb=0;
 endb=0;
 iblk = 0;
 ik = 0;
 ik1 = 0;
 ik2 = 0;
//    printf ("BLK     lineln = %d\n",lineln);
//    printf ("input = %s\n",input);

 for (k=4; k< lineln; k++)      /* scan line from = on to the end */
   {
   if(input[k] != ',')     /* comma separates begin from end  */
     {
     inblk[iblk] [ik] = input[k];
     ik++;
     if (iblk == 0) {
        ik1++;
     } else {
         ik2++;
     }
     }
   else          /*   comma found in line   */
     {
     iblk = 1;   /*   switch to end portion of work area    */
     ik   = 0;
     }
   }           /*   end of comma parser                   */
//   printf ("blkproc: ik1 = %d   ik2 = %d\n",ik1,ik2);

  for (k=0; k < ik1; k++)   /*  translate from char to int */
  {
    bgnb = 10L*bgnb + (inblk[0] [k]-'0');
  }
//  printf ("blkproc: bgnb = %d\n",bgnb);
  for (k=0; k < ik2; k++)   /*  translate from char to int */
  {
    endb = 10L*endb + (inblk[1] [k]-'0');
  }
//    printf ("blkproc: endb = %d\n",endb);
  *blk_bgn = bgnb;
  *blk_end = endb;
  }       /*  end of blk parser - there is no return value  */
/************************************************************/
char twodate_diff (int days) 
{
/* calc ERT from Jan 1, 1958 by adding number of elasped days */
    set_date(1, 1, 1958);  //ds1958

    skip_days(days);
    day_counter = 0;
    /* after this call next_day each day */

    *ert_date = print_date();
    printf ("twodate_diff:  ert_date = %s\n",ert_date);
    return *ert_date;
}
/****************************************************************/
int is_leap(int y) {
    return ((y % 4 == 0 && y % 100 != 0) || y % 400 == 0);
}
/****************************************************************/
void next_day(void)
{
    day += 1;
//    printf ("%d  next_day W:  day = %d month = %d year = %d\n",day_counter,day,month,year); 
    day_counter++;
    if (day > days_in_month[month]) {
        day = 1;
        month += 1;
//        printf ("%d  next_day X:  day = %d month = %d year = %d\n",day_counter,day,month,year);
        if (month > 12) {
//            printf ("%d  next_day Y:  day = %d month = %d year = %d\n",day_counter,day,month,year);
            month = 1;
            year += 1;
//            printf ("%d  next_day Z:  day = %d month = %d year = %d\n",day_counter,day,month,year);
            if (is_leap(year)) {
                days_in_month[2] = 29;
            } else {
                days_in_month[2] = 28;
            }
//            printf (" %d next_day 0:  day = %d month = %d year = %d\n",day_counter, day,month,year);
        }
//        printf ("%d next_day 1:  day = %d month = %d year = %d\n",day_counter,day,month,year);
    }
//    printf ("%d next_day 2:  day = %d month = %d year = %d\n",day_counter,day,month,year);
}
/****************************************************************/
void set_date(int d, int m, int y) 
{
    int days_in_month[2];

    m < 1 ? m = 1 : 0;
    m > 12 ? m = 12 : 0;
    d < 1 ? d = 1 : 0;
    d > days_in_month[m] ? d = days_in_month[m] : 0;
    if (is_leap(y)){
        days_in_month[2] = 29;
    } else {
        days_in_month[2] = 28;
    }
    day = d;
    month = m;
    year = y;
//    printf ("set_date:  day=%d month=%d year=%d\n",day,month,year);
}
/***************************************************************/
void skip_days(int x)
{
    
    int i;
//    printf ("skip_days:  x = %d\n",x);
    day = 0;
    for (i=0;i<x;i++) next_day();
}
/**************************************************************/
char print_date(void)
{
//    int month,day,year;

//    sprintf (ert_date,"day: %d month: %d year: %d\n", day, month, year);
    sprintf (ert_date,"%d-%d-%d",month,day,year);
    char *ed = (char *)&ert_date;
//    printf ("print_date:  ert_date = %s\n",ert_date);
    return *ed;
}
/******************************************************************/
char ms2hmd(long msecs)
{
//long hr, min, sec;
/* get ert_day time of day from number of millisecs past midnight */
//3600000 milliseconds in an hour
    hr = msecs / 3600000;
    msecs = msecs - 3600000 * hr;
//60000 milliseconds in a minute
    min = msecs / 60000;
    msecs = msecs - 60000 * min;
//
////1000 milliseconds in a second
    sec = msecs / 1000;
    msecs = msecs - 1000 * sec;

    if (min < 10 && sec < 10) {
            sprintf (ert_time,"%d:0%d:0%d.%d",hr,min,sec,msecs);
    } else if (min < 10) { 
            sprintf (ert_time,"%d:0%d:%d.%d",hr,min,sec,msecs);
    } else if (sec < 10) {
            sprintf (ert_time,"%d:%d:0%d.%d",hr,min,sec,msecs);
    } else {
            sprintf (ert_time,"%d:%d:%d.%d",hr,min,sec,msecs);
    }
    char *et = (char *)&ert_time;
//    printf ("ms2hdm: ert_time = %s\n",ert_time);

    return *et;

}
