/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*****************************************************************
 * TITLE: diertscecnv.c
 *
 * AUTHOR:  Unknown
 *          Aug 31, 1994
 *
 * MODIFIED:    Ray Bambery
 *          Aug 24, 2020 -  removed label after #ENDIF Header 
 *                          added prototypes to fix error:
 *                          error: static declaration of 'ahf_open' follows non-static declaration
 ***************************************************************
** MANUAL
**	DIERTSCECNV 3x "November 12, 1993"
**
** NAME
**	diertscecnv - convert from ert to scet or visa versa for
**	various missions.
**
** SYNOPSIS
**
**	  #include   <glbltme.h>
**
**	  void diertscecnv(ertsceary,msn,dptimes,sstrng,type_in,rtcode);
**
**	  long int *ertsceary;
**	  char     *msn;
**	  DBL      *dptimes;
**	  char     *sstrng;
**	  char     *type_in;
**	  char     *rtcode;
**
** PARAMETERS AT A GLANCE
**
**	  1. ertsceary - long int ertsceary[9] - an array of times  
**	     and spacecraft clocks.
**	  2. msn - char[4] - 3 character  string abbreviations for 
**	     missions, GLL, VGR, ULS etc.
**	  3. dptimes -  DBL[3] - double precisions times from epoch.
**	  4. sstrng - char[variable length] -  string input for  
**	     times, and/or clocks.
**	  5. type_in - char[variable length] - string of type of  
**	     time input to 'diertscecnv'.
**	  6. rtcode - char[5] - return code, 'NORM' for ok.
**	
** PARAMETERS IN DETAIL
**
**	  1. ertsceary - an array of times and spacecraft clocks.
**	     1. ertsceary[0] - SCET - seconds past epoch 
**	        (usually 1950)
**	     2. ertsceary[1] - SCET - fractional seconds past epoch.
**	     3. ertsceary[2] - SCET - milliseconds past epoch.
**	     4. ertsceary[3] - ERT  - seconds past epoch 
**	        (usually 1950)
**	     5. ertsceary[4] - ERT  - fractional seconds past epoch.
**	     6. ertsceary[5] - ERT  - milliseconds past epoch.
**	     7. ertsceary[6] - OWLT - One way light in seconds.
**	     8. ertsceary[7] - OWLT - fractional seconds.
**	     9. ertsceary[8] - OWLT - milliseconds.
**	              66 2/3 milliseconds.
**	           6 bits, line count, 10 bits.
**	   2. msn - ascii string containing three character
**	      abbreviation of missions names. This must be the
**	      first think input for open etc. The abbreviations are:
**	      1. VGR - voyager.
**	      2. ULS - Ulysses.
**	      3. GLL - Galileo.
**	   3. type_in - Not NULL or empty. If 'type_in' is
**	      1. ERT - Input in ertsceary[3], ertsceary[4]
**	      2. ERTM - input in ertsceary[3], ertsceary[5]
**	      3. DPERT -double precision, input in dptimes[1].
**	      4. SCET - input in ertsceary[0], ertsceary[1].
**	      5. SCETM - input in ertsceary[0], ertsceary[2].
**	      6. DPSCET - double precsion input in dptimes[0].
**	      7. OPEN - Open and initialize OWLT file.
**	         file name is input in 'sstrng'.
**	      8. CLOSE - Close OWLT file and reset parameters.
**	      9. CHAR - the input ascii and is in 'sstrng'.
**         4. dptimes - If type_in is 'DPERT','DPSCET', or 'DPSCLK',
**            dptimes contains.
**	      1. dptimes[0] - SCET - 'type_in' = 'DPSCET'.
**	      2. dptimes[1] - ERT - 'type_in' = 'DPERT'.
**	   5. sstrng - ascii string
**	      1. For 'type_in' = 'CHAR'.
**	         The allowed inputs are
**	         'ERT timestring'
**	         'SCET timestring'
**	         where 'timestring' is the legal inputs for
**	         'dictmcnv'
**	      2. For 'type_in' = 'OPEN'.
**	         'sstrng' contain the double precision file
**	         name of the OWLT file to open.
**	  6. rtcode - return code, 'NORM' for ok.
**	     
**	      
** GLOSSARY
**
**	 SCET - Spacecraft event time.
**	 ERT - Earth recieved time.
**	 OWLT - One way light time.
**	 GLL - Galileo.
**	 ULS - Ulysses.
**	 VGR - Voyager.
**	 MGN - Magellon.
**
** EXTERNS
**
**	  1. fdowlt - SFFOPDEF(fdowlt) - file descriptor for 
**	     OWLT file.
**	  2. owloffset - long int - Offset to start of data.
**	  3. msn - char[4] - 3 character mission id i.e. ULS, GLL, 
**	     VGR etc.
**	  4. msn_case_nmbr - int - for computing different 
**	     SCLK/SCETs. (ULS_INDX,GLL_INDX,VGR_INDX)
**	  5. OWLEPS -struct CLOWEP_STRUCT - epoch information
**	     used in creating double precision SCET and ERT 
**	     seconds past that epoch.
**
**
** STRUCTURES
**
**	  struct epoch_struct {
**	        char    char_epoch[24];
**	        long    int   epoch_tarray[SIZE_OF_TARRAY];
**	        DBL     epoch_dptimes[SIZE_OF_DPS];
**	        char    epoch_mon[4];
**	        char    epoch_dow[4];
**	        char    new_epoch[4];
**	        BOOL    find_epoch_times;
**	  };
**	  typedef struct epoch_struct EPOCH_STRUCT;
**	
**	  struct clowep_struct {	
**	        long    int    epoch_indx;
**	        char    epoch_type[12];
**	        char    epoch[24];
**	        EPOCH_STRUCT   clow_ep_save;
**	  };
**	
**	  typedef struct clowep_struct CLOWEP_STRUCT;
**	
**
**
**	  
**	  struct  owltrec 
**	  {
**	          union   {
**	                  DBL     scet;   
**	                  DBL     A0;     
**	          }su;
**	          DBL     down_leg;
**	          DBL     up_leg;
**	          union   {                       
**	                  DBL     ert;   
**	                  DBL     EA0;    
**	          }eu;    
**	          DBL     EA1;             
**	          short   int     station;
**	  };
**	  typedef struct owltrec  OWLTREC;
**
**
** DEFINES
**
**	These defines are set at compilation time.
**
**	  1. IBM_MAIN_FRAME - set if on IBM 3090, or any IBM 
**	     main frame.
**	  2. FBINARY - set for machines requiring binary opens.
**	     Use for IBM main frame machines and IBM PCs.
**	  3. MAC - set for Macintoshes using THINK C.
**	  4. NOPROTO - set for machines where there is no
**	     prototyping.
**	  5. NOSEEK - set for use buffered file management rather
**	     than seeking to a position in  a file. Use for
**	     IBM main frame machines and VAX VMS systems.
**	  6. SMLLMACH - set for machines using 'int' of 2 bytes.
**	     Use for Macintoshes, IBM PCs. Used for buffer 
**	     management.
**	  7. UNIX - set for UNIX operating systems on SUNSs etc.
**	  8. VMSVAX - set for DECS VMS VAXs.
**
**
** DESCRIPTION.
**	
**	The double precision One Way Light Time file is an SFDU
**	file and conists of the CCSDZ0001 label (which envelopes
**	other SFDUS), NJPLKL015	whose value is the meta data or header,
**	NJPLI0049 whose value are the double precision date needed
**	to convert from SCLK to SCET and visa versa.
**
**	'diertscecnv' reads the OWLT file and calculates all other 
**	outputs from the given input in 'type_in'.
**
**	'diertscecnv' opens and reads in the header for the OWLT file.
**	A sample header looks like.
**
**	
**
*/
#endif /*  Header */
#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"DCERTSC0") csect(STATIC,"DCERTSC1")
#endif

#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<string.h>
#include	<sfbufmgr.h>
#include	<glbltme.h>
#include	<glowlsce.h>
#include	<dictmerr.h>
#include	<diertsce.h>

/* prototypes */
void diertscecnv(long int *ertsceary,char *msn_in,DBL *dptimes,char *sstrng,char *type_in,char *rtcode);
static d8opowlt(char *msn_in,char *sstrng,char *rtcode);
static  int d8chkeps(void);
static d8owltfilein(char *msn_in,char *rtcode);
static d8setowlbfm(char *rtcode);
static d8crowltblks(char *rtcode);
static int d8closeowlt(char *rtcode);
//static d8crowltblks(char *rtcode);
static d8chrinp(char *sstrng,long int *ertsceary,DBL *dptimes,char *rtcode);
static d8dpinp(char *type_in,long int *ertsceary,DBL *dptimes,char *rtcode);
static d8ertsce(char *strng_pntr,long int *ertsceary,DBL *dptimes,char *rtcode);
static d8ertinp(char *type_in,long int *ertsceary,DBL *dptimes,char *rtcode);
static d8dperti(long int *ertsceary,DBL *dptimes,char *rtcode);

static d8dpscei(long int *ertsceary,DBL *dptimes,char *rtcode);
static d8cmpert(long int *ertsceary,DBL *dptimes,char *rtcode);
static d8sceinp(char *type_in,long int *ertsceary,DBL *dptimes,char *rtcode);
static d8sfeowlt(char *rtcode);
static d8fndowl(char *rtcode);
static d8fndowltblk(char *rtcode);
static d8fndowltrec(char *rtcode);
static d8cnvowl(long int *ertsceary,DBL *dptimes,char *rtcode);
static d8cnvsce(long int *ertsceary,DBL *dptimes,char *rtcode);
static d8cnvert(long int *ertsceary,DBL *dptimes,char *rtcode);









/********************************************************************/
void
diertscecnv(ertsceary,msn_in,dptimes,sstrng,type_in,rtcode)

  long int *ertsceary;
  char     *msn_in;
  DBL	   *dptimes;
  char     *sstrng;
  char     *type_in;
  char     *rtcode;
  
{

int	rtrn;
#if (IBM_MAIN_FRAME)
ibm_here = TRUE;
#else
ibm_here = FALSE;
#endif
/*begin 'disctclk'*/
strcpy(rtcode,"NORM");
if (strcmp(did_init,"YES"))
{
	/*initia;ize here*/
	if (strcmp(times_data_init,"YES"))
	{
		ditminit();
		if (ibm_here)
		{
			sfparser();
		}
	}
	strcpy(did_init,"YES");
}
if (strcmp(vax_ordr_called,"YES"))
{
	/*check on order of integters (VAX VMS etc vs SUN, MACINTOSH,
	 IBM main frame*/
	strcpy(vax_ordr_called,"YES");
	rtrn = mchintst(NULL,NULL);
	if (rtrn)
	{
		/*this is a VAX order reorder array*/
		vax4_or_othr[0] = 3;
		vax4_or_othr[1] = 2;
		vax4_or_othr[2] = 1;
		vax4_or_othr[3] = 0;
		vax2_or_othr[0] = 1;
		vax2_or_othr[1] = 0;
	}
}
/*********************************************************************
**	     1. ERT - Input in ertsceary[3], ertsceary[4]
**	     2. ERTM - input in ertsceary[3], ertsceary[5]
**	     3. DPERT -double precision, input in dptimes[1].
**	     4. SCET - input in ertsceary[0], ertsceary[1].
**	     5. SCETM - input in ertsceary[0], ertsceary[2].
**	     6. DPSCET - doble precsion input in dptimes[0].
**	    10. OPEN - Open and initialize scet sclk file.
**	    11. CHAR - character string in sstrng for times etc.
*********************************************************************/
scet_fnd = FALSE;
ert_fnd = FALSE;
switch (type_in[0])
{
	case 'O': 
	case 'o':
		/*open OWLT file*/
		d8opowlt(msn_in,sstrng,rtcode);
	break;
	case 'C':
	case 'c':
		/*close or char inputs*/
		if (type_in[1] == 'l'  ||  type_in[1] == 'L')
		{
			d8closeowlt(rtcode);
		}
		else
		{
			/*character string input*/
			d8chkeps();
			d8chrinp(sstrng,ertsceary,dptimes,rtcode);
		}
	break;
	case 'D':
	case 'd':
		/*double precision input here*/
		d8chkeps();
		d8dpinp(type_in,ertsceary,dptimes,rtcode);
	break;
	case 'E':
	case 'e':
		/*ert here*/
		d8chkeps();
		d8ertinp(type_in,ertsceary,dptimes,rtcode);
	break;
	case 'S':
	case 's':
		/*scet here*/
		d8chkeps();
		d8sceinp(type_in,ertsceary,dptimes,rtcode);
	break;
	default:
		strcpy(rtcode,err_rtrn_code[BDI_INDX]);
		return;
}

return;
}
/********************************************************************/
static
d8opowlt(msn_in,sstrng,rtcode)

char	*msn_in;
char	*sstrng;
char	*rtcode;

{

size_t	sizme;
SFFOPDEF(fdinit);
int	rtc;
int	recsize;
char	*malc;

/*begin diopscet*/
strcpy(owlt_file,sstrng);
d8owltfilein(msn_in,rtcode);
if (rtcode[3] == 'X')
{
	/*error */
	return;
}

return;
}	

/************************************************************************/
static	int
d8chkeps()

{

tepoch_indx = 10 + 2*OWLEPS.epoch_indx;
/*check epochs of owltfile*/
if (OWLEPS.epoch_indx != DPEP_OF_CHOICE)
{
	/*the epoch is either 1950 or 2000*/
	return;
}
/*ok check to see if the epoch for times is proper*/
if (!strcmp(EPS.char_epoch,OWLEPS.clow_ep_save.char_epoch))
{
	/*they are the same no need for a change*/
	return;
}
/*change epochs*/
EPS = OWLEPS.clow_ep_save;
return;
}
/******************************************************************/	
static
d8owltfilein(msn_in,rtcode)

char	*msn_in;
char	*rtcode;

{

size_t	sizme;
char	data_set_id[80];

SFFOPEN(fdowlt,owlt_file,SFRDONLY);
if SFFERROPEN(fdowlt)
{
	strcpy(rtcode,err_rtrn_code[OWO_INDX]);
	return;
}
/*get size of file*/
/*read in header records*/
owlt_file_size = sfilesiz(fdowlt);
/*get header*/
dihdrsceowl(fdowlt,data_set_id,SFFNULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error encountered*/
	return;
}
if (strcmp(data_set_id,OWLT_DATA_SET_ID))
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[DSO_INDX]);
	/*set error here*/
	return;
}
/*check on msn or space craft id*/
if (msn_in != NULL)
{	
	switch (msn_case_nmbr)
	{
		case ULS_INDX:
			if (msn_in[0] != 'u'  &&
			    msn_in[0] != 'U')
			{
				/*error here mission inconsistant*/
				strcpy(rtcode,err_rtrn_code[MTO_INDX]);
				return;
			}
		break;
		case GLL_INDX:
			if (msn_in[0] != 'g'   &&
			    msn_in[0] != 'G')
			{
				/*error here mission inconsistant*/
				strcpy(rtcode,err_rtrn_code[MTO_INDX]);
				return;
			}
		break;
		default:
			/*error here mission inconsistant*/
			strcpy(rtcode,err_rtrn_code[MTO_INDX]);
			return;
	}
}		
rec_size_owlt = sizeof(OWLTREC);
/*two files sclk/scet and owlt files*/
numowl_recs_per_blk = OWL_RECS_PER_BLK;
#if (NOSEEK)
/*set up structure rdinfo*/
d8setowlbfm(rtcode);
d8crowltblks(rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
#else
/*compute number of records*/
number_owlt_records = (owlt_file_size - owloffset)/rec_size_owlt;
/*malloc bloksize*/
sizme = numowl_recs_per_blk*rec_size_owlt +1;
owltarray = (char *)malloc(sizme);
if (owltarray == NULL)
{
	strcpy(rtcode,err_rtrn_code[MAL_INDX]);
	return;
}
/*place for reading records*/
SFFSEEK(fdowlt,owloffset,SFSEEK_SET);
std_owlt_offset = rec_size_owlt * numowl_recs_per_blk;
d8crowltblks(rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
#endif
return;
}

#if (NOSEEK)
/*********************************************************************/
static
d8setowlbfm(rtcode)

char	*rtcode;

{

int	i;
int	len;
int	len1;
int	rtc;

/*compute number records per block*/
/*MAXBUFL is the maximum buffer size.
  number of bytes must be some multiple of records
 */
len = MAXBUFL/rec_size_owlt;
len *= rec_size_owlt;
len1 = len/numowl_recs_per_blk;
if (!len1)
{
	len1 = 1;
}
numowl_recs_per_blk = len/len1;
std_owlt_offset = numowl_recs_per_blk * rec_size_owlt;
len /= std_owlt_offset;
len *= std_owlt_offset;
rdowlbuf.tot_bufs = 0;
rdowlbuf.tot_bytes = 0;
rdowlbuf.more_reads = FALSE;
rdowlbuf.start_at_byte = owloffset;
rdowlbuf.bytes_per_buf = len;
for ( i = 0 ; i < MAXNUMBUFRS ; i ++)
{
	/*initialize rdowlbuf structure*/
	rdowlbuf.bufs[i].pntr = NULL;
}
rdinfowl.rdbuf = (char *)&rdowlbuf;
rdinfowl.frd = fdowlt;
rtc = sfread_file_in(&rdinfowl);
if (rtc < 0)
{	
	/*error here*/
	strcpy(rtcode,err_rtrn_code[BFM_INDX]);
	return;
}
return;
}
/*********************************************************************/
/*this uses buffer management to create clock blocks*/
static
d8crowltblks(rtcode)

char	*rtcode;

{

size_t	sizme;
int	rtc;
int	i;
char	*malc;
DBL	*curr_owlt_pntr;
int	tot_bytes_left;
int	blksize;
	
sizme = (size_t)sizeof(BLKOWLT);
blk_size_owlt = std_owlt_offset;
number_owlt_blks = 0;
owlbufinfo.bufrmode = BOF;
/*point to first buffer*/
rtc = sfbufman(&owlbufinfo,&rdinfowl);
if (rtc < 0)
{
	/*error*/
	strcpy(rtcode,err_rtrn_code[BFM_INDX]);
	return;
} 
owlbufinfo.bufrmode = NEXT_BUFR;
while (TRUE)
{
	/*do for all buffers here and now*/
	tot_bytes_left = owlbufinfo.buflen;
	owltarray = owlbufinfo.buf;
	while(TRUE)
	{
		/*owlbufinfo.buf points to currebt buffer
		  owlbufinfo.buflen is the length i bytes
		  owlbufinfo.bufnumber is the buff number to 
		  store in CURROWLTBLK->bufcount*/
		curr_owlt_pntr = (DBL *)owltarray;
		SOWLTREC = (OWLTREC *)owltarray;
		/*malloc it here*/
		malc = (char *)malloc(sizme);
		if (malc == NULL)
		{
			strcpy(rtcode,err_rtrn_code[MAL_INDX]);
			return;
		}
		CURROWLTBLK = (BLKOWLT *)malc;
		rtc = 0;
		if (!number_owlt_blks)
		{
			/*start here*/
			start_owlt_blk = malc;
			CURROWLTBLK->prev = NULL;
		}
		else
		{
			CURROWLTBLK->prev = (char *)PREVOWLTBLK;
			PREVOWLTBLK->next = (char *)CURROWLTBLK;		
		}
		CURROWLTBLK->scet = SOWLTREC->su.scet;
		CURROWLTBLK->ert = SOWLTREC->eu.ert;
		CURROWLTBLK->next = NULL;
		CURROWLTBLK->bufcount = owlbufinfo.bufnumber;
		CURROWLTBLK->offset = owlbufinfo.buflen - 
			tot_bytes_left;
		if (tot_bytes_left < blk_size_owlt)
		{
			blksize = tot_bytes_left;
		}
		else
		{
			blksize = blk_size_owlt;
		}
		CURROWLTBLK->nmbr_recs = blksize/rec_size_owlt;
		/*set prev to curr*/
		PREVOWLTBLK = CURROWLTBLK;
		number_owlt_blks++;
		owltarray = &owlbufinfo.buf[(int)CURROWLTBLK->offset];
		/*do an offset from here*/
		tot_bytes_left -= blksize;
		if (tot_bytes_left <= 0)
		{
			break;
		}
	}
	/*get next*/
	if (!owlbufinfo.contflag)
	{
		break;
	}
	/*get next buffer*/
	rtc = sfbufman(&owlbufinfo,&rdinfowl);
	if (rtc < 0)
	{
		strcpy(rtcode,err_rtrn_code[BFM_INDX]);
		return;
	}
	
}
CURROWLTBLK = (BLKOWLT *)NULL;
SOWLTREC = (OWLTREC *)NULL;
return;
}
/********************************************************************/
static	int
d8closeowlt(rtcode)

char	*rtcode;

{

int	i;

/*close files and buffers*/
for ( i = 0 ; i < MAXNUMBUFRS ; i ++)
{
	if (rdowlbuf.bufs[i].pntr == NULL)
	{
		break;
	}
	free(rdowlbuf.bufs[i].pntr);
}
SFFCLOSE(fdowlt);
return;
}

#else
/***************************************************************/
static
d8crowltblks(rtcode)

char	*rtcode;

{

size_t	sizme;
int	rtc;
int	i;
char	*malc;
DBL	*curr_owlt_pntr;

sizme = (size_t)sizeof(BLKOWLT);
blk_size_owlt = std_owlt_offset;
number_owlt_blks = 0;
while (TRUE)
{
	while(TRUE)
	{
		rtc = SFFREAD(fdowlt,owltarray,rec_size_owlt);
		if (rtc < 0)
		{
			strcpy(rtcode,err_rtrn_code[OWR_INDX]);
			return;
		}
		if (!rtc)
		{
			/*ok no more set last etc*/
			CURROWLTBLK->nmbr_recs = 
				(owlt_file_size - CURROWLTBLK->offset)/
				rec_size_owlt;
			/*last block*/
			last_owlt_blk = malc;
			break;
		}
		curr_owlt_pntr = (DBL *)owltarray;
		if (!curr_owlt_pntr[OWLTEA1_FIELD])
		{
			continue;
		}
		break;
	}
	if (!rtc)
	{
		break;
	}
	SOWLTREC = (OWLTREC *)owltarray;
	/*malloc it here*/
	malc = (char *)malloc(sizme);
	if (malc == NULL)
	{
		strcpy(rtcode,err_rtrn_code[MAL_INDX]);
		return;
	}
	CURROWLTBLK = (BLKOWLT *)malc;
	rtc = 0;
	if (!number_owlt_blks)
	{
		/*start here*/
		start_owlt_blk = malc;
		CURROWLTBLK->prev = NULL;
	}
	else
	{
		CURROWLTBLK->prev = (char *)PREVOWLTBLK;
		PREVOWLTBLK->next = (char *)CURROWLTBLK;		
	}
	CURROWLTBLK->scet = SOWLTREC->su.scet;
	CURROWLTBLK->ert = SOWLTREC->eu.ert;
	CURROWLTBLK->next = NULL;
	CURROWLTBLK->nmbr_recs = std_owlt_offset/rec_size_owlt;
	CURROWLTBLK->offset = SFFTELL(fdowlt) - rec_size_owlt;
	/*set prev to curr*/
	PREVOWLTBLK = CURROWLTBLK;
	number_owlt_blks++;
	/*do an offset from here*/
	rtc = SFFSEEK(fdowlt,std_owlt_offset-rec_size_owlt,SFSEEK_CUR);
	if (rtc)
	{
		i = 0;
	}
}
CURROWLTBLK = (BLKOWLT *)NULL;
SOWLTREC = (OWLTREC *)NULL;
/*reset to position to offset*/
rtc = SFFSEEK(fdowlt,owloffset,SFSEEK_SET);
curr_owlt_seek_pos = owloffset;
return;
}
/***********************************************************************/
static	int
d8closeowlt(rtcode)

char	*rtcode;

{

char	*free_pntr;

CURROWLTBLK = (BLKOWLT *)start_owlt_blk;
for (;;)
{
	if (CURROWLTBLK == (BLKOWLT *)NULL)
	{
		break;
	}
	free_pntr = (char *)CURROWLTBLK;
	free(free_pntr);
	CURROWLTBLK = (BLKOWLT *)CURROWLTBLK->next;
}
SFFCLOSE(fdowlt);
return;
}
#endif
/********************************************************************/	
static	
d8chrinp(sstrng,ertsceary,dptimes,rtcode)

char	*sstrng;
long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

int	i;
int	j;
int	tot;
int	indx;
char	cmp_strng[8];
int	len;
int	rt_code;

/*********************************************************************
	parse string here find what kind

	1. ??? YYYY DOY HH:MM:SS.FFF (and other calender forms)
	2. ??? SP50 (or SP2000 or SPEPOCH) xxxxxxxxx.xxx
	3. ??? SPxH (for hex) XXXXXXXX XXXX (where x is 50 2000 or 
	   EPOCH). 
	Where ??? is either SCET or ERT
	4. SCLK the clock for mission
	   1. ULS - 'SCLK xxxxxxxx.xxx'
	   2. ULS - 'SCLKH (for hex) XXXXXXXX XXXX'
	   
	NOTE: See 'dictmcnv' for types of time for ERT and SCET inputs.
	
*********************************************************************/

strng_pntr = sstrng;
tot = strlen(sstrng);
ert_fnd = FALSE;
scet_fnd = FALSE;
/*actually find first word find if SCET or ERT*/
for (i = 0 ; i < tot ; i++)
{
	if (sstrng[i] == ' ')
	{
		continue;
	}
	/*we found something*/
	strng_pntr = strchr(&strng_pntr[i],' '); /*find blank*/
	if (strng_pntr == NULL)
	{
		/*error */
		strcpy(rtcode,err_rtrn_code[CHR_INDX]);
		return;
	}
	len = (long int)strng_pntr - (long int)&sstrng[i];
	if (len > 7) 
	{
		strcpy(rtcode,err_rtrn_code[STL_INDX]);
		return;
	}
	indx = i;
	for (j = 0 ; j < len ; j++)
	{
		if (islower(sstrng[i+j]))
		{
			cmp_strng[j] = (char)toupper(sstrng[i+j]);
		}
		else
		{
			cmp_strng[j] = sstrng[i+j];
		}
	}
	cmp_strng[len] = '\0';
	if (!strcmp(cmp_strng,"SCET"))
	{
		scet_fnd = TRUE;
		break;
	}
	else if (!strcmp(cmp_strng,"ERT"))
	{
		ert_fnd = TRUE;
		break;
	}
	strcpy(rtcode,err_rtrn_code[NOT_INDX]);
	return;
}
d8ertsce(strng_pntr,ertsceary,dptimes,rtcode);
return;
}
		
/************************************************************************/	
static
d8dpinp(type_in,ertsceary,dptimes,rtcode)

char	*type_in;
long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

/*input is dp someting or other*/
/*********************************************************************
**	     1. DPERT - double precision, input in dptimes[1].
**	     2. DPSCET - doble precsion input in dptimes[0].
*********************************************************************/

switch (type_in[4])
{
	case 'T':
	case 't':
		/*dper[T]*/
		d8dperti(ertsceary,dptimes,rtcode);
	break;
	case 'E':
	case 'e':
		/*dpsc[E]t*/
		d8dpscei(ertsceary,dptimes,rtcode);
	break;
	default:
		strcpy(rtcode,err_rtrn_code[BDI_INDX]);
}
return;
}
/*******************************************************************/
static
d8ertsce(strng_pntr,ertsceary,dptimes,rtcode)

char	*strng_pntr;
long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

int	i;
int	j;
int	k;
int	tot;
BOOL	hex;
int	len;
int	toti;
int	totl;
char	*pntr;
/*find next word see if SPetc*/

tot = strlen(strng_pntr);
for ( i = 0 ; i < tot ; i++)
{
	if (strng_pntr[i] == ' ')
	{
		continue;
	}
	pntr = strchr(&strng_pntr[i],' ');
	if (pntr == NULL)
	{
		strcpy(rtcode,err_rtrn_code[NOT_INDX]);
		return;
	}
	/*looking for SPEPOCH(H)*/
	totl = (long int)pntr - (long int)&strng_pntr[i];
	if (totl != 7 && len != 8)
	{
		break;
	}
	if (strng_pntr[i] != 'S'  && strng_pntr[i] != 's')
	{
		break;
	}
	if (strng_pntr[i+1] != 'P'   && strng_pntr[i+1] != 'p')
	{
		break;
	}
	if (strng_pntr[i+2] != 'E'   && strng_pntr[i+2] != 'e')
	{
		break;
	}
	/*should be SPEPOCH*/
	if (strng_pntr[i+7] == 'H'  || strng_pntr[i+7] == 'h')
	{
		hex = TRUE;
	}
	else
	{
		hex = FALSE;
	}
	len = strlen(time_type); /*50 or 2000*/
	strncpy(&strng_pntr[i],time_type,len);
	toti = i + 2 + len;
	for (j = toti ; j < totl ; j++)
	{
		strng_pntr[j] = ' '; /*set as blanks or white spaces*/
	}
	break;
}
/*convert*/
strcpy(typein,"CHAR");
/*check to see if we need to change structures*/
dictmcnv(tarray,mon,NULL,dpsecs,strng_pntr,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*set up according to scet or ert*/
if (ert_fnd)
{
	/*ok now set up and go*/
	ert = dpsecs[OWLEPS.epoch_indx];
	DPERT = ert;
	ERTSECS = tarray[tepoch_indx];
	ERTFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
	ERTMILLS = MILLSEC; /*milliseconds*/
	d8cmpert(ertsceary,dptimes,rtcode);
}
else
{
	/*scet here*/
	scet = dpsecs[    OWLEPS.epoch_indx];
	DPSCET = scet;
	SCESECS = tarray[tepoch_indx];
	SCEFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
	SCEMILLS = MILLSEC; /*milliseconds*/
	d8cmpsce(ertsceary,dptimes,rtcode);
}
return;
}
/******************************************************************/	
static
d8ertinp(type_in,ertsceary,dptimes,rtcode)

char	*type_in;
long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

int	len;

/*********************************************************************
**	     1. ERT - Input in ertsceary[3], ertsceary[4]
**	     2. ERTM - input in ertsceary[3], ertsceary[5]
***********************************************************************/

/******************************************************************
**	   tarray - long int tarray[14] - time array of different times 
**	     1. tarray[0]  - 4 digit year
**	     2. tarray[1]  - doy, day of year
**	     3. tarray[2]  - hoy, hour of year
**	     4. tarray[3]  - second of hour (hoy format)
**	     5. tarray[4]  - month of year
**	     6. tarray[5]  - day of month
**	     7. tarray[6]  - hour of day
**	     8. tarray[7]  - minutes of hour
**	     9. tarray[8]  - seconds of minute
**	    10. tarray[9]  - milliseconds
**	    11. tarray[10] - seconds past 1950
**	    12. tarray[11] - fractional part of seconds past 1950
**	    13. tarray[12] - seconds from 2000
**	    14. tarray[13] - fractional part of seconds from 2000
***********************************************************************/
if (type_in[3] == 'M'  ||  type_in[3] == 'm')
{
	/*milliseconds not fractional seconds*/
	/*set up to do time conversions here*/
	switch (    OWLEPS.epoch_indx)
	{
		case DPEP_OF_CHOICE:
			/*epoch of choice*/
			strcpy(typein,"SPEPOCHM");
			SECSFEPOCH = ERTSECS; 
		break;
		case DP2000_INDX:
			/* seconds past 2000*/
			strcpy(typein,"SP2000M");
			SECSF2000 = ERTSECS; 
		break;
		default:
			/* seconds past 50*/
			strcpy(typein,"SP50M");
			SECSF1950 = ERTSECS; 
		break;
	}
	MILLSEC = ERTMILLS;
}
else
{
	switch (    OWLEPS.epoch_indx)
	{
		case DPEP_OF_CHOICE:
			/*epoch of choice*/
			strcpy(typein,"SPEPOCH");
			SECSFEPOCH = ERTSECS; 
		break;
		case DP2000_INDX:
			/* seconds past 2000*/
			strcpy(typein,"SP2000");
			SECSF2000 = ERTSECS; 
		break;
		default:
			/* seconds past 50*/
			strcpy(typein,"SP50");
			SECSF1950 = ERTSECS; 
		break;
	}
	/*use fractional seconds*/
	tarray[tepoch_indx+1] = ERTFRACS;
}
dictmcnv(tarray,mon,NULL,dpsecs,NULL,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*ok now set up and go*/
ert = dpsecs[OWLEPS.epoch_indx];
DPERT = ert;
if (type_in[3] == 'M'  ||  type_in[3] == 'm')
{
	ERTFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
}
else
{
	ERTMILLS = MILLSEC; /*milliseconds*/
}
d8cmpert(ertsceary,dptimes,rtcode);
return;
}
/**************************************************************************/
static
d8dperti(ertsceary,dptimes,rtcode)

long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{
/*ert input in double precision*/
ert = DPERT;

/*do it now*/
d8cnvert(ertsceary,dptimes,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
d8cmpert(ertsceary,dptimes,rtcode);
return;
}
/**********************************************************************/
static
d8cmpert(ertsceary,dptimes,rtcode)

long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

ert_fnd = TRUE;
/**********************************************************************
struct	owltrec 
{
	union	{
		DBL	scet;	  spacecraft efent time
		DBL	A0;	  constant for function of x (sclk to 
				  scet)
	}su;
	DBL	down_leg;
	DBL	up_leg;
	union	{			
		DBL	ert;	  Earth recieve time
		DBL	EA0;	  constant for funtion of x (ert to 
				  scet)
	}eu;	
	DBL	EA1;		  the constant for x (function for 
				  ert to scet)
	short	int	station;  station number of received data
};
*********************************************************************/
owlt_value = ert;
owlt_field_indx = 3;
val_owlt_blk_indx = 1;
d8fndowltblk(rtcode); /*find block of data*/	
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
d8fndowltrec(rtcode); /*find record to do*/
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*compute ert using owlt file*/
d8sfeowlt(rtcode);
DPSCET = scet;
d8cnvsce(ertsceary,dptimes,rtcode);
if (rtcode[3] == 'X')
{
	return;
}
scet_fnd = TRUE; 
d8fndowl(rtcode);
DPOWLT = owlt;
d8cnvowl(ertsceary,dptimes,rtcode);
return;
}
/*******************************************************************/
static
d8sceinp(type_in,ertsceary,dptimes,rtcode)

char	*type_in;
long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

int	len;

/*********************************************************************
**	     1. SCET - input in ertsceary[0], ertsceary[1].
**	     2. SCETM - input in ertsceary[0], ertsceary[2].
**********************************************************************/


/******************************************************************
**	   tarray - long int tarray[14] - time array of different times 
**	     1. tarray[0]  - 4 digit year
**	     2. tarray[1]  - doy, day of year
**	     3. tarray[2]  - hoy, hour of year
**	     4. tarray[3]  - second of hour (hoy format)
**	     5. tarray[4]  - month of year
**	     6. tarray[5]  - day of month
**	     7. tarray[6]  - hour of day
**	     8. tarray[7]  - minutes of hour
**	     9. tarray[8]  - seconds of minute
**	    10. tarray[9]  - milliseconds
**	    11. tarray[10] - seconds past 1950
**	    12. tarray[11] - fractional part of seconds past 1950
**	    13. tarray[12] - seconds from 2000
**	    14. tarray[13] - fractional part of seconds from 2000
***********************************************************************/
if (type_in[4] == 'M'  ||  type_in[4] == 'm')
{
	switch (OWLEPS.epoch_indx)
	{
		case DPEP_OF_CHOICE:
			/*epoch of choice*/
			strcpy(typein,"SPEPOCHM");
			SECSFEPOCH = SCESECS; 
		break;
		case DP2000_INDX:
			/* seconds past 2000*/
			strcpy(typein,"SP2000M");
			SECSF2000 = SCESECS; 
		break;
		default:
			/* seconds past 50*/
			strcpy(typein,"SP50M");
			SECSF1950 =  SCESECS; 
		break;
	}
	/*milliseconds not fractional seconds*/
	/*set up to do time conversions here*/
	MILLSEC = SCEMILLS;
}
else
{
	/*use fractional seconds*/
	switch (OWLEPS.epoch_indx)
	{
		case DPEP_OF_CHOICE:
			/*epoch of choice*/
			strcpy(typein,"SPEPOCH");
			SECSFEPOCH = ERTSECS; 
		break;
		case DP2000_INDX:
			/* seconds past 2000*/
			strcpy(typein,"SP2000");
			SECSF2000 = SCESECS; 
		break;
		default:
			/* seconds past 50*/
			strcpy(typein,"SP50");
			SECSF1950 = SCESECS; 
		break;
	}
	tarray[tepoch_indx+1] = SCEFRACS;
}
dictmcnv(tarray,mon,NULL,dpsecs,NULL,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*ok now set up and go*/
scet = dpsecs[OWLEPS.epoch_indx];
DPSCET = scet;
if (type_in[3] == 'M'  ||  type_in[3] == 'm')
{
	SCEFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
}
else
{
	SCEMILLS = MILLSEC; /*milliseconds*/
}
d8cmpsce(ertsceary,dptimes,rtcode);
return;
}
/********************************************************************/
static
d8dpscei(ertsceary,dptimes,rtcode)

long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

/*double precision input of scet*/
scet = DPSCET;
d8cnvsce(ertsceary,dptimes,rtcode);
if (rtcode[3] == 'X')
{
	/*error here*/
	return;
}
/*call the rest now*/
d8cmpsce(ertsceary,dptimes,rtcode);
return;
}
 
static
d8cmpsce(ertsceary,dptimes,rtcode)

long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

/**********************************************************************
struct	owltrec 
{
	union	{
		DBL	scet;	  spacecraft efent time
		DBL	A0;	  constant for function of x (sclk to 
				  scet)
	}su;
	DBL	down_leg;
	DBL	up_leg;
	union	{			
		DBL	ert;	  Earth recieve time
		DBL	EA0;	  constant for funtion of x (ert to 
				  scet)
	}eu;	
	DBL	EA1;		  the constant for x (function for 
				  ert to scet)
	short	int	station;  station number of received data
};
*********************************************************************/
owlt_value = scet;
owlt_field_indx = 0;
val_owlt_blk_indx = 0;
d8fndowltblk(rtcode); /*find block of data*/
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
d8fndowltrec(rtcode); /*find record to do*/
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*compute ert using owlt file*/
d8efsowlt(rtcode);
DPERT = ert;
d8cnvert(ertsceary,dptimes,rtcode);
/*get owlt*/
d8fndowl(rtcode);
DPOWLT = owlt;
d8cnvowl(ertsceary,dptimes,rtcode);
return;
}
/***************************************************************/
static
d8sfeowlt(rtcode)

char	*rtcode;

{

/*get scet from ert*/
 /*SOWLTREC points to current record*/
 
 /*scet = SCET  + (ert - ERT)/EA1 */
 
 scet = SOWLTREC->su.scet + (ert - SOWLTREC->eu.ert)/SOWLTREC->EA1;
 
 return;
 }
/********************************************************************/ 
 static
 d8fndowl(rtcode)
 
 char	*rtcode;
 
 {
 
 /*find OWLT */
 /*OWLT = ert - scet */
 owlt = ert - scet;
 return;
 }
 
static
d8efsowlt(rtcode)
 
 char	*rtcode;
 
 {
 
 /*find ert from scet*/
 /* ert = ERT + (scet - SCET)*EA1 */
 
 ert = SOWLTREC->eu.ert + (scet - SOWLTREC->su.scet)*SOWLTREC->EA1;
 
 return;
 }
/*********************************************************************/
static
d8fndowltblk(rtcode)

char	*rtcode;

{

DBL	*blk_owlt_pntr;
DBL	*nxt_owlt_pntr;
long	int	pos;
int	rtc;
long	int	fromftell;

/*find block for next whatever*/
/*val_in_blk_indx is the index into structure BLKCLK for sclk, scet or
  ert field_indx is index into structure SCECLKREC for sclk, scet or 
  ert*/
  
if (CURROWLTBLK == (BLKOWLT *)NULL)
{
	owlt_frwrd = TRUE;
	CURROWLTBLK = (BLKOWLT *)start_owlt_blk;
}
else
{
	blk_owlt_pntr = (DBL *)CURROWLTBLK;
	if (owlt_value < blk_owlt_pntr[val_owlt_blk_indx])
	{
		owlt_frwrd = FALSE;
	}
	else if (CURROWLTBLK->next == NULL)
	{
		/*we have found it*/
		return;
	}
	else
	{
		nxt_owlt_pntr = (DBL *)CURROWLTBLK->next;
		if (owlt_value < nxt_owlt_pntr[val_owlt_blk_indx])
		{
			/*we have the right block*/
			return;
		}
		owlt_frwrd = TRUE;
		CURROWLTBLK = NEXTOWLTBLK;
	}
}
if (!owlt_frwrd)
{
	while (CURROWLTBLK != (BLKOWLT *)NULL)
	{
		/*go backwards to wahtever*/
		/*CURROWLTBLK has current info*/
		PREVOWLTBLK = (BLKOWLT *)CURROWLTBLK->prev;
		if (PREVOWLTBLK == (BLKOWLT *)NULL)
		{
			break;
		}
		CURROWLTBLK = PREVOWLTBLK;
		blk_owlt_pntr = (DBL *)CURROWLTBLK;
		if (owlt_value >= blk_owlt_pntr[val_owlt_blk_indx])
		{
			/*ok it is here*/
			break;
		}
	}
}
else
{
	/*search forward here*/
	
	while (CURROWLTBLK != (BLKOWLT *)NULL)
	{
		/*go backwards to wahtever*/
		/*CURROWLTBLK has current info*/
		NEXTOWLTBLK = (BLKOWLT *)CURROWLTBLK->next;
		if (NEXTOWLTBLK == (BLKOWLT *)NULL)
		{
			break;
		}
		nxt_owlt_pntr = (DBL *)CURROWLTBLK->next;	
		if (owlt_value < nxt_owlt_pntr[val_owlt_blk_indx])
		{
			/*we have the right block*/
			break;
		}
		CURROWLTBLK = NEXTOWLTBLK;
	}
}
#if (NOSEEK)
/*use buffered management*/
/*get current buffer*/
owlbufinfo.bufnumber = CURROWLTBLK->bufcount;
owlbufinfo.bufrmode = USE_BUFINDX;
rtc = sfbufman(&owlbufinfo,&rdinfowl);
if (rtc < 0)
{
	strcpy(rtcode,err_rtrn_code[BFM_INDX]);
	return;
}
owltarray = &owlbufinfo.buf[(int)CURROWLTBLK->offset];
number_owlt_in_curr_blk = CURROWLTBLK->nmbr_recs;
curr_blk_size_owlt = CURROWLTBLK->nmbr_recs * rec_size_owlt;
#else
/*read in proper block*/
/*firstposition for read*/
fromftell = SFFTELL(fdowlt);
pos = CURROWLTBLK->offset - fromftell;	
rtc = SFFSEEK(fdowlt,pos,SFSEEK_CUR);
if (rtc)
{
	strcpy(rtcode,err_rtrn_code[SEK_INDX]);
	return;
}
/*ok it is there*/
/*now read it in*/
rtc = SFFREAD(fdowlt,owltarray,CURROWLTBLK->nmbr_recs*rec_size_owlt);
if (rtc <= 0)
{
	strcpy(rtcode,err_rtrn_code[OWR_INDX]);
	return;
}
curr_owlt_seek_pos = CURROWLTBLK->offset + rtc;
/*compute number of records in this block*/
number_owlt_in_curr_blk = rtc/rec_size_owlt;
curr_blk_size_owlt = rtc;
#endif
curr_owlt_rec = NULL;
prev_owlt_rec = NULL;
next_owlt_rec = NULL;
return;
}

/************************************************************************/
static
d8fndowltrec(rtcode)

char	*rtcode;

{

DBL	*curr_owlt_pntr;
DBL	*next_owlt_pntr;
long	int	last_rec_add;
long	int	next_rec_add;

if (curr_owlt_rec == NULL)
{
	owlt_frwrd = TRUE;
	curr_owlt_rec = owltarray;
	SOWLTREC = (OWLTREC *)curr_owlt_rec;
}
else
{
	curr_owlt_pntr = (DBL *)curr_owlt_rec;
	if (curr_owlt_pntr[owlt_field_indx] > owlt_value)
	{
		owlt_frwrd = FALSE;
	}
	else if (next_owlt_rec == NULL)
	{
		/*found*/
		return;
	}
	else 
	{
		next_owlt_pntr = (DBL *)next_owlt_rec;
		if (owlt_value < next_owlt_pntr[owlt_field_indx])
		{
			/*found */
			return;
		}
		owlt_frwrd = TRUE;
		curr_owlt_rec = next_owlt_rec;
		SOWLTREC = (OWLTREC *)curr_owlt_rec;
	}
}
if (!owlt_frwrd)
{
	/*back wards search*/
	while (curr_owlt_rec != NULL)
	{
		prev_owlt_rec = curr_owlt_rec - rec_size_owlt;
		if ((long int)prev_owlt_rec < (long int)owltarray)
		{
			prev_owlt_rec = NULL;
			next_owlt_rec = curr_owlt_rec + rec_size_owlt;
			break;
		}
		next_owlt_rec = curr_owlt_rec;
		curr_owlt_rec = prev_owlt_rec;
		curr_owlt_pntr = (DBL *)curr_owlt_rec;
		if (!curr_owlt_pntr[OWLTEA1_FIELD])
		{
			continue;
		}
		if (curr_owlt_pntr[owlt_field_indx] <= owlt_value)
		{
			/*found*/
			prev_owlt_rec = curr_owlt_rec - rec_size_owlt;
			break;
		}
	}
}
else
{
	/*forward*/
	last_rec_add = (long int)&owltarray[curr_blk_size_owlt];
	while (curr_owlt_rec != NULL)
	{
		next_owlt_rec = curr_owlt_rec + rec_size_owlt;
		next_rec_add = (long int)next_owlt_rec;
		if (next_rec_add >= last_rec_add) 
		{
			/*over out of space*/
			next_owlt_rec = NULL;
			break;
		}
		next_owlt_pntr = (DBL *)next_owlt_rec;
		if (!next_owlt_pntr[OWLTEA1_FIELD])
		{
			prev_owlt_rec = curr_owlt_rec;
			curr_owlt_rec = next_owlt_rec;
			continue;
		}
		if (next_owlt_pntr[owlt_field_indx] > owlt_value)
		{
			/*found*/
			prev_owlt_rec = curr_owlt_rec - rec_size_owlt;
			break;
		}
		prev_owlt_rec = curr_owlt_rec;
		curr_owlt_rec = next_owlt_rec;
	}
}
SOWLTREC = (OWLTREC *)curr_owlt_rec;
return;
}
/***************************************************************/
static
d8cnvert(ertsceary,dptimes,rtcode)

long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

/*convert dp seconds of ert to long ints*/

switch (OWLEPS.epoch_indx)
{
	case DPEP_OF_CHOICE:
		/*epoch of choice*/
		strcpy(typein,"DPEPOCH");
	break;
	case DP2000_INDX:
		/* seconds past 2000*/
		strcpy(typein,"DP2000");
	break;
	default:
		/* seconds past 50*/
		strcpy(typein,"DP50");
	break;
}
dpsecs[OWLEPS.epoch_indx] = ert;
dictmcnv(tarray,mon,NULL,dpsecs,NULL,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
ERTSECS = tarray[tepoch_indx]; /*seconds past epoxh*/
ERTFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
ERTMILLS = MILLSEC; /*milliseconds*/
return;
}
/**************************************************************************/
static
d8cnvsce(ertsceary,dptimes,rtcode)

long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{


/*convert dp seconds of scet to long ints*/

switch (OWLEPS.epoch_indx)
{
	case DPEP_OF_CHOICE:
		/*epoch of choice*/
		strcpy(typein,"DPEPOCH");
	break;
	case DP2000_INDX:
		/* seconds past 2000*/
		strcpy(typein,"DP2000");
	break;
	default:
		/* seconds past 50*/
		strcpy(typein,"DP50");
	break;
}
dpsecs[OWLEPS.epoch_indx] = scet;
dictmcnv(tarray,mon,NULL,dpsecs,NULL,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
SCESECS = tarray[tepoch_indx]; /*seconds past epoxh*/
SCEFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
SCEMILLS = MILLSEC; /*milliseconds*/
return;
}
/******************************************************************/
static
d8cnvowl(ertsceary,dptimes,rtcode)

long	int	*ertsceary;
DBL	*dptimes;
char	*rtcode;

{

/*convert OWLTto integer formats*/

union {
	long	int	fracsl;
	unsigned short	int	fracss[2];
} frc;
DBL	fracsd;
DBL	dblowl;

OWLTSECS = owlt; /*long int of owlt*/
fracsd = owlt - (DBL)OWLTSECS;
dblowl = fracsd *1000.;
dblowl += .5;
OWLTMILLS = dblowl;
frc.fracss[ORDS_INDX0] = 0;
/*fractional portion of seconds*/
fracsd *= float_hex_1000;
fracsd += .5;
frc.fracss[ORDS_INDX1] = fracsd;
OWLTFRACS  = frc.fracsl;
return;
}

