/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*****************************************************************
 * TITLE: sedrate.c
 *
 * AUTHOR:  Unknown
 *          Aug 31, 1994
 *
 * MODIFIED:    Ray Bambery
 *          Aug 24, 2020 -  removed label after #ENDIF Header 
 *                  add prototypes
  ******************************************************************
            */
/*
** MANUAL
**	DICLKCNV 3x "May 6, 1993"
**
** NAME
**	diclkcnv - universal conversions for spacecraft clocks.
**
** SYNOPSIS
**
**	  #include   <glbltme.h>
**	  #include   <glowlsce.h>
**
**	  void diclkcnv(carray,msn_in,dpclk,cstrng,type_in,type_out,
**	                strng_out,rtcode);
**
**	  long int *carray;
**	  char     *msn_in;
**	  DBL      *dpclk;
**	  char     *cstrng;
**	  char     *type_in;
**	  char     *type_out;
**	  char     *strng_out;
**	  char     *rtcode;
**
** PARAMETERS
**	  1. carray - an array of spacecraft clocks. Dimensioned by 6
**	  2. msn_in - 3 character mission id. i.e. GLL, VGR, ULS etc.
**	  3. dpclk - DBL precisions SCLK.
**	  4. cstrng - string input for clocks.
**	  5. type_in - type of SCLK input to 'diclkcnv'.
**	  6. type_out  - For strng_out. NULL, Blank or empty no 
**	     strng_out.
**	  7. strng_out - character output for type_out.
**	  8. rtcode - return code, 'NORM' for ok.
**	     
** PARAMETER DESCRIPTION
**
**	  1. carray - an array of spacecraft clocks.
**	    1. carray[0] thru carray[3] - SCLK components, msn 
**	        dependent.
**	        1. ULS
**	           1. carray[0] - corrected counter, incremented every
**	              2 seconds.
**	           2. carray[1] - fractional part of counter.
**	        2. GLL 
**	           1. carray[0] - rim counter, incremented every
**	              60 2/3 seconds.
**	           2. carray[1] - Mod 91 counter, incremented every 
**	              2/3 seconds.
**	           3. carray[2] - mod 10 counter, incremented every
**	              66 2/3 milliseconds.
**	           4. carray[3] - mod 8 counter, incremented every
**	              8 2/3 milliseconds.
**	        3. VGR
**	           1. carray[0] - Mod 16 counter.
**	           2. carray[1] - mod 60 counter.
**	           3. carray[2] - line counter.
**	    2. carray[4], carray[5], actual SCLK, mission dependent.
**             6 bytes.
**	        1. ULS - counter, 32 bits, binary fraction,16 bits.
**	        2. GLL - rim counter, 24 bits, mod 91 counter, 8 bits,
**	           mod 10 counter, 8 bits, mod 8 counter, 8 bits.
**	        3. VGR - Mod 16 counter, 16 bits, mod 60 counter,
**	           6 bits, line count, 10 bits.
**	  2. msn_in - 3 character mission id. i.e. GLL, VGR, ULS etc.
**	  3. dpclk - DBL precisions SCLK.
**	  4. cstrng - string input for clocks.
**	  5. type_in - type of SCLK input to 'diclkcnv'.
**	     1. SCLK - input in carray[0] etc msn dependent.
**	     2. RAW - input in carray[4],carray[5]
**	     3. DPSCLK - Doble precision SCLK input in dpclk.
**	     4. CHAR - character string in cstrng for SCLK.
**	  6. type_out  - For strng_out. NULL, Blank or empty no 
**	     strng_out.
**	     1. DPCLK - double precision clock.
**	        'x.xxxxxxxxxxxxEyy'. Input is dpclk.
**	     2. HEX - HEX SCLK output for clock.
**	        'xxxxxxxx xxxx' - 8 and 4 characters (6 bytes).
**	        Input is SCLKRAW.
**	     3. COUNTS - clocks seperated by '.' Mission dependent.
**	        1. ULS - xxxxxxxx.xxx
**	        2. VGR - xxxxxx.xx.xxx
**	        3. GLL - xxxxxxxx.xx.x.x
**	  7. strng_out - character output for type_out.
**	  8. rtcode - return code, 'NORM' for ok.
**	     
**	      
** EXTERNS
**
**	 1. msn_case_nmbr - int - output from diclkcnv. 
**
*/
#endif /*  Header */
#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"DCCLKCN0") csect(STATIC,"DCCLKCN1")
#endif


#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<string.h>
#include	<sftypes.h>
#include	<sfbufmgr.h>
#include	<glbltme.h>
#include	<glowlsce.h>
#include	<dictmerr.h>
#include	<diclkcnv.h>

void
diclkcnv(carray,msn_in,dpclk,cstrng,type_in,type_out,strng_out,rtcode)

long int *carray;
char	*msn_in;
DBL      *dpclk;
char     *cstrng;
char     *type_in;
char     *type_out;
char     *strng_out;
char     *rtcode;

{

int	rtrn;

/*begin 'disctclk'*/
strcpy(rtcode,"NORM");
/*initia;ize here*/
if (strcmp(times_data_init,"YES"))
{
	ditminit();
}

if (strcmp(vax_ordr_called,"YES"))
{
	/*check on order of integters (VAX VMS etc vs SUN, MACINTOSH,
	 IBM main frame*/
	strcpy(vax_ordr_called,"YES");
	rtrn = mchintst(NULL,NULL);
	if (rtrn)
	{
		/*this is a VAX order reorder array*/
		vax4_or_othr[0] = 3;
		vax4_or_othr[1] = 2;
		vax4_or_othr[2] = 1;
		vax4_or_othr[3] = 0;
		vax2_or_othr[0] = 1;
		vax2_or_othr[1] = 0;
	}
}
if (msn_in != NULL)
{
	if (msn_in[0] != '\0'  &&  strcmp(msn,msn_in))
	{
		d4omsn(msn_in,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		if (type_in == NULL  &&  type_out == NULL)
		{
			return;
		}
	}
}
is_hex = FALSE;
sclk_hex = FALSE;
use_type_in = FALSE;
if (type_in != NULL)
{
/*********************************************************************
**	     1. SCLK - input in carray[0] etc msn dependent.
**	     2. RAW - input in carray[4],carray[5]
**	     3. DPSCLK - Doble precision SCLK input in dpclk.
**	     4. CHAR - character string in cstrng for SCLK.
**********************************************************************/
	use_type_in = TRUE;
	switch (type_in[0])
	{
		case 'S':
		case's':
			/*sclk input*/
			d4clkinp(carray,dpclk,rtcode);
		break;
		case 'R':
		case 'r':
			/*raw input*/
			d4clkraw(carray,dpclk,rtcode);
		break;
		case 'D':
		case 'd': 
			/*double precision sclk*/
			d4clkdp(carray,dpclk,rtcode);
		break;
		case 'C':
		case 'c':
			d4chrclk(carray,dpclk,cstrng,rtcode);
		break;
		default:
			use_type_in = FALSE;
		break;
	}
}
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
if (strng_out == NULL)
{
	if (!use_type_in)
	{
		/*error here*/
		strcpy(rtcode,err_rtrn_code[BDI_INDX]);
	}
	return;
}
/*output string*/
d4clkout(carray,dpclk,type_out,strng_out,rtcode);
}
			
static
d4chrclk(carray,dpclk,cstrng,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*cstrng;
char	*rtcode;

{

int	len;
DBL	mills;
int	rt_code;

/*ok now dissamble this by parsing*/
strng_pntr = cstrng;
tot_sclk_fields = 0;
sclk_hex = FALSE;
/*find if hex
  cstrng = 'HEX xxxxxxxx xxxx'  or
  cstrng = '0Xxxxxxxxx 0Xxxxx'
*/
while (strng_pntr[0] == ' ')
{
	strng_pntr++;
}
len = strlen(strng_pntr);
rt_code = dictprs(strng_pntr,len,
	          strngsclkpar,spcl_chars);
if (rt_code)
{
	strcpy(rtcode,pars_error);
	return;
}
if (is_hex)
{
	/*change to hex*/
	c4ng2hex(carray,dpclk,rtcode);
	if (rtcode[3] == 'X')
	{
		/*error*/
		return;
	}
	d4clkraw(carray,dpclk,rtcode);
	return;	
}
if (sclk_hex)
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[DHX_INDX]);
	return;
}
switch(msn_case_nmbr)
{
	case ULS_INDX:
		/*Ulyses*/
		/*get raw output for uls*/
		d4chruls(cstrng,carray,dpclk,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
	break;
	case GLL_INDX:
		/*Galileo*/
		/*get raw output for GLL*/
		d4chrgll(cstrng,carray,dpclk,rtcode);
	break;
	case VGR_INDX:
		/*Voyager*/
		/*get raw output for voyager*/
		d4chrvgr(cstrng,carray,dpclk,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
	break;
	default:
	break;
}
return;
d4chkclk(carray,rtcode);
}

static
d4clkinp(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{
/*input is in carray[0] thru [3] (depending on mission)*/

switch(msn_case_nmbr)
{
	case ULS_INDX:
		/*Ulyses*/
		/*place into raw*/
		d4ulstrw(carray,(char *)&ULSRAW,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4ulstdp(carray,dpclk,rtcode);
	break;
	case GLL_INDX:
		/*Galileo*/
		d4glltrw(carray,(char *)&GLLRAW,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4glltdp(carray,dpclk,rtcode);
	break;
	case VGR_INDX:
		/*Voyager*/
		d4vgrtrw(carray,(char *)&VGRRAW,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4vgrtdp(carray,dpclk,rtcode);
	break;
	default:
	break;

}
return;
d4chkclk(carray,rtcode);
}

static
d4clkdp(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{
/*input is in dpclk */

switch(msn_case_nmbr)
{
	case ULS_INDX:
		/*Ulyses*/
		/*place into raw*/
		d4ulsfdp(carray,dpclk,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4ulstrw(carray,(char *)&ULSRAW,rtcode);
	break;
	case GLL_INDX:
		/*Galileo*/
		d4gllfdp(carray,dpclk,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4glltrw(carray,(char *)&GLLRAW,rtcode);
	break;
	case VGR_INDX:
		/*Voyager*/
		d4vgrfdp(carray,dpclk,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4vgrtrw(carray,(char *)&VGRRAW,rtcode);
	break;
	default:
	break;

}
d4chkclk(carray,rtcode);
return;
}

static
d4clkraw(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{
/*input is in carray[0] thru [3] (depending on mission)*/

switch(msn_case_nmbr)
{
	case ULS_INDX:
		/*Ulyses*/
		/*place into raw*/
		d4ulsfrw(carray,(char *)&ULSRAW,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4ulstdp(carray,dpclk,rtcode);
	break;
	case GLL_INDX:
		/*Galileo*/
		d4gllfrw(carray,(char *)&GLLRAW,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4glltdp(carray,dpclk,rtcode);
	break;
	case VGR_INDX:
		/*Voyager*/
		d4vgrfrw(carray,(char *)&VGRRAW,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		d4vgrtdp(carray,dpclk,rtcode);
	break;
	default:
	break;

}
d4chkclk(carray,rtcode);
return;
}
static
d4chkclk(carray,rtcode)

long	int	*carray;
char	*rtcode;

{

switch(msn_case_nmbr)
{
	case ULS_INDX:
		/*Ulyses*/
		/*place into raw*/
		d4chkuls(carray,rtcode);
	break;
	case GLL_INDX:
		/*Galileo*/
		d4chkgll(carray,rtcode);
	break;
	case VGR_INDX:
		d4chkvgr(carray,rtcode);
	break;
	default:
	break;

}
return;
}

static
d4clkout(carray,dpclk,type_out,strng_out,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*type_out;
char	*strng_out;
char	*rtcode;

{

int	hex_int;
char	*hex_pntr;
int	i;
char	*strng;

/****************************************************
**	      1. DPCLK - double precision clock.
**	         'x.xxxxxxxxxxxxEyy'. Input is dpclk.
**	      2. HEX - HEX SCLK output for clock.
**	         'xxxxxxxx xxxx' - 8 and 4 characters (6 bytes).
**	         Input is SCLKRAW.
**	      3. COUNTS - clocks seperated by '.' Mission dependent.
**	         1. ULS - xxxxxxxx.xxx
**	         2. VGR - xxxxxx.xx.xxx
**	         3. GLL - xxxxxxxx.xx.x.x
**
***********************************************************/

switch (type_out[0])
{
	case 'D':
	case 'd':
		/*double precision output*/
		sprintf(strng_out,"%.13E");
	break;
	case 'H':
	case 'h':
		/*hex output*/
		strng = strng_out;
		hex_pntr = (char *)&SCLKRAW;
		for (i = 0 ; i < 12 ; i += 2)
		{
			hex_int = hex_pntr[0];
			sprintf(strng,"%02X",hex_int&0Xff);
			strng += 2;
			if (i == 6)
			{
				strng[0] = ' ';
				strng++;
			}
			hex_pntr++;
		}
	break;
	case 'C':
	case 'c':
		switch (msn_case_nmbr)
		{
			case ULS_INDX:
				ulsd4splay(carray,strng_out);
			break;
			case GLL_INDX:
				glld4splay(carray,strng_out);
			break;
			case VGR_INDX:
				vgrd4splay(carray,strng_out);
			break;
		}
	break;
	default:
		if (!use_type_in)
		{
			/*error here*/
			strcpy(rtcode,err_rtrn_code[BDI_INDX]);
		}
		return;
}
return;
}			
		 
static
c4ng2hex(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

int	hex1_indx;
int	hex2_indx;
char	hex_chars[3];
int	hexint;
char	*hex_pntr;
int	i;
int	j;

hex_pntr = (char *)&SCLKRAW;
hex1_indx = 1;
if (tot_sclk_fields == 4)
{
	/*cstrng = 0Xyyyyyyyy 0Xyyyy*/
	hex2_indx = 3;
}
else
{
	/*******************
	 cstrng = HEX yyyyyyyy yyyy or
	 cstrng = 0Xyyyyyyyy yyyy
	 **********************/
	hex2_indx = 2;	
}
if (sclk_field_len[hex1_indx] != 8 ||
    sclk_field_len[hex2_indx] != 4)
{
	strcpy(rtcode,err_rtrn_code[HXN_INDX]);
	return;
}
hex_chars[2] = '\0';
for ( i = 0 ; i < 8 ; i += 2)
{
	memcpy(hex_chars,&sclk_field_pntrs[hex1_indx][i],2);
	sscanf(hex_chars,"%x",&hexint);
	hex_pntr[0] = hexint;
	hex_pntr++;
}
for ( i = 0 ; i < 4 ; i += 2)
{
	memcpy(hex_chars,&sclk_field_pntrs[hex2_indx][i],2);
	sscanf(hex_chars,"%x",&hexint);
	hex_pntr[0] = hexint;
	hex_pntr++;
}
return;
}

static
d4chruls(strng,carray,dpclk,rtcode)

char	*strng;
long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

char	cnvstrng[20];
int	i;
DBL	mills;
long	int	mill;
DBL	dpuls;
/*ulysis here*/
if (tot_sclk_fields != 2)
{
	strcpy(rtcode,err_rtrn_code[ULC_INDX]);
	return;
}
/*counter for ULS*/
strncpy(cnvstrng,sclk_field_pntrs[0],sclk_field_len[0]); 
cnvstrng[sclk_field_len[0]] = '\0';
sscanf(cnvstrng,"%ld",&ULSCOUNT);
/*millisecs input*/
strncpy(cnvstrng,sclk_field_pntrs[1],sclk_field_len[1]); 
cnvstrng[sclk_field_len[1]] = '\0';
sscanf(cnvstrng,"%ld",&ULSMILLS);
/*convert from mills to fracs*/
mills = ULSMILLS;
mills /= 1000.;
dpuls = mills * float_hex_1000;
dpuls += .5;
ULSFRAC = dpuls;
/*get raw output for uls*/
d4ulstrw(carray,(char *)&ULSRAW,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
dpclk[0] = (DBL)ULSCOUNT + mills;
return;
}

static
d4chrvgr(strng,carray,dpclk,rtcode)

char	*strng;
long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

char	cnvstrng[20];
int	i;
DBL	mills;
long	int	mill;
DBL	dpvgr;

/*ulysis here*/
if (tot_sclk_fields != 3)
{
	strcpy(rtcode,err_rtrn_code[VGC_INDX]);
	return;
}
/*Mod 16 counter*/
strncpy(cnvstrng,sclk_field_pntrs[0],sclk_field_len[0]); 
cnvstrng[sclk_field_len[0]] = '\0';
sscanf(cnvstrng,"%ld",&VGRMOD16);
/*Mod 60 counter*/
strncpy(cnvstrng,sclk_field_pntrs[1],sclk_field_len[1]); 
cnvstrng[sclk_field_len[1]] = '\0';
sscanf(cnvstrng,"%ld",&VGRMOD60);
/*Line counter (mod 800, BUT starts at 1 not zero)*/
strncpy(cnvstrng,sclk_field_pntrs[2],sclk_field_len[2]); 
cnvstrng[sclk_field_len[2]] = '\0';
sscanf(cnvstrng,"%ld",&VGRLNCNT);
/*get raw output for vgr*/
d4vgrtrw(carray,(char *)&VGRRAW,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
d4vgrtdp(carray,dpclk,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
return;
}

static
d4chrgll(strng,carray,dpclk,rtcode)

char	*strng;
long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

char	cnvstrng[20];
int	i;
DBL	mills;
long	int	mill;
DBL	dpgll;

/*ulysis here*/
if (tot_sclk_fields != 4)
{
	strcpy(rtcode,err_rtrn_code[GLC_INDX]);
	return;
}
/*rim counter*/
strncpy(cnvstrng,sclk_field_pntrs[0],sclk_field_len[0]); 
cnvstrng[sclk_field_len[0]] = '\0';
sscanf(cnvstrng,"%ld",&GLLRIMCNT);
/*Mod 91 counter*/
strncpy(cnvstrng,sclk_field_pntrs[1],sclk_field_len[1]); 
cnvstrng[sclk_field_len[1]] = '\0';
sscanf(cnvstrng,"%ld",&GLLMOD91);
/*Mod 10 counter*/
strncpy(cnvstrng,sclk_field_pntrs[2],sclk_field_len[2]); 
cnvstrng[sclk_field_len[2]] = '\0';
sscanf(cnvstrng,"%ld",&GLLMOD10);
/*Mod 8 counter*/
strncpy(cnvstrng,sclk_field_pntrs[3],sclk_field_len[3]); 
cnvstrng[sclk_field_len[3]] = '\0';
sscanf(cnvstrng,"%ld",&GLLMOD8);
/*get raw output for gll*/
d4glltrw(carray,(char *)&GLLRAW,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
d4glltdp(carray,dpclk,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
return;
}


static
d4glltrw(carray,sclk_pntr,rtcode)

long	int	*carray;
char	*sclk_pntr; 
char	*rtcode;

{

int i;

/*'d4glltrw' GLL mission from integer to raw*/

/*********************************************************************
**	    GLL - Raw format (bit stream) place in sclk_pntr.		
**	    rim counter, 24 bits, mod 91 counter, 8 bits,
**	    mod 10 counter 8 bits, mod 8 counter 8 bits.
**	    1. carray[0] - rim counter, incremented every
**	       60 2/3 seconds.
**	    2. carray[1] - Mod 91 counter, incremented every 
**	       2/3 seconds.
**	    3. carray[2] - mod 10 counter, incremented every
**	       66 2/3 milliseconds.
**	    4. carray[3] - mod 8 counter, incremented every
**	       8 2/3 milliseconds.
**********************************************************************/

/*carray[0] (24 bits), RIM counter is 3 bytes worth of info*/
shu.lngi = carray[0];
for (i = 0 ; i < 3 ; i++)
{
	sclk_pntr[i] = shu.chr[vax4_or_othr[i+1]];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}

/* carray[1] is 8bits */
shu.lngi = carray[1];
sclk_pntr[3] = shu.chr[vax4_or_othr[3]];

/* carray[2] is 8bits */
shu.lngi = carray[2];
sclk_pntr[4] = shu.chr[vax4_or_othr[3]];

/* carray[3] is 8bits */
shu.lngi = carray[3];
sclk_pntr[5] = shu.chr[vax4_or_othr[3]];

return;
}

static
d4vgrtrw(carray,sclk_pntr,rtcode)

long	int	*carray;
char	*sclk_pntr; 
char	*rtcode;

{

int	i;
char	shftr[2];
int	nshft;

/*'d4vgrtrw' VGR mission from integer iput output to sclk_pntr*/

/********************************************************************
**	  VGR - Raw format (bit stream) place in sclk_pntr.	   **
**	     1. MOD 16 - 16 bits (carray[0]).			   **
**	     2. MOD 60 - 6 bits  (carray[1]).			   **
**	     3. Line Count - 10 bits (carray[2]).		   **
********************************************************************/


/*carray[0] is 16 bits of MOD 16 data*/
shu.lngi = carray[0];
for (i = 0 ; i < 2 ; i++)
{
	sclk_pntr[i] = shu.chr[vax4_or_othr[i+2]];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}

/*carray[7] 6 bits of info*/
/* shift last byte 2 bits to left*/
nshft = 2;
d4sffl(carray[1],(char *)shftr,nshft);
sclk_pntr[2] = shftr[0];
/*carray[2] 10 bits of info*/
shu.lngi = carray[2];
for (i = 0 ; i < 2 ; i++)
{
	shftr[i] = shu.chr[vax4_or_othr[i+2]];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}
/*OR shift[0] with sclk_pntr[2]*/
sclk_pntr[2] |= shftr[0];
sclk_pntr[3] = shftr[1];

return;
}


static
d4ulstrw(carray,sclk_pntr,rtcode)

long	int	*carray;
char	*sclk_pntr; 
char	*rtcode;
{

int	i;
char	shftr[2];
int	nshft;

/*'d4ulstrw' ULS mission from integer iput output to sclk_pntr*/

/*********************************************************************
**	   ULS - Raw format (bit stream) place in sclk_pntr.	    **
**	     1. Current Corrected Counter - 4 bytes (carray [0]).   **
**	     2. Fractional count - 16 bits (carray[1]).		    **
*********************************************************************/

/*carray[0] is 32 bits of corrected counter*/
shu.lngi = ULSCOUNT;
for (i = 0 ; i < 4 ; i++)
{
	sclk_pntr[i] = shu.chr[vax4_or_othr[i]];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}

/*carray[1] is 16 bits of fractional count*/
shu.lngi = ULSFRAC;
for (i = 0 ; i < 2 ; i++)
{
	sclk_pntr[i+4] = shu.chr[vax4_or_othr[i+2]];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}

return;
}


static
d4gllfrw(carray,sclk_pntr,rtcode)

long	int	*carray;
char	*sclk_pntr; 
char	*rtcode;

{

int i;

/*'d4gllfrw' GLL mission to integer from sclk_pntr*/

/*********************************************************************
**	    GLL - Raw format (bit stream) place in sclk_pntr.		
**	    rim counter, 24 bits, mod 91 counter, 8 bits,
**	    mod 10 counter 8 bits, mod 8 counter 8 bits.
**	    1. carray[0] - rim counter, incremented every
**	       60 2/3 seconds.
**	    2. carray[1] - Mod 91 counter, incremented every 
**	       2/3 seconds.
**	    3. carray[2] - mod 10 counter, incremented every
**	       66 2/3 milliseconds.
**	    4. carray[3] - mod 8 counter, incremented every
**	       8 2/3 milliseconds.
**********************************************************************/
shu.lngi = 0;
/*carray[0] (24 bits), RIM counter is 3 bytes worth of info*/
for (i = 0 ; i < 3 ; i++)
{
	shu.chr[vax4_or_othr[i+1]] = sclk_pntr[i];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}
carray[0] = shu.lngi;
shu.lngi = 0;

/* carray[1] is 8bits */
shu.chr[vax4_or_othr[3]] = sclk_pntr[3];
carray[1] = shu.lngi;

/* carray[2] is 8bits */
shu.chr[vax4_or_othr[3]] = sclk_pntr[4];
carray[2] = shu.lngi;
shu.lngi = 0;

/* carray[3] is 8bits */
shu.chr[vax4_or_othr[3]] = sclk_pntr[5];
carray[3] = shu.lngi;

return;
}

static
d4vgrfrw(carray,sclk_pntr,rtcode)

long	int	*carray;
char	*sclk_pntr; 
char	*rtcode;

{

int	i;
char	shftr[2];
int	nshft;

/*'d4vgrfrw' VGR mission to integer from sclk_pntr*/

/*********************************************************************
**	  VGR - Raw format (bit stream) from sclk_pntr.	   	    **
**	     1. MOD 16 - 16 bits (carray[0]).			    **
**	     2. MOD 60 - 6 bits  (carray[1]).			    **
**	     3. Line Count - 10 bits (carray[2]).		    **
**********************************************************************/


/*carray[6] is 16 bits of MOD 16 data*/
shu.lngi = 0;
for (i = 0 ; i < 2 ; i++)
{
	shu.chr[vax4_or_othr[i+2]] = sclk_pntr[i];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}
carray[0] = shu.lngi;

/*carray[1] 6 bits of info*/
/* shift 2 bits to right*/
nshft = 2;
shftr[0] = sclk_pntr[2]; /*6bits of mod 16 + 2 bits of mod10*/
d4sffr(&carray[1],(char *)shftr,nshft);

/*carray[2] 10 bits of info*/
shftr[0] = sclk_pntr[2]; /*6 bits of garbage + 2 bits of line count*/
shftr[1] = sclk_pntr[3]; /* 8 bits of line count*/
shftr[0] &= 0x03; /*2 least sig bits left*/
shu.lngi = 0;
for (i = 0 ; i < 2 ; i++)
{
	shu.chr[vax4_or_othr[i+2]] = shftr[i];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}
carray[2] = shu.lngi;

return;
}


static
d4ulsfrw(carray,sclk_pntr,rtcode)

long	int	*carray;
char	*sclk_pntr; 
char	*rtcode;
{

int	i;
char	shftr[2];
int	nshft;

/*'d4ulsfrw' ULS mission from sclk_pntr to long ints*/

/*********************************************************************
**	   ULS - Raw format (bit stream) place in sclk_pntr.	    **
**	     1. Current Corrected Counter - 4 bytes (carray [0]).   **
**	     2. Fractional count - 16 bits (carray[1]).		    **
*********************************************************************/

/*carray[0] is 32 bits of corrected counter*/
for (i = 0 ; i < 4 ; i++)
{
	shu.chr[vax4_or_othr[i]] = sclk_pntr[i];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}
ULSCOUNT = shu.lngi;
shu.lngi = 0;
/*carray[1] is 16 bits of fractional count*/
for (i = 0 ; i < 2 ; i++)
{
	shu.chr[vax4_or_othr[i+2]] = sclk_pntr[i+4];
	/*the 'vax4_or_othr' takes care of byte ordering for various
	  architectures*/
}
ULSFRAC = shu.lngi;
ULSMILLS = 0;
return;
}

static
d4vgrtdp(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{


/*given integer mod 16, mod60 and line counts now get dpclk for 
  output mod 60 counter turns over every 60 counts and bumps 
  mod 16 counter line counter start at 1 counts to 800 turns 
  over mod 60 counter*/
dpclk[0] = (DBL)VGRMOD16*60. + (DBL)VGRMOD60 + (DBL)VGRLNCNT/801.;
return;
}

static	
d4ulstdp(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

DBL	mills;
DBL	dpuls;

/*ULSCOUNT is the 32 bit counter*/
/*If ULSMILLS or ULSFRAC is non zero then get mills*/
if (ULSMILLS != 0)
{
	mills = ULSMILLS;
	mills /= 1000.;
	dpuls = mills * float_hex_1000;
	dpuls += .5;
	ULSFRAC = dpuls;
}
else if (ULSFRAC != 0)
{
	mills = (DBL)ULSFRAC/float_hex_1000;
	dpuls = mills * 1000.;
	dpuls += .5;
	ULSMILLS = dpuls;
}
else
{
	mills = 0;
}
/*mills contains proper dp to add for sclk*/
dpclk[0] = (DBL)ULSCOUNT + mills;
return;
}
	

static	
d4glltdp(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

/*********************************************************************
**	    GLL - Raw format (bit stream) place in sclk_pntr.		
**	    rim counter, 24 bits, mod 91 counter, 8 bits,
**	    mod 10 counter 8 bits, mod 8 counter 8 bits.
**	    1. carray[9] - rim counter, incremented every
**	       60 2/3 seconds.
**	    2. carray[10] - Mod 91 counter, incremented every 
**	       2/3 seconds.
**	    3. carray[11] - mod 10 counter, incremented every
**	       66 2/3 milliseconds.
**	    4. carray[12] - mod 8 counter, incremented every
**	       8 2/3 milliseconds.
**	    double precision
**	       whole part is rim counter
**	       fractional part = ((mod_91*10 + mod_10)*8 + mod_8)/7280
**	       7280 = 91*8*10
**********************************************************************/
dpclk[0] = (DBL)GLLRIMCNT + 
	(DBL)((GLLMOD91*10 + GLLMOD10)*8 + GLLMOD8)/7280.;
return;
}


static
d4ulsfdp(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

/*convert ULS to integer formats*/

union {
	long	int	fracsl;
	unsigned short	int	fracss[2];
} frc;
DBL	fracsd;
DBL	dpuls;

ULSCOUNT = dpclk[0]; /*long int of sclk*/
if (dpclk[0] < 0)
{
	fracsd = (DBL)ULSCOUNT - dpclk[0];
}
else
{
	fracsd = dpclk[0] - (DBL)ULSCOUNT;
}
dpuls =  fracsd *1000. +.5;
ULSMILLS = dpuls;
frc.fracss[ORDS_INDX0] = 0;
/*fractional portion of seconds*/
fracsd *= float_hex_1000;
fracsd += .5;
frc.fracss[ORDS_INDX1] = fracsd ;
ULSFRAC  = frc.fracsl;
return;

}


static
d4vgrfdp(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

DBL dpcnt;

long	int	sclkint;
/*given floating point number find mod 16, mod 60 and line count 
  for voyager*/
sclkint =dpclk[0];
VGRMOD16 = sclkint/60;
VGRMOD60 = sclkint % 60;
dpcnt = (dpclk[0] - (DBL)sclkint)*801. + .5;
VGRLNCNT = (long int)dpcnt;
return;
}

static	
d4gllfdp(carray,dpclk,rtcode)

long	int	*carray;
DBL	*dpclk;
char	*rtcode;

{

long	int	li;
DBL	dbl;

/*********************************************************************
**	    from dp gll sclk to components
**	    GLL - Raw format (bit stream) place in sclk_pntr.		
**	    rim counter, 24 bits, mod 91 counter, 8 bits,
**	    mod 10 counter 8 bits, mod 8 counter 8 bits.
**	    1. carray[9] - rim counter, incremented every
**	       60 2/3 seconds.
**	    2. carray[10] - Mod 91 counter, incremented every 
**	       2/3 seconds.
**	    3. carray[11] - mod 10 counter, incremented every
**	       66 2/3 milliseconds.
**	    4. carray[12] - mod 8 counter, incremented every
**	       8 2/3 milliseconds.
**	    double precision
**	       whole part is rim counter
**	       fractional part = ((mod_91*10 + mod_10)*8 + mod_8)/7280
**	       7280 = 91*8*10
**********************************************************************/
  
GLLRIMCNT =dpclk[0];
if (dpclk[0] < 0)
{
	dbl = (DBL)GLLRIMCNT -dpclk[0];
}
else
{
	dbl =dpclk[0] - (DBL)GLLRIMCNT; /*decimal fraction portion*/
}
dbl *= 7280.;
dbl += .5;
li = (long int)dbl;
GLLMOD8 = li % 8;
li /= 8;
GLLMOD91 = li/10;
GLLMOD10 = li % 10;
return;
}

static
d4chkgll(carray,rtcode)

long	int	*carray;
char	*rtcode;

{

/*********************************************************************
**	    GLL - Raw format (bit stream).		
**	    rim counter, 24 bits, mod 91 counter, 8 bits,
**	    mod 10 counter 8 bits, mod 8 counter 8 bits.
**	    1. carray[0] - rim counter, incremented every
**	       60 2/3 seconds.
**	    2. carray[1] - Mod 91 counter, incremented every 
**	       2/3 seconds.
**	    3. carray[2] - mod 10 counter, incremented every
**	       66 2/3 milliseconds.
**	    4. carray[3] - mod 8 counter, incremented every
**	       8 2/3 milliseconds.
**********************************************************************/

/*rim counter 24 bits i.e. 00ffffff or 256** 3 -1 = 16777215 */
if (GLLRIMCNT > 16777215)
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[GLO_INDX]);
	return;
}

/*mod 91 counter 0 to 90*/
if (GLLMOD91 < 0  ||  GLLMOD91 > 90)
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[GLO_INDX]);
	return;
}

/*mod 10 counter*/
if (GLLMOD10 < 0  ||  GLLMOD10 > 9)
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[GLO_INDX]);
	return;
}

/*mod 8*/
if (GLLMOD8 < 0  ||  GLLMOD8 > 7)
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[GLO_INDX]);
	return;
}

return;
}

static
d4chkvgr(carray,rtcode)

long	int	*carray;
char	*rtcode;

{

/*********************************************************************
**	  VGR - Raw format (bit stream)     	    		    **
**	     1. MOD 16 - 16 bits (carray[0]).			    **
**	     2. MOD 60 - 6 bits  (carray[1]).			    **
**	     3. Line Count - 10 bits (carray[2]).		    **
**********************************************************************/

/*16 bits 0000ffff or 256 ** 2 - 1 = 65535*/
if (VGRMOD16 < 0  ||  VGRMOD16 > 65535)
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[VGO_INDX]);
	return;
}

/*MOD 60 count*/
if (VGRMOD60 < 0  ||  VGRMOD60 > 59)
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[VGO_INDX]);
	return;
}

/*line count goes from 1 to 800*/
if (VGRLNCNT < 1  ||  VGRLNCNT > 800)
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[VGO_INDX]);
	return;
}

return;
}
	


static
d4chkuls(carray,rtcode)

long	int	*carray;
char	*rtcode;

{
/*no check at this time*/
return;
}

	
static
d4omsn(msn_in,rtcode)

char	*msn_in;
char	*rtcode;

{

strcpy(msn,msn_in);
switch (msn_in[0])
{
	case 'U':
	case 'u':
		/*Ulysis*/
		msn_case_nmbr = ULS_INDX;
	break;
	case 'V':
	case 'v':
		/*Voyager*/
		msn_case_nmbr = VGR_INDX;
	break;
	case 'G':
	case 'g':
		/*Gallileo*/
		msn_case_nmbr = GLL_INDX;
	break;
	default:
		/*unknown*/
	break;
}
return;
}

int  
strngsclkpar(indx_of_wrd,len_wrd,is_spcl_chars)

int  indx_of_wrd;   /*index in buffer*/
int  len_wrd;       /*length, in bytes, of 1st word.*/
BOOL is_spcl_chars; /*TRUE this is a special character
                      FALSE it is not*/
                        
{
int	i;

if (len_wrd < 0)
{
	return(1);
}
if (is_spcl_chars)
{
	if (strng_pntr[indx_of_wrd] == 'x'  ||
	    strng_pntr[indx_of_wrd] == 'X')
	{
		is_hex = TRUE;
	}
	return(0);
}
if (tot_sclk_fields >= 4)
{
	strcpy(pars_error,err_rtrn_code[FLD_INDX]);
	return(-1);
}
sclk_field_type[tot_sclk_fields] = 'i';
sclk_field_pntrs[tot_sclk_fields] = &strng_pntr[indx_of_wrd];
sclk_field_len[tot_sclk_fields] = len_wrd;
for ( i = 0 ; i < len_wrd ; i++)
{
	if (!isdigit(sclk_field_pntrs[tot_sclk_fields][i]))
	{
		if (!isxdigit(sclk_field_pntrs[tot_sclk_fields][i]))
		{	
			/*this is not a digit or hex*/
			sclk_field_type[tot_sclk_fields] = 'c';
			break;
		}
		sclk_hex = TRUE;
	}
}
		
tot_sclk_fields++;
return(0);
}

static	
glld4splay(carray,clkstrng)

long	int	*carray;
char	*clkstrng;

{

/*********************************************************************
**	    from dp gll sclk to components
**	    GLL - Raw format (bit stream) place in sclk_pntr.		
**	    rim counter, 24 bits, mod 91 counter, 8 bits,
**	    mod 10 counter 8 bits, mod 8 counter 8 bits.
**	    1. sarray[9] - rim counter, incremented every
**	       60 2/3 seconds.
**	    2. sarray[10] - Mod 91 counter, incremented every 
**	       2/3 seconds.
**	    3. sarray[11] - mod 10 counter, incremented every
**	       66 2/3 milliseconds.
**	    4. sarray[12] - mod 8 counter, incremented every
**	       8 2/3 milliseconds.
**	    double precision counter
**	       whole part is rim counter
**	       fractional part = ((mod_91*10 + mod_10)*8 + mod_8)/7280
**	       7280 = 91*8*10
**********************************************************************/

/*7+1+2+1+1+1 = 13 need 30 - 7 = 23  */
sprintf(clkstrng,"%ld.%02ld.%ld.%ld",
	GLLRIMCNT,
	GLLMOD91,
	GLLMOD10,
	GLLMOD8);
return;
}


static
ulsd4splay(carray,clkstrng)

long	int	*carray;
char	*clkstrng;

{

/*********************************************************************
**	   ULS - Raw format (bit stream) place in sclk_pntr.	    **
**	     1. Current Corrected Counter - 4 bytes (sarray [6]).   **
**	     2. Fractional count - 16 bits (sarray[7]).		    **
*********************************************************************/
sprintf(clkstrng,"%ld.%03ld",
	ULSCOUNT,
	ULSMILLS);
return;
}

vgrd4splay(carray,clkstrng)

long	int	*carray;
char	*clkstrng;

{


/********************************************************************
**	  VGR - Raw format (bit stream) place in sclk_pntr.	   **
**	     1. MOD 16 - 16 bits (sarray[6]).			   **
**	     2. MOD 60 - 6 bits  (sarray[7]).			   **
**	     3. Line Count - 10 bits (sarray[8]).		   **
********************************************************************/
sprintf(clkstrng,"%ld.%02ld.%03ld",
	VGRMOD16,
	VGRMOD60,
	VGRLNCNT);
return;
}


static
d4cnvfhx(strng,intout,len)

char	*strng;
long	int	*intout; /*will cast to specifications*/
int	len;

{

/*convert from hex chracters*/
int	j;
int	k;
char	hexchars[3];
union
{
	long	int	li;
	short	int	si[2];
	char	ch[4];
} iu;

/*len is 4 (bytes) or 2 (bytes)*/
iu.li = 0;
if (len == 4)
{
	k = 0;
}
else
{
	k = 2;
}
for ( j = 0 ; j < len ; j++)
{
	strncpy(hexchars,(char *)&strng[j*2],2);
	/*find hex  from character string*/
	sscanf(hexchars,"%X",
	       &iu.ch[vax4_or_othr[k]]);
	k++;
}
intout[0] = iu.li;
return;
}
 
static	int
d4sffr(lngi,shft,nshft)

long	int	*lngi;
char	*shft;
int	nshft;

{
int	nbytes;
int	i;
int	k;

/*shift to long from character pointer. Shift right >>*/
nbytes = nshft/8;
if (nshft % 8)
{
	nbytes++;
}
k = 4 - nbytes;
if (SIZE_INT == 2)
{
	/*ok do this one*/
	if (nbytes > 2) 
	{
		return(-1);
	}
	shu.lngi = 0;
	for ( i = 0 ; i < nbytes ; i++)
	{
		shu.chr[vax4_or_othr[i+k]] = shft[i]; 
		/*this places it in proper byte no matter what 
		  archtecture*/
	}
	/*now right shift it*/
	lngi[0] = (long int)(shu.shi[1] >> nshft);
}
else if (SIZE_INT == 4)
{
	/*this is easier*/
	for ( i = 0 ; i < nbytes ; i++)
	{
		shu.chr[vax4_or_othr[i+k]] = shft[i]; 
		/*this places it in proper byte no matter what 
		  archtecture*/
	}
	/*now right shift it*/
	lngi[0] = shu.lngi >> nshft;
}
else
{
	return(-1);
}
return(0);
}

static	int
d4sffl(lngi,shft,nshft)

long	int	lngi;
char	*shft;
int	nshft;

{

int	nbytes;
int	i;
int	k;

/*shift left (<<) from long to character pointer*/
nbytes = nshft/8;
if (nshft % 8)
{
	nbytes++;
}
k = 4 - nbytes;
if (SIZE_INT == 2)
{
	if (nbytes > 2) 
	{
		return(-1);
	}
	shu.lngi = lngi;
	shu.shi[1] <<= nshft;
	for ( i = 0 ; i < nbytes ; i++)
	{
		shft[i] = shu.chr[vax4_or_othr[k+i]]; 
		/*this places it in proper byte no matter what 
		  archtecture*/
	}
}
else if (SIZE_INT == 4)
{
	/*this is easier*/
	shu.lngi = lngi;
	shu.lngi <<= nshft;
	for ( i = 0 ; i < nbytes ; i++)
	{
		shft[i] = shu.chr[vax4_or_othr[k+i]]; 
		/*this places it in proper byte no matter what 
		  archtecture*/
	}
}
else
{
	return(-1);
}
return(0);
}


