/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*
** MANUAL
**	CCSDSCNV 3x "January 13, 1984"
**
** NAME
**	ccsdscnv - convert CCSDS time codes to text output.
**
** SYNOPSIS
**
**	  int ccsdscnv(type_ccsds,ccsds_code,epoch,strng_out);
**
**	  char  *type_ccsds;
**	  char  *ccsds_code;
**	  char  *epoch;
**	  char  *strng_out;
**
** PARAMETERS
**	
**	  1. type_ccsds - char * - points to one of the following.
**	     1. 'CCSDSDAYSG' Day segmented time code.
**	     2. 'CCSDSDOYSG' Calendar DOY segmented time code.
**	     3. 'CCSDSBUNSG' Unsegmented time code.
**	     4. 'CCSDSYMDSG' Calendar DOY segmented time code.
**	  2. ccsds_code - char * - points to CCSD input of type
**	     'type_ccsds'.
**	  3. epoch - char *& - points to the epoch in ASCII.
**	  4. strng_out - char * - points to the address of where
**	     place the output strng.
**
** DESCRIPTION
**
**	Consultive Commitee for Space Data Systes (CCSDS) time formats.
**	The CCSDS time code formats (none ASCII) contain a preamble
**	field 'P' and a time field 'T'. The preamble field is optional
**	and is not used. For CCSDS the binary times 
**	are past the user supplied epoch. If there is no user supplied 
**	epoch then the default epoch is 1950. The different non ASCII 
**	CCSDS inputs, identified by 'ccsds_type' are input as a byte
**	stream in 'ccsds_code'. These CCSDS times are as follows.
**	
**	The CCSDS unsegmented time code is a binary time calculated
**	from some epoch. If there is no user supplied epoch then 1950 
**	is used. This time is the same as integer seconds past an epoch
**	plus the fractional seconds but is input and out put as a byte
**	stream. The time consists of 4 bytes of course time (4 byte
**	integer seconds past an epoch) then 2 bytes of fractional 
**	seconds. 'ccsds_type' is 'CCSDSBUNSG' and the input is in
**	'ccsds_code'. The ASCII string output in 'strng_out' is
**	   'xxxxxxxx.xxx seconds past {epoch}'
**
**	The CCSDS day segmented time code is a binary time calculated
**	from some epoch. If there is no user supplied epoch then 1950 
**	is used. This time consists of 3 bytes of days past past the 
**	epoch then 4 bytes of milliseconds in the day. 'ccsds_type' is
**	'CCSDSDAYSG' and the input byte stream is in 'ccsds_code'.
**	The ASCII string output in 'strng_out' is 
**	   'xxxxx days xxxxx milliseconds past {epoch}' 
**
**	CCSDS calendar segmented time codes. There are two different
**	calendar segmented time codes. Both time codes are in segmented
**	BCD (Binary Coded Decimal). They are input in 'ccsds_code'. 
**	They are:
**	  1. CCSDSYMDSG - Year Month Day segmented and consists
**	     of YYYYMNDDHHMMSSFFFz. The ASCII string output in 
**	     'strng_out' is 'YYYY MN DD HH:MM:SS.FFF
**	  2. CCSDSDOYSG - DOY segmented time and consists of
**	     YYYYzDOYHHMMSSFFFz. Where 'z' is 4 bits set to zero.
**	     The ASCII string output in 'strng_out' is 
**	     'YYYY DOY HH:MM:SS.FFF'
**	     
**	{epoch} is 'epoch'. If epoch is NULL then {epoch} is Jan 1
**	1950 00:00:00.
**
*/
#endif Header

#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"CCSDSCN0") csect(STATIC,"CCSDSCN1")
#endif
#include	<stdio.h>
#include	<string.h>
#include	<sftypes.h>

#if (NOPROTO)
extern	int	d5frac2mills();
extern	int	mchintst();
#else
extern	int	d5frac2mills(long int *,long int *,long int *,DBL *);
extern	int	mchintst(int *,int *);
#endif

#define		DAYSG_INDX		0
#define		DOYSG_INDX		1
#define		UNSG_INDX		2
#define		YMDSG_INDX		3

static	char	*ccsds_types[] = {
"CCSDSDAYSG",  /* Day segmented time code. */
"CCSDSDOYSG",  /*Calendar DOY segmented time code.*/
"CCSDSBUNSG",   /*Unsegmented time code.*/
"CCSDSYMDSG",   /*Calendar DOY segmented time code.*/
NULL};
static	char	byte_order[4];
static	int	bi[4];

int
ccsdscnv(ccsds_type,ccsds_code,epoch,strng_out)

char	*ccsds_type;
char	*ccsds_code;
char	*epoch;
char	*strng_out;

{

DBL	dbl_mills;
int	i;
long	int	daysecs;
long	int	millsetc;
union	{
	long	int	li;
	char	chr[4];
} u;
char	use_epoch[24];


/*find byte order*/
if (strcmp(byte_order,"YES"))
{
	mchintst(NULL,bi);
	strcpy(byte_order,"YES");
}
i = 0;
for (;;)
{
	if (ccsds_types[i] == NULL)
	{
		return(-1); /*not found*/
	}
	if (!strncmp(ccsds_types[i],ccsds_type,10))
	{
		break;
	}
	i++;
}

switch (i)
{
	case DAYSG_INDX:
/**********************************************************************
**	The CCSDS day segmented time code is a binary time calculated
**	from some epoch. If there is no user supplied epoch then 1950 
**	is used. This time consists of 3 bytes of days past past the 
**	epoch then 4 bytes of milliseconds in the day. 'ccsds_type' is
**	'CCSDSDAYSG' and the input byte stream is in 'ccsds_code'.
**	The ASCII string output in 'strng_out' is 
**	   'xxxxx days xxxxx milliseconds past {epoch}' 
***********************************************************************/
		/*check for minus sign first bit in ccsds_code[0]*/
		if (ccsds_code[0]&0x80)
		{
			/*minus here*/
			u.li = -1;
		}
		else
		{
			u.li = 0;
		}
		/*start with least sig and work up*/
		u.chr[bi[0]] = ccsds_code[2];
		u.chr[bi[1]] = ccsds_code[1];
		u.chr[bi[2]] = ccsds_code[0];
		daysecs = u.li;
		u.chr[bi[0]] = ccsds_code[6];
		u.chr[bi[1]] = ccsds_code[5];
		u.chr[bi[2]] = ccsds_code[4];
		u.chr[bi[3]] = ccsds_code[3];
		millsetc = u.li;
		sprintf(strng_out,"%ld days %ld milliseconds past ",
			daysecs,millsetc);
		switch(ccsds_type[10])
		{
			case '5': /*1950*/
				sprintf(&strng_out[strlen(strng_out)],
					"1950");
			break;
			case '2': /*2000*/
				sprintf(&strng_out[strlen(strng_out)],
					"2000");
			break;
			case 'E': /*epoch of choice*/
				sprintf(&strng_out[strlen(strng_out)],
					"%s",epoch);
			break;
			default:
			break;
		}
	break;
	case DOYSG_INDX:
/**********************************************************************
**	In BCD (Binary Coded Decimal)
**	CCSDSDOYSG - DOY segmented time and consists of
**	YYYYzDOYHHMMSSFFFz. Where 'z' is 4 bits set to zero.
**	The ASCII string output in 'strng_out' is 
**	   'YYYY DOY HH:MM:SS.FFF'
***********************************************************************/
		sprintf(strng_out,
			"%02x%02x%02x%02x %02x:%02x:%02x.%02x%02x",
			ccsds_code[0]&0xff, /*YY*/
			ccsds_code[1]&0xff, /*YY*/
			ccsds_code[2]&0xff, /*0D*/
			ccsds_code[3]&0xff, /*OY*/
			ccsds_code[4]&0xff, /*HH*/
			ccsds_code[5]&0xff, /*MM*/
			ccsds_code[6]&0xff, /*SS*/
			ccsds_code[7]&0xff, /*FF*/
			ccsds_code[8]&0xff);/*F0*/
		/*place blanks where 'z's are*/
		strng_out[4] = ' ';
		strng_out[21] = ' ';
	break;
	case UNSG_INDX:
/**********************************************************************
**	The CCSDS unsegmented time code is a binary time calculated
**	from some epoch. If there is no user supplied epoch then 1950 
**	is used. This time is the same as integer seconds past an epoch
**	plus the fractional seconds but is input and out put as a byte
**	stream. The time consists of 4 bytes of course time (4 byte
**	integer seconds past an epoch) then 2 bytes of fractional 
**	seconds. 'ccsds_type' is 'CCSDSBUNSG' and the input is in
**	'ccsds_code'. The ASCII string output in 'strng_out' is
**	   'xxxxxxxx.xxx seconds past {epoch}'
***********************************************************************/
		u.chr[bi[0]] = ccsds_code[3];
		u.chr[bi[1]] = ccsds_code[2];
		u.chr[bi[2]] = ccsds_code[1];
		u.chr[bi[3]] = ccsds_code[0];
		daysecs = u.li;
		u.li = 0;
		u.chr[bi[0]] = ccsds_code[5];
		u.chr[bi[1]] = ccsds_code[4];
		d5frac2mills(&daysecs,&u.li,&millsetc,&dbl_mills);
		if (millsetc < 0)
		{
			millsetc *= -1;
		}
		sprintf(strng_out,"%ld.%03ld seconds from ",
			daysecs,millsetc);
		switch(ccsds_type[10])
		{
			case '5': /*1950*/
				sprintf(&strng_out[strlen(strng_out)],
					"1950");
			break;
			case '2': /*2000*/
				sprintf(&strng_out[strlen(strng_out)],
					"2000");
			break;
			case 'E': /*epoch of choice*/
				sprintf(&strng_out[strlen(strng_out)],
					"%s",epoch);
			break;
			default:
			break;
		}
	break;
	case YMDSG_INDX:
/**********************************************************************
**	In BCD (Binary Coded Decimal)
**	CCSDSYMDSG - Year Month Day segmented and consists
**	of YYYYMNDDHHMMSSFFFz. The ASCII string output in 
**	'strng_out' is 'YYYY MN DD HH:MM:SS.FFF
***********************************************************************/
		sprintf(strng_out,
			"%02x%02x %02x %02x %02x:%02x:%02x.%02x%02x",
			ccsds_code[0]&0xff, /*YY*/
			ccsds_code[1]&0xff, /*YY*/
			ccsds_code[2]&0xff, /*MN*/
			ccsds_code[3]&0xff, /*DD*/
			ccsds_code[4]&0xff, /*HH*/
			ccsds_code[5]&0xff, /*MM*/
			ccsds_code[6]&0xff, /*SS*/
			ccsds_code[7]&0xff, /*FF*/
			ccsds_code[8]&0xff);/*F0*/
		/*place blanks where 'z's are*/
		strng_out[23] = ' ';		
	break;
	default:
		return(-1);
}
	
return(0);
}
