/***********************************************************************
TITLE:	sclkscet.c
CIN:	sclk2scet.c,v 1.1.1.1 1994/08/31 19:22:02 root Exp

FILE NAME:	sclk2scet.c
AUTHOR:		David A. Wagner
CREATED: 	January 4, 1991
LAST UPDATE: 	1994/08/31 19:22:02
VERSION:	1.1.1.1
COMPILER:	cc and msc 6.0
OS:		SunOS 4.1 or MSDOS 3.3
REFERENCES:
WARNINGS:	Must call read_sclk() first to load SCLK_TABLE with a
		valid mission SCLKSCET file.
BUGS:		Currently no default on epoch: chould be NOW or
		last row (SCET0) in SCLKSCET file.
UNIT TEST:	Compile with -DDEBUG and -DUNIT_TEST
WAIVERS:

CHANGE HISTORY:
  sclk2scet.c,v
 * Revision 1.1.1.1  1994/08/31  19:22:02  root
 * Initial Release
 *
 * Revision 1.4  1992/12/31  04:17:12  kevin
 * corrected RCS keywords
 *
 * Revision 1.3  1991/06/17  21:42:48  dsfa
 * changed sclkrows to sclk_rows
 *
 * Revision 1.2  1991/06/05  17:50:41  dsfa
 * changed readsclk to read_sclk
 *
 * Revision 1.1  1991/03/13  17:42:03  dsfa
 * Initial revision
 *
 *
 *      routine                 called_by
 *      sclktoscet              
***********************************************************************/
#ifdef Header
/*
** MANUAL
**	sclkscet 3 "1.1.1.1"
** NAME
**	sclktoscet -- convert a SCLK value to SCET using SCLKSCET file
** SYNOPSIS
**	sclktoscet(sclk, scet, epoch)
**		double sclk;
**		double * scet;
**		double epoch;
**
** DESCRIPTION
**	__ sclktoscet() converts the given SCLK value to SCET.  This assumes
**	that the function __ read_sclk() has already been called and an
**	appropriate SCLK calibration file (SCLKSCET file) has already been
**	read into memory and is accessible through the table pointer SCLK_TABLE.
**	Since SCLK is not guaranteed to be unique over the life of a mission
**	the function requires an epoch to disambiguate the given sclk
**	(in general).  The epoch is used to determine a starting point in the
**	SCLKSCET file for a search for the given SCLK (a minimum SCET).
**	What actually happens is that the __ read_sclk function calculates a
**	unique dbsclk value for each sclk value read from the sclkscet file
**	by adding a value such that every successive dbsclk is greater than
**	its predecessor.  __ sclktoscet will first look up the epoch and
**	use the dbsclk and sclk found in that row to determine the offset
**	value.  This value is added to the given sclk to arrive at a dbsclk
**	which can then be looked up unambiguously in the table.
**
**	__ sclktoscet then uses the located coefficients to convert the
**	given sclk to scet according to the following linear formula:
**
**	 SCET = SCET0(dbsclk) + SCLKRATE(dbsclk) * (sclk - SCLK0(dbsclk))
**
** WARNINGS
**	Currently the function __ read_sclk will determine the mission of the
**	given SCLKSCET file by reading the file header.  This identification
**	is stored in a global variable, __ sclk_mission but is not checked
**	internally for correctness for the current application.  It is up to
**	the application to verfiy that the correct SCLKSCET file was read.
**
** FILES
**	SCLKSCET files are assumed to be in SFOC SFDU format.
**
** AUTHOR
**	David A. Wagner, Jet Propulsion Laboratory
**
** SEE ALSO
**	__ read_sclk(3) __ sclkfind(3)
*/
#endif Header


#include <stdio.h>
#include "sclkscet.h"
#include "scegen.h"

//-------------------------------------------------
int
sclktoscet(sclk, scet, epoch)
	double		sclk;
	double		* scet;
	double		epoch;
{
	sclktblptr	found;
	sclktbl		key;
	double		dbsclk;

	/* sanity checks */
	if (SCLK_TABLE == NULL)
		return(ERROR);

	DEBUG_PRN("DEBUG: Converting sclk = %lf\n", sclk);

	/* Since sclk is not guaranteed to be unique we need to provide
	** an ambiguity eliminator.  In this case we use the supplementary
	** argument 'epoch' to figure out how to convert sclk to dbsclk.
	** we then search for coefficients in the table by dbsclk.
	** dbsclk is simply sclk plus some offset.
	** Thus, the first step is to look up epoch and determine the
	** offset between sclk0 and dbsclk0.  That offset is added to
	** sclk to get a dbsclk which we then use to look the coefficients
	** to use in the conversion to scet.
	*/
	key.scet0 = epoch;
	if (epoch <= 0.0)
	{
		found = SCLK_TABLE;
	} else {
		if ((found = sclkfind(&key, compare_scet)) == NULL)
			return(ERROR);
	}

	/* dbsclk = sclk + offset */
	dbsclk = sclk + (found->dbsclk0 - found->sclk0);

	DEBUG_PRN2("DEBUG: epoch dbsclk = %lf sclk = %lf\n", found->dbsclk0, found->sclk0);
	DEBUG_PRN("DEBUG: sclk converts to dbsclk of %lf\n", dbsclk);

	key.dbsclk0 = dbsclk;
	if ((found = sclkfind(&key, compare_dbsclk)) == NULL)
		return(ERROR);

	DEBUG_PRN("DEBUG: using sclk0 = %lf\n", found->sclk0);
	*scet = found->scet0 + found->sclkrate * (sclk - found->sclk0);
	return(OK);
}
//------------------------------------------------------------------------

#ifdef UNIT_TEST

main()
{
	double sclk, scet;
	sclktblptr	x;
	char buff[100];
	int n;

	if (read_sclk("mosclk.dat") <= 0)
	{
		fprintf(stderr, "TEST: read_sclk failed");
		exit();
	}

	printf("Parsed file Ok:");
	printf("Table = %p\n", SCLK_TABLE );

	for (n = 0, x = SCLK_TABLE; n < sclk_rows; n++, x++)
	{
		fprintf(stderr, "k0[%d] = %lf  dk0 = %lf t0[] = %lf\n", 
			n,
			x->sclk0,
			x->dbsclk0,
			x->scet0 );
	}

	/* look up some values... */
	if ( sclktoscet((sclk=86500.0), &scet, SCLK_TABLE->scet0) == ERROR )
	{
		fprintf(stderr, "Didn't work!!!\n");
		exit();
	}

	printf("converted scet = %lf\n", scet);
	seconds_to_ascii_time(scet, 0, buff);
	printf("sclk = %lf  scet = %s\n", sclk, buff);

	fflush(stderr);
	exit();
}
#endif

