
/***********************************************************************
TITLE:	scetsclk.c
CIN:	scet2sclk.c,v 1.1.1.1 1994/08/31 19:22:02 root Exp

FILE NAME:	scet2sclk.c
AUTHOR:		David A. Wagner
CREATED: 	January 4, 1991
LAST UPDATE: 	1994/08/31 19:22:02
Modified:   Ray Bambery - 11-18-2020
VERSION:	1.1.3  
COMPILER:	gcc 4.5.3
OS:		    Solaris 10.2
REFERENCES:
WARNINGS:	Must call readsclk() first to load SCLK_TABLE with a
		valid mission SCLKSCET file.
BUGS:
UNIT TEST:	Compile with -DDEBUG and -DUNIT_TEST
WAIVERS:

CHANGE HISTORY:
  scet2sclk.c,v
 * Revision 1.1.1.1  1994/08/31  19:22:02  root
 * Initial Release
 *
 * Revision 1.2  1992/12/31  04:17:12  kevin
 * corrected RCS keywords
 *
 * Revision 1.1  1992/02/21  17:10:49  dsfa
 * Initial revision
 *
 *      routine              called_by
 *      scettosclk
 *
***********************************************************************/
#ifdef Header
/*
** MANUAL
**	scetsclk 3 "1.1.1.1"
** NAME
**	scettosclk -- convert a SCET value to SCLK using SCLKSCET file
** SYNOPSIS
**	scettosclk(scet, sclk)
**		double scet;
**		double * sclk;
**
** DESCRIPTION
** __ scettosclk() converts the given SCET value to SCLK.  This assumes
** that the function __ readsclk() has already been called and an
** appropriate SCLK calibration file (SCLKSCET file) has already been
** read into memory and is accessible through the table pointer SCLK_TABLE.
** NROFF .TP
** __ scettosclk then uses the located coefficients to convert the
** given sclk to scet according to the following linear formula:
**	SCLK = SCLK0(scet) + (scet - SCET0(scet))/ SCLKRATE(scet)
**
** WARNINGS
** Currently the function __ readsclk will determine the mission of the
** given SCLKSCET file by reading the file header.  This identification
** is stored in a global variable, __ sclkmission but is not checked
** internally for correctness for the current application.  It is up to
** the application to verfiy that the correct SCLKSCET file was read.
**
** FILES
**	SCLKSCET files are assumed to be in SFOC SFDU format.
**
** AUTHOR
**	David A. Wagner, Jet Propulsion Laboratory
**
** SEE ALSO
**	__ readsclk(3) __ sclkfind(3)
*/
#endif Header


#include <stdio.h>
#include <math.h>
#include "sclkscet.h"
#include "sclkConvert.h"
//------------------------------------------------------------------------
int
scettosclk(scet, sclk)
	double		scet;
	double		* sclk;
{
	sclktblptr	found;
	sclktbl		key;

	/* sanity checks */
	if (SCLK_TABLE == NULL)
		return(ERROR);

	DEBUG_PRN("Converting scet = %lf\n", scet);

	key.scet0 = scet;
	if ((found = sclkfind(&key, compare_scet)) == NULL)
		return(ERROR);

	DEBUG_PRN2("using sclk0=%lf rate=%lf\n", found->sclk0, found->sclkrate);
	*sclk = found->sclk0 + (scet - found->scet0) / found->sclkrate;
	return(0);
}

//--------------------------------------------------------------
#ifdef UNIT_TEST
char buff[100] = "1992-002T00:00:00.000";

main()
{
	double sclk, scet;
	sclktblptr	x;
	int n;

	if (readsclk("mosclk.dat") <= 0)
	{
		fprintf(stderr, "TEST: readsclk failed");
		exit();
	}

	DEBUG_MSG("Parsed file Ok:");
	DEBUG_PRN("Table = %p\n", SCLK_TABLE );

	for (n = 0, x = SCLK_TABLE; n < sclkrows; n++, x++)
	{
		fprintf(stderr, "k0[%d]=%lf dk=%lf t0=%lf r=%lf\n", 
			n,
			x->sclk0,
			x->dbsclk0,
			x->scet0,
			x->sclkrate	);
	}

	std_time_to_seconds(buff, &scet);

	/* look up some values... */
	if ( scettosclk(scet, &sclk) == ERROR )
	{
		fprintf(stderr, "Didn't work!!!\n");
		exit();
	}

	printf("scet2sclk: converted sclk = %lf\n", sclk);
	any_sclk_from_double(5, sclk, buff);
	printf("scet2sclk: converted sclk = (%lf) = %s\n", sclk, buff);

	fflush(stderr);
	exit();
}

#endif




