/*
 * NOSA HEADER START
 *
 * The contents of this file are subject to the terms of the NASA Open
 * Source Agreement (NOSA), Version 1.3 only (the "Agreement").  You may
 * not use this file except in compliance with the Agreement.
 *
 * You can obtain a copy of the agreement at
 *   docs/NASA_Open_Source_Agreement_1.3.txt
 * or
 *   https://cdaweb.gsfc.nasa.gov/WebServices/NASA_Open_Source_Agreement_1.3.txt.
 *
 * See the Agreement for the specific language governing permissions
 * and limitations under the Agreement.
 *
 * When distributing Covered Code, include this NOSA HEADER in each
 * file and include the Agreement file at
 * docs/NASA_Open_Source_Agreement_1.3.txt.  If applicable, add the
 * following below this NOSA HEADER, with the fields enclosed by
 * brackets "[]" replaced with your own identifying information:
 * Portions Copyright [yyyy] [name of copyright owner]
 *
 * NOSA HEADER END
 *
 * Copyright (c) 2010-2023 United States Government as represented by
 * the National Aeronautics and Space Administration. No copyright is
 * claimed in the United States under Title 17, U.S.Code. All Other
 * Rights Reserved.
 *
 */

package gov.nasa.gsfc.spdf.cdas.test;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.Authenticator;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;

import jakarta.xml.bind.JAXBContext;
import jakarta.xml.bind.JAXBException;
import jakarta.xml.bind.JAXBElement;
import jakarta.xml.bind.Marshaller;
import jakarta.xml.bind.Unmarshaller;

import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.DatatypeConfigurationException;

import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.client.Invocation;
import jakarta.ws.rs.client.Invocation.Builder;
import jakarta.ws.rs.client.WebTarget;


import org.w3._1999.xhtml.Html;

import gov.nasa.gsfc.cdaweb.schema.DatasetRequest;
import gov.nasa.gsfc.cdaweb.schema.DataRequest;
import gov.nasa.gsfc.cdaweb.schema.DataResult;
import gov.nasa.gsfc.cdaweb.schema.EnumTextFormat;
import gov.nasa.gsfc.cdaweb.schema.FileDescription;
import gov.nasa.gsfc.cdaweb.schema.ObjectFactory;
import gov.nasa.gsfc.cdaweb.schema.TextRequest;
import gov.nasa.gsfc.cdaweb.schema.TimeInterval;


/**
 * This class represents an example client of Coordinated Data Analysis
 * System (CDAS) Web services.  It utilizes the JAX-RS 2 client API.
 *
 * @author B. Harris
 */
public class JaxRsExample {

    /**
     * CDAS web service target.
     */
    private WebTarget cdas;

    /**
     * Client of CDAS RESTful Web services.
     */
    private Client client = ClientBuilder.newClient();

    /**
     * Last modified date from most recent response.
     */
    String lastModified = null;

    private static final String USER_AGENT = "JaxRsExample";


    /**
     * JAXB context for CDAS XML data.
     */
    private JAXBContext cdasJaxbContext =
        JAXBContext.newInstance("gov.nasa.gsfc.cdaweb.schema");

    /**
     * JAXB context for XHTML data.
     */
    private JAXBContext xhtmlJaxbContext =
        JAXBContext.newInstance("org.w3._1999.xhtml");


    private DatatypeFactory datatypeFactory = null;

    private ObjectFactory cdasFactory = new ObjectFactory();


    /**
     * Creates a JaxRsExample.
     *
     * @param endpoint base URL of CDAS Web services
     */
    public JaxRsExample(
        String endpoint)
        throws JAXBException {

        cdas = client.target(endpoint);

        try {

            datatypeFactory = DatatypeFactory.newInstance();
        }
        catch (DatatypeConfigurationException e) {

            System.err.println(
                "DatatypeFactory.newInstance failed with exception: " +
                e.getMessage());
        }

    }


    /**
     * Executes this example.
     */
    public void execute()
        throws Exception {

        DataResult dataResult = getTextData("sp_phys");

        if (dataResult == null) {

            System.out.println("getTextData returned no data");
            return;
        }

        System.out.println("DataResult:");
        marshal(System.out, dataResult);

/*
        display("Statuses", dataResult.getStatus());
        display("Messages", dataResult.getMessage());
        display("Warnings", dataResult.getWarning());
        display("Errors", dataResult.getError());

        System.out.println("\nResult Files:");
        for (FileDescription fileDescription : 
             dataResult.getFileDescription()) {

            String resultFilename = fileDescription.getName();
            System.out.println("  " + resultFilename);

            try {

                URL resultUrl = new URL(resultFilename);

                BufferedReader resultReader = 
                    new BufferedReader(
                            new InputStreamReader(
                                    resultUrl.openStream()));

                String line;
                for (int i = 0; i < 35 && 
                     ((line = resultReader.readLine()) != null); 
                     i++) {

                    System.out.println(line);
                }
            }
            catch (MalformedURLException e) {

                System.out.println(e.getMessage());
            }
        }
*/
    }

    private void display(String title, List<String> values) {

        if (values.size() > 0) {

            System.out.println("\n" + title + ":");

            for (String msg : values) {

                System.out.println("  " + msg);
            }
        }
    }


    private DataResult getTextData(
        String dataview) 
        throws JAXBException {

        TimeInterval timeInterval = cdasFactory.createTimeInterval();
        timeInterval.setStart(
            datatypeFactory.newXMLGregorianCalendar(
                "2013-06-01T00:00:00.000Z"));
        timeInterval.setEnd(
            datatypeFactory.newXMLGregorianCalendar(
                "2013-06-01T01:00:00.000Z"));

        DatasetRequest datasetRequest = cdasFactory.createDatasetRequest();
        datasetRequest.setDatasetId("AC_H2_MFI");
        datasetRequest.getVariableName().add("Magnitude");

        TextRequest textRequest = cdasFactory.createTextRequest();
        textRequest.setTimeInterval(timeInterval);
        textRequest.setDatasetRequest(datasetRequest);
        textRequest.setFormat(EnumTextFormat.CSV_2);

        JAXBElement<TextRequest> textRequestElement =
            cdasFactory.createTextRequest(textRequest);

        DataRequest dataRequest = cdasFactory.createDataRequest();
        dataRequest.setDataRequestEntity(textRequestElement);

// marshal(System.out, dataRequest);

        Entity<DataRequest> dataRequestEntity =
            Entity.entity(dataRequest, MediaType.APPLICATION_XML);

        Invocation.Builder request =
            cdas.path("dataviews").path(dataview).path("datasets").
                 request(MediaType.APPLICATION_XML);

        if (lastModified != null) {

            System.out.println("Setting " + 
                HttpHeaders.IF_MODIFIED_SINCE + " to " + lastModified);

            request.header(HttpHeaders.IF_MODIFIED_SINCE, lastModified);
        }
        Invocation invocation =
            request.header("User-Agent", USER_AGENT).
                buildPost(dataRequestEntity);

        System.out.println("Invoking HTTP POST");

        Response response = invocation.invoke();
        int status = response.getStatus();
        System.out.println("status = " + status);
        if (status == Response.Status.OK.getStatusCode()) {

            lastModified = response.getHeaderString(
                               HttpHeaders.LAST_MODIFIED);
            System.out.println("lastModified = " + lastModified);

            return response.readEntity(DataResult.class);
        }
        return null;

    }


    private void marshal(OutputStream out, Object value) 
        throws JAXBException {

        Marshaller marshaller = null;

        if (value instanceof gov.nasa.gsfc.cdaweb.schema.DataResult ||
            value instanceof gov.nasa.gsfc.cdaweb.schema.TextRequest) {

            marshaller = cdasJaxbContext.createMarshaller();
        }
        else if (value instanceof org.w3._1999.xhtml.Html) {

            marshaller = xhtmlJaxbContext.createMarshaller();
        }
        else {

            System.err.println("Don't know how to marshall " +
                value.getClass().getName());
        }
        marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT,
                               Boolean.TRUE);

        marshaller.marshal(value, out);
    }


    /**
     * Instantiates an instance of this class and then invokes the
     * {@link #execute(String)} method.
     *
     * @param args command line arguments.  
     *            args[0] is the base URL of CDAS Web services
     * @throws Exception if an Exception occurs
     */
    public static void main(String[] args) 
        throws Exception {

        if (args.length < 1) {

            System.out.println("ERROR: missing argument(s)");
            System.out.println("USAGE: JaxRsExample WS-URL");
            return;
        }
        JaxRsExample example = new JaxRsExample(args[0]);

        example.execute();
        // do it again to test If-Modified-Since/Last-Modified
        example.execute();
    }
}
